(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             genterm.ml                                   *)
(****************************************************************************)
#open "std";;
#open "univ";;
#open "initial";;
#open "extraction";;
#open "initial";;
#open "term";;
#open "termfw";;
#open "more_util";;

type param =
    CONTENTS of contents
  | INT of int
  | NAME of name
;;

let print_param = function
    (CONTENTS Null) -> [< '"Null" >]
  | (CONTENTS Pos) -> [< '"Pos" >]
  | (CONTENTS Data) -> [< '"Data" >]
  | (INT n) -> [< 'string_of_int n >]
  | (NAME (Name id)) -> [< 'string_of_id id >]
  | (NAME (Anonymous)) -> [< '"<>" >]
;;

let parse_param = function
    [< '`N` ; '`u` ; '`l` ; '`l` >] -> (CONTENTS Null)
  | [< '`P` ; '`o` ; '`s` >] -> (CONTENTS Pos)
  | [< '`D` ; '`a` ; '`t` ; '`a` >] -> (CONTENTS Data)
  | [< number n >] -> (INT n)
  | [< p_atom s >] -> NAME (Name (id_of_string s))
  | [< '`<` ; '`>` >] -> NAME(Anonymous)
;;  

let injection = function
    (CONTENTS _) -> "CONTENTS"
  | (INT _) -> "INT"
  | (NAME _) -> "NAME"
;;

type 'a genterm =
    VAR of identifier
  | OP of string * 'a list * 'a genterm list
  | SLAM of identifier option list * 'a genterm
;;

let isVAR = function
    (VAR _) -> true
  | _ -> false
;;

let get_new_id(IDENT(v,n),others) =
    let matches = filter (function IDENT(s,_) -> (s=v)) others in
        if matches = [] then IDENT(v,0)
        else IDENT(v,1+(max n (maxNEList (map (function IDENT(_,n)->n) matches))))
;;

   
let rec drop = function
    [] -> []
  | (NONE::t) -> drop t
  | ((SOME id)::t) -> id::(drop t)
;;


let rec make_iterated_ap = function
    (m,[]) -> m
  | (m,h::t) -> make_iterated_ap(OP("APP",[],[m;h]),t)
;;  

let rec genify_rec = function avoid -> function env -> function
    Meta(l) -> OP("META",[],[])
  | Var(Decl(Name s,_,_)) -> let id = s in (VAR id)
  | Rel(n) -> 
    (match nth_from_one(n,env) with
     SOME id -> VAR id
   | NULL ->anomaly "genify was fed a term with an anonymous bound var which subsequently appeared in the body of term.")
  | Prop(c) -> OP("PROP",[CONTENTS c],[])
  | Type(c,u) -> OP("TYPE",[CONTENTS c],[])
  | App(l,r) -> OP("APP",[],[genify_rec avoid env l;genify_rec avoid env r])
  | Lambda(Name s,ty,body) ->
    let bid = get_new_id(s,avoid) in
        OP("LAMBDA",[],[genify_rec avoid env ty;
                         SLAM([SOME bid],genify_rec (bid::avoid)
                                                    ((SOME bid)::env)
                                                    body)])
  | Lambda(Anonymous,ty,body) ->
    OP("LAMBDA",[],[genify_rec avoid env ty;
                     SLAM([NONE],genify_rec avoid (NONE::env) body)])
  | Prod(Name s,dom,rang) ->
    let bid = get_new_id(s,avoid) in
        OP("PROD",[],[genify_rec avoid env dom;
                       SLAM([SOME bid],genify_rec (bid::avoid) ((SOME bid)::env) rang)])
  | Prod(Anonymous,dom,rang) ->
    OP("PROD",[],[genify_rec avoid env dom;
                   SLAM([NONE],genify_rec avoid (NONE::env) rang)])
  | Ind(stamp,a,l,_,_) ->
    let bid = get_new_id(id_of_string"X",avoid) in
    let genify_cons_type c =
        SLAM([SOME bid],genify_rec (bid::avoid) ((SOME bid)::env) c)
    in
        OP("IND",[NAME stamp],[genify_rec avoid env a;
                     SLAM([SOME bid],
                          OP("CONSTRUCTORS",[],
                             map (genify_rec (bid::avoid) ((SOME bid)::env)) l))])
  | Construct(n,c) -> OP("CONSTR",[INT n],[genify_rec avoid env c])
  | Rec(_,P::cl,c) ->
    let M = OP("ELIM",[],[genify_rec avoid env c;
                          genify_rec avoid env P]) in
    let branches = map (genify_rec avoid env) cl in
        make_iterated_ap(M,branches)
  | Implicit -> OP("IMPLICIT",[],[])
  | Const(Def(Name s,_,_)) -> let id = s in VAR id
;;

let genify t = genify_rec (globals t) [] t
;;


let lifted_iso(bl1,bl2) =
    try (for_all
      (fun (NONE,NONE) -> true | (SOME id,SOME id') -> true)
      (combine(bl1,bl2)))
    with _ -> false
;;

let pure_index(a,l) =
    try index a l with _ -> -1
;;

let alpha_eq(t1,t2) = aux([],t1,[],t2)
    where rec aux = function
    (bl1,OP(op1,pl1,tl1),bl2,OP(op2,pl2,tl2)) ->
    (op1 = op2) & (pl1 = pl2) & auxlists(bl1,tl1,bl2,tl2)
  | (bl1,VAR id1,bl2,VAR id2) ->
    if (pure_index(id1,bl1) = -1) then (id1=id2)
    else pure_index(id1,bl1) = pure_index(id2,bl2)
  | (bl1,SLAM(idl1,body1),bl2,SLAM(idl2,body2)) ->
    (length idl1) = (length idl2)
    & lifted_iso(idl1,idl2)
    & aux(revc(drop idl1,bl1),body1,
          revc(drop idl2,bl2),body2)

and auxlists = function
    (bl1,h1::t1,bl2,h2::t2) ->
    aux(bl1,h1,bl2,h2) & auxlists(bl1,t1,bl2,t2)
  | (_,[],_,[]) -> true
  | (_,_,_,_) -> false
;;

let fold curriedf l base = list_it (fun a -> fun b -> curriedf(a,b)) l base
;;

let free_vars t = aux(t,[]) where rec aux = function
  (VAR id,accum) -> listset__add accum id
| (OP(_,_,tl),accum) -> fold aux tl accum
| (SLAM(ids,body),accum) ->
        (listset__diff (aux(body,[])) (drop ids))@accum
;;

let lower_id id others =
    get_new_id (id_of_string (stringpart_of_id id),others)
;;
let lower_id_option others = fun
    NONE -> NONE
 | (SOME id) -> SOME(lower_id id others)
;;

let lower_indices = lower [] where rec lower = function env -> function
    (VAR id) -> (try VAR(assoc id env) with _ -> (VAR id))
  | (OP(opn,pl,tl)) -> OP(opn,pl,map (lower env) tl)
  | (SLAM(ids,body) as t) ->
    let fv = free_vars t in
    let new_ids = map (lower_id_option fv) ids in
        SLAM(new_ids,lower ((combine(drop ids,drop new_ids))@env) body)
;;
