(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             fwtofml.ml                                   *)
(****************************************************************************)

#open "std";;
#open "fmlterm";;
#open "initial";;
#open "termfw";;
#open "printer";;
#open "fwfomatch";;

let vardepth = ref 0;;

let initvar () = (vardepth:=0);;

let varname () = vardepth:=!vardepth+1;
                 IDENT("VAR", !vardepth);;

let funcname () = vardepth:=!vardepth+1;
                 IDENT("REC", !vardepth);;

let opvarname = fun 
      (Fconst(Name(s),_))->s
    | (Fvar(Name(s),_))->s
    | _ -> varname();;

let rec iskind = function Fomega       -> true
                        | Fprod(_,_,m) -> iskind m
                        | Fconst(_,c)  -> iskind c
                        | _            -> false;;

let rec propenv l = propl
   where rec propl = function
        Fvar(_,K)      -> iskind K
      | Fconst(_,c)    -> propl c
      | Frel(n)        -> iskind (nth l n)
      | Fapp(c1,_)     -> propl c1
      | Flambda(_,t,c) -> propenv (t::l) c
      | Fprod(_,t,c)   -> propenv (t::l) c
      | Find(_,_,_,_,_)  -> true
      | _              -> false;;
let isprop = propenv [];;

exception Notind;;

let rec find_ind = function
    Flambda(_,_,c) -> find_ind c
  | Find(_,_,l,_,_)  -> l
  | Fconst(_,t)    -> find_ind t
  | _              -> raise Notind;;

let rec hdind = function
      Find(_,_,_,_,_) -> true
    | Fapp(c,_) -> hdind c
    | Flambda(_,_,c)->hdind c
    | _ -> false;;

let rec specif_ind = function
                 Fapp(c1,_)        -> specif_ind c1
               | Fconst(Name(s),c) -> if hdind c then s else specif_ind c
               | Flambda(_,_,c) -> specif_ind c
               | _                 -> raise Notind;;

let isind c = try find_ind c; true with Notind -> false;;

let rec find_ind_weak = function
    Flambda(_,_,c) -> find_ind c
  | Find(_,_,l,_,_)    -> l
  | _              -> raise Notind;;

let isind_weak c = try find_ind_weak c; true with Notind -> false;;


let nameind c = try  (specif_ind c)
                with Notind -> anomaly "should be an inductive type 2";;

let nbconstr c = try length(find_ind c)
                 with Notind -> anomaly "should be an inductive type 3";;

let rec isconstr = function Fconst(_,c)   -> isconstr c
                          | Flambda(_,_,c)-> isconstr c
                          | Fapp(c,_)     -> isconstr c
                          | Fconstr(_,c)  -> true
                          | _             -> false;;

(* exception NotTranslatable;; *)
let error_trad () = raise NotTranslatable;;


let TYPTOSUBST = ref ([("NUM",FmConst("Int"));
                       ("BOOL",FmConst("Bool"))]:(string * fmltype) list);;
let typtosubst s f =  TYPTOSUBST := (s,f)::!TYPTOSUBST;;

let fwtofmlvartype s = try (assoc s !TYPTOSUBST) with Not_found -> (FmError s);;


let fwtofmlabbr = trad [] 
where rec trad l =
        function Fprod(_,t,c) -> if iskind t
                                 then (
(* print_string "WARNING"; print_newline(); *)
                                            trad (""::l) c)
                                 else FmArr(trad l t, trad (""::l) c)
               | Frel m       -> FmVar (nth l m)
               | Flambda(Name(s),t,c) ->
                 let sid = string_of_id s in
                     FmLambda(sid,trad (sid::l) c)
               | Fconst(Name(s),T)  -> FmConst (string_of_id s)
               | Fapp(c1,c2)  -> if iskind c2 then trad l c1 else
                                 FmApp(trad l c1, trad l c2)
               | Fvar(Name s,_)    -> fwtofmlvartype (string_of_id s)
               | _            -> error_trad();;


let fwtofmla l = trad
   where rec trad n =
        function Fprod(_,t,c) -> if iskind t then trad (n+1) c
                                 else FmArr(trad n t,trad (n+1) c)
               | Frel m       -> if m=(n+1) then match l with
                         [] -> anomaly "fwtofmla"
                       | c::m -> it_list (fun x y -> FmApp(x,y)) (FmConst c) 
                                         (map (fun x -> FmVar x) (rev m))
                                 else if m>(n+1) then FmVar (nth l (m-n))
                                 else error_trad()
               | Fconst(Name(s),T)  -> FmConst (string_of_id s)
               | Fapp(c1,c2)  -> if iskind c2 then trad n c1 else
                                 FmApp(trad n c1, trad n c2)
               | Fvar(Name s,T)    -> FmConst (string_of_id s)
(*               |  Flambda(Name(s),t,c) -> FmLambda(n,trad (n::l) c) *)
               | _            -> error_trad();;

let rec fwtofmltype l =
        function Flambda(Name(s),_,c) ->
        let sid = string_of_id s in
            FmLambda(sid,fwtofmltype (sid::l) c)
               | x                    -> fwtofmla l 0 x;;


let fwtofmlc l = trad 0 []
  where rec trad n acc =
        function Fprod(_,t,c) -> if iskind t then trad (n+1) acc c
                                 else trad (n+1) ((fwtofmla l n t)::acc) c
               | Frel m       -> rev acc
               | _            -> error_trad();;


let fwtofmlind name namec = trad []
   where rec trad l = function
     Flambda(Name(n),_,c) ->
     let nid = string_of_id n in
         FmLambda(nid,trad (nid::l) c)
   | Fapp(c1,c2)          -> if iskind c2 then trad l c1
                             else FmApp(trad l c1, fwtofmltype l c2)
   | Find(_,_,lc,_,_)          -> FmInd(name,
                                    map2 (fun n c -> n,fwtofmlc (name::l) c)
                                         namec lc)
   | _                    -> error_trad();;

let name_var n = "Var"^(string_of_int n);;

let names_var n = map name_var (range n);;

exception Occur;;

let occurvar name C =
 let rec frec = function
    FmApp(c1,c2)  -> frec c1; frec c2
   |FmLambda(_,c) -> frec c
   |FmArr(c1,c2)  -> frec c1; frec c2
   |FmConst(s)      -> if s=name then raise Occur
   |_             -> ()
 in try frec C;false with Occur -> true;;


let fmltyptransfa name A = (if occurvar name A then Reca (trans Fmarg 0 A)
                         else Noreca)
    where rec trans a p = function
        FmArr(_,B)-> let x = name_var p in
                     Fmlambda(x,trans (Fmapp(a,Fmvar x)) (p+1) B)
      | FmConst(n)  -> Fmapp(Fmrec,a)
      | FmApp(B,_) -> trans a p B
      |  _         -> error_trad () ;;

let Fmapplist = it_list (fun x y -> Fmapp(x,y));;

let Fmlambdalist c l = it_list (fun c s -> Fmlambda(s,c)) c (rev l);;


let TOSUBST = ref ([("TRUE",Fmext True);
                     ("FALSE",Fmext False);
                     ("PLUS",Fmext Plus);
                     ("EQNUM",Fmext Equal);
                     ("LESSER",Fmext Lesser);
(*
                     ("NUM_recs",Fmext int_elim);
                     ("NUM_recd",Fmext int_elim);
*)
                     ("IF",Fmext If);
                     ("eq_rec",Fmlambda("a",
                                Fmlambda("h",
                                Fmlambda("b",Fmvar"h"))));
                    ("Acc_rec",Fmlambda("f",
                                Fmlambda("x",
                                Fmlocal("F",
                                FmRec(Fmlambda("y",
                                  Fmapp(Fmapp(Fmvar"f",Fmvar"y"),Fmvar"F"))),
                                  Fmapp(Fmvar"F",Fmvar("x"))))));
                    ("SUCC",Fmlambda("x",Fmapp(Fmapp(Fmext Plus,Fmext(Num 1)),
                                                     Fmvar"x")))
                            ]:(string * fmlterm) list);;
let tosubst s f =  TOSUBST := (s,f)::!TOSUBST;;

let make_fmconstr s i (n,l) = let abs_var = names_var (length l) in
    n, Fmlambdalist (Fmconstruct(i,s,map (fun x -> Fmvar x) abs_var)) abs_var;;

let rec makefmind = function
  FmLambda(_,c) ->  makefmind c
| FmInd(name,l) ->  (name,
                    map (function n,la -> n,map (fmltyptransfa name) la) l)
| FmApp(c,_)    ->  makefmind c
| _             ->  anomaly "should be an inductive type"
;;

let add_fmind f = Fmindenv:= f::!Fmindenv;;

let find_fmind s = assoc s !Fmindenv;;
let reset_fmind () = Fmindenv:=[];;

let constrarity i s = length (snd (nth (find_fmind s) i));;
let fwtofmlvar s = try (assoc s !TOSUBST) with Not_found -> Fmerror;;

let genvarlist n = 
    let rec genvarl = function
              0 -> [] 
            | n ->(varname())::(genvarl (n-1))
 in genvarl n;;


(* let fwtofmlterm M = trad (fglobals M) [] [] M  where *)

let rec trad ln lt largs =
 let apply = function c -> it_list (fun x y -> Fmapp(x,y)) c largs in
 function
    Flambda(Name(s),t,c) -> if iskind t then trad (s::ln) (t::lt) largs c
                              else let s' = next_ident_away s ln in
                                   let s'id = string_of_id s' in
                               apply (Fmlambda(s'id,trad (s'::ln) (t::lt) [] c))
 |  Fapp(c,c')           -> if propenv lt c' then trad ln lt largs c
                            else trad ln lt ((trad ln lt [] c')::largs) c
 |  Frel(m)              -> apply (Fmvar(string_of_id(nth ln m)))
 |  Fconst(Name(s),_)    -> apply (Fmconst (string_of_id s))
 |  Fvar(Name(s),_)      -> apply (Fmconst (string_of_id s))
 |  Frec(l,t)            -> let T = ftype_term lt t in
                            let s = nameind T in
                            let ((_,_,ctl,Fspecifr(_,rl),_),_)=
                                                    (find_finductype T) in
                            let NA = funcname() in
                            let rec build_match n = function 
                                  [] -> [] 
                                | (ct::l) -> let a = farity (nth ctl n)
                                                            (T::lt)
                                             in
                                             let lv = genvarlist a in
                                              (n,
                                               map string_of_id lv,
                                               if mem_assoc n rl
                                             then
                                                it_list
                                                 (fun x y -> Fmapp(x,y))
                                                 (Fmapp
                                                   (Fmapp(
                                                     trad ln lt []
                                                        (assoc n rl)
                                                      ,ct),
                                                    Fmvar (string_of_id NA)))
                                                 (map (fun x -> Fmvar (string_of_id x)) lv)
                                             else 
                                                 it_list
                                                    (fun x y -> Fmapp(x,y))
                                                    ct
                                                    (map (fun x -> Fmvar (string_of_id x)) lv)
                                              )::(build_match (n+1) l)
                                in if rl = [] then (* non-recursive case *)
                                   Fmmatch (trad ln lt [] t,string_of_id s,
                                            build_match 1
                                            (map (trad ln lt []) (tl l)))
                                   else let VN = varname() in
                                    Fmlocal(string_of_id NA,
                                            FmRec
                                             (Fmlambda
                                               (string_of_id VN,
                                                Fmmatch
                                                 (Fmvar (string_of_id VN),
                                                   string_of_id s,
                                                   build_match 1
                                                    (map
                                                      (trad ln lt [])
                                                      (tl l))))),
                                           Fmapp(Fmvar (string_of_id NA),trad ln lt [] t))
 |  Fconstr(i,c)         -> let s = nameind c in
                            let n = (*constrarity i s*) 
                              (let ((_,_,ctl,Fspecifr(_,rl),_),_)=
                                            (find_finductype c) in
                               farity (nth ctl i) (c::lt))
                              in
                            if n > (length largs) then
                            let abs_var = names_var (n-(length largs))
                            in Fmlambdalist 
                                (Fmconstruct(i,string_of_id s,largs@ map (fun x -> Fmvar x) abs_var))
                               abs_var
                            else let (l,m) = chop_list n largs in
                              Fmapplist (Fmconstruct(i,string_of_id s,l)) m

 |  x                    -> fprterm x; anomaly "should be a proof";;

let fwtofmlterm M = let M' = refold_up_indtypes M in
    initvar (); trad (fglobals M') [] [] M';;
