(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                              *)
(*                            Projet Coq                    *)
(*                                              *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                    Lyon                       *)
(*                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                            fwfomatch.ml                                  *)
(****************************************************************************)

(* The F-omega terms for extraction of programs *)

#infix "o";;
#open "std";;
#open "pp";;
#open "stdpp";;
#open "initial";;
#open "univ";;
#open "extraction";;
#open "term";;
#open "printer";;
#open "machine";;
#open "termfw";;

(*let e s = construct (parse_fterm s);PP(print_val());let (Inf(_,ft)) = read_inf() in ppfterm ft;ft;;*)

(* fclosedto n M == true if and only if there is an
 * occurrence of Ref i, i in [1..n], in M.
 *)
let succp (i,j) = (succ i,succ j);;
exception Occur;;
let fclosedto n term = 
  let rec occur_rec((i,n) as p) = function
    Frel(m)         -> if i<=m & m<=n then raise Occur
  | Fapp(c,c')      -> occur_rec p c; occur_rec p c'
  | Flambda(_,c,c') -> occur_rec p c; occur_rec (succp p) c'
  | Fprod(_,c,c')   -> occur_rec p c; occur_rec (succp p) c'
  | Find(_,c,l,_,_)   -> occur_rec p c; do_list (occur_rec (succp p)) l
  | Fconstr(_,c)    -> occur_rec p c
  | Frec(l,c)       -> occur_rec p c; do_list (occur_rec p) l
  | _               -> ()
  in try (occur_rec (1,n) term; true) 
     with Occur -> false;;

(* alpha conversion : ignore print names *)
let rec eq_fomega pair = if eq(pair) then true else
    match pair with
     (Flambda(_,t1,c1),Flambda(_,t2,c2)) -> eq_fomega(t1,t2) & eq_fomega(c1,c2)
   | (Fprod(_,t1,c1),Fprod(_,t2,c2))     -> eq_fomega(t1,t2) & eq_fomega(c1,c2)
   | (Fapp(t1,u1),Fapp(t2,u2))           -> eq_fomega(t1,t2) & eq_fomega(u1,u2)
   | (Frel p1,Frel p2)                   -> p1=p2
   | (Fvar(na1,t1),Fvar(na2,t2))                   -> eq(na1,na2) & eq_fomega(t1,t2)
   | (Fconst p1,Fconst p2)               -> eq(p1,p2)
   | (Find(stamp1,c1,l1,_,_),Find(stamp2,c2,l2,_,_)) -> stamp1 = stamp2 &
                                      eq_fomega(c1,c2) &
                                      for_all2eq (curry eq_fomega) l1 l2
   | (Fconstr(i,c1),Fconstr(j,c2)) -> (i=j) & eq_fomega(c1,c2)
   | (Frec(l1,c1),Frec(l2,c2))       -> eq_fomega(c1,c2) &
                                          for_all2eq (curry eq_fomega) l1 l2
   | _                                 -> false;;


(* Input: two terms:
 * The pattern: P, with exactly [1..n] free references.
 * the term:    Q, with 1..m free references
 *
 * The goal: Is there an instanciation of P with n terms T1..Tn, each
 *           containing free references from [1..m], such that
 *           P[T1..Tn] =alpha Q ?
 *
 * The references which are locally bound are always 1..d.
 * Thus, for the Motif, the references which are bound to
 * pattern variables are ... [d+1...d+nvars]
 *
 * If we have rel(m), Pat, at a depth of D, and m not in [1..D],
 * then m is a free reference.  We can bind it to lift(-d,Pat), if
 * (1) Pat does not contain any current binding at depths [1..d]
 * (2) any current binding is alpha-convertible with lift(-d,Pat)
 *)

let stamp_geq = fun
    _ Anonymous -> true
  | (Name s1) (Name s2) -> s1 = s2
  | _ _ -> false
;;

let fwfomatch nvars Motif Expr =
    fomatch_rec(Motif,0,Expr,[]) where rec fomatch_rec = function
(Fapp(c1,c1'),d,
 Fapp(c2,c2'),sigma) -> fomatch_rec(c1,d,c2,
                        fomatch_rec(c1',d,c2',sigma))
|(Flambda(_,c1,c1'),d,
  Flambda(_,c2,c2'),sigma) -> fomatch_rec(c1',d+1,c2',
                             fomatch_rec(c1,d,c2,sigma))
|(Frel m,d,Expr,sigma) ->
    if 1<=m & m<=d then
        if (Frel m) = Expr then sigma else failwith "fomatch: conflict"
    else if d+1<=m & m<=nvars+d & fclosedto d Expr then
        if listmap__in_dom sigma (m-d) then
            if eq_fomega(flift (-d) Expr,listmap__map sigma (m-d)) then sigma
            else failwith "fomatch: conflict"
        else listmap__add sigma (m-d,flift (-d) Expr)
    else failwith "fomatch: dangling reference"

|((Fvar _) as Motif,_,Expr,sigma) ->
 if eq_fomega(Motif,Expr) then sigma else failwith "fomatch: conflict"

|((Fconst _) as Motif,_,Expr,sigma) ->
 if eq_fomega(Motif,Expr) then sigma else failwith "fomatch: conflict"

|(Fomega,_,Fomega,sigma) -> sigma

|(Fprod(_,c1,c1'),d,
  Fprod(_,c2,c2'),sigma) -> fomatch_rec(c1',d+1,c2',
                            fomatch_rec(c1,d,c2,sigma))

|(Find(stamp1,t1,tl1,_,_),d,
  Find(stamp2,t2,tl2,_,_),sigma) ->
  if stamp_geq stamp1 stamp2 then
      fomatch_rec(t1,d,t2,
      fomatch_list(tl1,succ d,tl2,sigma))
  else failwith "fomatch: conflict"

|(Fconstr(n1,t1),d,Fconstr(n2,t2),sigma) -> if n1 = n2 then
                                                fomatch_rec(t1,d,t2,sigma)
                                            else failwith "fomatch: conflict"

|(Frec(l1,t1),d,Frec(l2,t2),sigma) -> fomatch_rec(t1,d,t2,
                                      fomatch_list(l1,d,l2,sigma))

| _ -> failwith "fomatch: conflict"

and fomatch_list = function
 ([],d,[],sigma) -> sigma
|(h1::tl1,d,h2::tl2,sigma) -> fomatch_list(tl1,d,tl2,
                              fomatch_rec(h1,d,h2,sigma))
| _ -> failwith "fomatch: conflict"
;;

let rec fomatch_rec = fun
nvars (Fapp(c1,c1'),d,
 Fapp(c2,c2'),sigma) -> fomatch_rec nvars (c1,d,c2,
                        fomatch_rec nvars (c1',d,c2',sigma))
| nvars (Flambda(_,c1,c1'),d,
  Flambda(_,c2,c2'),sigma) -> fomatch_rec nvars (c1',d+1,c2',
                             fomatch_rec nvars (c1,d,c2,sigma))
| nvars (Frel m,d,Expr,sigma) ->
    if 1<=m & m<=d then
        if (Frel m) = Expr then sigma else failwith "fomatch: conflict"
    else if d+1<=m & m<=nvars+d & fclosedto d Expr then
        if listmap__in_dom sigma (m-d) then
            if eq_fomega(flift (-d) Expr,listmap__map sigma (m-d)) then sigma
            else failwith "fomatch: conflict"
        else listmap__add sigma (m-d,flift (-d) Expr)
    else failwith "fomatch: dangling reference"

| nvars ((Fvar _) as Motif,_,Expr,sigma) ->
 if eq_fomega(Motif,Expr) then sigma else failwith "fomatch: conflict"

| nvars ((Fconst _) as Motif,_,Expr,sigma) ->
 if eq_fomega(Motif,Expr) then sigma else failwith "fomatch: conflict"

| nvars (Fomega,_,Fomega,sigma) -> sigma

| nvars (Fprod(_,c1,c1'),d,
  Fprod(_,c2,c2'),sigma) -> fomatch_rec nvars (c1',d+1,c2',
                            fomatch_rec nvars (c1,d,c2,sigma))

| nvars (Find(stamp1,t1,tl1,_,_),d,
  Find(stamp2,t2,tl2,_,_),sigma) ->
  if stamp1 = stamp2 then
      fomatch_rec nvars (t1,d,t2,
      fomatch_list nvars(tl1,succ d,tl2,sigma))
  else failwith "fomatch: conflict"

| nvars (Fconstr(n1,t1),d,Fconstr(n2,t2),sigma) -> if n1 = n2 then
                                                fomatch_rec nvars (t1,d,t2,sigma)
                                            else failwith "fomatch: conflict"

| nvars (Frec(l1,t1),d,Frec(l2,t2),sigma) -> fomatch_rec nvars (t1,d,t2,
                                      fomatch_list nvars(l1,d,l2,sigma))

| _ _ -> failwith "fomatch: conflict"

and fomatch_list = fun
nvars ([],d,[],sigma) -> sigma
| nvars (h1::tl1,d,h2::tl2,sigma) -> fomatch_list nvars(tl1,d,tl2,
                              fomatch_rec nvars (h1,d,h2,sigma))
| _ _ -> failwith "fomatch: conflict"
;;

let fwfomatch nvars Motif Expr = fomatch_rec nvars (Motif,0,Expr,[]);;

(* for each name, typenv gives the number of parameters,
 * the pattern which must be matched against, and the original fterm.
 *)
let (typenv: (string * ((int * fterm) * fterm)) list ref) = ref [];;

let reset_indtypes () = (typenv := []);;

let motif_ize fty = aux(0,fty) where rec aux = function
    (n,Flambda(_,ty,body)) -> aux(succ n,body)
  | (n,(Find _) as rslt) -> rslt
;;

let try_find f = try_find_f
 where rec try_find_f = function
     [] -> failwith "try_find"
  | h::t -> (try f h with Failure _ -> try_find_f t)
;;

let find_named_ty fty =
    let lookat (s,((len,motif),ftyconst)) = (s,fwfomatch len motif fty,len,ftyconst) in
    let (s,sigma,len,ftyconst) = try_find lookat (!typenv) in
        fapplist ftyconst (map (listmap__map sigma) (range len))
;;


exception Notind;;
let rec find_ind = function
    Flambda(_,_,c) -> find_ind c
  | Find(_,_,l,_,_)  -> l
  | Fconst(_,t)    -> find_ind t
  | _              -> raise Notind;;
let isind c = try find_ind c; true with Notind -> false;;exception Notind;;
let is_indconst (Fconst(_,ft)) = isind ft;;

let rec is_indtyexp = function
    Flambda(_,_,c) -> is_indtyexp c
  | Find _ -> true
  | _ -> false
;;

let (rename_env : (identifier * fterm) list ref) = ref [];;
let reset_renamings () = (rename_env := []);;

let add_renaming(id,ft) =
    (if listmap__in_dom (!rename_env) id then
         error ("tried to add a renaming of "
                  ^ (string_of_id id) ^ " twice - please do a Reset All in Fml."));
    rename_env := (listmap__add (!rename_env) (id,ft))
;;

let lookup_renaming id =
    listmap__map (!rename_env) id
;;

let refold_indtypes fatally = refold where rec refold = function
    (Fapp(t1,t2)) -> Fapp(refold t1,refold t2)
  | (Flambda(n,a,b)) -> Flambda(n,refold a,refold b)
  | (Fprod(n,a,b)) -> Fprod(n,refold a,refold b)
  | (Find(stamp,ft,ftl,fr,ftl')) ->
    let ft = Find(stamp,refold ft,map refold ftl,refold_sr fr,map refold ftl') in
     (try refold(find_named_ty ft)
      with Failure _ ->
          if fatally then (PPNL [< 'S"System error: cannot divine how ";
                                 'S"to refold the type ";
                                 fterm0 [] ft >];
                           error "Fml: Are you sure you gave a top-level inductive definition for this type?")
          else ft)
  | (Fconstr(n,ft)) -> Fconstr(n,refold ft)
  | (Fconst(Name id,_) as fc) -> (try lookup_renaming id
                                 with Not_found -> fc)
  | (Frec(ftl,ft)) -> Frec(map refold ftl,refold ft)
  | (Fvar(na,ft)) -> (Fvar(na,refold ft))
  | ft -> ft
and refold_sr (Fspecifr(n,l)) =
    Fspecifr(n,map (function (i,t) -> (i,refold t)) l)
;;
let refold_up_indtypes = refold_indtypes true;;
let try_refold_up_indtypes = refold_indtypes false;;

let process_type (Fconst(Name id,fty)) =
let fty' = try_refold_up_indtypes fty in
let ftyconst = Fconst(Name id,fty') in
    add_renaming(id,ftyconst);
    (if is_indtyexp fty' then
         let s = string_of_id id in
         (if listmap__in_dom (!typenv) s then
              error ("adding inductive type " ^ s ^ " for a second time - please do a Reset All in Fml."));
         typenv := listmap__add (!typenv) (s,(motif_ize fty',ftyconst))
     else if not(is_silent()) then
         PPNL [< 'S"Replacing type " ; print_id id ; 'S" by " ;
                  fterm0 [] fty' >]);
    ftyconst
;;

let process_term (Fconst(Name id,ft)) =
    let fc = Fconst(Name id,refold_up_indtypes ft)
    in add_renaming(id,fc);fc
;;
