(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                            extraction.ml                                 *)
(****************************************************************************)

#open "initial";;
#open "univ";;
#open "termfw";;

(* The constructive engine can be used for three purposes :
    - checking that some term is typed in Fomega 
                (a subsystem of the Calculus of Constructions)
    - checking that a "logical" proof is well-formed.
    - writing a "constructive" proof and extracting its computational meaning
                (an Fomega program)
Each term c of the Calculus of Construction contains a computational 
information that is :
    Inf(Fw,c')        if c is already typed in Fw and if c' is its translation 
                      as an fterm.
    Logic             if c is a "logical" term (without computational meaning).
    Inf(Extracted,c') if c is a constructive proof and c' the extracted fterm.

The information is initialized at the level of propositional variables :
    A : Prop(Data)   A is a type variable of Fomega.
    A : Prop(Null)   A is a logical propositional variable.
    A : Prop(Spec)   A is a constructive propositional variable.

This file contains the rule for computing the information during an 
application, generalization or abstraction.

When processing a term c equal to  (M N) or (x:N)M or [x:N]M,
three cases are distinguished :
  If M is a logical term then c is also a logical term.
  If M is in Fw then N must be in Fw and the operation must be valid 
            in Fw (there are restriction on the level of M and N).
  If M is a constructive term then depending of the level of N and of its
            information, the information of c will be the same as the 
            information of M or computed using the fterm extracted from N.

If M is an informative term of level lev and nature n, let if lev' is the 
level of N, we associate to the information of N three possibilities :
    
    Error : The operation is illegal in Fomega.
    Invar : The information of c is equal to the information of M.
    Comp(n') : The information of c must be computed using the fterm n' 
            coming from the information of N.
*)
(* type contents = Pos | Null | Data;; *)

(* type nature = Fw | Extracted;; *)
(* type information = Logic | Inf of nature * fterm;; *)

let compatible nat1 nat2 = (nat1 = Extracted) or (nat2 = Fw);;

(* type compatibility = Comp of fterm | Error | Invar;; *)

let compat lev n lev' = function 
    Inf(n',c') -> if le_level lev lev' then
                          if compatible n n' then Comp(c') else Error
                  else if n = Extracted then Invar else Error
  | Logic      -> if n = Fw then Error else Invar;;

let inf_const n = function Inf(f,c) -> Inf(f,Fconst(n,c)) | x -> x;;

let inf_var n = function Inf(f,c) -> Inf(f,Fvar(n,c)) | x -> x;;


let inf_value = function Inf(f,Fconst(_,c))->Inf(f,c) | x -> x;;

let inf_vartype = function Inf(f,Fvar(_,t))->Inf(f,t) | x-> x;;

let inf_kind = function  Null -> Logic 
                       | Pos  -> Inf(Extracted,Fomega)
                       | Data -> Inf(Fw,Fomega);;

let inf_apply lev1 lev2 = fun
    Logic _ -> Logic
  | (Inf(n1,c1) as i1) inf2 ->
        match compat lev1 n1 lev2 inf2 with
            Comp(c2) -> Inf(n1,Fapp(c1,c2))
          | Invar    -> i1
          | Error    -> error "Application not valid in Fw";;

let inf_replace str i1 i2 = infrep (i1,i2)
    where infrep = function
        Inf(n1,c1),(Inf(n2,c2) as i2) ->
                if foccur_eq str c2 then Inf(n2,fsubst_norm str c1 c2)
                else i2
      | (_,i2) -> i2;;

let inf_abs_var lev varlev = fun
    Logic _ -> Logic
  | (Inf(n,c) as i) infvar ->
    match compat lev n varlev infvar with
            Comp(Fvar(name,tv)) ->
                         Inf(n,Flambda(name,tv,fsubst_var name c))
          | Comp(_)  -> anomaly"Should be a Fvariable"
          | Invar    -> i
          | Error    -> error "Abstraction not valid in Fw 1";;

(* generalize a type M. lev indicates if M is a type of proof or of object *)

let inf_generalize lev varlev = fun
    Logic _ -> Logic
  | (Inf(n,c) as i) infvar ->
    match compat lev n varlev infvar with
            Comp(Fvar(name,tv)) ->
                              Inf(n,Fprod(name,tv,fsubst_var name c))
          | Comp(_) -> anomaly"Should be a variable"
          | Invar    -> i
          | Error    -> error "Generalization not valid in Fw 1";;

let inf_gen_rel lev varlev = fun
    Logic _ -> Logic
  | (Inf(n,c) as i) infvar ->
    match compat lev n varlev infvar with
            Comp(Fvar(name,tv)) -> Inf(n,Fprod(name,tv,c))
          | Invar    -> Inf(n,fpop c)
          | Error    -> error "Generalization not valid in Fw 2"
          | _        -> anomaly "Should be a variable";;


let inf_abs_rel lev varlev = fun
    Logic _ -> Logic
  | (Inf(n,c) as i) infvar ->
    match compat lev n varlev infvar with
            Comp(Fvar(name,tv)) -> Inf(n,Flambda(name,tv,c))
          | Invar    -> Inf(n,fpop c)
          | Error    -> error "Abstraction not valid in Fw 2"
          | _        -> anomaly "Should be a variable";;
 
let inf_rel d = function Inf(n,_) -> Inf(n,Frel d) | x -> x;;
 
let inf_abs_apply lev varlev = fun
    Logic _ -> Logic, Logic
  | (Inf(n,Fconst(name,c)) as i) infvar ->
    (match compat lev n varlev infvar with
            Comp(Fvar(hypname,tv) as fvar) ->
                let c' = Fconst(name,Flambda(hypname,tv,fsubst_var hypname c))
                in Inf(n,c'),Inf(n,Fapp(c',fvar))
          | Invar    -> i,i
          | Error    -> error "Abstraction not valid in Fw 3"
          | _        -> anomaly "Should be a variable")
  | _ _ -> anomaly "inf_abs_apply";;
 
let inf_gen_apply lev varlev = fun
    Logic _ -> Logic, Logic
  | (Inf(n,Fvar(name,c)) as i) infvar ->
    (match compat lev n varlev infvar with
            Comp(Fvar(hypname,tv) as fvar) ->
                let c' = Fvar(name,Fprod(hypname,tv,fsubst_var hypname c))
                in Inf(n,c'),Inf(n,Fapp(c',fvar))
          | Invar    -> i,i
          | Error    -> error "Generalization not valid in Fw 3"
          | _        -> anomaly "Should be a variable")
  | _ _ -> anomaly "inf_abs_apply";;
    
let is_fw = function Inf(Fw,_) -> true | _ -> false;;
let is_informative = function Inf(_,_) -> true | _ -> false;;
let logic () = Logic;;

let inf_app f = function Inf (x,c) -> Inf (x,f c) | x -> x;;

(* Inductive types in Fw *)

let abstract_inf name = inf_app (function c -> fnf (fsubst_var name c));;

let extract = function Inf(_,c) -> c | _ -> anomaly "not informative";;

let inf_indtype stamp infvar inflconstr =
    inf_app (fun t -> fmake_ind stamp t (map extract inflconstr)) infvar;;

let inf_indconstr infind i = 
    inf_app (function ind -> Fconstr(i,ind)) infind;;

let inf_make_elim infind  = function
    Null -> Logic
  | Pos  -> (match infind with 
        Inf(_,i) -> Inf(Extracted,fmake_elim i)
       | _       -> error "Elimination incompatible with contents")
  | Data -> (match infind with
            Inf(_,i) -> Inf(Fw,fmake_elim i)
           | _       -> error "Elimination incompatible with contents");;

let inf_make_const str = inf_app (function c -> Fconst(Name(str),c));;

let inf_make_elimination infc = function
    Inf(n,P) -> (match infc with 
                   Inf(_,c) -> Inf(n,fmake_elimination c P)
                 | _        -> anomaly "inf_make_elimination")
  | x -> x;;

let inf_inst_elim = function
    Logic::_ -> (function t -> Logic)
 | (Inf(b,_):: _ as linf) -> (function 
                Inf(_,t) -> Inf(b,Frec(map extract linf,t))
              | _        -> anomaly "Constructor should be informative")
 | []    -> anomaly "elim should have at least one argument";;
