(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                           coqtoplevel.ml                                 *)
(****************************************************************************)

#open "pp";;
#open "stdpp";;
#open "std";;
#open "tactics";;
#open "initial";;
#open "univ";;
#open "machine";;
#open "command";;
#open "vernac";;
#open "lexer";;
#open "matching";;
#open "sys";;
#open "more_util";;
#open "verify";;

(* parse_vernac parses an expression in vernacular,
   Finishes if the expression is correct,
   raises UserError if an error occurs,
   raises Anomaly if an anomaly is detected,
   raises Drop if the expression is Drop. *)

(* exception Exit;; *)

let toplevel_print_newline() = if is_silent () then
 (print_string "."; print_flush()) else print_newline() ;;

let cnt = nref 0;;

let stream_of_channel ic =
  cnt.v <- 0;
  stream_from (fun _ -> cnt.v <- cnt.v+1; input_char ic)
;;

let print_syntax_error() =
  prerr_string "Syntax error";
  if cnt.v >= 0 then (
    prerr_string ", char ";
    prerr_int cnt.v
  );
  prerr_endline ""
;;

(* read_loop reads vernacular expressions while no error occurs.
   it raises various exceptions if an error occurs,
   and raises Drop if the last expression is Drop. *)

let buffer = nref(create_string 80);;
let nbuffer = nref (0);;

let read_and_keep_char in_chan () =
  let c = input_char in_chan in
     let len = string_length buffer.v in
     if nbuffer.v >= len then (buffer.v <- buffer.v ^ (create_string len); ());
     set_nth_char buffer.v nbuffer.v c;
     nbuffer.v <- nbuffer.v + 1;
     cnt.v <- cnt.v + 1;
     c;;

let rec discard_dots ts =
    let rec aux = function
        [< 'Tdot >] -> [< 'S"." >]
      | [< 't ; s >] -> [< 'S(string_of_token rev_vernac_keywords t) ;
                           'S" " ;
                            aux s >]
      | [< >] -> [< >]
    in
        try PPNL [< 'S"Discarding " ; aux ts >]
        with _ -> (prerr_string "Nasty syntax error in lexer! - resynchronizing.")
;;

let read_loop ts =
    while true do
    print_newline();
    set_record Delayed; (* standard option : we record in sequence *)
    nbuffer.v <- 0;
    cnt.v <- 0;
    if !transcript & refining() then
        (record_line "\n" 1;
         let vc = make_verify_command() in
             record_line vc (string_length vc));
    parse_vernac  ts;
    record_line buffer.v nbuffer.v
    done;;

let clean_buffer() = (* PREVIOUS_BUFFER := "\n" *) ();;

let cl_message s = clean_buffer(); message s;;

let loop_handler f discarder ts =  
try f ts with
  Drop                 -> clean_buffer(); raise(Drop)
| Parse_failure        -> clean_buffer(); raise Parse_failure
| End_of_file          -> clean_buffer(); raise End_of_file
| Sys_error msg        -> cl_message("OS error: " ^ msg); raise (Sys_error msg)
| UserError(s)         -> (clean_buffer();
                           match s with
                             "simpl" -> report()
                           | s       -> message("Error " ^ s))
| Parse_error          -> clean_buffer(); discarder ts; print_syntax_error()
(*
| caml_system(str,_)   -> cl_message ("Error in CAML expression : " ^ str)
*)
| Break                -> cl_message "Interrupted while parsing vernac"
| INCONSISTENCY        -> cl_message "You have reached a paradox";
                            raise Exit
| Anomaly(s)           -> cl_message("System Error " ^ s ^ ". Please report");
                            raise Exit
(*
| match_failure(s1,s2) -> cl_message "System Error match. Please report";
                            message s1; message s2; raise Exit
*)
| Undeclared           -> cl_message "System Error search. Please report";
                            raise Exit
| Failure(s)           -> cl_message ("System Error " ^ s ^ ". Please report");
                            raise Exit
| Format(s)            -> cl_message ("Formatter Error " ^ s ^ ". Please report");
                            raise Exit
| reraise              -> cl_message "Unknown exception. Please report";
                            raise(reraise);;

let main_loop in_chan = 
    let ts = token_stream vernac_keywords (stream_from (read_and_keep_char in_chan)) in
     loop_handler read_loop discard_dots ts;;

(* main_loop reads vernacular expressions while no error occurs.
   Finishes if a non-fatal error occurs,
   raises Exit if a fatal error occurs,
   raises Drop if the last expression is Drop,
   raises io_failure _ if such an error is encountered *)

let constr_loop () = while true do (main_loop std_in) done;; 

(* constr_loop reads vernacular expressions while no fatal error occurs.
   Raises Exit if a fatal error occurs,
   raises Drop if the last expression is Drop.,
   raises io_failure _ if such an error is encountered *)

let coq () = set_prompt "Coq < ";
             try constr_loop ()
             with (End_of_file|Parse_failure) -> sys__exit 0 | _ -> set_prompt "ml #";();;

(* coq has to be used in interactive mode. Finishes if a fatal error occurs, 
   if Drop. is encountered, if EOF is encountered. Prints an error message 
   and gives a new prompt "Coq < " if a non-fatal error occurs. *)

let filename pathname =
  let len = length_string pathname
  in slash_rec 0  where rec
     slash_rec n = match succ(scan_string pathname "/" n) with
          0 -> sub_string pathname n len
        | p -> slash_rec p;;

let read_batch_loop ts =
    while true do
    toplevel_print_newline();
    parse_vernac ts
    (* Note : No recording *)
    done;;

let echo_chars_to ochan cs = [< echo_rec cs >] where rec echo_rec = function
    [< 'c ; s >] -> (output_char ochan c; [< 'c ; echo_rec s >])
  | [< >] -> [< >]
;;

let main_batch_loop in_chan =
    let ts = token_stream vernac_keywords (stream_of_channel in_chan)
    in
     loop_handler read_batch_loop discard_dots ts;;

let load_vernacular name = 
 let in_chan = open_with_suffix open_in name ".v"
     and silence = is_silent()
     in  let cleanup () = (make_silent silence;
                           try close_in in_chan with _ -> ())
         in try clean_buffer();
                make_silent true;
                main_batch_loop in_chan; 
                cleanup()
            with    Parse_failure    -> cleanup()
                  | Drop             -> cleanup()
                  | Exit             -> cleanup()
                  | reraise          -> cleanup(); raise reraise;;

let load_vernacular_from_loadpath name = 
 let in_chan = open_with_suffix_from_path open_in (search_paths()) name ".v"
     and silence = is_silent()
     in  let cleanup () = (make_silent silence;
                           try close_in in_chan with _ -> ())
         in try clean_buffer();
                make_silent true;
                main_batch_loop in_chan; 
                cleanup()
            with    Parse_failure    -> cleanup()
                  | Drop             -> cleanup()
                  | Exit             -> cleanup()
                  | reraise          -> cleanup(); raise reraise;;

let noisy_batch_loop in_chan = 
    let ts = token_stream vernac_keywords (echo_chars_to std_out (stream_from (read_and_keep_char in_chan))) in
     loop_handler read_loop discard_dots ts;;

let load_vernacular_noisily name = 
 let in_chan = open_with_suffix open_in name ".v"
     and silence = is_silent()
     in  let cleanup () = (make_silent silence;
                           try close_in in_chan with _ -> ())
         in try clean_buffer();
                make_silent false;
                noisy_batch_loop in_chan; 
                cleanup()
            with    Parse_failure    -> cleanup()
                  | Drop             -> cleanup()
                  | Exit             -> cleanup()
                  | reraise          -> cleanup(); raise reraise;;

forward_load_vernacular.v <- load_vernacular;;
forward_load_vernacular_from_loadpath.v <- load_vernacular_from_loadpath;;
forward_load_vernacular_noisily.v <- load_vernacular_noisily;;

(* load_vernacular is a vernacular loader. 
   Finishes if any error occurs.
   Finishes if Drop or EOF is encountered. *)

let go = coq and V = load_vernacular and VV = load_vernacular_noisily;; (* more user friendly *)

set_prompt "ml #";;
