(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                            compile.ml                                    *)
(****************************************************************************)

#open "std";;
#open "initial";;
#open "pretty";;
#open "fmlterm";;
#open "fmlenv";;
#open "fwtofml";;
#open "lazy";;
#open "lexer";;
#open "pp";;
#open "stdpp";;
#open "pp_control";;
#open "more_util";;
#infix "o";;

let make_caml_file = nref (forward : string -> unit);;
let optimal = nref (forward : unit -> (string * fmlterm) list);;

(***************************************************************************)
(* nFML : langage destine a l'etude et l'execution de programmes specifies *)
(*                    dans le Calcul des Constructions                     *)
(***************************************************************************)
let varindex = ref 0;;
let reserve_flag = ref true;;
(****************************************************************************)
(*                           Saisie au clavier                              *)
(****************************************************************************)

type fcobj =
      FCapp of fcobj * fcobj
    | FClambda of string * fcobj
    | FCvar of string
    | FCelim of string
    | FCext of Ext
    | FCconst of string
    | FCconstruct of int * string * (fcobj list)
    | FCconstr of string
    | FCerror
    | FCrec of fcobj
    | FClocal of string * fcobj * fcobj
    | FCmatch of fcobj * (string * (string list) * fcobj) list
    | FCexcept of fcobj * fcobj
                        ;;

let rec FCbind = fun [] -> I
                    |(x::b) -> (fun t -> (FClambda(x,(FCbind b t))));;

type fccommand =
      Affectation of string * fcobj
    | RecAffectation of string * fcobj
    | Affecttype of string * (string list) * fmltype
    | Inductive of string * fmltype
    | Typedef of string * fmltype
    | Compute of fcobj
    | Instanciate of string * fmltype
    | Realize of string * fmlterm
    | Extr_from_to of string * string
    | Extr_until of string
    | Extr_from of string
    | Extr_all 
    | Reset
    | Reset_All
    | Wait
    | Print of string
    | CAML_File of string
    | Optimise
    | free_vars
    | Save of string
    | Load of string
                        ;;

let rec clotype1 s = function
      (FmLambda(s1,t))->(FmLambda(s1,clotype1 s t))
    | (FmApp(t1,t2))->(FmApp(clotype1 s t1,clotype1 s t2))
    | (FmArr(t1,t2))->(FmArr(clotype1 s t1,clotype1 s t2))
    | (FmConst s1)->if s=s1 then (FmVar s) else (FmConst s1)
    | (FmInd(s1,cl))->FmInd(s1,map (fun (s2,tl)->
                                         (s2,(map (clotype1 s) tl)))
                                   cl)
    |x -> x
                        ;;
let rec clotype = function
      (FmLambda(s,t))->(FmLambda(s,clotype(clotype1 s t)))
    | t -> t
                        ;;

let  FCsubst s = let rec subst =
function
      (FCapp(t1,t2)) -> FCapp(subst t1,subst t2)
    | (FClambda(s1,t)) as t0 -> if s=s1 then t0 else FClambda(s1,subst t)
    | (FCvar s1) as v -> v
    | (FCelim s1) as E -> E
    | (FCext E) as e -> e
    | (FCconst s1) as c -> if s1=s then FCvar s1 else c
    | (FCconstruct(n,s1,cl)) -> FCconstruct(n,s1,map subst cl)
    | (FClocal(s1,(FCrec t1),t2)) as t -> if s=s1
                                            then t
                                            else
                                              FClocal(s1,
                                                      FCrec(subst t1),
                                                      subst t2)
    | (FClocal(s1,t1,t2)) -> if s=s1 then FClocal(s1,subst t1,t2)
                                     else FClocal(s1,subst t1,subst t2)
    | (FCmatch(t,pl)) -> FCmatch(subst t,
                                 map(fun(s,l,t1)->(s,l,subst t1)) pl)
    | FCexcept(t1,t2) -> FCexcept(subst t1,subst t2)
    | t -> t 
 in subst ;;


(***************************************************************************)
(*                                    FML -> fml                           *)
(***************************************************************************)

(* identification des variables et des constructeurs deja definis          *)

exception fml_undefined_term of string;;
exception fml_undefined_type of string;;
exception fml_no_type ;;

let fst3 (a,b,c) = a;;

let get_Fmtype pl = let rec get_Fmtype1 = fun
      [] -> raise fml_no_type
    | ((s,cl)::l) -> if not(length cl = length pl)
                         then get_Fmtype1 l
                          else let cl1 = map fst cl and pl1=(map fst3 pl)in
                             if (for_all (fun x->mem x cl1) pl1) &
                                (for_all (fun x->mem x pl1) cl1)
                              then (s,cl)
                              else get_Fmtype1 l
  in get_Fmtype1 !Fmindenv;;


let rec to_fmlterm1 l t = match t with
      (FCapp(t1,t2))->FCapp(to_fmlterm1 l t1,to_fmlterm1 l t2)
    | (FClambda(s,t))->FClambda(s,to_fmlterm1 (s::l) t)
    | (FCext E)->FCext E
    | (FCvar s) as v -> v
    | (FCconstr s) -> let rec getctr =(fun [] -> raise (fml_undefined_term
                                                   s)
                                    | (a::l) -> if (mem(s)(map fst(snd a)))
                                                then
                                                  (FCconstruct
                                               (index s (map fst(snd a)),
                                                      fst a, []))
                                                else (getctr l))
                in getctr !Fmindenv
    | (FCconst s) as c -> if (mem s l) then FCvar s else c
    | (FCconstruct(n,s,tl)) -> (FCconstruct(n,s,map (to_fmlterm1 l) tl))
    | (FCelim s)->FCelim s
    | FCerror -> FCerror
    | (FClocal(s,(FCrec t1),t2)) -> FClocal(s,FCrec (to_fmlterm1 (s::l) t1),
                                              (to_fmlterm1 (s::l) t2))
    | (FClocal(s,t1,t2)) -> FClocal(s, to_fmlterm1 l t1, to_fmlterm1 (s::l) t2)
    | (FCrec t) -> message "Warning anormal use of 'rec'";
                   FCrec (to_fmlterm1 l t)
    | (FCmatch(x,pl)) -> FCmatch(to_fmlterm1 l x,
                                 map (fun (s,vl,t)->
                                                (s,vl,to_fmlterm1 (l@vl) t))
                                     pl)
    | FCexcept(t1,t2) -> FCexcept(to_fmlterm1 l t1,to_fmlterm1 l t2)
                                            ;;



let to_fmlterm3 ob = (let rec to_fmlterm2 t  = match t  with
      (FCapp(t1,t2))->Fmapp((to_fmlterm2 t1),(to_fmlterm2 t2))
    | (FClambda(s,t))->Fmlambda(s,to_fmlterm2 t)
    | (FCvar s)->Fmvar s
    | (FCext E)->(Fmext E)
    | (FCconst s)->if (mem_assoc s !Fmenv)
                     then Fmconst s 
                     else raise(fml_undefined_term s)
    | (FCconstruct (n,IT,l))->(Fmconstruct (n,IT,map to_fmlterm2 l))
(*    | (FCelim s)->if (mem s (map fst (!Fmindenv)))
                    then Fmelim (s,[])
                    else raise (fml_undefined_type s)  *)
    | FCerror -> Fmerror
    | (FClocal(s,t1,t2)) -> (Fmlocal(s,to_fmlterm2 t1,to_fmlterm2 t2))
    | (FCrec t) -> FmRec (to_fmlterm2 t)
    | (FCmatch(t,pl)) -> let T = get_Fmtype pl in Fmmatch(to_fmlterm2 t,
                                  fst T,
                                  let L = map fst (snd T) in
                                  map (fun (s,vl,ob)->
                                        (index s L,
                                         vl,to_fmlterm2 ob))
                                      pl)
    | FCexcept(t1,t2) -> Fmexcept(to_fmlterm2 t1,to_fmlterm2 t2)
    | _ -> failwith "to_fmlterm3"
in
  varindex:=0;
  to_fmlterm2 (to_fmlterm1 [] ob)) 
                    ;;

let rec to_fmlterm6 = function
  (Fmconstruct (n,s,tel)) -> (function
      [] -> if (length (snd(nth(assoc s !Fmindenv) n)))=(length tel)
                then((Fmconstruct (n,s,map to_fmlterm4 tel)))
                else (varindex:=!varindex+1;
                     let v="x_ex"^(string_of_int !varindex)
                    in
                     Fmlambda(v,
                              to_fmlterm6
                                (Fmconstruct (n,s,(tel@[(Fmvar v)])))
                                []))
    | (t::l) -> if (length (snd(nth (assoc s !Fmindenv) n)))=(length tel)
                    then (Fmapp(to_fmlterm6 (Fmconstruct 
                                                (n,s,map to_fmlterm4 tel))
                                            (rev(tl(rev(t::l)))),
                                (last(t::l))))
                    else (to_fmlterm6
                            (Fmconstruct (n,s,(tel@[t])))
                            l))
| _ -> failwith "to_fmlterm6"
                     
(*
and  to_fmlterm7 (Fmelim (s,tel)) = function
      [] -> if (length (assoc s !Fmindenv))+1=(length tel)
                then (Fmelim (s,tel))
                else (varindex:=!varindex+1;
                     let v="x_ex"^(string_of_int !varindex)
                    in
                     Fmlambda(v,
                              to_fmlterm7
                              (Fmelim (s,(tel@[Fmvar v])))
                                []))
    |t::l -> if (length (assoc s !Fmindenv))+1=(length tel)
                then (Fmapp(to_fmlterm7 (Fmelim (s,tel)) (rev(tl(rev(t::l))))
                            ,last(t::l)))
                else (to_fmlterm7
                        (Fmelim (s,tel@[t]))
                        l)
                       
*)

and to_fmlterm4 ob = match ob with
      (Fmapp (t1,t2)) -> to_fmlterm5 ob ob []
    | (Fmconstruct (n,s,[])) -> to_fmlterm5 ob ob []
(*    | (Fmelim (s,[])) -> to_fmlterm5 ob ob []  *)
    | (Fmlambda (s,t)) -> (Fmlambda (s,to_fmlterm4 t))
    | (Fmconstruct (n,s,tl)) ->(Fmconstruct (n,s,map to_fmlterm4 tl))
(*    | (Fmelim (s,tl)) -> (Fmelim (s,map to_fmlterm4 tl)) *)
    | (Fmlocal(s,t1,t2)) -> Fmlocal(s,to_fmlterm4 t1,to_fmlterm4 t2)
    | (FmRec t) -> FmRec (to_fmlterm4 t)
    | t -> t

and to_fmlterm5 t0 t l = match t with
      (Fmapp (t1,t2)) -> to_fmlterm5 t0 t1 (t2::l)
    | (Fmconstruct (n,s,tl)) ->  to_fmlterm6 t l
(*    | (Fmelim (s,tl)) -> to_fmlterm7 t l  *)
    | _ -> t0
       ;;                

let  to_fmlterm = to_fmlterm4 o to_fmlterm3;;

(***************************************************************************)
(*                       Manipulation des termes                           *)
(***************************************************************************)

exception fmtypemismatch;;

let rec free_var = function 
      (Fmvar s) -> [s]
    | (Fmapp(t1,t2)) -> union (free_var t1) (free_var t2)
    | (Fmconstruct (n,s,l)) -> list_it union (map free_var l) []
    | (Fmlambda(s,t)) -> subtract (free_var t) [s]
    | (Fmlocal(s,FmRec t1,t2)) -> subtract(union(free_var t1)(free_var t2))[s]
    | (Fmlocal(s,t1,t2))->union(free_var t1)(subtract (free_var t2) [s])
    | (Fmmatch(t,s,pl)) -> union (free_var t)
                                 (list_it union 
                                          (map 
                                            (fun (n,l,t)->subtract
                                                            (free_var t)
                                                            l)
                                            pl)
                                           [])
    | Fmexcept(t1,t2) -> union (free_var t1)(free_var t2)
    | _ -> [];;

let new_name t =
 let V = free_var t in
    let rec nw s = let s'=s^"'"in if mem s' V then nw s' else s'
           in nw;;

let mems = fun (s:string) -> mem s;;

let rec subl s s1 = fun [] -> []
                      | (s2::l)->if s2=s then s1::l else s2::(subl s s1 l);;

let rec fmsubst s1 t = 
  let V = free_var t in
  let rec subst1 = function
      (Fmlambda(s,t1)) as t2 -> if s=s1 then t2
                                 else if mems s V 
                                    then subst1
                                           (Fmlambda(new_name t1 s,
                                                     fmsubst s
                                                         (Fmvar(new_name t1 s))
                                                         t1))
                                    else Fmlambda(s,(subst1 t1))
    | (Fmapp (t1,t2)) -> Fmapp (subst1 t1,subst1 t2)
    | Fmexcept(t1,t2) -> Fmexcept(subst1 t1,subst1 t2)
    | (Fmvar s) as t2 -> (* if mem s V then print_string ("warning  "^s); *)
                         if s=s1 then t else t2
    | (Fmconstruct (n,s,l)) -> ((Fmconstruct (n,s,(map subst1 l))))
    | (Fmconst s) as t2 -> t2
    | (Fmext E) as t1 -> t1
    | (Fmlocal(s,FmRec t1,t2)) as t3 ->if s=s1 then t3
                                         else
                                            if mem s V
                                              then let s'=new_name
                                                            (Fmapp(t1,t2))
                                                            s
                                                        in fmsubst s1 t
                                                 (Fmlocal(s',
                                                    FmRec
                                                      (fmsubst s 
                                                           (Fmvar s')
                                                           t1),
                                                    fmsubst s
                                                          (Fmvar s')
                                                           t2))
                                               else Fmlocal(s,FmRec
                                                                (subst1 t1),
                                                                subst1 t2)
    | (Fmlocal(s,t1,t2)) as t3 -> if s=s1 then t3
                                    else if mem s V
                                           then let s' = new_name t2 s in
                                            fmsubst s1 t
                                             (Fmlocal(s',
                                                      t1,
                                                      fmsubst s 
                                                          (Fmvar s')
                                                          t2))
                                           else Fmlocal(s,subst1 t1,subst1 t2)
    | (FmRec t1) -> FmRec(subst1 t1)
    | (Fmmatch(t2,T,pl)) -> Fmmatch(subst1 t2,T,
            (map (let rec subspat = function
                   (n,l,t1) as t3-> let I = intersect V l in
                     match I with
                       []->if mem s1 l
                            then t3
                            else (n,l,subst1 t1)
                      |(v::l1)->let v'=new_name t1 v in
                                 subspat (n,subl v v' l,fmsubst v(Fmvar v') t1)
                  in subspat)
                  pl))
    | t -> t        
 in subst1
                                    ;;
let rec fmlistsubst l = function
      ([],t) -> t
    | ((s::sl),t) -> fmlistsubst (tl l) (sl,fmsubst s (hd l) t);; 


let fmbetared = function
   (Fmapp((Fmlambda(s,t1)),t2))->(fmsubst s t2 t1)
(*
 | (Fmmatch(Fmconstruct(n,s1,l),s2,pl))->if s1=s2
                                            then fmlistsubst l (assoc n pl)
                                            else raise fmtypemismatch
*)
 | t -> t
                                    ;;

let last5 s = sub_string s ((length_string s) - 5) 5;;

let rec is_constr = function
      (Fmconstruct(_,_,_)) -> true
    | (Fmlambda(s,t1))->is_constr t1
    | _ -> false;;


let to_exp s = ((length_string s)>5 &
               ((last5 s)="_recs" or (last5 s)="_recd")) or
               (is_constr (assoc s !Fmenv)) or
                (s="well_founded_recursion");;


let to_exp1 s = ((length_string s)>5 &
               ((last5 s)="_recs" or (last5 s)="_recd")) or
               (is_constr (assoc s !Fmenv));;

let rec fmexp = function 
      (Fmlambda(s,t1)) -> Fmlambda(s,fmexp t1)
    | (Fmapp (t1,t2)) -> Fmapp(fmexp t1,fmexp t2)
    | (Fmconstruct (n,s,l)) -> Fmconstruct(n,s,map fmexp l)
    | (Fmmatch(t,T,pl)) -> Fmmatch(fmexp t,T,map
                                              (fun (n,l,t)->
                                                  (n,l,fmexp t))
                                               pl)
    | (Fmconst s) as t -> if to_exp s then fmexp(assoc s !Fmenv)
                                      else t
    | Fmexcept(t1,t2) -> Fmexcept(fmexp t1,fmexp t2)
    | t -> t;;

let rec fmexp1 = function
      (Fmlambda(s,t1)) -> Fmlambda(s,fmexp1 t1)
    | (Fmapp (t1,t2)) -> Fmapp(fmexp1 t1,fmexp1 t2)
    | (Fmconstruct (n,s,l)) -> Fmconstruct(n,s,map fmexp1 l)
    | (Fmmatch(t,T,pl)) -> Fmmatch(fmexp1 t,T,map
                                              (fun (n,l,t)->
                                                  (n,l,fmexp1 t))
                                               pl)
    | (Fmconst s) as t -> if to_exp1 s then fmexp1(assoc s !Fmenv)
                                      else t
    | t -> t;;


 
let superfl s = let T=assoc s !Fmindenv in
           length T=1 & length(snd(hd T))=1;;

let rec fmred1 = function
      (Fmlambda(s,t1)) -> 
            (match t1 with
                (Fmapp(t,Fmvar s1))-> if s=s1&(not(mem s(free_var(fmred1 t))))
                                    then fmred1 t
                                    else Fmlambda(s,fmred1 t1)
              | _ -> Fmlambda(s,fmred1 t1))
    | (Fmapp (t1,t2)) -> (match fmred1 t1 with
                            (Fmlambda(s,t3))->fmred1(fmsubst s (fmred1 t2)
                                                              (fmred1 t3))
                          | (Fmmatch(t,T,pl)) ->
                              fmred1(
                                Fmmatch(t,T,map (fun(n,sl,t0)->
                                        (n,sl,fmred1(Fmapp(t0,t2))))
                                                pl))
                          | t->Fmapp(t,fmred1 t2))
    | (Fmconstruct (n,s,l)) -> if superfl s
                                then fmred1(hd l)
                                else  Fmconstruct (n,s,map fmred1 l)
    | (Fmmatch(t,T,pl)) -> if superfl T 
                            then (match pl with
                             [(_,[s],t1)]->fmred1(fmsubst s
                                                                (fmred1 t)
                                                                t1)
                            | _ -> failwith "fmred1 Fmmatch")
                            else  Fmmatch(fmred1 t,T,map 
                                               (fun (n,l,t)->
                                                       (n,l,fmred1 t))
                                               pl)
    | Fmlocal(s,(FmRec(Fmlambda(x,(Fmmatch(Fmvar x1,S,pl) as t))) as t1),
              (Fmapp(Fmvar s1,a) as t2))->
        if x=x1 & 
           s=s1 & not (mem s (free_var t)) 
                                    then Fmmatch(a,S,pl)
                                    else Fmlocal(s,fmred1 t1,fmred1 t2)
    | Fmlocal(s,t1,t2) -> Fmlocal(s,fmred1 t1,fmred1 t2)
    | FmRec(t)->FmRec(fmred1 t)
    | t -> fmbetared t;;


let rec fmred2 = function
       (Fmapp (t1,t2)) -> (match (fmred2 t1) with
                            (Fmlambda(s,t3))->fmred2(fmsubst s (fmred2 t2)
                                                              (fmred2 t3))
                          | t->Fmapp(t,fmred2 t2))
    | (Fmmatch(t,T,pl)) -> Fmmatch(fmred2 t,T,map
                                               (fun (n,l,t)->
                                                       (n,l,fmred2 t))
                                               pl)
    | t -> fmbetared t;;

let rec hd_var = function
   Fmapp(t1,t2) -> hd_var t1
 | Fmvar(_) -> true
 | _ -> false ;;

let rec fmtermsubst t0 ts t=
   if t0=t then ts else match t with
      (Fmlambda(s,t1)) ->  if t0 = (Fmvar s)
                                then (Fmlambda(s,t1))
                                else (Fmlambda(s,(fmtermsubst t0 ts t1)))
    | (Fmapp (t1,t2)) -> (Fmapp (fmtermsubst t0 ts t1,fmtermsubst t0 ts t2))
    | Fmexcept(t1,t2) -> Fmexcept(fmtermsubst t0 ts t1,fmtermsubst t0 ts t2)
    | (Fmvar s) -> if t0 = (Fmvar s) then ts else (Fmvar s)
(*    | (Fmelim (s,l)) -> (Fmelim (s,(map (fmtermsubst t0 ts) l)))  *)
    | (Fmconstruct (n,s,l)) -> ((Fmconstruct (n,s,
                                            (map (fmtermsubst t0 ts) l))))
    | (Fmconst s) -> (Fmconst s)
    | (Fmext E) -> (Fmext E)
    | Fmrec -> Fmrec
    | Fmarg -> Fmarg
    | Fmerror -> Fmerror
    | (Fmlocal(s,t1,t2)) -> if t0=Fmvar s then let t1' =
                                               match t1 with
                                                  (FmRec t) -> t1
                                                | t -> fmtermsubst t0 ts t
                                             in (Fmlocal(s,t1',t2))
                                           else (Fmlocal(s,
                                                         fmtermsubst t0 ts t1,
                                                         fmtermsubst t0 ts t2))

    | Fmmatch(t1,T,pl) -> Fmmatch(fmtermsubst t0 ts t1,
                                 T,
                                 map (fun(x,l,ti) -> 
                                            (x,l,
                                              (match t0 with
                                            Fmvar(s) -> if (mem s l) then ti
                                               else fmtermsubst t0 ts ti 
                                          | t0 -> fmtermsubst t0 ts ti)))
                                pl)
    | FmRec(t) -> FmRec(fmtermsubst t0 ts t)
                                    ;;

let rec strict_vars = function
   Fmconstruct(_,_,l)->list_it union (map strict_vars l) []
 | Fmlambda(s,t) -> strict_vars t
 | Fmvar(s) -> [s]
 | Fmconst(_) -> []
 | Fmmatch(t,T,pl) -> union (strict_vars t)
                            (list_it intersect
                                     (map (fun(_,l,t)->subtract(strict_vars t)
                                                               l)
                                          pl)
                                     [])
 | Fmexcept(t1,t2) -> union(strict_vars t1)(strict_vars t2)
 | FmRec(t) -> strict_vars t
 | Fmlocal(s,FmRec(t1),t2) -> subtract(union(strict_vars t1)
                                            (strict_vars t2))
                                      [s]
 | Fmlocal(s,t1,t2) -> union (strict_vars t1)
                             (subtract
                                (strict_vars t2)
                                 [s])
 | Fmapp(Fmconst(s),t) ->  (strict_vars t)
 | Fmapp(Fmvar(s),t) -> [s]
 | Fmapp(t1,t2) -> if hd_var t1 then strict_vars t1
                                else union(strict_vars t2)(strict_vars t1)
 | _ -> [];;

let rec fml_abs_var = function
   Fmlambda(s,t) -> add_set s (fml_abs_var t)
 | Fmmatch(t,_,pl) -> list_it union
                              (map (fun(_,l,t)->subtract(fml_abs_var t) l)
                                   pl)
                             []
 | _ -> [];;

let is_strict t =(t<>Fmerror)&([] = subtract (fml_abs_var t)(strict_vars t));;

let fmred = fmred1 o fmexp o fmred1 o fmexp1;;


let optimal_old () =
  let rec op = function
    [] -> []
 |  (s,t)::l -> let t0 = fmred t in
                 if (not (is_constr t)) & (is_strict t0 or  (l=[]))
                    then (s,t0)::(op l)
                    else op (map(fun(s1,t)-> (s1,fmred
                                                (fmtermsubst (Fmconst s)
                                                             t0
                                                             t)))
                                 l)
    in rev(op (rev !Fmenv));;

let opti() = Fmenv:=optimal_old();();;



(***************************************************************************)
(*                          Pretty-Printer to LML                          *)
(***************************************************************************)
(*

let LML_name = ref "extract";;
let LMLreserved = ref ["cons";"nil";"true";"false";
                       "and";"Max";"Min";"Or";"Sum";"Times";"abs";"all";
                      "difference";"exists";"for";"from";"fst";"hd";"head";
                      "last";"odd";"sort";"split";"tl";"tail"];;

let lmlsafe s = if (mem s !LMLreserved) then (s^"_ren") else s;;
*)


let Gaml() = CurLang:=GAML; exp_typ := true;;

let Lml() = CurLang:=LML; exp_typ := false;;



(*****************************************************************************)
(*                          execution des commandes                          *)
(*****************************************************************************)

let rec is_rec s = function
      (FmConst s1)->(s=s1)
    | (FmArr(t1,t2))->(is_rec s t2)
    | (FmApp(t1,t2))->(is_rec s t1)
    | (FmVar s1) -> false
    | _ -> failwith "is_rec"                                ;;

let rec to_ind2 n = function
      (FmArr(t1,t2)) -> (to_ind2 (n+1) t2)
    | _ -> n
                                ;;

let rec to_ind1 n = function
      x -> if n=0 then x
                  else to_ind1 (n-1)(Fmlambda("x_ex"^(string_of_int n),x))
                               ;;
let rec to_ind3 = function 
      0 -> Fmarg
    | n -> Fmapp(to_ind3 (n-1),Fmvar ("x_ex"^(string_of_int n)))
                               ;;
let to_ind0 s t = if (is_rec s t)
                              then Reca
                                     (let n = (to_ind2 0 t) in
                                      (to_ind1 n
                                               (Fmapp(Fmrec,(to_ind3 n)
                                                ))))
                              else Noreca
                                ;;

                                   
let to_ind = function
  (FmInd(s,cl)) -> (s,(map (fun(s1,t)->(s1,(map (to_ind0 s) t))) cl))
| _ -> failwith "to_ind"
                            ;;

let rec const_to_var x = subs 
where rec subs = fun
      (FmConst s) -> if s=x then FmVar x else FmConst s
    | (FmArr (t1,t2)) -> (FmArr(subs t1,subs t2))
    | (FmLambda (s,t)) -> (FmLambda (s,subs (const_to_var s t)))
    | (FmInd(s,cl)) -> (FmInd(s,map (fun(s1,tl)->(s1,map subs tl)) cl))
    | (FmApp (t1,t2)) -> (FmApp(subs t1,subs t2))
    | t -> t
                ;;

let to_fmltype2 s = fun
      (FmInd (s1,cl)) -> FmInd (s,cl)
    | t -> t
            ;;

let rec to_fmltype1 s l t =
    match l with 
          [] -> to_fmltype2 s t
        | (x::l1) -> FmLambda (x,to_fmltype1 s l1 (const_to_var x t))
            ;;

let to_fmltype s l t =  (to_fmltype1 s l t);;

let funconstr t = 
    let rec funcons1 s n = fun
          [] -> ()
        | (a::l) -> add_fml (fst a)(to_fmlterm (FCconstruct(n,s,[])));
                    funcons1 s (n+1) l
    in if !reserve_flag then funcons1 (fst t) 1 (snd t)
                            ;;
let header() = match !CurLang with
          LML -> ""
        | GAML -> "module main \n export \n val main;;";;

let ender1 s = match !CurLang with
          LML -> [< >]
        | GAML -> [< 'S "let main = " >];;

let ender2 s = match !CurLang with
          LML -> [< >]
        | GAML -> [< 'S ";; \n end" >];;

let rec type_is_ok = function
      (FmConst c) -> if (mem_assoc c !Fmtypenv) or (c="Int")
                          then ()
                          else raise (fml_undefined_type c)
    | (FmArr(t1,t2)) -> type_is_ok t1; type_is_ok t2
    | (FmApp(t1,t2)) -> type_is_ok t1; type_is_ok t2
    | (FmInd(n,l)) -> do_list ((do_list type_is_ok) o snd) l
    | (FmLambda(_,t)) -> type_is_ok t
    | _ -> ();;

let Tsubst t1 s t2 =
let rec Ts = function
      FmConst (c) as C -> C
    | FmArr(t1,t2)->FmArr((Ts t1),(Ts t2))
    | FmInd(n,l)->FmInd(n,map (fun(x,t)->x,map Ts t) l)
    | FmApp(t1,t2)->FmApp(Ts t1,Ts t2)
    | (FmVar(v) as V) -> if v=s then t2 else V
    | (FmLambda(v,t) as T) -> if s=v then T else FmLambda(v,Ts t)
    | t -> t
in Ts t1;;

let rec Tnorm = function
      FmConst (c) as C-> if c = "Int" or (mem_assoc c !Fmindenv)
                            then C
                            else Tnorm (assoc c !Fmtypenv)
    | FmArr (t1,t2) -> FmArr(Tnorm t1,Tnorm t2)
    | FmInd(n,l) -> FmInd(n,map (function (s,t)->(s,map Tnorm t)) l)
    | FmApp(t1,t2) -> let t2' = Tnorm t2 in
                       (match (Tnorm t1) with
                          FmLambda(s,t3) -> Tsubst t3 s t2'
                        | t4 -> FmApp (t4,t2'))
    | FmLambda(s,t)->FmLambda(s,Tnorm t)
    | FmVar (_) as v -> v
    | t -> t;;

let FML_save = nref (forward : string -> unit);;
let FML_load = nref (forward : string -> unit);;

let rec optype = function
      FmLambda(s,t) -> optype(t)
    | t             -> t
;;
let fmltop = function
    Affectation(s,ob) ->
    let ter = (to_fmlterm ob) in
        ((add_fml s ter);
         PPNL [< 'S (s^" is defined:") >];
         PPNL(Prfmlterm ter))

  | RecAffectation(s,ob) ->
    let ter = FmRec(to_fmlterm (FCsubst s ob)) in
        ((add_fml s ter);
         PPNL [< 'S (s^" is defined:") >];
         PPNL(Prfmlterm ter))

  | Affecttype(s,l,t) ->
    let tp = (to_fmltype s l t) in
        (type_is_ok tp;
         (add_fmltype s tp);
         (match t with
          (FmInd(s1,cl)) ->
          let ity = to_ind(optype tp) in
              (Fmindenv := (ity::(!Fmindenv));
               funconstr ity)
            |_ -> ());
         PPNL [< 'S ("type "^s^" is defined:") >];
         PPNL(Prfmltype tp))

  | Compute(ob) ->
    let ter =  (to_fmlterm ob) in
    let gp = copy_gp dflt_gp in
    let fp = copy_fp std_fp in
    let lml_output_chan =
        open_trapping_failure open_out (!LML_name) (match !CurLang with
                                                    LML->".m"
                                                  | GAML->".gl") in
             with_output_to fp lml_output_chan;
             fully_pp gp;
             PPNL_WITH gp fp
             [< 'S (header());
              prlist
              (fun t ->
                   if is_indt (snd t) or (not !exp_typ) then 
                       [< LML_type_def (fst t,Tnorm (snd t));
                        'FNL;
                        'S(match !CurLang with
                                 LML -> "in"
                               | GAML -> ";;");
                        'FNL >]
                   else [< >])
                   (rev !Fmtypenv);
                   prlist (fun t ->
                               [< LML_term_def t;'FNL;
                                'S (match !CurLang with
                                          LML -> "in"
                                        | GAML -> ";;");
                                'FNL >])
                   (rev !Fmenv);
                   ender1();
                   LML_term ter;
                   ender2() >];
             close_out lml_output_chan

  | Inductive (s,IT) ->
             type_is_ok (const_to_var s IT);
                 (add_fmltype s IT);
                 let ity = to_ind (optype IT) in
                     (Fmindenv := (ity::(!Fmindenv)));
                     funconstr ity;
                     PPNL [< 'S ("type "^s^" is defined:") >];
                     PP(Prfmltype IT)

    | Instanciate(s,t) -> type_is_ok t; (typtosubst s t); ()
    | Realize(s,t) -> (tosubst s t); ()
    | Extr_all -> extract_all()
    | Extr_from_to(s1,s2) -> extract_from_to (id_of_string s1) (id_of_string s2)
    | Extr_until(s) -> extract_until (id_of_string s)
    | Extr_from(s) -> extract_from (id_of_string s)
    | Reset -> fm_reset()
    | Reset_All -> fm_reset_all()
    | CAML_File(s) ->  make_caml_file.v s
    | Print s ->
      PPNL
      [< if mem_assoc s !Fmenv then
             [< 'S (s^" = "); Prfmlterm (assoc s !Fmenv) >]
         else if mem_assoc s !Fmtypenv then
             [< 'S (s^" = "); Prfmltype (assoc s !Fmtypenv) >]
         else [< 'S ("The constant "^s^" is not defined") >]
      >]

    | free_vars -> PP(print_extracted_vars())
    | Optimise -> (Fmenv:=optimal.v();())
    | Save(s) -> FML_save.v s
    | Load(s) -> fm_reset_all(); FML_load.v s
    | _ -> anomaly "Incomplete fmltop";;

exception wait          ;;
exception FMLdrop       ;;
exception FMLreserved of string ;;
exception FMLundef of string ;;
exception FMLbadar of string * bool * int * int ;;

let FML_prdot_flag = ref false;;

let is_reserved s = (!reserve_flag)&
                    ((length_string s)>3) &
                    (eq_string(sub_string s 0 3,"VAR")) &
                    (list_it (fun x y->(x<58 & x>47 & y))
                             (tl(tl(tl(explode_ascii s))))
                             true)
                                ;;

let FML_pr_env () =
    prlist (fun (s,t)->
                 [< 'S(s^" = ");
                  Prfmlterm t;
                  if !FML_prdot_flag then
                      [< 'S "." >]
                  else [< >];
                      'FNL >])
    (rev !Fmenv);;

let FML_pr_typ () =
    prlist(fun (s,t)->
                [< 'S (s^" ");
                 Prfmltype t;
                 if !FML_prdot_flag then
                     [< 'S "." >]
                 else [< >];
                     'FNL >])
    (rev !Fmtypenv);;


let eval_to_fcobj _ =
  prerr_endline "<<<eval_to_fcobj>>>";
  exit 1
;;

type fml_keyword =
  KAll | KCAML | KConstr | KDrop | Kelse | Kend | KEnv | KExit | KExtract | KFile | Kif | Kin | KInductive | Kinductive | KInstantiate | Klet | KLoad | Kmatch |KOptimise | KPrint | KRealize | KREC | Krec | KReset | Kthen | KTypes | Kwith | KWrite
 | Kfrom | Kuntil | Kfree | Kvars | Kto | Ksave | KLML | KGAML
;;

let (fml_keywords,rev_fml_keywords) =
  let t = make_vect 11 [] in
  let rev_t = make_vect 11 [] in
  do_list (fun (s,tok) -> hash_add_assoc (s,tok) t;
                          hash_add_assoc (tok,s) rev_t) [
    "All", KAll;        "Attach", KInstantiate;
        "CAML", KCAML;      "Constr", KConstr;
        "Drop", KDrop;
    "else", Kelse;      "end", Kend;        "Env", KEnv;
        "Exit", KExit;
    "Extract", KExtract;
    "Free", Kfree; "From",Kfrom; "Until",Kuntil;
    "File", KFile;      "if", Kif;
    "in", Kin;      "Inductive", KInductive;
        "inductive", Kinductive;
        "Instanciate", KInstantiate;
        "Instantiate", KInstantiate;
    "let", Klet;        "Load", KLoad;      "match", Kmatch;
        "Optimise", KOptimise; "Optimize", KOptimise;
    "Print", KPrint;    "Realize", KRealize;
        "REC", KREC;        "rec", Krec;
    "Reset", KReset;
    "Save", Ksave;
    "then", Kthen;
    "To",Kto; "Types", KTypes;
    "Vars", Kvars;
        "with", Kwith;      "Write", KWrite;
    "LML",KLML; "GAML",KGAML
  ];
  (t,rev_t)
;;


let plist elem = plist_rec
  where rec plist_rec = function
    [< elem e; plist_rec l >] -> e::l
  | [< >] -> []
;;

let ne_plist elem = function
  [< elem e; (plist elem) l >] -> e::l
;;

let list_with_sep sep elem = do_rec
  where rec do_rec = function
    [< elem e; (function [< sep(); do_rec l >] -> l | [< >] -> []) l >] -> e::l
;;

let right_assoc op subexp =
  let rec do_loop x = function
    [< op f; subexp y; (do_loop y) z >] -> f x z
  | [< >] -> x in
  function [< subexp x; (do_loop x) y >] -> y
;;

let left_assoc_app f subexp =
  let rec do_loop x = function
    [< subexp y; (do_loop (f x y)) z >] -> z
  | [< >] -> x in
  function [< subexp x; (do_loop x) y >] -> y
;;

let binop_app op t1 t2 = FCapp(FCapp(FCext op,t1),t2);;

let rec Fmltop_command = function
      [< 'Tkw KDrop; 'Tdot >] -> raise FMLdrop
    | [< 'Tkw KEnv; 'Tdot >] -> full_print(FML_pr_env()); raise wait
    | [< 'Tkw KTypes; 'Tdot >] -> full_print(FML_pr_typ()); raise wait
    | [< 'Tkw KLML ; 'Tdot >] -> (message "Extraction into LML code";
                                  Lml(); raise wait)
    | [< 'Tkw KGAML ; 'Tdot >] -> (message "Extraction into Gaml code";
                                  Gaml(); raise wait)
    | [< ccommand C; 'Tdot >] -> C
    | [< 'Tdot >] -> (message "Syntax Error : Unparsed ``.''";raise wait)
    | [< '_ >] -> raise Parse_error

and ccommand = function
      [<
        'Tkw Klet;
        begin function
          [< 'Tkw Krec >] -> true
        | [< >] -> false
        end is_rec;
        IDENT1 x; 'Tequal; term t1
      >] ->
        if is_rec then RecAffectation(x,t1) else Affectation(x,t1)
(*
    | "let"; IDENT1 x; "="; "REC"; term t1 -> (RecAffectation(x,t1))
*)
    | [< 'Tkw KPrint; 'Tident s >] -> Print s

    | [< 'Tkw Kfree; 'Tkw Kvars >] -> free_vars

    | [<
        IDENT1 s;
        begin function
          [<
            'Tequal;
            begin function
              [< 'Tkw KREC >] -> true
            | [< >] -> false
            end is_rec;
            term t1
          >] ->
            if is_rec then RecAffectation(s,t1) else Affectation(s,t1)
        | [<
            begin function
              [< 'Tequalequal >] -> []
            | [< varlist l; 'Tequalequal >] -> l
            end l;
            begin function
              [< 'Tkw Kinductive; constr C >] ->
                (Inductive
                  (s,
                   (clotype(list_it (fun s t->FmLambda(s,t)) l (FmInd(s,C))))))
            | [< ttype t >] ->
                (Affecttype(s,l,t))
            end x
          >] -> x
        end x
      >] -> x
    | [< 'Tkw KInstantiate; IDENT1 x; ctype t >] -> Instanciate(x,t)
    | [< 'Tkw KRealize; IDENT1 x; term t >] -> Realize(x,(to_fmlterm t))
    | [<
        'Tkw KInductive; IDENT1 s;
        begin function
          [< varlist l; 'Tequal; constr C >] ->
            (Inductive
               (s,
                (clotype(list_it (fun s t->FmLambda(s,t)) l (FmInd(s,C))))))
        | [< 'Tequal; constr C >] ->
            (Inductive
               (s,
                    clotype(FmInd(s,C))))
        end x
      >] -> x
(*
    | "Write"; "File"; term t -> Compute t
    | "Write"; IDENT s; "File"; term t -> LML_name:=s;
                                          Compute t
*)
    | [<'Tkw KWrite ;
       begin function
       [< 'Tkw KCAML; 'Tkw KFile;
        (function
         [< 'Tident s >] -> s
       | [< 'Tstring s >] -> s) s >] ->
       CAML_File s
     | [< 'Tkw KFile  ; term t >] -> Compute t
     | [< 'Tident s ; 'Tkw KFile ; term t >] -> (LML_name := s;Compute t)
       end x >] -> x

    | [<
        'Tkw KExtract;
        begin function
         [< 'Tkw KAll >] -> Extr_all
       | [< 'Tkw Kfrom ; 'Tident s1 ;
            (function
             [< 'Tkw Kto ; 'Tident s2 >] -> Extr_from_to(s1,s2)
           | [< >] -> Extr_from s1) x >] -> x
       | [< 'Tkw Kuntil ; 'Tident s >] -> Extr_until s
       | [< >] -> Extr_from (string_of_id last_id_initial)
       end x
     >] -> x
(*
    | "Extract"; "From"; IDENT s1; "To"; IDENT s2 -> Extr_from_to(s1,s2)
    | "Extract"; "Until"; IDENT s -> Extr_until s
    | "Extract"; "From"; IDENT s -> Extr_from s
*)

    | [< 'Tkw KReset;
       begin function
        [< 'Tkw KAll >] -> Reset_All
      | [< >]-> Reset
       end x >] -> x

    | [< 'Tkw KOptimise >] -> Optimise

    | [< 'Tkw Ksave ; 'Tident s >] -> Save s

    | [< 'Tkw KLoad;
       (function
        [< 'Tident s >] -> s
      | [< 'Tstring s >] -> s) s >] -> Load s

and varlist x =
      ne_plist IDENT1 x

and term x =
      left_assoc_app (fun t1 t2 -> FCapp(t1,t2))
term1 x

and term1 = function
      [< 'Tlbracket; binder b; 'Trbracket; term1 t >] -> FCbind b t
    | [<
        'Tkw Kif; term1 t1; 'Tkw Kthen; term1 t2;
        'Tkw Kelse; term1 t3
      >] -> FCapp(FCapp(FCapp(FCext If,t1),t2),t3)
    | [<
        'Tkw Klet; 'Tkw Krec; IDENT1 x; 'Tequal;
        term1 t1; 'Tkw Kin; term1 t2
      >] -> FClocal(x,FCrec t1,t2)
    | [<
        'Tkw Kmatch; term t; 'Tkw Kwith; patlist pl;
        'Tkw Kend; 'Tkw Kmatch
      >] -> FCmatch(t,pl)
    | [< term2 t >] -> t

and term2 x =
      right_assoc (function [< binop op >] -> binop_app op)
cterm x

and cterm = function
(*
      "(";  tterm t1; cterm t2; ")" -> FCapp(t1,t2)
*)
      [< 'Tsharp; 'Tkw KExit >] -> FCerror
(*
    | "let"; "rec"; IDENT1 x; "="; cterm t1; "in";cterm t2 ->
                                                     FClocal(x,FCrec t1,t2)
    | "let"; IDENT1 x; "="; cterm t1; "in";cterm t2 -> FClocal(x,t1,t2)
*)
    | [< extterm t >] -> t
(*
    | "'"; IDENT1 x -> FCconstr x
*)
    | [< IDENT1 x >] -> FCconst(x)
(*
    | "["; binder b; "]"; cterm t -> FCbind b t
*)
    | [< 'Tlparen; term t; 'Trparen >] -> t
(*
    | "match"; term t; "with"; patlist pl; "end"; "match" -> FCmatch(t,pl)
*)
    | [<
        'Tkw KConstr;
        'Tlbrace; 'Tint n; 'Tcomma; 'Tident x; 'Trbrace;
        begin function
          [< 'Tless; termlist l; 'Tgreater >] -> l
        | [< >] -> []
        end l
      >] ->
            if mem_assoc x (!Fmindenv) then FCconstruct(n,x,l)
            else raise (FMLundef x)
(*
    | "Try"; term t1; "With"; term t2 -> FCexcept(t1,t2)
    | "#"; {parse_caml_expr0()} t -> eval_to_fcobj t
*)

and extterm = function
      [< 'Tint n >] -> FCext(Num n)
(*
    |cterm O1; binop op ; cterm O2 -> FCapp(FCapp(FCext(op),O1),O2)
    |"TRUE" -> FCext True
    |"FALSE" -> FCext False
    |"If_EXT" -> FCext If
    |"if"; cterm t1; "then"; cterm t2; "else"; cterm t3 ->
            FCapp(FCapp(FCapp(FCext If,t1),t2),t3)
*)

and binop = function
      [< 'Tplus >] -> Plus
    | [< 'Tstar >] -> Times
    | [< 'Tminus >] -> Minus
    | [< 'Tslash >] -> Div
    | [< 'Tless >] -> Lesser
    | [< 'Tequal >] -> Equal

(*
and tterm = parse
      cterm t -> t
    | tterm t1; cterm t2 -> FCapp(t1,t2)
*)

and termlist x =
    list_with_sep (function [< 'Tcomma >] -> ()) term x

and binder = function
      [<
        IDENT1 x;
        begin function
          [< 'Tcolon; IDENT1 y >] -> [x]
        | [< 'Tcomma; binder b >] -> x::b
        | [< >] -> [x]
        end b
      >] -> b

and constr =
    let simple_constr = function
        [<
          IDENT1 s;
          begin function
            [< tctypel tl >] -> tl
          | [< >] -> []
          end tl
        >] -> [(s,tl)]
    in
    function
      [<
        simple_constr c1;
        begin function
          [< 'Tplus; constr c >] -> c1@c
        | [< 'Tbar; constr c >] -> c1@c
        | [< >] -> c1
        end x
      >] -> x

and ttype = function (* type is renamed ttype *)
     [< ctype t >] -> t
(* | "inductive"; constr cl -> FmInd ("",cl)  *)

and tctypel x =
    ne_plist tctype x

and ctype x = list_it (fun x -> x) [
    right_assoc (function [< 'Tminusgreater >] -> fun t1 t2 -> FmArr(t1,t2));
    left_assoc_app (fun t1 t2 -> FmApp(t1,t2))
] tctype x

and tctype = function
      [< IDENT1 c >] -> FmConst c
    | [< 'Tlparen; ctype t; 'Trparen >] -> t

and patlist x =
    list_with_sep (function [< 'Tbar >] -> ()) pattern x

and pattern = function
      [< IDENT1 x; 'Tminusgreater; term t >] -> (x,[],t)
    | [<
        'Tlparen; IDENT1 x; varlist l; 'Trparen;
        'Tminusgreater; term t
      >] -> (x,l,t)

and IDENT1 = function
       [< 'Tident x >] -> if is_reserved x then raise (FMLreserved x)
                              else x
                                                ;;

let parse_fml ts =
  Fmltop_command ts
;;

let toplevel_print_newline() =
    if is_silent ()
     then (print_string".";print_flush())
     else print_newline()
;;

let fmlread_loop ts = toplevel_print_newline(); fmltop (parse_fml ts);;

let fml_loop_handler f arg = try f arg with 
      wait -> ()
    | FMLundef (s) -> message ("The inductive type "^s^" is undefined.")
    | fml_undefined_type (s) -> message ("The type "^s^" is undefined.")
    | fml_undefined_term (s) -> message ("The constant "^s^" is undefined.")
    | FMLreserved (s) -> message (s^" is reserved by FML")
    | FMLbadar (s,b,n,m) -> if b then 
                              message ("The constructor "^s^" has arity "^
                                       (string_of_int n)^
                                        " and cannot be used with "^
                                        (string_of_int m)^" arguments")
                                  else
                              message ("Type "^s^" has "^(string_of_int n)^
                                       " constructors and cannot be used with "
                                        ^(string_of_int m)^" arguments")
    | reraise -> raise(reraise) ;;

let fml_loop ts = while true do
    fml_loop_handler fmlread_loop ts
done  ;;

let rec discard_dots ts =
    let rec aux = function
        [< 'Tdot >] -> [< 'S"." >]
      | [< 't ; s >] -> [< 'S(string_of_token rev_fml_keywords t) ;
                           'S" " ;
                            aux s >]
      | [< >] -> [< >]
    in
        PPNL [< 'S"Discarding " ; aux ts >]
;;

let rec fml () =
    let oprompt = get_prompt () in
    set_prompt "Fml < ";
    let cs = stream_of_channel std_in in
    let ts = token_stream fml_keywords cs in
    try fml_loop ts
    with  FMLdrop -> set_prompt oprompt
        | Parse_failure | End_of_file -> print_newline(); set_prompt oprompt
        | Parse_error -> print_syntax_error(); discard_dots ts; reset_lexer cs; set_prompt oprompt; fml()
        | reraise -> set_prompt oprompt; raise(reraise);;

let fml_file = ref std_out;;

FML_save.v <-
    (fun s ->
         let fml_file = open_trapping_failure open_out s ".f" in
         let fp = copy_fp std_fp in
         let gp = copy_gp dflt_gp in
             with_output_to fp fml_file;
             fully_pp gp;
             FML_prdot_flag := true;
             reserve_flag := false;
             PPNL_WITH gp fp
             [< FML_pr_typ ();
              FML_pr_env();
              'S "Drop." >];
             FML_prdot_flag := false;
             reserve_flag := true;
             close_out fml_file);;


FML_load.v <- fun s ->
    reserve_flag := false;
    let in_chan = open_with_suffix open_in s ".f" in
    let quit_load () =
        reserve_flag := true;
        close_in in_chan in
    let cs = stream_of_channel in_chan in
    let ts = token_stream fml_keywords cs in
    try fml_loop ts; quit_load()
    with
         FMLdrop -> quit_load()
      |  Not_found -> quit_load()
      |  reraise -> quit_load (); raise(reraise);;
