(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                             command.ml                                   *)
(****************************************************************************)

#open "std";;
#open "extraction";;
#open "term";;
#open "machine";;
#open "initial";;
#open "pretty";;
#open "pp";;
#open "stdpp";;


(* type command =
  RefC of identifier
| PropC of contents
| TypeC of contents
| AppC of command * command
| LambdaC of identifier * command * command
| ProdC of identifier * command * command
| ArrowC of command * command
| ConstrC of int * command
| IndC of identifier * command * command list
| ElimCdep of command * command * command list
| ElimCnodep of command * command
| ElimC of command * command 
| LetC of identifier * command * command;; *)

let with_heavy_rollback f arg =
    (raw_save_state "*construct*" "";
     try let rslt = f arg in (forget_state false "*construct*"; rslt)
     with reraise -> (raw_restore_state "*construct*"; forget_state false "*construct*"; raise reraise))
;;

let with_light_rollback f arg =
     (save_light();
      try let rslt = f arg in (forget_light(); rslt)
      with reraise -> (restore_light(); raise reraise))
;;

let construct t = save_light();
                   (try construct_rec t; forget_light ()
                       with reraise -> restore_light (); raise(reraise))
where rec construct_rec = function
   RefC(name)       -> consider name
 | PropC(c)         -> proposition c
 | TypeC(c)         -> new_type c
 | AppC(c1,c2)      -> construct_rec c2;stack_value();construct_rec c1;pop_apply()
 | LambdaC(s,c1,c2) -> construct_rec c1;assume s 0;construct_rec c2;abs_var()
 | ProdC(s,c1,c2)   -> construct_rec c1;assume s 0;construct_rec c2;gen_var()
 | ArrowC(c1,c2)    -> construct_rec c1;postulate 0;construct_rec c2;gen_var()
 | ConstrC(i,c)     -> construct_rec c; constructor i
 | IndC(stamp,s,c,lt)     -> construct_rec c;assume s 0;
                       map (fun c -> construct_rec c; postulate 0) lt;
                       inductype stamp (length lt)
 | ElimCdep(cind,ckind,lconstr)-> let k = (construct_rec ckind; read_val()) 
                       and lc = map (fun c -> construct_rec c; read_val()) lconstr
                       in construct_rec cind; dep_elimination k lc
 | ElimCnodep(cind,ckind) -> let k = (construct_rec ckind; read_val()) 
                       in construct_rec cind; nodep_elimination k
 | ElimC(c,P)       -> construct_rec P; stack_value();construct_rec c;
                       elimination()
 | LetC(s,c1,c2)    -> construct_rec c1;declare s 1;construct_rec c2;pop_const();;

let toplevel_message s = if not(is_silent()) then message s ;;


(* For demos of the constructions machine.  e.g. e<<(A:Prop)A->A>>;; *)
let e c = construct c;PP(print_val());;

(* Commands of the interface *)

(* 1| Theorems *)
let theorem s n = open_section s;
                  push_scope (Strength n);;

let conjecture c = construct c; stack_cast();;

let statement c = conjecture c;
                  open_section ("Proof_of_"^(current_section ()));;

let proof p =  
   let proof_ident = current_section () in 
            construct p;
            close_section' proof_ident false;
   let ident = current_section () and (Strength n) = pop_scope () in 
            toplevel_message ("Checking " ^ ident);
            verify(); 
            declare (id_of_string ident) n;
            toplevel_message "Proof completed";
            close_section ident false;;

(* 2| Constant definitions *)

let definition s n = open_section s;
                     push_scope (Strength n);;

let body c = 
   let ident = current_section () and (Strength n) = pop_scope () in 
      construct c; 
      declare (id_of_string ident) n;
      toplevel_message (ident ^ " is defined");
      close_section ident false;;

let body_typ term typ =
   let ident = current_section () and (Strength n) = pop_scope () in 
      conjecture typ;
      construct term;
      close_section' ident false;
      toplevel_message ("Checking " ^ ident);
      verify(); 
      toplevel_message (ident ^ " is well typed");
      declare (id_of_string ident) n;;

(* 3| Variable definitions *)

let hypothesis s n = open_section s;
                     push_scope (Strength n);;

let def_var c = 
   let ident = current_section () and (Strength n) = pop_scope () in
      construct c; 
      assume (id_of_string ident) n; 
      close_section ident false;
      toplevel_message (ident ^ " is assumed");;

(* 4| Inductive definitions *)

exception OccurC;;

let occurC name c = (try occurrec c;false with OccurC -> true) where rec
    occurrec = function
        ProdC(n,e1,e2)   -> occurrec e1; if n<>name then occurrec e2
      | LambdaC(n,e1,e2) -> occurrec e1; if n<>name then occurrec e2
      | AppC(e1,e2)      -> occurrec e1; occurrec e2 
      | ArrowC(e1,e2)    -> occurrec e1; occurrec e2
      | LetC(n,e1,e2)    -> occurrec e1; if n<>name then occurrec e2
      | RefC(n)          -> if n = name then raise OccurC
      | ElimCdep(e1,e2,le) -> occurrec e1; occurrec e2; do_list occurrec le
      | ElimCnodep(e1,e2)-> occurrec e1; occurrec e2
      | ElimC(e1,e2)     -> occurrec e1; occurrec e2
      | ConstrC(_,e)     -> occurrec e
      | IndC(_,n,e,le)     -> occurrec e; if n<>name then do_list occurrec le
      | _                -> ();;

(* for sig, we use (sig A [x](P x)) instead of (sig A P) *)
let rec eta_reduce = function
  (LambdaC(n1,_,AppC(t,RefC n2)) as c) -> if n1=n2 & not (occurC n2 t)
                                            then eta_reduce t
                                            else c
| c                                      -> c;;

let rec compare = function
    (AppC(c1,c2),AppC(d1,d2)) -> c2 = eta_reduce d2 & compare(c1,d1)
 |  (RefC(n1),RefC(n2))       -> n1=n2
 |  _                         -> false;;

(* check that name appear only in subsexpressions (name largs) and
substitute name instead of (name largs) *)

let substC_list largs name c =
    if largs = [] then c else
    let cname = (RefC name) in
    let name_largs = it_list (fun app (na,_) -> AppC(app,RefC na)) cname largs
    in substrec c where rec
    substrec c = if occurC name c then
      match c with
        ProdC(n,e1,e2)   -> ProdC(n,substrec e1, if n = name then e2 else
                                                 substrec e2)
      | LambdaC(n,e1,e2) -> LambdaC(n,substrec e1, if n = name then e2 else
                                                   substrec e2)
      | AppC(e1,e2)      -> if compare(name_largs,c) then cname
                            else AppC(substrec e1,substrec e2)
      | ArrowC(e1,e2)    -> ArrowC(substrec e1,substrec e2)
      | LetC(n,e1,e2)    -> LetC(n,substrec e1, if n = name then e2 else
                                                substrec e2)
      | RefC(n)          -> error
                         ((string_of_id n)^" should only appear applied to its parameters")
      | ElimCdep(e1,e2,le) -> ElimCdep(substrec e1, substrec e2,
                                       map substrec le)
      | ElimCnodep(e1,e2)->  ElimCnodep(substrec e1,substrec e2)
      | ElimC(e1,e2)     ->  ElimC(substrec e1,substrec e2)
      | ConstrC(i,e)     ->  ConstrC(i,substrec e)
      | IndC(stamp,n,e,le)     ->  IndC(stamp,n,substrec e, if n=name then le
                                               else map substrec le)
      | _                -> c
      else c;;

let structural elim stamp_tok labsc recname arityc lname_command strength =
   (save_light (); 
    try build (); forget_light ()
    with reraise -> restore_light (); raise(reraise))
   where build () =
    open_section "inductive";
    do_list (fun (na,c) -> construct c; assume na (read_sec())) labsc;
    construct arityc;
    assume recname (read_sec());
    do_list (function (na,c) -> construct (substC_list labsc recname c);
                                assume na (read_sec()))
            lname_command;
    recursive stamp_tok elim (length lname_command) strength;
    toplevel_message ((string_of_id recname) ^ " is inductively defined");
    close_section "inductive" false;;

let structural_elim = structural true
and structural_noelim = structural false;;

let generative elim labsc recname arityc lname_command strength =
   (save_light (); 
    try build (); forget_light ()
    with reraise -> restore_light (); raise(reraise))
   where build () =
    open_section "inductive";
    do_list (fun (na,c) -> construct c; assume na (read_sec())) labsc;
    construct arityc;
    assume recname (read_sec());
    do_list (function (na,c) -> construct (substC_list labsc recname c);
                                assume na (read_sec()))
            lname_command;
    recursive (Name recname) elim (length lname_command) strength;
    toplevel_message ((string_of_id recname) ^ " is inductively defined");
    close_section "inductive" false;;

let generative_elim = generative true
and generative_noelim = generative false;;

(* 5| Dropping back to the meta-language *)

(* exception Drop;; *)
let drop () = raise Drop;;(* Back to CAML *)
