(*****************************************************************************)
(*                                                                           *)
(*        Miscellaneous Functions for a Coq - Centaur interface              *)
(*                             Yves Bertot                                   *)
(*                                                                           *)
(*****************************************************************************)
#open "std";;
#open "initial";;
#open "term";;
#open "tactics";;
#open "pp_control";;
#open "printer";;
#open "vernac";;
#open "pp";;
#open "stdpp";;
#open "search";;
#open "machine";;
#open "pretty";;
#open "coqtoplevel";;
#open "more_util";;

let Cmessage x = PPNL [< prs x; line(); 'S "End message">];;

let Cgoal command = goal_default command; show();;

let Cadd_unfold l = add_unfold_default l; PPNL [< 'S "Hint succeeded">];;

let Cadd_resolution l = add_resolution_default l ; PPNL [< 'S "Hint succeeded">];;

let Cadd_rewrite b l = add_rewrite_default b l ; PPNL [< 'S "Hint succeeded">];;

let Cadd_trivial_list l = add_trivial_list_default l;
                          PPNL [< 'S "Hint succeeded">];;

let Cerase_tacs l = erase_tacs_default l; PPNL [< 'S "Erase succeeded" >];;
let Cerase_tacs_rewrite b l = erase_tacs_rewrite_default b l;
                 PPNL [< 'S "Erase succeeded" >];;

let Cprint_search () = 
    PPNL [< 'S "B-E-G-I-N---H-I-N-T---L-I-S-T::>"; line() >];
    (do_vect print_namedtac_list SEARCH
      where print_namedtac_list = do_list print_namedtac
      where print_namedtac = function
       (name_head,ntlist) -> PPNL [< prs ("For "^name_head^" -> ");
                              prlist (function (name_tac,(pr,_))->
                                      [< 'S"("; prs (name_tac^"),");'INT pr;
                                       prs" " >]) ntlist >]);
                       PPNL [< 'S "E-N-D---H-I-N-T---L-I-S-T" >];;

let rec string_of_loc = function
     [] -> "#"
   | [a] -> (string_of_int a)
   | a::tail -> (string_of_int a) ^ "." ^ (string_of_loc tail);;

let rec string_of_loc = function
     [] -> "#"
   | [a] -> (string_of_int a)
   | a::tail -> (string_of_int a) ^ "." ^ (string_of_loc tail);;


let Cpr_goal (loc,sign,c) = 
 [< prs (string_of_loc loc); line()>];;

let Cpr_subgoals gls =
  let rec pr_rec = 
    function []      -> [< >] 
               | g::rest -> [< Cpr_goal g; line(); pr_rec rest>]
  in [< prs "S-T-A-T-E"; line();
     prs ":>" ; line();
     prs (string_of_int (current_goal())); line();
   prs ":" ; line(); pr_rec gls; prs "E-N-D---S-T-A-T-E">];;

let Cprint_state () = PPNL([< Cpr_subgoals (list_pf (get_goals())); line()>]);;

let Cpr_specific_goal (loc,sign,c) = 
 let rec pr_rec_sign = function
     []      -> [< >]
      |  u::rest -> [< print_decl u; line(); pr_rec_sign rest >]
 in [< prs "S-P-E-C-I-F-I-C---S-U-B-G-O-A-L"; line();
    prs ":<"; line();
    prs (string_of_loc loc);line();
    prs ":";line();
    pr_rec_sign sign; line(); prs "======";
    line(); prterm c; line();
    prs "E-N-D---S-P-E-C-F-I-C"; line()>];;

let Cprint_goal (n:int) = PPNL(Cpr_specific_goal (nth (list_pf (get_goals())) n));;

let Cprint_crible s = try (global s;
                    [< prs "T-H-E-O-R-E-M-S[PREMISES_LIST]"; line();
                    print_crible_extend s;
                    prs "E-N-D---T-H-E-O-R-E-M-S" >])
               with Undeclared -> error ((string_of_id s) ^ " not declared");;

let Cprint_val () =
     [< prs "Checking_Value";
     prs "[TYPED_FORMULA]";line();
     print_val_default();
     prs "End_Checking_Value">];;

let Cprint_name name = try (match (global name) with
        Const(Def(_,Judge(c,t,_),_)) -> 
             [< prs "P-R-I-N-T-I-N-G_V-A-L-U-E";
                prs "[VERNAC]";line();
                prs "Theorem "; prs (string_of_id name) ; prs ".";line(); 
                prs "Statement ";   pr t; prs "."; line();
                prs " Proof " ;pr (simplify c);prs".";line();
                prs " E-N-D_P-R-I-N-T-I-N-G_V-A-L-U-E">]
      | Var(Decl(_,Judge(t,_,_),_)) -> 
             [< prs "Checking_Value";
                prs "[TYPED_FORMULA]";line();
                prs (string_of_id name) ; prs ":";
                prterm t;
                line();
                prs "End_Checking_Value">]
      | _ -> error "print_name")
    with Undeclared -> error ((string_of_id name) ^ " not declared");;


let set_default_functions() = set_print_fun print_state1;
                            set_print_search_fun print_search_default;
                            set_add_trivial_list_fun add_trivial_list_default;
                            set_add_unfold_fun add_unfold_default;
                            set_add_resolution_fun add_resolution_default;
                            set_add_rewrite_fun add_rewrite_default;
                            set_message_fun message_default;
                            set_show_fun show_default;
                            set_print_goal_fun print_goal_default;
                            set_print_crible_fun print_crible_default;
                            set_print_val_fun print_val_default;
                            set_print_name_fun Cprint_name;
                            set_goal_fun goal_default;
                            set_require_fun require_default;;

let reset_functions () = reset_print_fun ();
                            reset_print_search_fun ();
                            reset_add_trivial_list_fun ();
                            reset_add_unfold_fun ();
                            reset_add_resolution_fun ();
                            reset_add_rewrite_fun ();
                            reset_message_fun ();
                            reset_show_fun ();
                            reset_print_goal_fun ();
                            reset_print_crible_fun ();
                            reset_print_val_fun ();
                            reset_print_name_fun ();
                            reset_goal_fun ();
                            reset_require_fun ();;


let Crequire id fname =
    let pl = search_packages () in
        if not (listset__memb pl id) then
        begin
        (try (try set_default_functions();
             set_message_fun Cmessage;
             (message ("(Centaur Flag in) Requiring " ^ fname);
              load_vernacular_from_loadpath fname;
              message ("(Centaur Flag out) Finished requiring " ^ fname)) with
         Failure _ ->
         error ("(Centaur Flag out) Require of package " ^
                 id ^ " from file " ^ fname ^ " failed")) 
         with
         reraise -> message "(Centaur Flag out)"; 
                    reset_message_fun();
                    reset_functions();
                    raise reraise);
        reset_message_fun ();
        reset_functions();
        let pl' = search_packages () in
            if not(listset__memb pl' id) then
                error ("Require of package " ^ id ^ " failed") else
                message ("Acknowledge Require of package " ^ id)
       end
       else
       message ("Acknowledge Require of package " ^ id);;

let set_C_functions () = set_print_fun Cprint_state; 
         set_print_search_fun Cprint_search;
         set_add_trivial_list_fun Cadd_trivial_list;
         set_message_fun Cmessage;
         set_add_unfold_fun Cadd_unfold;
         set_add_rewrite_fun Cadd_rewrite;
         set_add_resolution_fun Cadd_resolution;
         set_erase_tacs_fun Cerase_tacs;
         set_erase_tacs_rewrite_fun Cerase_tacs_rewrite;
         set_goal_fun Cgoal;
         set_show_fun Cprint_state;
         set_print_goal_fun Cprint_goal;
         set_print_crible_fun Cprint_crible;
         set_print_val_fun Cprint_val;
         set_print_name_fun Cprint_name;
         set_require_fun Crequire;;

let Centaur_go () = try set_C_functions ();
         go();
         reset_functions()
     with reraise -> reset_functions(); raise reraise;;


