(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                              caml.ml                                     *)
(****************************************************************************)
(*                                                                          *)
(*             A generator of ML abstract syntax out of FML                 *)
(*                                                                          *)
(****************************************************************************)

#open "std";;
#open "fmlterm";;
#open "fmlenv";;
#open "optimise";;
#open "hashtbl";;
#open "pp_control";;
#open "pp";;
#open "stdpp";;
#open "more_util";;

let max_line_len = 70;;

let list_with_sep elem sep = do_rec
  where rec do_rec = function
    e::l -> [< elem e; (function [] -> [< >] | l -> [< sep(); do_rec l >]) l >]
  | [] -> failwith "caml..list_with_sep"
;;

let Plit = STRING;;
let Pbox(n,s) = [< 'WS n ; 'CUT ; HOV 0 s >];;
let Pbrk = SPC;;
let Pubrk = SPC;;

let ext_to_ML = function
   Num(n) -> [< 'Plit(string_of_int n) >]
 | Plus -> [< 'Plit "(fun x y -> x+y)" >]
 | Minus -> [< 'Plit "(fun x y -> x-y)" >]
 | Times -> [< 'Plit "(fun x y -> x*y)" >]
 | Div ->  [< 'Plit "(fun x y -> x/y)" >]
 | Lesser -> [< 'Plit "(fun x y -> x<y)" >]
 | Equal -> [< 'Plit "(fun x y -> x=y)" >]
 | True -> [< 'Plit "true" >]
 | False -> [< 'Plit "false" >]
 | If -> [< 'Plit "(fun b x y -> if b then x else y)" >]
 | _ -> failwith "ext_to_ML";;

let rec no_lam = function
    FmLambda(_,t)->no_lam t
  | FmInd(_,_) as t -> t
  | _ -> failwith "error: no inductive fml type";;

let CAMLreserved = ["abstype"; "and"; "as"; "at"; "begin"; "case";
                    "continue"; "directive"; "do"; "done"; "dynamic";
                    "else"; "end"; "exception"; "fail"; "failwith";
                    "force"; "freeze"; "from"; "fun"; "function"; "future";
                    "if"; "in"; "it"; "lazy"; "let"; "match"; "mlet"; "mod";
                    "mutable"; "not"; "of"; "or"; "parallel"; "prefix";
                    "protect"; "raise"; "rec"; "reraise"; "segment";
                    "strict"; "tags"; "then"; "try"; "type"; "value";
                    "vector"; "where"; "while"; "with"; "end";
                    "overload"; "forward"; "printer"; "grammar"; "system";
                    "autoload"; "module"];;

let is_CAMLreserved =
  let v = make_vect 80 [] in
  do_list (fun s ->
    let i = (hash s) mod (vect_length v) in
    v.(i) <- s::v.(i)
  ) CAMLreserved;
function s ->
  mem s (v.((hash s) mod (vect_length v)))
;;

let is_constructor s = mem s ["true";"false";"Var";"If"];;

let camlsafe s = if (is_CAMLreserved s) or (is_constructor s)
                 then (s^"_ren") else s;;

type 'a option = None | Some of 'a;;

let occur sc =
  let rec oc = function
      (Fmvar s) -> if s=sc then 1 else 0
    | (Fmapp(t1,t2)) -> (oc t1)+(oc t2)
    | (Fmexcept(t1,t2)) -> (oc t1)+(oc t2)
    | (Fmconstruct (n,s,l)) -> list_it (fun t n->(oc t)+n) l 0
    | (Fmlambda(s,t)) -> if s = sc then 0 else oc t
    | (Fmlocal(s,FmRec t1,t2)) -> if s=sc then 0
                                          else (oc t1)+(oc t2)
    | (Fmlocal(s,t1,t2))-> if s=sc then oc t1
                                   else (oc t1)+(oc t2)
    | (Fmmatch(t,s,pl)) ->
             list_it
                (fun (_,vl,t) n -> if mem sc vl then n
                                                else n+(oc t))
                pl 
               (oc t)
    | (FmRec t) -> oc t
    | _ -> 0
in oc;;

let rec fml_to_ML = function
   Fmlambda(s,t) ->
     begin match
       match t with
         Fmmatch(Fmvar s',_,_) ->
         if s = s' & 1 = (occur s t) then Some t else None
       | _ -> None
     with
       Some(Fmmatch(_,s,pl)) -> [<
           'Plit "function"; Pbox(1,
             list_with_sep (to_MLpat s) (fun _ -> [< 'Pubrk; 'Plit "| " >]) pl
           )
         >]
     | _ -> [<
           'Plit "fun "; 'Plit (camlsafe s); 'Plit " ->";
           Pbox(1, fml_to_ML t)
         >]
     end
 | Fmapp(Fmapp(Fmapp(Fmext(If),b),t1),t2) -> [<
      'Plit "if "; fml_to_ML b; 'Plit " then";
      Pbox(1, fml_to_ML t1); 'Pbrk; 'Plit "else";
      Pbox(1, fml_to_ML t2)
    >]
 | Fmapp((Fmapp(Fmext(op),t1)) as t'1,t2) ->
      if mem_assoc op !LMLextinfix then [<
        fml_to_simpleML t1; 'Plit(assoc op !LMLextinfix);
        fml_to_simpleML t2
      >]
      else  [< fml_to_nofunML t'1; 'Plit " "; fml_to_simpleML t2 >]
 | Fmapp(t1,t2) -> [< fml_to_nofunML t1; 'Plit " "; fml_to_simpleML t2 >]
 | Fmvar(s) -> [< 'Plit (camlsafe s) >]
 | Fmconst(s) -> [< 'Plit (camlsafe s) >]
 | Fmconstruct(n,s,l) ->
      begin match no_lam(assoc s !Fmtypenv) with
        FmInd(_,sl) -> [<
          'Plit (fst (nth sl n)); 'Plit "_C"; constr_args l
        >]
      | _ -> failwith "caml..fml_to_ML (Fmconstruct)"
      end
 | Fmext(E) -> ext_to_ML E
 | Fmerror -> [< 'Plit "fail" >]
 | Fmlocal(s,FmRec(t1),t2) -> [<
      'Plit "let rec "; 'Plit (camlsafe s); 'Plit " =";
      Pbox(1, fml_to_ML t1); 'Plit " in"; 'Pbrk;
      fml_to_ML t2
    >]
 | Fmlocal(s,t1,t2) -> [< 'Plit "(* <:CAML<let #(MLvarpat s) = #(fml_to_ML t1) in
                                            #(fml_to_ML t2)>> *)" >]
 | Fmmatch (t,s,pl) -> [<
      'Plit "match "; fml_to_ML t; 'Plit " with";
      Pbox(1,
        list_with_sep (to_MLpat s) (fun _ -> [< 'Pubrk; 'Plit "| " >]) pl
      )
   >]
 | Fmexcept(t1,t2) ->
[< 'Plit "(*     <:CAML<exception E in
          let rec f x = try #(fml_to_ML t2) (unchecked_coercion x)
                             with E(x)->f x
                                | e -> raise(e)
          in try #(fml_to_ML t1) (fun x -> raise E(unchecked_coercion x))
             with E(x) -> f x
                | e -> raise(e)>> *)" >]
 | _ -> failwith "caml..fml_to_ML"

and fml_to_nofunML = function
  Fmlambda(_,_) | Fmerror | Fmmatch (_,_,_) as x ->
    [< 'Plit "("; fml_to_ML x; 'Plit ")" >]
| x -> fml_to_ML x

and fml_to_simpleML = function
  Fmvar _ | Fmconst _ as x -> fml_to_ML x
| x -> [< 'Plit "("; fml_to_ML x; 'Plit ")" >]

and fst_apply = function 
    (Fmapp(t1,t2)) -> fst_apply t1
  | t -> fml_to_ML t

and rest_apply = function
    (Fmapp(t1,t2)) -> (rest_apply t1)@[fml_to_ML t2]
  | _ -> []

and to_MLpat s (n,sl,t) = 
    match no_lam (assoc s !Fmtypenv) with
      FmInd(_,cl) -> [<
        'Plit (fst (nth cl n)); 'Plit "_C"; constr_pat_args sl; 'Plit " ->";
        Pbox(1, fml_to_nofunML t)
      >]
    | _ -> failwith "caml..to_MLpat"

and constr_pat_args = function
  [] -> [< >]
| [e] -> [< 'Plit " " ; constr_pat_arg e >]
| l -> let rec aux = function
         [e] -> constr_pat_arg e
       | h::t -> [< 'Plit "(" ; constr_pat_arg h;
                    'Plit "," ; aux t ; 'Plit ")" >]
       in aux l

and constr_pat_arg x = [< 'Plit (camlsafe x) >]

and constr_args = function
  [] -> [< >]
| x::[] -> [< 'Plit " ("; fml_to_simpleML x ; 'Plit ")" >]
| l -> let rec aux = function
         [e] -> [< 'Plit "(" ; fml_to_nofunML e ; 'Plit ")" >]
       | h::t -> [< 'Plit "((" ; fml_to_nofunML h;
                    'Plit ")," ; aux t ; 'Plit ")" >]
         in aux l
;;

let to_ML_decl = function 
   (s,FmRec(t)) -> [<
     'Plit "let rec "; 'Plit (camlsafe s); 'Plit " ="; Pbox(1, fml_to_ML t)
   >]
 | (s,t) -> [<
     'Plit "let "; 'Plit (camlsafe s); 'Plit " ="; Pbox(1, fml_to_ML t)
   >];;

let rec fml_to_MLtype  = function
   FmLambda(_,_) -> failwith "intern abstraction in type"
 | FmApp (t1,t2) -> [<
      type_args ((rest_type t1)@[fml_to_MLtype t2]);
      fst_type t1
   >]
 | FmArr (t1,t2) -> [< fml_to_MLtype t1; 'Plit " -> "; fml_to_MLtype t2 >]
 | FmVar(s) -> [< 'Plit "'"; 'Plit s >]
 | FmConst("Int") -> [< 'Plit "int" >]
 | FmConst(s) -> [< 'Plit s >]
 | FmInd(s,cl) -> failwith "internal occurrence of inductive type"
 | FmError s -> failwith ("uninstanciated type "^s)
and fst_type = function
   FmApp(t,_) -> fst_type t
 | FmVar(s) -> [< 'Plit "'"; 'Plit s >]
 | FmConst(s) -> [< 'Plit s >]
 | _ -> failwith "not a constant type in heap position"
and rest_type  = function
    FmApp(t1,t2)->(rest_type t1)@[fml_to_MLtype t2]
 | _ -> []
and type_args = function
  [] -> [< >]
| x::[] -> [< x; 'Plit " " >]
| l -> [<
      'Plit "("; list_with_sep I (fun _ -> [< 'Plit "," >]) l;
      'Plit ") "
    >]
;;

(*
let rec fml_to_MLtype1  = function
   FmLambda(_) -> failwith "intern abstraction in type"
 | FmApp (t1,t2) -> MLconsttyp(fst_type t1,(rest_type t1)@[fml_to_MLtype1 t2])
 | FmArr (t1,t2) ->  MLconsttyp("->",[fml_to_MLtype1 t1;
                                   fml_to_MLtype1 t2])
 | FmVar(s) -> MLvartyp s
 | FmConst("Int") -> MLconsttyp ("int",[])
 | FmConst(s) -> if is_indt (assoc s !Fmtypenv)
                    then MLconsttyp (s,[])
                    else fml_to_MLtype1 (assoc s !Fmtypenv)
 | FmInd(s,cl) -> failwith "internal occurrence of inductive type"
 | FmError s -> failwith ("uninstanciated type "^s)
and fst_type = function
   FmApp(t,_) -> fst_type t
 | FmVar(s) -> s
 | FmConst(s) -> if is_indt (assoc s !Fmtypenv)
                    then s
                    else fst_type (assoc s !Fmtypenv)
 | _ -> failwith "not a constant type in heap position"
and rest_type  = function
   FmApp(t1,t2)->(rest_type t1)@[fml_to_MLtype1 t2]
 | _ -> [];;
*)

let to_MLconstr (s,tl) = match tl with
   [] -> [< 'Plit(s^"_C") >]
 | l -> let rec mkprod = function
                [t] -> fml_to_MLtype t
              | t::l -> [< fml_to_MLtype t;
                           'Plit " * ("; mkprod l ; 'Plit")" >]
              | _ -> failwith "to_MLconstr"
            in [< 'Plit (s^"_C of "); mkprod l >];;


(*
let to_MLconstr1 (s,tl) = match tl with
   [] -> MLconstruct0 (s^"_C")
 | l -> let rec mkprod = (function
                [t] -> fml_to_MLtype1 t
              | t::l -> MLconsttyp("*",[fml_to_MLtype1 t; mkprod l]))
            in MLconstruct (s^"_C",(mkprod l));;
*)

let type_decl_args = function
  [] -> [< >]
| s::[] -> [< 'Plit "'"; 'Plit s; 'Plit " " >]
| l ->
    let type_decl_arg s = [< 'Plit "'"; 'Plit s >] in [<
      'Plit "("; list_with_sep type_decl_arg (fun _ -> [< 'Plit "," >]) l;
      'Plit ") "
    >]
;;

let to_MLtype_decl (s,t) = 
let rec to_decl l = function
  FmLambda(N,t) -> to_decl (N::l) t
| FmInd(s,cl) -> [<
    'Plit "type "; type_decl_args l; 'Plit s; 'Plit " =";
    Pbox(1,
       list_with_sep I (fun _ -> [< 'Pubrk; 'Plit "| " >]) (map to_MLconstr cl)
     )
  >]
| t -> [<
    'Plit "type "; type_decl_args l; 'Plit s; 'Plit " == ";
    fml_to_MLtype t
  >]
       in to_decl [] t;;

(*
let to_MLtype_decl1 (s,t) = 
let rec to_decl l = function
   FmLambda(N,t) -> to_decl (N::l) t
 | FmInd(s,cl) -> MLdecl(MLtype[MLconcrete_type(s,l,map to_MLconstr1 cl)])
 | t -> MLdecl(MLtype[MLabbrev_type(s,l,fml_to_MLtype1 t)])
        in to_decl [] t;;
*)


let make_caml_file s =
    let caml_prog = open_trapping_failure open_out s ".ml" in
    let fp = copy_fp std_fp in
    let gp = copy_gp dflt_gp in
        with_output_to fp caml_prog;
        fully_pp gp;
        PPNL_WITH gp fp
        [< 'S "let fail = obj__magic (fun _ -> failwith \"fail\");;\n";
         prlist (fun x -> [< to_MLtype_decl x;
                           'S ";;";
                           'FNL >])
         (rev !Fmtypenv);
         prlist (fun x -> [< to_ML_decl x;
                           'S ";;";
                           'FNL >])
         (rev (optimal())) >];
        close_out caml_prog;;

let extract_caml _ =
  prerr_endline "<<<extract_caml>>>";
  exit 1
;;

compile__make_caml_file.v <- make_caml_file;;
