(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                              annote.ml                                   *)
(****************************************************************************)
#infix "THEN";;
#infix "THENL";;
#infix "ORELSE";; 
#open "std";;
#open "initial";;
#open "univ";;
#open "termfw";;
#open "term";;
#open "program";;
#open "command";;
#open "tactics";;
#open "machine";; 
#open "print_program";;
#open "trad";;
#open "extraction";;
#open "matching";;

(* let OCCUR = ref ([]:((int list)*fterm) list) ;; *)

let annot_of_com sign c = annot_of_fterm (fterm_of_com sign c);;

let alambda name typ c = Alambda(name,typ,(asubst_var name c)) ;;

let alambdacom name typ c = Alambdacom(name,typ,(asubst_var name c)) ;;

(********************************************************)
(* Extraction of the proof term associated to a program *)
(********************************************************)
let constr_of_annot (_,sign,cl as gl) = crec where rec crec = function
    Avar(Name(n),c)   ->  (try caml_assoc_sign n sign with Failure _ ->
                  (try (val_of n) with Undeclared -> error((string_of_id n)^" not declared(2)")))
  | Aconst(Name(n),c) ->  global n
  | Arel(n)           ->  Rel(n)
  | Aomega            ->  Prop(Pos)  
  | Aapp(t1,t2)       ->  App(crec t1,crec t2)
  | Alambda(x,A,B)    ->  Lambda(x,constr_of_fterm gl A,crec B)
  | Alambdacom(x,A,B) ->  crec B
  | Aprod(x,A,B)      ->  Prod(x,constr_of_fterm gl A,constr_of_fterm gl B)
  | Aind(stamp,A,lc,_,_)    ->  let A' = constr_of_fterm gl A
                          and lc' = map (constr_of_fterm gl) lc in
                            make_ind stamp A' lc'
  | Aconstr(i,ind)    ->  Construct(i,find_ind cl)
  | Arec(P,lf,c)      ->  Rec(false,((constr_of_fterm gl P)::(map crec lf)),crec c)
  | Arecursion(wf,A,P,R)->  App(App(App(constr_of_fterm gl wf,constr_of_fterm gl A),constr_of_fterm gl P),R)
  | Annot(c,P)        ->  crec c
  | _                 ->  anomaly "shouldn't appear(constr_of_annot)" ;;

let constr_of_pure_annot sign = crec [] where rec crec l = function
    Avar(Name(n),c)   ->  let c' = constr_of_pure_fterml l sign c
                          and c'' = try caml_assoc_sign n sign with Failure _ ->
                  (try val_of n with Undeclared -> error((string_of_id n)^" not declared(3)"))
                          in if (conv_x c' (type_of c'')) then
                              c''
                            else
                              raise (NONPUR "not a pure program")
  | Aconst(Name(n),c) ->  let c' = constr_of_pure_fterml l sign c
                          and c'' = global n in
                            if (conv_x c' c'') then
                              c''
                            else
                              raise (NONPUR "not a pure program")
  | Arel(n)           ->  Rel(n)
  | Aomega            ->  Prop(Pos)
  | Aapp(t1,t2)       ->  App(crec l t1,crec l t2)
  | Alambda(x,A,B)    ->  let A' = constr_of_pure_fterml l sign A in
                            Lambda(x,A',crec (A'::l) B)
  | Alambdacom(x,A,B) ->  crec l B
  | Aprod(x,A,B)      ->  let A' = constr_of_pure_fterml l sign A in
                            Prod(x,A',constr_of_pure_fterml (A'::l) sign B)
  | Aind(stamp,A,lc,_,_)    ->  let A' = constr_of_pure_fterml l sign A in
                          let lc' = map (constr_of_pure_fterml (A'::l) sign) lc in
                            make_ind_rel stamp l A' lc'
  | Aconstr(i,ind)    ->  Construct(i,constr_of_pure_fterml l sign ind)
  | Arec(P,lf,c)      ->  Rec(false,((constr_of_pure_fterml l sign P)::(map (crec l) lf)),crec l c)
  | Arecursion(wf,A,P,R)->  App(App(App(constr_of_pure_fterm sign wf,constr_of_pure_fterm sign A),constr_of_pure_fterm sign P),R)
  | Annot(c,P)        ->  crec l c
  | Aimplicit         ->  Implicit 
  | _                 ->  anomaly "shouldn't appear(constr_of_pure_annot)" ;;

let red_const pg = 
let rec redrec pg =
  match pg with
    Aprod(n,A,B)      -> Aprod(n,fterm_of_annot(redrec(annot_of_fterm A)),fterm_of_annot(redrec (annot_of_fterm B)))
  | Aapp(c1,c2)       -> anf(aapplist (redrec c1) [redrec c2])
  | Aind(stamp,c1,c2,c3,c4) -> let c2' = map annot_of_fterm c2 in
                          let c2'' = map redrec c2' in
                            Aind(stamp,c1,map fterm_of_annot c2'',c3,c4)
  | Aconst(_,x)       ->  redrec (annot_of_fterm x)
  | _                 ->  let (x,l) = areduce_app pg in
                            aapplist x l
in let (x,l) = areduce_app pg in
    aapplist (redrec x) (map redrec l);;

(********************************************)
(* Transformation of a command to an annote *)
(********************************************)
let rec annot_of_acom (_,sign,_ as gl) = crec (fsign_of_sign sign) sign 
 where rec crec vl vl'= crecvl where rec
  crecvl = function
    ApurC(c)          -> annot_of_com vl c
  | AappC(c1,c2)      -> Aapp(crecvl c1,crecvl c2)
  | AlambdaC(s,c1,c2) -> let u0 = (annot_of_com vl c1) in
                          let u1 = fterm_of_annot u0 
                          and name = Name(s) in
                            (try let u1' = constr_of_pure_annot vl' u0 in
               alambda name u1 (crec ((name,Fvar(name,u1))::vl) 
                       ((name,(creation_var name u1')):: vl') c2)
                            with
                              NONPUR _ -> 
             (* A non pur program should 
               not be used in an annotation, avoid the cost of executing (red_const u0) :
              let u1' = constr_of_annot gl (red_const u0) in *)
               alambda name u1 (crec ((name,Fvar(name,u1))::vl) 
                               ((name,Implicit):: vl') c2))
  | AlambdacomC(s,c1,c2) -> (let u1 = constr_of_com vl' c1 
                             and name = Name(s) in
               alambdacom name u1 (crec vl ((name,(creation_var name u1))::vl') c2))
  | ArecC(c,P)        -> amake_elimination2 (crecvl c) (annot_of_com vl P)
  | ArecursionC(RefC(wf),A,P,R)-> let R' = constr_of_com vl' R in
     (match (type_of R') with
        Prod(_,B,(Prod(_,C,Prop(_)))) -> let A' = constr_of_com vl' A in
           if (conv_x A' B) & (conv_x B C) then
              Arecursion(fterm_of_ident(val_of wf),fterm_of_com_use vl' A,
                         fterm_of_com_use vl' P,R') 
           else
              error "Ill-typed recursion"
      | _                              -> error "Ill-typed recursion")
  | AnnotC(c,P)       -> let c' = crecvl c and P' = constr_of_com vl' P in
                          (match (type_of P') with
                            Prop(c) -> Annot(c',P')
                          | _       -> error "Ill-typed annotation") ;;


(* List (occurrence,annoted associated program) *)
let OCCUR = ref ([]:((int list)*annote) list) ;;


(* Clear of OCCUR *)
let clear_OCCUR () = OCCUR:=[] ; () ;;


(* Undo of the last operation on OCCUR *)
let undo_OCCUR () = OCCUR:=tl !OCCUR ; () ;;


(***************************************************************)
(* Correspondance between program type and goal extracted type *)
(***************************************************************)
let prog_assoc_old c ((x,sign,cl) as gl) =
  (match x with
    [] -> clear_OCCUR()
   | _ -> ());
  let v = anf (annot_of_acom gl c) in
   try let t = atype_of2 v in
    (match (info_of_constr cl) with
      Inf(n,t')   ->  (if (aconv2 t (annot_of_fterm t')) then 
                            OCCUR:=(x,v)::!OCCUR
                      else  error "types of program and goal incompatibles") ; 
                            IDTAC gl
    | Logic       ->  error "no informative sense for the goal")
   with
    UserError _ -> error "Ill-typed program"  ;;

let prog_assoc c ((x,sign,cl) as gl) =
  (match x with
    [] -> clear_OCCUR()
   | _ -> ());
  let v = anf (annot_of_acom gl c) in
   let t = (try atype_of2 v with UserError _ -> error "Ill-typed program") in
    (match (info_of_constr cl) with
      Inf(n,t')   ->  (if (aconv2 t (annot_of_fterm t')) then 
                            OCCUR:=(x,v)::!OCCUR
                      else  error "types of program and goal incompatibles") ; 
                            IDTAC gl
    | Logic       ->  error "no informative sense for the goal") ;;

(*****************)
(* Introductions *)
(*****************)
let aderoule_intro pg (u,sign,cl as gl) = 
  match pg with
    Alambda(x,A,B)  ->  let (nr,ni) = nb_red_intro 0 0 gl cl in
                          let pf =  (DOTHEN nr red (DO ni intro_global)) gl in
                            (match (list_pf pf) with
                              [(u',(n,(Var(Decl(_,_,Inf(_,v'))) as c))::sign,_)] 
                                    -> OCCUR:=(u',asubst2 (annot_of_fterm v') c B)::!OCCUR ; pf
                            | _     ->  anomaly "shouldn't appear(deroule 1)")
  | _               -> anomaly "shouldn't appear(deroule 2)" ;;

(* Only for non dependent introductions *)
let rec intro_to_dep_log A = introrec where rec introrec (_,_,cl as gl) = 
 match cl with
    Prod(x,A',B) ->  if conv_x A A' then
                       intro_global gl
                      else
                        if is_logical [] A then
                          (intro_global THEN introrec) gl
                        else
                          error "invalid form of goal"
 |  t            -> (red THEN introrec) gl ;;

(* Find the first variable in the signature of a prooftree *)
let find_var pf = match list_pf pf with
                    [(u,(_,c)::_,_)] -> u,c
                  | _     ->  anomaly "shouldn't appear(find_var)";;

(* For annoted lambda *)
let aderoule_intro_com pg (_,_,cl as gl) = 
  match pg with
    Alambdacom(x,A,B)  -> let pf = intro_to_dep_log A gl in
                            let (u,c) = find_var pf in
                               OCCUR:=(u,asubstconstr1 c B)::!OCCUR ; pf
  | _               -> anomaly "shouldn't appear(deroule 3)" ;;


(*********************************************)
(* Trivial between a fterm and a proposition *)
(*********************************************)
let atrivial_prop arg prop gl = 
  let arg' = constr_of_annot gl arg in
      try let t = type_of arg' in
            if (conv_x t prop) then
              arg'
            else
              raise (NONTRIVIAL "appliquer une autre methode")
      with
        UserError _  -> raise (NONTRIVIAL "appliquer une autre methode") ;;

(* Arguments list of an application *)
let alist_arg = 
  let rec lrec la = function
    Aapp(c1,c2) ->  lrec (c2::la) c1
  | Alambdacom(x,A,B) -> lrec la B
  | Annot(c,P)  ->  lrec la c
  | x           ->  (x,la) 
  in lrec [] ;;


(* Total reduction of products of an annote *)
let rec ared_all = function
    Aprod(s,c1,c2)  ->  Aprod(s,c1,fred_all c2)
  | t               ->  let (x,l) = areduce t in
                          match x with
                            Aprod(_,_,_)  ->  ared_all x
                          | _             ->  aapplist x l ;;


let acont m lmeta gl t = function
  Prod(x,A',B') -> if dependent 1 B' then
                    let t2' = constr_of_annot gl t in
                      (try let T = type_of t2' 
                        in m,subst1 t2' B',lmeta
                    with UserError _ -> m+1,subst1 (Meta([m])) B',(([m],info_of_constr_meta lmeta A')::lmeta))
                   else
                    m,B',lmeta
| _             -> anomaly "shouldn't appear (acont)" ;;


(* try to match P' with c ignoring the non-informative hypothesis *)

let try_matching_concl subst P' = mrec where rec mrec c =
   try (if occur_meta c then raise NONSPECIF else try_matching subst P' c)
   with NONSPECIF  -> 
      (match c with Prod(_,t,c') -> if is_logical [] t 
                                    then 
                                      if closed c' then
                                        mrec c' 
                                      else
                                        raise NONSPECIF
                                    else raise NONSPECIF
                  | _ -> raise NONSPECIF);;
   


(* Find the specification of a term if available *)
let specif_of_annot (_,sign,cl as gl) T = 
 let (n1,P,lmeta) = specrec 1 [] T where rec specrec n lmeta t =
  (match t with
    Avar(Name(x),_)   ->  n,type_of(try caml_assoc_sign x sign with 
                              Failure _ -> (try (val_of x) with 
                                Undeclared -> error((string_of_id x)^" not declared"))),lmeta
  | Aconst(Name(x),_) ->  n,type_of (global x),lmeta
  | Annot(c,P)        ->  0,P,[]
  | Aapp(t1,t2)       -> let (k,T1,lmeta') = specrec n lmeta t1 in
                          let (c,l,lmeta'') = go_to_info k lmeta' T1 in
                        (match c with
                          Prod(x,A,B) -> if occur_meta A then
                                          let (m,A',_) = specrec l [] t2 in
                                            let subst = try_matching_concl [] A A' in
                                              acont m lmeta'' gl t2 (instance subst c)
                                         else
                                          acont l lmeta'' gl t2 c
                        | _           ->  anomaly "shouldn't appear (specif_of_annot)")
  | _                 ->  raise NONSPECIF)
  in P ;;


(***************************)
(* Exploration of products *)
(***************************)
let rec is_arity P = match P with
    Prop(_)     -> true
  | Prod(x,A,B) -> is_arity (subst1 A B)
  | _           -> false ;;

let aexplore gl Hl La subst = exrec Hl La subst []
  where rec exrec hl la subst cor = 
   match (hl,la) with
    (n,P)::hl',a::la' -> 
          let P' = instance subst P in
            if (is_logical Hl P') then
              exrec hl' la subst cor
            else
              if (is_arity P') then
                exrec hl' la' subst ((n,a)::cor)
              else
              let subst' = if occur_meta P' then 
                    (try let c = specif_of_annot gl a in 
                         try_matching_concl subst P' c 
                    with NONSPECIF -> subst) else subst in
              let P'' = instance (map (fun(x,y)->(x,constr_of_annot gl y)) cor)
                       (instance subst' P')
              in if occur_meta P'' then exrec hl' la' subst' ((n,a)::cor)
                 else (try let c' = atrivial_prop a P'' gl in
                                    exrec hl' la' ((n,c')::subst') cor
                               with
                                NONTRIVIAL _ ->
                                    exrec hl' la' subst' ((n,a)::cor))
  | _,_               ->  (subst,cor) ;;

(**************************************************)
(* Resolution Tactics with a partial substotution *)
(**************************************************)
(* First order matching *)
let arefine_subst subst lpg c t (n,sign,c1 as gl) =
  let (cl,hl) = forme_clausale n t
  in let (subst',cor) = aexplore gl hl lpg subst
  in  let subst'' = matching subst' cl c1 in
    do_list (function (oo,pr) -> 
                let P = instance subst'' (caml_assoc oo hl) in
                  if occur_meta P then error "Open subgoal generated"
                  else
                    if (aconv2 (atype_of2 pr) (annot_of_fterm (fterm_of(info_of_constr P)))) then
                      OCCUR:=(oo,pr)::!OCCUR
                    else
                      error "Non compatible types") cor;
    inst_pf subst'' (APP(PF(c)::(map_sign sign hl)));;


(* Second order matching *)
let arefinew_subst subst lpg c t (n,sign,c1 as gl) =
  let (cl,hl) = forme_clausale n t
  in let (subst',cor) = aexplore gl hl lpg subst
  in let cl' = instance subst' cl 
  in  let subst'' =  match decomp_app cl' with
                  (Meta(p),oplist)  -> 
                        (try matching subst' cl' c1
                         with
                          UserError _ ->
                            let typp = instance subst' (caml_assoc p hl) 
                              in add_matching subst' 
                                          (p,abstract_list_all typp c1 oplist))
                | _                 -> anomaly "shouldn't appear(refinew)"
 in do_list (function (oo,pr) -> 
                let P = instance subst'' (caml_assoc oo hl) in
                  if occur_meta P then error "Open subgoal generated"
                  else
                    if (aconv2 (atype_of2 pr) (annot_of_fterm(fterm_of(info_of_constr P)))) then
                      OCCUR:=(oo,pr)::!OCCUR
                    else
                      error "Non compatible types") cor;
    inst_pf subst'' (APP(PF(c)::(map_sign sign hl)));;

let arefinew_subst_intros subst lpg c t (n,sign,c1 as gl) =
  let (cl,hl) = forme_clausale n t
  in let (subst',cor) = aexplore gl hl lpg subst
  in  match decomp_app (instance subst' cl) with
                  (Meta(p),oplist)  ->
  let ni = nb_intros_liees_list oplist c1 in
  DOTHEN ni intro_global (arefinew_subst subst lpg c t) gl
                | _ -> (intros_global THEN arefine_subst subst lpg c t) gl;;

let refine_type_list lpg t (n,sign,c1 as gl) = 
    (* check that t is well formed *)
    match hnftype(type_of t) with
    (Prop(_)|Type(_,_)) ->
        let (cl,hl) = forme_clausale n (hnftype t)
        in let (subst,cor) = aexplore gl hl lpg []
        in let subst' = matching subst cl c1 in
          do_list (function (oo,pr) -> 
                let P = instance subst' (caml_assoc oo hl) in
                  if occur_meta P then error "Open subgoal generated"
                  else
                    if (aconv2 (atype_of2 pr) (annot_of_fterm (fterm_of(info_of_constr P)))) then
                      OCCUR:=(oo,pr)::!OCCUR
                    else
                      error "Non compatible types") cor;
         inst_pf subst' (APP(INCOMPLET(0::n,sign,t)::(map_sign sign hl)))
  | _ -> error "Not a type" ;;


(******************************)
(* Association (subgoals,pgs) *)
(******************************)
let rec assoc_OCCUR_old = function
    (u,sign,cl)::l,(f::lf as LF)  ->  
         let t = atype_of f  
   in (match info_of_constr cl with
        Inf(_,t')  -> (if (aconv2 t (annot_of_fterm t')) then OCCUR:= (u,f):: !OCCUR
                      else anomaly "types of program and goal incompatibles");
                     assoc_OCCUR_old (l,lf)
     | Logic       -> assoc_OCCUR_old (l,LF))
  | _,[] -> ()
  | [],_ -> anomaly"not enough subgoals for programs" ;;

let rec assoc_OCCUR = function
    (u,sign,cl)::l,(f::lf as LF)  ->  OCCUR:= (u,f):: !OCCUR;
                                      assoc_OCCUR (l,lf)
  | _,[] -> ()
  | [],_ -> anomaly"not enough subgoals for programs" ;;

(* larsdep and largsnodep construction for elim-with *)
let aconstr_largs sign l prod = crec [] [] [] (l,prod)
  where rec crec largsdep largsnodep lrel = function
    [],_                      -> (largsdep,largsnodep)
  |(f::lf as LF), Prod(x,A,B) ->  
          if is_logic (info_of_constr_rel lrel A) then 
            let (cA,lA) = contents_level_of_arity 
                            (type_wfconstr (map fst3 lrel) A) in
              crec largsdep largsnodep ((A,cA,lA)::lrel) (LF,B)
          else if closed B then 
            let (cA,lA) = contents_level_of_arity 
                            (type_wfconstr (map fst3 lrel) A) in
              crec largsdep (f::largsnodep) ((A,cA,lA)::lrel) (lf,B)
          else let p = constr_of_pure_annot sign f in
                crec (p::largsdep) largsnodep lrel (lf,subst1 p B)
  | _           ->  anomaly "shouldn't appear(constr_largs)" ;;


(* P transformed to b=b->P *)
let elim_pattern c gl =
  let prop = App(App(App(global (id_of_string "eq"),type_of c),c),c)
  in ((cut_tac prop) THENL trivial THEN (pattern_list [[2],c])
                     THEN (tactics__elimination [] c)) gl ;;

let do_intro_elim_patterna ni c lf gl =
  let pf = (DOTHEN ni intro_global (elim_pattern c)) gl in
            assoc_OCCUR (list_pf pf,lf) ; pf ;;


(* Elim-with *)
let do_intro_elima ni c ldep subpgl gl =
  let pf = (DOTHEN ni intro_global (tactics__elimination ldep c)) gl in
    assoc_OCCUR (list_pf pf,subpgl) ; pf ;;


(* Elim_type *)
let do_elim_type ni P lf gl =
  let pf = (DOTHEN ni intro_global (tactics__elimination_type P)) gl in
    assoc_OCCUR (list_pf pf,lf) ; pf ;;

let do_elim_type_pg ni P lf pg gl =
  let pf = (DOTHEN ni intro_global (tactics__elimination_type P)) gl in
    assoc_OCCUR (list_pf pf,lf@pg) ; pf ;;

let rec cut_prod = function
    Prod(x,A,B) -> cut_prod B
  | t           -> t;;

(*****************************)
(* When the program is a Rec *)
(*****************************)
let aelim_app lf pg (_,sign,cl as gl) = 
 if is_annot pg then
  try let c = constr_of_pure_annot sign pg
  in if dependant c cl then
      let ni = nb_intros_liees c cl in 
        do_intro_elima ni c [] lf gl
     else 
      let ni = nb_intros cl in
        do_intro_elim_patterna ni c lf gl
  with
    NONPUR _ -> let ni = nb_intros cl 
                and (hdt,l) = alist_arg pg in 
                  let c' = constr_of_annot gl hdt in
                    let t = red_all (type_of c') in
                     (try let (largsdep,largsnodep) = aconstr_largs sign l t in
                        do_intro_elima ni c' (rev largsdep) (lf@largsnodep) gl
                       with NONPUR _ -> (try let P = specif_of_annot gl pg in
                                              let P' = cut_prod P in 
                                               do_elim_type_pg  ni P' lf [pg] gl
                                         with NONSPECIF -> aprterm pg; error"insufficient specification of this part of program."))
 else
  let ni = nb_intros cl in
    (try let P = specif_of_annot gl pg in
          let P' = cut_prod P in
            let pg' = remove_annote pg in
              do_elim_type_pg ni P' lf [pg'] gl
     with NONSPECIF -> aprterm pg; error"insufficient specification of this part of program.") ;;

let rec is_match_rec lf l gl = 
  match (lf,l) with
    (f::lf',(S::l')) ->
      let rec imrec n S f =
      (match (S,f) with
        (Fprod(x,A,B),Lambda(x',A',B')) -> 
            if foccurn n A then
              imrec (n+1) B B'
            else
              (match B' with
                Lambda(x'',A'',B'') -> if noccurn 1 B'' then
                                          imrec (n+1) B B''
                                        else
                                          true
              | _                    -> false)
      | _                                 -> false)
   in let b = imrec 1 S f in if b then true else is_match_rec lf' l' gl
  | ([],[])     -> false
  | _           -> error "non coherent match";;

let rec search_nb_lam l=
  match l with
    S::l' -> 
      let rec mrec n S =
      (match S with
        Prod(x,A,B) -> if noccurn n A then
                        mrec (n+1) B
                       else
                        mrec (n+2) B
      | _           -> n-1)
   in (mrec 1 S)::search_nb_lam l' 
  | []    -> [];;

let rec change_lam_at n f name x = 
  if (n=0) then Alambda(name,x,asubst_var name (alift 1 f))
  else (match f with
          Alambda(name',A,B) -> Alambda(name',A,change_lam_at (n-1) B name x)
        | _                  -> anomaly "shouldn't appear (change_lam_at)");;

let rec modif_lf name ln lf x =
  match (lf,ln) with
    (f::lf',n::ln') -> (change_lam_at n f name x)::modif_lf name ln' lf' x
  | ([],[])         -> [] ;;

let occurs_in_annot s a = listset__memb (fglobals (fterm_of_annot a)) s;;

let full_generalize c lf (_,sign,_ as gl) =
  let rec fullrec s l =
      match s with
        (Name(s'') as name,x)::s' -> 
              if (dependant c (type_of x)) then
                      (if (is_logic (fst (extract [] [] x))) then
                        (x::(fst (fullrec s' l))), l
                      else 
                        (if (exists (occurs_in_annot s'') lf) then
                          let ((_,_,L,_,_),_) = find_inductype(type_of c) in
                          let n = search_nb_lam L in
                          (match (info_of_constr x) with
                            Inf(_,inf) -> let t = ftype_of(inf) in
                                            let l' = modif_lf name n lf t in 
                                              (x::(fst (fullrec s' l'))), l'
                          | _          -> anomaly "shoudn't appear(full_gen)")
                         else
                          (fst (fullrec s' l)), l))
               else
                     (fst (fullrec s' l)), l
      | []        -> [],l
    in fullrec sign lf ;;

let generalize_rec pg gl =
  match pg with
    Arec(_,lf',c') ->
      (match (constr_of_fterm gl (fterm_of_annot pg)) with
        Rec(_,P::lf,c as pg') -> 
          let ((_,_,l,_,_),_) = find_ainductype(atype_of2 c') in 
            if not(is_match_rec lf l gl) then
              (match c with
                Var(x)  -> let (l',lf'') = full_generalize c lf' gl in
                              lf'',c,l'
              | _       -> lf',c,[])
            else 
              lf',c,[]
      | _                    -> anomaly "shoudn't appear(gen_rec 1)")
  | _             -> anomaly "shoudn't appear(gen_rec 2)";;

let generalize_rec_tac lf c c' l gl = 
    match c' with
      Var(x)  -> ((tactics__generalize_tac l) THEN (aelim_app lf c)) gl 
    | _       -> aelim_app lf c gl;;

let aprogram_rec pg gl = 
  match pg with
    Arec(P,lf,c) -> let (lf',c',l) = generalize_rec pg gl in
                      generalize_rec_tac lf' c c' l gl 
  | _            -> anomaly "shouldn't appear (aprogram_rec)" ;;


(**********************************************)
(* When the program is a Rec in an applicaion *)
(**********************************************)
let atactic_gen t (u,sign,cl as gl) =
  match t with
    Aapp(t1,t2) -> 
    (try let c = constr_of_pure_annot sign t2 in
          let T = type_of c in
            let B_gen = prod_name (Name(id_of_string"x"),T,subst_term c cl) in
              let gl' = (0::u,sign,B_gen) in
                assoc_OCCUR ([gl'],[t1]) ;
                APP [INCOMPLET(gl');PF(c)]
    with
      NONPUR _ -> (try let A = specif_of_annot gl t2 in
                        let gl1 = (0::u,sign,arrow A cl)
                        and gl2 = (1::u,sign,A) in
                          assoc_OCCUR([gl1;gl2],[t1;t2]) ;
                          APP [INCOMPLET(gl1);INCOMPLET(gl2)] 
                  with 
                    NONSPECIF -> aprterm t2; error "insufficient specification of this part of program"))
  | _          -> anomaly "shoudn't appear(atactic_gen)" ;;


(**********************************************)
(* When the program is the application of Wfi *) 
(**********************************************)
let nb_info_prod t = nrec 0 t where rec nrec n T = match T with 
  Prod(x,A,B) -> if is_logic(fst(extract [] [] A)) or (is_arity A) then
                    nrec n (subst1 (creation_var x A) B)
                 else
                    nrec (n+1) (subst1 (creation_var x A) B)
| _           -> n ;;

let rec apply_wfi pg (u,sign,c1 as gl) = 
  let (hdt,l) = alist_arg pg in
  match hdt with
    Arecursion(wf,A,P,R)  -> let c = constr_of_fterm gl wf in
     let subst = [(2::u),R] in
      let (b,t) = red_all_lie (type_of c) in
        let n = nb_info_prod t in
          if (n<(length l)) then 
              (match pg with
                Aapp(t1,t2) -> (atactic_gen pg THEN apply_wfi t1) gl
              | _           -> anomaly "shouldn't appear(apply_wfi)")
          else
            let l' = [annot_of_fterm A;annot_of_fterm P] in
              (try (intros_global THEN arefine_subst subst (l'@l) c t) gl
                 with UserError _ -> arefinew_subst_intros subst (l'@l) c t gl)
  | _                     ->  anomaly "shouldn't appear(apply_wfi)" ;;


(**************************************)
(* When the program is an application *) 
(**************************************)
let alist_arg_annot = 
  let rec lrec la = function
    Aapp(c1,c2) ->  lrec (c2::la) c1
  | Alambdacom(x,A,B) -> lrec la B
  | x           ->  (x,la) 
  in lrec [] ;;

let aprogram_app pg (u,sign,c1 as gl) = 
  let (hdt,l) = alist_arg_annot pg in
  match hdt with
    Arec(P,lf,c)  -> atactic_gen pg gl
  | Arecursion(c) -> apply_wfi pg gl
  | Annot(c,P)    -> let pf = refine_type_list l P gl in
                        assoc_OCCUR(list_pf pf,[c]) ; pf
  | _             -> let c = constr_of_annot gl hdt in
      let (b,t) = red_all_lie (type_of c) in
       if b then arefinew_subst_intros [] l c t gl
           else  (change_pf (red_all c1) THEN intros_global
                  THEN arefine_subst [] l c t) gl ;;

(* Number of informative arguments for a products list *)
let anb_info_arg = 
  let rec nrec ni = function
    Aprod(x,A,B)  ->  nrec (ni+1) (annot_of_fterm B)
  | _             ->  ni 
in nrec 0 ;;


(* One reduction step on an annotation *)
let aone_step_reduce x = anf (redrec [] x)
where rec redrec largs x = match x with
    Alambda(n,t,c)  ->  (match largs with
                            []      -> error "Not reducible 1"
                         |  a::rest -> aapplist (asubst1 a c) rest)
  | Alambdacom(n,t,c) -> Alambdacom(n,t,redrec largs c)
  | Aapp(c1,c2)     ->  redrec (c2::largs) c1
  | Aconst(_,x)     ->  aapplist (annot_of_fterm x) largs
  | Arec(c)         ->  (try aapplist (ared_elimination c) largs
                          with Felim -> error "Not reducible 2")
  | Annot(c,P)      ->  Annot(redrec largs c,P)
  | _               ->  error "Not reducible 3" ;;


(****************)
(* Basic Tactic *)
(****************)

let (GO: tactic -> tactic) = GO where rec GO = fun T g ->
   ((T THEN (GO T)) ORELSE auto) g ;;

let rec apply t l =
  match l with
    a::l' -> apply (anf(Aapp(t,a))) l'
  | _     -> t ;;

let construct_abs ind pg = 
  match (ind,pg) with
    (Find(_,A,l,_,_),Arec(P,lf,c)) ->
     let rec crec l i =
      (match l with
        a::l' -> let rec drec n a p = (match a with
                  Fprod(x,B,C) -> if foccurn n B then
                            let name = Name(IDENT("x",-1)) in
                          (Flambda(name,B,(drec (n+1) C (Arec(P,lf,Arel(n))))))
                                  else
                           let name1 = Name(IDENT("H",-1)) in
                           let name2 = Name(IDENT("H",-1)) in   
                          (Flambda(name1,(fsubst1 ind B),(Flambda(name2,(fsubst1 ind a),(drec (n+1) C (Arec(P,lf,Arel(n+1))))))))
                | _            -> fterm_of_annot p)
                 in (drec 1 a pg)::(crec l' (i+1))
     | []      ->  [])
    in crec l 1
  | _              -> anomaly "shoudn't appear(construct_abs)" ;;

let modif_rec pg = match pg with
    Arec(P,lf,c) -> let t = atype_of2 c in
                    let (ind,_) = find_ainductype t in 
                      let lc = map annot_of_fterm (construct_abs (Find(ind)) pg) in
                        apply (amake_elimination c (annot_of_fterm P)) lc
  | _            -> anomaly "shouldn't appear(modif_rec)" ;;

let rec modif_lam A t =
   (match A,t with
      Fprod(s,c1,c2),Fprod(s',c1',c2')  -> let c3 = modif_lam c2 c2' in
                                            Fprod(s,c1,c3)
    | B,t -> (try let ((_,_,l,_,_),_) = find_finductype t in
                    (match l with
                      [Fprod(_,C,Frel(_))] -> if (fconv B C) then t else A
                    | _                                  -> A)
              with
                Finduc -> A));;

let construct_abs_constr ind = 
  match ind with
    Find(_,A,l,_,_) ->
     let rec crec l i =
      (match l with
        a::l' -> let rec drec n a constr = (match a with
                  Fprod(x,B,C) -> if foccurn n B then
                          (Flambda(Name(id_of_string"x"),B,(drec (n+1) C (Fapp(constr,Frel(n))))))
                                  else
                           let name = Name(IDENT("H",-1)) in
                             (Flambda(Name(id_of_string"x"),(fsubst1 ind B),(Flambda(name,(fsubst1 ind a),(drec (n+1) C (Fapp(constr,Frel(n+1))))))))
                | _            -> constr)
                 in (drec 1 a (Fconstr(i,ind)))::(crec l' (i+1))
     | []      ->  [])
    in crec l 1
  | _              -> anomaly "shoudn't appear(construct_abs_constr)" ;;

exception REC;;

let is_rec term = let rec recrec = function
    Aapp(c1,c2)          ->  recrec c1 
  | Arecursion(wf,A,P,R) ->  raise REC
  | x                    ->  ()
in try (recrec term ; false) with REC -> true ;;

let rec change_rec pg t = match pg with
    Aapp(c1,c2)          ->  Aapp(change_rec c1 t,c2)
  | Arecursion(wf,A,P,R) ->  Arecursion(wf,A,t,R)
  | _                    ->  anomaly "shouldn't appear(change_rec)";;

let modif_arg pg l = 
 let rec mrec l1 l2 n pg l =
  match (pg,l) with
   (Aapp(t1,t2),A::l') ->     
      let t = atype_of2 t2 in
     if not(aconv A t) then
          let (ind,_) = find_ainductype t in
            let lc = (map annot_of_fterm (construct_abs_constr(Find(ind)))) in
          let p = apply (amake_elimination t2 t) lc in
                  mrec (p::l1) (A::l2) (n+1) t1 l'
             else
                  mrec l1 l2 n t1 l'
 | _ -> l1,l2,n
in mrec [] [] 0 pg l;;

let rec modif_bad_arg pg l = 
  match (pg,l) with
   (Aapp(t1,t2),A::l') ->     
      (try let t = atype_of2 t2 in
          Aapp(modif_bad_arg t1 l',t2)
       with
        UserError _ -> 
          (match t2 with
             Arec(P,lf,c) -> Aapp(t1,modif_rec t2)
           | _            -> anomaly "shouldn't appear(modif_arg)"))
 | _ -> pg ;;

let rec new_app n l pg = match (pg,l) with
   (Aapp(t1,t2),A::l') -> 
     let t = atype_of2 t2 in 
      if not(aconv A t) then     
        Aapp((new_app (n-1) l' t1),Arel(n))
      else
        Aapp((new_app n l' t1),t2)
 | (t',_)           -> t' ;;

let rec imbrique_rec app = function
  [Arec(P,[Alambda(x,A,_)],c)]    -> Arec(P,[Alambda(x,A,app)],c)
| (Arec(P,[Alambda(x,A,_)],c))::l -> Arec(P,[Alambda(x,A,(imbrique_rec app l))],c)
| _                               -> anomaly "shouldn't appear(imbr_rec)";;

let construct_lf n P = crec 1 where rec crec m = 
  if (n=m) then [Aconstr(n,P)] else Aconstr(m,P)::crec (m+1) ;;

let modif_app_if pg t (_,sign,cl as gl) = 
try let x = constr_of_pure_annot sign pg in pg
with
  NONPUR _ -> 
 let P = atype_of2 pg in
 (try 
  match pg with
    Aconst(c,Fconstr(i,ind)) -> pg
  | Aconstr(i,ind)           -> pg
  | _                        -> 
  if (fconv t (fterm_of_annot P)) & not(conv_x cl (specif_of_annot gl pg)) then
   let (ind,_) = find_ainductype P in 
       let lf = construct_abs_constr (Find(ind)) in
         Arec((fterm_of_annot P),map annot_of_fterm lf,pg) 
  else
    pg
 with 
  NONSPECIF -> pg) ;;

(*let modif_app_rel lrel pg t (_,_,cl as gl) = 
  let A = fterm_of_annot(atype_constr lrel pg) in
    (try let (((_,_,l,_,_) as ind),_) = find_finductype t in
        let fi = Find(ind) in
          (match l with
                  [Fprod(x,B,Frel(_))] -> 
                    if (fconv A B) then 
              if (is_rec pg) then 
                change_rec pg t
              else
                Aapp(Aconstr(1,fi),pg)
                    else pg
                | _                    -> pg)
    with  Finduc -> pg);;
*)
let construct_arg ind pg = 
  match ind with
    Find(_,A,l,_,_) ->
     let rec crec l i =
      (match l with
        a::l' -> let rec drec n a p = (match a with
                  Fprod(x,B,C) -> if foccurn n B then
                          (Flambda(Name(id_of_string"x"),B,(drec (n+1) C (Arel(n)))))
                                  else
                           let name = Name(IDENT("H",-1)) in
                             (Flambda(Name(id_of_string"x"),(fsubst1 ind B),(Flambda(name,(fsubst1 ind a),(drec (n+1) C (Arel(n+1)))))))
                | _            -> fterm_of_annot p)
                 in (drec 1 a pg)::(crec l' (i+1))
     | []      ->  [])
    in crec l 1
  | _              -> anomaly "shoudn't appear(construct_arg)" ;;

let modif_var lrel pg t = 
  let A = fterm_of_annot(atype_constr lrel pg) in
    try let (((_,_,l,_,_) as ind),_) = find_finductype A in
      (match l with
        [Fprod(x,B,Frel(_))] -> 
          if (fconv B t) then
            let lc = map annot_of_fterm (construct_arg (Find(ind)) pg) in
              apply (amake_elimination pg (annot_of_fterm t)) lc
          else
              pg
      | _                    -> pg)
     with
      Finduc -> pg ;;

let modif_app_rel lrel pg t = 
  let A = fterm_of_annot(atype_constr lrel pg) in
    (try let (((_,_,l,_,_) as ind),_) = find_finductype t in
        let fi = Find(ind) in
          (match l with
                  [Fprod(x,B,Frel(_))] -> 
                    if (fconv A B) then 
              if (is_rec pg) then 
                change_rec pg t
              else
                Aapp(Aconstr(1,fi),pg)
                    else modif_var lrel pg t
                | _                    -> modif_var lrel pg t)
    with  Finduc -> modif_var lrel pg t);;

let modif_bad_app pg t l2 =
  modif_app_rel l2 pg t ;;

let rec type_list pg = 
  match pg with
    Aapp(t1,t2) -> type_list t1
  | t           -> let rec trec t =
                      (match t with
                         Aprod(x,A,B) -> (annot_of_fterm A)::(trec (annot_of_fterm (fsubst1 (Fvar(x,A)) B)))
                       | _            -> [])
                   in trec (atype_of2 t) ;; 

let rec modif_list_rec t = function
    Arec(P,lf,c)::l -> Arec(t,lf,c)::(modif_list_rec t l)
  | []              -> []
  | _               -> anomaly "shouldn't appear(modif_list_rec)";;

let modif_app pg t = 
try modif_app_rel [] pg t
  with
    UserError _ ->  let l = type_list pg in
                      (try let (l1,l2,n) = modif_arg pg (rev l) in
                            let app = new_app n (rev l) pg in 
                             let app' = modif_bad_app app t l2 in
                              let l1' = if not(eq(fterm_of_annot(atype_constr l2 app),t)) then
                                          modif_list_rec t l1
                                        else
                                          l1 in
                              imbrique_rec app' (rev l1')
                       with
                        UserError _ -> modif_bad_arg pg (rev l)) ;;

let modif_app_rec pg t = 
match pg with
  Arec(P,lf,c) -> (try Arec(P,lf,modif_app_rel [] c t)
                  with
                    UserError _ ->  let l = type_list c in
                      (try let (l1,l2,n) = modif_arg c (rev l) in
                            let app = new_app n (rev l) c in 
                             let app' = modif_bad_app app t l2 in
                              imbrique_rec (Arec(P,lf,app')) (rev l1)
                       with
                        UserError _ -> modif_bad_arg pg (rev l)))
| _            -> anomaly "shouldn't appear(modif_app_rec)" ;;

let rec aprogram_tac pg (u,sign,cl as gl) =
 let aprogram_app_ext pg gl = 
  let (hdt,l) = alist_arg pg in
    let n = anb_info_arg (atype_of2 hdt) in
      if (n>(length l)) then
        (match (ared_all(atype_of2 pg)) with
           Aprod(Name(y),A,B)  ->  
             aprogram_tac (Alambda(Name(y),A,Aapp(pg,Arel(1)))) gl
         | Aprod(_,A,B)        ->  
             aprogram_tac (Alambda(Name(IDENT("n",-1)),A,Aapp(pg,Arel(1)))) gl
         | _                   -> aprogram_app pg gl)
      else aprogram_app pg gl
in
  let aprogram_app_tac pg (_,sign,cl as gl) =
     (match (info_of_constr cl) with
        Inf(_,t) -> 
            (try 
             let x=constr_of_pure_annot sign pg in
              let pg' = modif_app pg t in
               let pg'' = modif_app_if pg' t gl in
               (match pg'' with
                  Arec(P,lf,c) -> aprogram_rec pg'' gl
                | _            -> aprogram_app_ext pg'' gl)
            with
         NONPUR _ -> 
           (try 
         aprogram_app_ext pg gl
        with
                  UserError _ -> 
            let pg' = modif_app pg t in
                     let pg'' = modif_app_if pg' t gl in
              (match pg'' with
             Arec(P,lf,c) -> aprogram_rec pg'' gl
               | _            -> aprogram_app_ext pg'' gl)))
      | _         -> anomaly "shouldn't appear(program_tac)")
in
  match pg with
    Alambda(x,A,B)    ->  (match (info_of_constr cl) with
                            Inf(_,t) -> let t' = (modif_lam A t) in  
                                        aderoule_intro (Alambda(x,t',B)) gl
                          | _       -> anomaly "shouldn't appear(program_tac)")
  | Alambdacom(x,A,B) ->  aderoule_intro_com pg gl
  | Arec(P,lf,c)      ->  (match (info_of_constr cl) with
                            Inf(_,t) ->
                              (try aprogram_rec pg gl with
                               UserError _ -> 
                                  (try let pg' = modif_rec pg in
                                        aprogram_tac pg' gl
                                     with
                                      UserError _ -> 
                                        let pg'' = modif_app_rec pg t in
                                          aprogram_rec pg'' gl))
                          | _       -> anomaly "shouldn't appear(program_tac)")
  | Annot(c,P)        ->  let pf = cut_tac P gl in
                            let t = fterm_of_annot(atype_of2 c) in
                            assoc_OCCUR(list_pf pf, [Alambda(Name(id_of_string"x"),t,Arel(1));c]) ; pf
  | Aconst(c)         ->    
          (match (info_of_constr cl) with
              Inf(_,t) -> 
                  let t' = fterm_of_annot(atype_of2 pg)
                  and pg' = modif_app pg t
                  in (match pg' with
                        Aconst(c') -> aprogram_tac (aone_step_reduce pg') gl
                      | _          -> 
                            if not(fconv t t') & (fconv2 t t') then
                              aprogram_tac pg' gl
                            else (try aprogram_app_tac pg gl
                                  with UserError s -> 
                                      let pg' = (try aone_step_reduce pg 
                                                 with
                                                  UserError _ -> error s) in
                                              aprogram_tac pg' gl))
           | _       -> anomaly "shouldn't appear(program_tac)")
  | _                 ->  (try aprogram_app_tac pg gl
                            with UserError s -> 
                                let pg' = (try aone_step_reduce pg 
                                           with UserError _ -> error s) in
                                  aprogram_tac pg' gl) ;;


let aprogram_with pg (u,sign,cl as gl) = 
  aprogram_tac (annot_of_acom gl pg) gl ;;

let Program (u,sign,cl as gl) = 
  let pg = try caml_assoc u !OCCUR with Failure _ -> error "no program associated to this occurrence" in
    aprogram_tac pg gl ;;

let Program_all gl = GO Program gl ;;

let see_program n = let l = list_pf (get_goals()) in
    try let (u,_,_) = nth l n in aprterm (caml_assoc u !OCCUR)
    with Failure "item" -> error "no such goal"
      |  Failure "assoc" -> error "no program associated to this goal"
;;

let See_program () = see_program 1;;

