(****************************************************************************)
(*              The Calculus of Inductive Constructions                     *)
(*                                                                          *)
(*                            Projet Coq                                    *)
(*                                                                          *)
(*                  INRIA                        ENS                        *)
(*           Rocquencourt                        Lyon                       *)
(*                                                                          *)
(*                              Coq V5.8                                    *)
(*                            Jan 1st 1993                                  *)
(****************************************************************************)
(*                               pp.ml                                      *)
(****************************************************************************)
#open "prelude";;
#open "impqueue";;
#open "pp_control";;

let inc n = (n := (!n) + 1);;
let dec n = (n := (!n) - 1);;

(* exception Format of string;; *)

let blank_line = make_string 80 ` `;;

type block_type =
    pp_hbox   (* Horizontal block no line breaking *)
  | pp_vbox   (* Vertical block each break leads to a new line *)
  | pp_hvbox  (* Horizontal-vertical block: same as vbox, except if this block
                 is small enough to fit on a single line *)
  | pp_hovbox (* Horizontal or Vertical block: breaks lead to new line
                 only when necessary to print the content of the block *)
  | pp_fits   (* Internal usage: when a block fits on a single line *)
;;

type pp_token = 
     pp_text of string            (* normal text *)
   | pp_break of int * int        (* complete break *)
   | pp_begin of int * block_type (* beginning of a block *)
   | pp_end                       (* end of a block *)
   | pp_newline                   (* to force a newline inside a block *)
   | pp_if_broken                 (* to do something only if this very
                                     line has been broken *)
;;

(* The Queue: contains all formatting elements.
   elements are tuples (size,token,length), where
    size is set when the size of the block is known
    len is the declared length of the token *)

type pp_queue_elem = {Elem_size : int ref; Token : pp_token; Length : int}
;;

(* Scan stack
   each element is left_total,queue element where left_total is the value
   of pp_left_total when the element has been enqueued *)

type pp_scan_elem = scan_elem of int * pp_queue_elem
;;

(* Formatting Stack:
   used to break the lines while printing tokens *)
(* The formatting stack contains the description of
   the currently active blocks *)

type pp_format_elem = format_elem of block_type * int
(* block type tells the way of breaking,offset is where on the line
   the block has to be printed (starts) *)
;;

type pp_global_params =
    GP of pp_gp_aux
and pp_gp_aux = {infinity : int;
                 line_length : int;
                 line_break : int;
                 margin : int;
                 min_space_left : int;
                 max_indent : int;
                 max_depth : int;
                 limit_depth : int;
                 ellipsis : string;
                 format_ellipsis : bool}
;;

type 'a pp_formatter_params =
    FP of 'a pp_fp_aux
and 'a pp_fp_aux = {output : (string -> int -> int -> unit);
                    flush_out : unit -> unit;
                    print: 'a -> (int * string stream)}
;;

let freeze_gp (gp:pp_control__pp_global_defaults) =
    GP{infinity = !(gp.pp_control__infinity);
     line_length = !(gp.pp_control__line_length);
     line_break = !(gp.pp_control__line_break);
     margin = !(gp.pp_control__margin);
     min_space_left = !(gp.pp_control__min_space_left);
     max_indent = !(gp.pp_control__max_indent);
     max_depth = !(gp.pp_control__max_depth);
     limit_depth = !(gp.pp_control__limit_depth);
     ellipsis = !(gp.pp_control__ellipsis);
     format_ellipsis = !(gp.pp_control__format_ellipsis)}
;;

let freeze_fp fp =
    FP{output = !(fp.pp_control__output);
       flush_out = !(fp.pp_control__flush_out);
       print = !(fp.pp_control__print)}
;;

type pp_state_data =
    {scan_stack : pp_scan_elem list ref;
     format_stack : pp_format_elem list ref;
     space_left : int ref;
     current_indent : int ref;
     left_total : int ref;
     right_total : int ref;
     curr_depth : int ref;
     queue : pp_queue_elem queue ref}
;;

let (dflt_sd:pp_state_data) =
    {scan_stack = ref [];
     format_stack = ref [];
     space_left = ref 72;
     current_indent = ref 0;
     left_total = ref 1;
     right_total = ref 1;
     curr_depth = ref 0;
     queue = ref (newq())}
;;

type 'a ppcmd_token =
    ppcmd_print of 'a
  | ppcmd_box of int * block_type * ('a ppcmd_token stream)
  | ppcmd_print_break of int * int
  | ppcmd_white_space of int
  | ppcmd_force_newline
  | ppcmd_print_if_broken
  | ppcmd_open_box of int * block_type
  | ppcmd_close_box
;;

type 'a ppdir_token =
    ppdir_ppcmds of 'a ppcmd_token stream
  | ppdir_print_newline
  | ppdir_print_flush
;;

let make_pp = fun (GP gp) (FP fp) ->
let (sd:pp_state_data) = {scan_stack = ref [];
                          format_stack = ref [];
                          space_left = ref 72;
                          current_indent = ref 0;
                          left_total = ref 1;
                          right_total = ref 1;
                          curr_depth = ref 0;
                          queue = ref (newq())}
in
    let output_newline() = (fp.output) "\n" 0 1
    in

    let output_string s = (fp.output) s 0 (string_length s)
    in
(*
  The queue:
  size is set when the size of the block is known
  len is the declared length of the token.
*)
    let pp_clear_queue () = 
        ((sd.left_total) := 1; (sd.right_total) := 1;
         clearq (!(sd.queue)))
    in
(* Enter a token in the pretty-printer queue *)

    let pp_raise_format s =  (raise (Format s))
    in
    let pp_enqueue ({Length=len;_} as token) =
        ((sd.right_total) := (!(sd.right_total)) + len;
         enq token (!(sd.queue)))
    in

(* To format blanks *)
(* Rk : it should be necessary to forbid too big breaks
   (namely greater than (gp.margin)) *)

    let display_blanks n =
        if n <= 80 then
            ((fp.output)) blank_line 0 n
        else
            output_string (make_string n ` `)
    in

(* to format a break, indenting a new line *)

    let break_new_line offset width =
        ((sd.space_left) := width - offset;
         output_newline ();
         (sd.current_indent) := ((gp.margin)) - (!(sd.space_left));
         display_blanks (!(sd.current_indent)))
    in

(* To force a line break inside a block: no offset is added *)
    let break_line width = break_new_line 0 width
    in

(* format a break that fits on the current line *)
    let break_same_line width =
        ((sd.space_left) := (!(sd.space_left)) - width;
         display_blanks width)
    in

(*
  To indent no more than (gp.max_indent) a block
  which is begun after (gp.max_indent) is rejected on the left
  by simulating a break.
  Same thing is done when a string continue after right margin *)

    let pp_force_newline () = 
        match (!(sd.format_stack)) with
            format_elem(bl_ty,width)::_ ->
                if width > (!(sd.space_left)) then
                    (match bl_ty with
                         pp_fits -> () | pp_hbox -> () | _ -> break_line width)
                else ()
                       | _ ->  pp_raise_format "Cannot break without a box"
    in

(* To skip a token if line has been broken *)
    let pp_skip_token () =
        (* When calling pp_skip_token the queue cannot be empty *)
        match deq (!(sd.queue)) with
            {Elem_size=ref size; Length=len; _} ->
                ((sd.left_total) := (!(sd.left_total)) - len;
                 (sd.space_left) := (!(sd.space_left)) + size)
    in

(* to format a token *)
    let format_pp_token size = fun
        (pp_text s) -> ((sd.space_left) := (!(sd.space_left)) - size;
                      output_string s)
            
      | (pp_begin (off,ty)) ->
            let offs = (!(sd.space_left)) - off in
            let offset =
                    if ((gp.margin)) - offs > ((gp.max_indent)) then
                        (* can't open a block right there ! *)
                        (pp_force_newline();(!(sd.space_left)) - off)
                    else offs in
            let bl_type =
                    (match ty with
                         pp_vbox -> pp_vbox
                       | _ -> if size > (!(sd.space_left)) then
                             ty
                              else pp_fits)
            in
                (sd.format_stack) := format_elem (bl_type,offset)::(!(sd.format_stack))

        
      | pp_end ->
            (sd.format_stack) :=
            (match (!(sd.format_stack)) with
                 x::l -> l
               | _ -> pp_raise_format "Attempt to close the last box")
                 
      | pp_newline ->
            (match (!(sd.format_stack)) with
                 format_elem (_,width) :: _ -> break_line width
               | _ ->  pp_raise_format "Cannot break without a box")
                 
      | pp_if_broken ->
            if (!(sd.current_indent)) <> ((gp.margin)) - (!(sd.space_left)) then
                pp_skip_token ()
            else ()
                
      | (pp_break (n,off)) ->
            (match (!(sd.format_stack)) with
                 format_elem (ty,width) :: _ ->
                     (match ty with
                          pp_hovbox ->
                              if size > (!(sd.space_left)) then
                                  break_new_line off width
                              else
                                  (* break the line here leads to new indentation ? *)
                                  if ((!(sd.current_indent)) > ((gp.margin)) - width + off) then
                                      break_new_line off width
                                  else
                                      break_same_line n
                        | pp_hvbox -> break_new_line off width
                        | pp_fits -> break_same_line n
                        | pp_vbox  -> break_new_line off width
                        | pp_hbox  -> break_same_line n)
                          
               | _ -> pp_raise_format "Cannot break without a box")
    in

(* Print if token size is known or printing is lagging
   Size is known if not negative
   Printing is lagging if the text waiting in the queue requires
   more room to format than exists on the current line
*)

    let rec advance_left () =
        match first_elem (!(sd.queue)) with
            NONE -> ()
          | SOME {Elem_size= ref size; Token=tok; Length=len} ->
                if size < 0 & ((!(sd.right_total)) - (!(sd.left_total)) <= (!(sd.space_left))) then ()
                else 
                    (deq (!(sd.queue));
                     format_pp_token (if size < 0 then (gp.infinity) else size) tok;
                     (sd.left_total) := len + (!(sd.left_total));
                     advance_left())
    in

(* To enqueue a string : try to advance *)
    let enqueue_string_as n s =
        (pp_enqueue {Elem_size=ref n; Token=pp_text s; Length=n};
         advance_left())
    in

    let enqueue_string s = enqueue_string_as (string_length s) s
    in

(*
  Routines for scan stack
  determine sizes of blocks
*)
(* scan_stack is never empty *)
    let clear_scan_stack =
        let dummy =
            [scan_elem (-1,{Elem_size=ref (-1); Token=pp_text""; Length=0})]
        in
            fun () -> ((sd.scan_stack) := dummy)
    in

    let scan_push () =
        match last_elem (!(sd.queue)) with
            SOME head ->
                (sd.scan_stack) := scan_elem ((!(sd.right_total)),head)::(!(sd.scan_stack))
   | _ -> pp_raise_format "bad scan-stack element, please report it"
    in


(*
  Set size of block on scan stack : 
  if ty = true then size of break is set else size of block is set
  in each case (sd.scan_stack) is popped *)
    let set_size ty =
        match (!(sd.scan_stack)) with
            scan_elem (left_tot,({Elem_size=ref size;Token=tok;_} as queue_elem))::t ->
                (* test if scan stack contains any data that is not obsolete *)
                if left_tot < (!(sd.left_total)) then
                    clear_scan_stack ()
                else
                    (match tok with
                         pp_break (_,_) ->
                             if ty then
                                 (queue_elem.Elem_size := (!(sd.right_total)) + size;
                                  (sd.scan_stack):=t)
                             else ()
                       | pp_begin (_,_) ->
                             if not ty then
                                 (queue_elem.Elem_size := (!(sd.right_total)) + size;
                                  (sd.scan_stack):=t)
                             else ()
                       | _ -> pp_raise_format "bad scan-stack element, please report it")
          | _ -> pp_raise_format "empty scan stack"
    in

(**************************************************************

  Procedures to control pretty-printer from outside

 **************************************************************)

(* To format a string *)
    let print_as n s =
        if (!(sd.curr_depth)) < ((gp.max_depth)) then (enqueue_string_as n s)
        else ()

(*
  To open a new block :
  the user may set the depth bound (gp.max_depth)
  any text nested deeper is printed as the character &
  The user may set the limit of number of blocks opened simultaneously
  using set_limit_depth : any text nested deeper is printed as ...
  and pretty_printer fails after a newline.
*)
    in

    let rec pp_open_box (indent,br_ty) =
        (inc (sd.curr_depth);
         if (!(sd.curr_depth)) < ((gp.max_depth)) then
             (pp_enqueue
              {Elem_size=ref (- (!(sd.right_total)));
               Token=pp_begin(indent,br_ty);
               Length=0};
              scan_push())
         else if (!(sd.curr_depth)) > ((gp.limit_depth)) then 
             (enqueue_string " ...";print_newline();
              raise (Format "limit_format_depth exceeded"))
              else if (!(sd.curr_depth)) = ((gp.max_depth)) then
                  enqueue_string ((gp.ellipsis))
                   else if ((gp.format_ellipsis)) then
                       enqueue_string ((gp.ellipsis))
                        else
                            ())
        
   and pp_open_sys_box () = 
       (pp_enqueue
        {Elem_size=ref(- (!(sd.right_total)));
         Token=pp_begin(0,pp_hovbox);
         Length=0};
        inc (sd.curr_depth);
        scan_push())

(* close a block, setting sizes of its subblocks *)
    and close_box () =
        (if (!(sd.curr_depth)) < ((gp.max_depth)) then 
             (pp_enqueue {Elem_size=ref 0;
                          Token=pp_end;
                          Length=0};
              set_size true;
              set_size false)
         else ();
             dec (sd.curr_depth))

(* Initialize pretty-printer. *)
    and pp_rinit_def () =
        (pp_clear_queue (); clear_scan_stack();
         (sd.current_indent) := 0;
         (sd.curr_depth) := 0;(sd.space_left) := ((gp.margin));
         (sd.format_stack) := [];
         pp_open_sys_box ())

(* Print a new line after printing all queued text
   (same for format_flush but without a newline)    *)
    and print_newline_gen b () =
        (close_box ();(sd.right_total) := (gp.infinity);advance_left ();
         if b then output_newline () else ();
             (fp.flush_out)();pp_rinit_def())

    and print_newline () = print_newline_gen true ()

    and print_flush () = print_newline_gen false ()
    in

(* Breaks: indicate where a block may be broken.
   If line is broken then offset is added to the indentation of the current
    block else (the value of) width blanks are printed.
   To do (?) : add a maximum width and offset value *)

    let print_break (width,offset) =
        if (!(sd.curr_depth)) < ((gp.max_depth)) then 
            (pp_enqueue
             {Elem_size=ref(- (!(sd.right_total)));
              Token=pp_break (width,offset);
              Length=width};
             set_size true;
             scan_push ())
        else ()
    in

(* To get a newline when one does not want to close the current block *)

    let force_newline () =
        if (!(sd.curr_depth)) < ((gp.max_depth)) then
            (pp_enqueue {Elem_size=ref 0;
                         Token=pp_newline;
                         Length=0};
             advance_left())
        else ()
    in


(* To format something if the line has just been broken *)
    let print_if_broken () =
        if (!(sd.curr_depth)) < ((gp.max_depth)) then
            (pp_enqueue {Elem_size=ref 0;
                         Token=pp_if_broken;
                         Length=0};
             advance_left())
                                  else ()
    in

    let rec pp_cmd = function
        (ppcmd_print tok) ->
        let (n,ss) = fp.print tok in
            (print_as n "";
             do_stream (function s -> print_as 0 s)  ss)
      | (ppcmd_box(n,bty,ss)) ->
        (pp_open_box(n,bty);
         do_stream pp_cmd ss;
         close_box())
      | ppcmd_open_box(n,bty) -> pp_open_box(n,bty)
      | ppcmd_close_box -> close_box()
      | ppcmd_white_space n -> print_as n (make_string n ` `)
      | (ppcmd_print_break(m,n)) -> print_break(m,n)
      | (ppcmd_force_newline) -> force_newline()
      | (ppcmd_print_if_broken) -> print_if_broken() in

    let pp_dir = function
        ppdir_ppcmds cmdstream -> do_stream pp_cmd cmdstream
      | ppdir_print_newline -> print_newline()
      | ppdir_print_flush -> print_flush()

in
    (pp_rinit_def();
     fun dirstream -> (try (do_stream pp_dir dirstream)
                       with ((Format s) as reraise) -> (pp_rinit_def();
                                                        raise reraise)))
;;


let PR x = ppcmd_print x;;
let BRK x = ppcmd_print_break x;;
let FNL = ppcmd_force_newline;;
let PifB = ppcmd_print_if_broken;;
let WS n = ppcmd_white_space n;;

let H n s = [< 'ppcmd_box(n,pp_hbox,s) >];;
let V n s = [< 'ppcmd_box(n,pp_vbox,s) >];;
let HV n s = [< 'ppcmd_box(n,pp_hvbox,s) >];;
let HOV n s = [< 'ppcmd_box(n,pp_hovbox,s) >];;

let PPCMDS x = ppdir_ppcmds x;;
let PNL = ppdir_print_newline;;
let FLSH = ppdir_print_flush;;


let PPDIR_WITH gp fp =
    make_pp (freeze_gp gp) (freeze_fp fp)
;;

let PP_WITH gp fp strm = PPDIR_WITH gp fp [< 'PPCMDS strm ; 'FLSH >];;
let PPNL_WITH gp fp strm = PPDIR_WITH gp fp [< 'PPCMDS strm ; 'PNL >];;


let with_output_to fp ch =
    (fp.pp_control__output := (output_substring ch);
     fp.pp_control__flush_out := (fun () -> flush ch))
;;

let with_depth gp n =
    (gp.pp_control__max_depth := n;
     if !(gp.pp_control__limit_depth) < n then
         gp.pp_control__limit_depth := succ n)
;;
let SPC = BRK(1,0);;
let CUT = BRK(0,0);;

let HB n = ppcmd_open_box(n,pp_hbox);;

let VB n = ppcmd_open_box(n,pp_vbox);;
let HVB n = ppcmd_open_box(n,pp_hvbox);;
let HOVB n = ppcmd_open_box(n,pp_hovbox);;
let CLOSE = ppcmd_close_box;;
