(* $Id: rtdef.mlip,v 1.9 92/10/16 15:29:41 ddr Exp $ *)

#open "xlib";;
#open "time";;

type 'a option = None | Some of 'a;;

type xdata = {
    dpy                       : Display;
    scr                       : int;
    vis                       : Visual;
    black                     : int;
    white                     : int;
    rootw                     : Window;
    root_width                : int;
    root_height               : int;
    depth                     : int;
    cmap                      : Colormap;
    connection_number         : int;
    gc 	                      : GC;
    xevent                    : xevent;
    mutable win_but           : window_button_state;
    mutable popped_up         : Window list;
    mutable end_func          : (unit -> unit) list;
    ginfo                     : (string * global_info) list vect;
    wid_by_win                : (Window * widget) list vect;
    wid_by_name               : (string * widget) list vect;
    font_by_name              : (string * font) list vect
}

and widget = {
  wid_xd            : xdata;
  win               : Window;
  mutable x         : int;
  mutable y         : int;
  mutable width     : int;
  mutable height    : int;
  border            : int;
  mutable is_mapped : bool;
  mutable user_info : user_info;
  wdesc             : widget_desc;
  info              : local_info;
  mutable children  : widget list
}

and global_info == globinfo -> string and globinfo = C'GI
and local_info == locinfo -> string and locinfo = C'LI
and user_info = C'UI of (unit -> string)

and xevent = {
  mutable x_win     : int;
  mutable y_win     : int;
  mutable x_root    : int;
  mutable y_root    : int;
  mutable button    : int
}

and widget_desc = {
  wsize         : xdata -> int * int * int;
  wcreate       : xdata * Window * widget_desc *
                  int * int * int * int * int -> widget;
  wdestroy      : widget -> unit;
  wdispatch     : widget * XEvent -> unit;
  filler        : bool
}

and font = {
  font_xd : xdata;
  fs      : XFontStruct;
  fid     : Font;
  gc_mask : int;
  ascent  : int;
  descent : int;
  fwidth  : int;
  fheight : int
}

and orientation = C'Horizontal | C'Vertical

and pixmap = {
  pixm_xd       : xdata;
  pixmap        : Pixmap
}

and color = {
  col_xd        : xdata;
  pixel         : int
}

and cursor = {
  curs_xd       : xdata;
  cursor        : Cursor
}

and xargs = {
  mutable xdl           : xdata list;
  mutable fd_list       : (int * (unit -> unit)) list;
  initial_time          : timeb;
  mutable current_time  : int;
  mutable timeout       : int option;
  mutable timeout_fun   : unit -> unit;
  mutable running       : bool
}

and background = C'NoneBg | C'PixmapBg of Pixmap | C'ColorBg of int

and drawable = C'WidgetDr of widget | C'PixmapDr of pixmap

and attribute =
  C'BackgroundAtt of background
| C'FillerAtt | C'NameAtt of string
| C'WidthAtt of int | C'HeightAtt of int | C'BorderAtt of int
| C'BorderBackgAtt of background

and window_button_state =
  C'WB_None | C'WB_Win of Window | C'WB_WinBut of Window
| C'WB_WinButExit of Window | C'WB_WinButOther of Window
| C'WB_But | C'WB_ButWin
;;

(* $Id: rt.mlih,v 1.1 92/10/16 15:31:34 ddr Exp $ *)

(*** Main functions *)

(* connection to the display *)

value rt_run : string -> (xdata -> 'a) -> 'a;;
value rt_initialize : string -> xdata;;
value rt_end : xdata -> unit;;

(* widgets *)

value rt_create_widget : xdata * string * string * widget_desc -> widget;;
value rt_map_widget : widget -> unit;;
value rt_unmap_widget : widget -> unit;;
value rt_destroy_widget : widget -> unit;;
value rt_change_background : widget * background -> unit;;


(* events handling *)

value rt_main_loop : xargs -> unit;;
value rt_stop_main_loop : xargs -> unit;;
value rt_treat_one_event : xargs -> unit;;
value rt_treat_pending_events : xargs -> unit;;
value rt_args : xdata list -> xargs;;
value rt_select_xdata : xargs * xdata -> unit;;
value rt_unselect_xdata : xargs * xdata -> unit;;
value rt_select_file : xargs * int * (unit -> unit) -> unit;;
value rt_unselect_file : xargs * int -> unit;;
value rt_current_time : xargs -> int;;
value rt_set_timeout : xargs * int -> unit;;
value rt_set_timeout_fun : xargs * (unit -> unit) -> unit;;
value rt_reset_timeout : xargs -> unit;;

(* informations about displays *)

value rt_display_name : string -> string;;
value is_colored : xdata -> bool;;
value screen_width : xdata -> int;;
value screen_height : xdata -> int;;

(* informations about widgets *)

value widget_x : widget -> int;;
value widget_y : widget -> int;;
value widget_width : widget -> int;;
value widget_height : widget -> int;;
value widget_border : widget -> int;;
value widget_named : xdata -> string -> widget;;
value is_mapped : widget -> bool;;
value widget_size : xdata * widget_desc -> int * int;;

(* informations about events *)

value xevent_x : xdata -> int;;
value xevent_y : xdata -> int;;
value xevent_x_root : xdata -> int;;
value xevent_y_root : xdata -> int;;
value xevent_button : xdata -> int;;

(* widget attributes *)

value BackgroundAtt : background -> attribute;;
value BorderBackgAtt : background -> attribute;;
value FillerAtt : attribute;;
value NameAtt : string -> attribute;;
value WidthAtt : int -> attribute;;
value HeightAtt : int -> attribute;;
value BorderAtt : int -> attribute;;

(* widget backgrounds *)

value NoneBg : background;;
value PixmapBg : pixmap -> background;;
value ColorBg : color -> background;;

(* drawables *)

value WidgetDr : widget -> drawable;;
value PixmapDr : pixmap -> drawable;;

(* orientation *)

value Horizontal : orientation;;
value Vertical : orientation;;


(*** Button widget *)

value ButtonD : string * (widget -> unit) -> widget_desc;;
value PopupD : string * (widget -> unit) -> widget_desc;;
value CommD : string * (widget -> unit) -> widget_desc;;
value ButtonA : attribute list -> string * (widget -> unit) -> widget_desc;;
value PopupA : attribute list -> string * (widget -> unit) -> widget_desc;;
value CommA : attribute list -> string * (widget -> unit) -> widget_desc;;

value button_border : int ref;;
value button_band : int ref;;
value button_bold : int ref;;
value button_font : string ref;;


(*** Pack widget *)

value PackD : orientation * widget_desc list -> widget_desc;;
value PackA : attribute list -> orientation * widget_desc list -> widget_desc;;

value pack_border : int ref;;
value pack_band : int ref;;
value pack_inter : int ref;;


(*** Scroll widget *)

value ScrollD :
  orientation * int * int * int * (widget * int * int -> unit) -> widget_desc;;
value ScrollA : attribute list ->
  orientation * int * int * int * (widget * int * int -> unit) -> widget_desc;;

value scroll_set : widget * int -> unit;;
value scroll_val : widget -> int;;

value scroll_border : int ref;;
value scroll_width : int ref;;
value scroll_band : int ref;;


(*** Text widget *)

value TextD : int * int * int *
  (widget * string -> unit) *
  (widget * int * int * int -> unit) -> widget_desc;;
value TextA : attribute list -> int * int * int *
  (widget * string -> unit) *
  (widget * int * int * int -> unit) -> widget_desc;;

value text_send_string : widget * string -> unit;;
value text_scroll : widget -> unit;;
value text_clear : widget -> unit;;
value text_goto : widget * int * int -> unit;;
value text_current_line : widget -> int;;
value text_current_column : widget -> int;;
value text_home : widget -> unit;;
value text_shift : widget * int -> unit;;
value text_shift_value : widget -> int;;
value text_set_mark : widget * int * int -> unit;;
value text_get_text : widget * int * int -> string;;

value rt_get_cut_buffer : xdata -> string;;
value rt_set_cut_buffer : xdata * string -> unit;;

value text_border : int ref;;
value text_band : int ref;;
value text_font : string ref;;


(*** Title widget *)

value TitleD : string -> widget_desc;;
value TitleA : attribute list -> string -> widget_desc;;

value title_border : int ref;;
value title_band : int ref;;
value title_font : string ref;;


(*** Select widget *)

value SelectD : widget_desc list -> widget_desc;;
value SelectA : attribute list -> widget_desc list -> widget_desc;;

value select_raise : widget * int -> unit;;


(*** Raw widget *)

type property;;

value RawD : int * int * int * property list -> widget_desc;;
value RawA : attribute list -> int * int * int * property list -> widget_desc;;

value ExposePr : (widget * int * int * int * int -> unit) -> property;;
value KeyPr : (widget * string -> unit) -> property;;
value ButtonPressedPr : (widget * int * int * int -> unit) -> property;;
value ButtonReleasedPr : (widget -> unit) -> property;;
value ButtonMotionPr : (widget * int * int -> unit) -> property;;
value PointerMotionPr : (widget * int * int -> unit) -> property;;
value EnterWindowPr : (widget -> unit) -> property;;
value LeaveWindowPr : (widget -> unit) -> property;;


(*** Pixmap *)

value rt_create_pixmap : xdata * int * int -> pixmap;;
value rt_select_pixmap : pixmap -> unit;;


(*** Color *)

value rt_create_color : xdata * int * int * int -> color;;
value rt_change_color : color * int * int * int -> unit;;
value rt_closest_color : xdata * int * int * int -> color;;
value rt_black_color : xdata -> color;;
value rt_white_color : xdata -> color;;
value rt_select_color : color -> unit;;
value color_pixel : color -> int;;
value rt_query_color : xdata * int -> int * int * int;;


(*** Pattern *)

type pattern;;

value rt_create_pattern : xdata * string * int * int -> pattern;;
value rt_select_pattern : pattern * int * int -> unit;;
value rt_select_pattern_mask : pattern * int * int -> unit;;
value rt_select_pattern_clip : pattern * int * int -> unit;;
value rt_unselect_pattern_clip : xdata -> unit;;


(*** Drawing *)

value rt_draw_point : drawable * int * int -> unit;;
value rt_draw_line : drawable * int * int * int * int -> unit;;
value rt_draw_lines : drawable * (int * int) list -> unit;;
value rt_fill_polygon : drawable * (int * int) list -> unit;;
value rt_fill_rectangle : drawable * int * int * int * int -> unit;;
value rt_draw_rectangle : drawable * int * int * int * int -> unit;;
value rt_fill_arc : drawable * int * int * int * int * int * int -> unit;;
value rt_draw_arc : drawable * int * int * int * int * int * int -> unit;;
value rt_clear_widget : widget -> unit;;
value rt_clear_area : widget * int * int * int * int -> unit;;
value rt_draw_string : drawable * int * int * string -> unit;;
value rt_erase_draw_string : drawable * int * int * string -> unit;;
value rt_copy_area :
  drawable * drawable * int * int * int * int * int * int -> unit;;

value rt_set_line_width : xdata * int -> unit;;
value rt_set_backing_store : widget -> unit;;


(*** Image *)

type image;;

value rt_create_image : xdata * string * int * int * int -> image;;
value rt_put_image : drawable * image * int * int * int * int * int * int
  -> unit;;
value rt_get_image : drawable * int * int * int * int -> image;;
value rt_get_pixel : image * int * int -> int;;


(*** Cursor *)

value rt_create_cursor :
  xdata * string * string * int * int *
  (int * int * int) * (int * int * int) * int * int -> cursor;;
value rt_define_cursor : widget * cursor -> unit;;
value rt_undefine_cursor : widget -> unit;;


(*** Font *)

value rt_load_query_font : xdata * string -> font;;
value rt_select_font : font -> unit;;
value rt_text_width : font * string -> int;;
value rt_font_size : font -> int;;


(*** Resource *)

value rt_get_default : xdata * string * string -> string;;


(*** Miscellaneous *)

value rt_create_subwidget : widget * int * int * widget_desc -> widget;;
value rt_move_widget : widget * int * int -> unit;;
value rt_resize_widget : widget * int * int -> unit;;
value rt_move_resize_widget : widget * int * int * int * int -> unit;;
value rt_reparent_widget : widget * widget * int * int -> unit;;

type position;;

value UserPosition : int * int -> position;;
value AutoPosition : position;;

value rt_create_located_widget : xdata * string * string * position *
  widget_desc -> widget;;

value rt_create_transient_widget : widget * string * widget_desc -> widget;;
value rt_map_transient_widget : widget * int * int -> unit;;
value rt_create_popup_widget : xdata * widget_desc -> widget;;
value rt_map_popup_widget : widget * int * int * int -> unit;;

value popup_border : int ref;;

value rt_sync : xdata -> unit;;
value rt_query_pointer : widget -> int * int * int list;;
value rt_get_bell_params : xdata -> int * int * int;;
value rt_set_bell_params : xdata * int * int * int -> unit;;
value rt_bell : xdata * int -> unit;;


(*** User information *)

type 'a uoption;;
type 'a user_info_func == ('a -> user_info) * (widget -> 'a);;

value UNone : 'a uoption;;
value user_info : string -> 'a uoption ref -> 'a user_info_func;;
value rt_set_user_info : widget * user_info -> unit;;
