(* $Id: rt_main.ml,v 1.13 92/10/08 16:45:00 ddr Exp $
 *
 * Rogloglo Toolkit
 *)

#open "rtdef";;
#open "xlib";;
#open "std";;
#open "time";;
#open "util";;

let rt_initialize name =
  let dpy = XOpenDisplay name in
  if is_null_Display dpy then failwith "Can't open display";
  let scr = XDefaultScreen dpy
  and rootw = XDefaultRootWindow dpy in
  let black = XBlackPixel(dpy, scr)
  and white = XWhitePixel(dpy, scr) in
  let xgcv = (gstr()).xgcv in
  set_XGCValues_background(white, xgcv);
  set_XGCValues_foreground(black, xgcv);
  let gc = XCreateGC(dpy, rootw, GCBackground lor GCForeground, xgcv) in
  {
    dpy = dpy;
    scr = scr;
    vis = XDefaultVisual(dpy, scr);
    black = black;
    white = white;
    rootw = rootw;
    root_width = (XDisplayWidth(dpy, scr));
    root_height = (XDisplayHeight(dpy, scr));
    depth = XDefaultDepth(dpy, scr);
    cmap = XDefaultColormap(dpy, scr);
    connection_number = XConnectionNumber dpy;
    gc = gc;
    xevent = {x_win=0; y_win=0; x_root=0; y_root=0; button=0};
    win_but = C'WB_None;
    popped_up = [];
    end_func = [];
    ginfo = (make_vect 17 []);
    wid_by_win = (make_vect 53 []);
    wid_by_name = (make_vect 53 []);
    font_by_name = (make_vect 17 [])
  }
;;

let rt_end xd =
  do_list (function f -> f()) xd.end_func;
  XCloseDisplay xd.dpy
;;

let rt_run name func =
  let xd = rt_initialize name in
  let v =
    try
      func xd
    with
      X_failure str ->
        failwith ("X failure: " ^ str)
    | X_io_error (dname, errno) ->
        failwith (
          "X fatal io error on display \"" ^ dname ^
          "\", errno = " ^ (string_of_int errno)
        )
    | x -> rt_end xd; raise x
  in
  rt_end xd; v
;;

let rt_args xdl = {
  xdl = xdl;
  fd_list = [];
  initial_time = ftime();
  current_time = 0;
  timeout = None;
  timeout_fun = (fun _ -> failwith "timeout");
  running = false
}
;;

exception bad_wid;;

let treat_next_event xd xev =
  XNextEvent(xd.dpy, xev);
  let win = XAnyEvent_window(XEvent_xany xev) in
  try
    let wid = try hash_assoc win xd.wid_by_win with _ -> raise bad_wid in
    wid.wdesc.wdispatch(wid, xev)
  with bad_wid ->
    ()
;;

let try_pending xdl xev =
  it_list (fun ok xd ->
    if XPending xd.dpy <> 0 then (
      treat_next_event xd xev; true
    ) else ok
  ) false xdl

and dispatch xa xev fds =
  do_list (fun (fd, fd_fun) ->
    if FD_ISSET(fd, fds) <> 0 then fd_fun()
  ) xa.fd_list;
  do_list (fun xd ->
    if FD_ISSET(xd.connection_number, fds) <> 0 then
      treat_next_event xd xev
  ) xa.xdl
;;

let rt_treat_one_event xa =

  let init_fds fds =
    FD_ZERO fds;
    let max_fd = it_list (fun max_fd (fd, _) ->
      FD_SET(fd, fds);
      max max_fd fd
    ) 0 xa.fd_list in
    it_list (fun max_fd xd ->
      let fd = xd.connection_number in
      FD_SET(fd, fds);
      max max_fd (fd)
    ) max_fd xa.xdl

  in
  if xa.xdl = [] & xa.fd_list = [] & xa.timeout = None then
    failwith "rt_treat_one_event: no xdata, no fd, no timeout";
  let gstr = gstr() in
  xa.current_time <- timeb_sub (ftime()) xa.initial_time;
  match xa.timeout with
    None ->
      if not (try_pending xa.xdl gstr.xev) then (
        let max_fd = init_fds gstr.fds in
        let _ = fselect((max_fd+1), gstr.fds, (-1)) in
        xa.current_time <- timeb_sub (ftime()) xa.initial_time;
        dispatch xa gstr.xev gstr.fds
      )
  | Some tmout ->
      if not (try_pending xa.xdl gstr.xev) then (
        if xa.current_time < tmout then (
          let max_fd = init_fds gstr.fds in
          let tm = tmout - xa.current_time in
          let ns = fselect((max_fd+1), gstr.fds, tm) in
          if ns = 0 then (
            xa.current_time <- tmout;
            xa.timeout <- None;
            xa.timeout_fun()
          )
          else (
            xa.current_time <- timeb_sub (ftime()) xa.initial_time;
            dispatch xa gstr.xev gstr.fds
          )
        )
        else (
          xa.current_time <- tmout;
          xa.timeout <- None;
          xa.timeout_fun()
        )
      )
;;

let rec rt_treat_pending_events xa =
  if try_pending xa.xdl (gstr()).xev then rt_treat_pending_events xa
;;

let rt_main_loop xa =
  xa.running <- true;
  while xa.running do rt_treat_one_event xa done

and rt_stop_main_loop xa =
  xa.running <- false

and rt_set_timeout_fun(xa, timeout_fun) =
  xa.timeout_fun <- timeout_fun

and rt_set_timeout(xa, timeout) =
  xa.timeout <- Some timeout

and rt_reset_timeout xa =
  xa.timeout <- None

and rt_current_time xa =
  xa.current_time

and rt_select_xdata(xa, xd) =
  xa.xdl <- add_setq xd xa.xdl

and rt_unselect_xdata(xa, xd) =
  xa.xdl <- filter_neg (prefix == xd) xa.xdl

and rt_select_file(xa, fd, fd_fun) =
  xa.fd_list <- (fd, fd_fun)::(
    filter_neg (fun (fd1, _) -> fd = fd1) xa.fd_list
  )

and rt_unselect_file(xa, fd) =
  xa.fd_list <- filter_neg (fun (fd1, _) -> fd = fd1) xa.fd_list
;;

let rt_create_widget (xd, wname, iname, wdesc) =
  let (width, height, border) = wdesc.wsize xd in
  let wid = wdesc.wcreate(
    xd, xd.rootw, wdesc, 0, 0, width, height, border
  ) in
  set_std_prop (wid, wname, iname, 0, 0, width, height, PPosition lor USSize);
  set_wm_and_class_hints wid;
  wid

and rt_map_widget wid =
  wid.is_mapped <- true;
  XMapWindow(wid.wid_xd.dpy, wid.win)

and rt_unmap_widget wid =
  wid.is_mapped <- false;
  XUnmapWindow(wid.wid_xd.dpy, wid.win)

and rt_destroy_widget wid =
  let xd = wid.wid_xd
  and win = wid.win in
  let rec destroy wid =
    do_list (fun wid -> destroy wid) wid.children;
    wid.wdesc.wdestroy wid
  in
  destroy wid;
  XDestroyWindow(xd.dpy, win)
;;

let rt_change_background(wid, backg) =
  let xd = wid.wid_xd in
  match backg with
    C'NoneBg -> XSetWindowBackgroundPixmap(xd.dpy, wid.win, XNone)
  | C'PixmapBg p -> XSetWindowBackgroundPixmap(xd.dpy, wid.win, p)
  | C'ColorBg c -> XSetWindowBackground(xd.dpy, wid.win, c)
;;

let widget_named xd wname =
  try hash_assoc wname xd.wid_by_name
  with _ -> failwith ("widget_named " ^ wname)
;;

let NoneBg = C'NoneBg
and PixmapBg p = C'PixmapBg p.pixmap
and ColorBg c = C'ColorBg c.pixel
;;

let WidgetDr w = C'WidgetDr w
and PixmapDr p = C'PixmapDr p
;;

let BackgroundAtt v = C'BackgroundAtt v
and FillerAtt = C'FillerAtt
and NameAtt v = C'NameAtt v
and WidthAtt v = C'WidthAtt v
and HeightAtt v = C'HeightAtt v
and BorderAtt v = C'BorderAtt v
and BorderBackgAtt v = C'BorderBackgAtt v
;;

let screen_width xd = xd.root_width
and screen_height xd = xd.root_height
and rt_display_name dname =
  let s = XDisplayName dname in
  string_of_C_String(s, C_String_length s)
and is_colored xd = Visual_class xd.vis = PseudoColor
;;

let widget_x wid = wid.x
and widget_y wid = wid.y
and widget_width wid = wid.width
and widget_height wid = wid.height
and widget_border wid = wid.border
and is_mapped wid = wid.is_mapped
and widget_size(xd, wdesc) =
  let (w, h, _) = wdesc.wsize xd in
  w, h
;;

let xevent_x xd = xd.xevent.x_win
and xevent_y xd = xd.xevent.y_win
and xevent_x_root xd = xd.xevent.x_root
and xevent_y_root xd = xd.xevent.y_root
and xevent_button xd = xd.xevent.button
;;

let Vertical = C'Vertical
and Horizontal = C'Horizontal
;;
