(* $Id: misc.ml,v 1.10 92/10/08 16:44:55 ddr Exp $
 *
 * Rogloglo toolkit: miscellaneous functions
 *)

#open "xlib";;
#open "std";;
#open "rtdef";;
#open "util";;

let rt_create_subwidget (pwid, x, y, wdesc) =
  let xd = pwid.wid_xd in
  let (width, height, border) = wdesc.wsize xd in
  let wid = wdesc.wcreate(xd, pwid.win, wdesc, x, y, width, height, border) in
  pwid.children <- wid::pwid.children;
  wid

and rt_move_widget =
  let mask = it_list (prefix +) (0) [CWX; CWY; CWStackMode]
  and xwc = mallocated_var alloc_XWindowChanges (ref None) in
function (wid, x, y) ->
  let xd = wid.wid_xd in
  let xwc = xwc() in
  set_XWindowChanges_x(x, xwc);
  set_XWindowChanges_y(y, xwc);
  set_XWindowChanges_stack_mode(Above, xwc);
  XConfigureWindow(xd.dpy, wid.win, mask, xwc);
  wid.x <- x; wid.y <- y

and rt_reparent_widget(wid, pwid, x, y) =
  XReparentWindow(wid.wid_xd.dpy, wid.win, pwid.win, x, y)

and rt_resize_widget(wid, width, height) =
  let width = max 1 width and height = max 1 height in
  XResizeWindow(wid.wid_xd.dpy, wid.win, width, height)
;;

let rt_move_resize_widget(wid, x, y, width, height) =
  let xd = wid.wid_xd in
  let width = max 1 width and height = max 1 height in
  XRaiseWindow(xd.dpy, wid.win);
  XMoveResizeWindow(
    xd.dpy, wid.win,
    x, y, width, height
  );
  wid.x <- x; wid.y <- y
;;

let xswa = mallocated_var alloc_XSetWindowAttributes (ref None);;

type position =
  C'AutoPosition
| C'UserPosition of int * int
;;

let UserPosition(x, y) = C'UserPosition(x, y)
and AutoPosition = C'AutoPosition
;;

let rt_create_located_widget (xd, wname, iname, position, wdesc) =
  let (width, height, border) = wdesc.wsize xd in
  let (x, y, flag) = (match position with
    C'UserPosition(x, y) -> (x, y, USPosition)
  | C'AutoPosition -> (0, 0, PPosition)
  ) in
  let wid = wdesc.wcreate(
    xd, xd.rootw, wdesc, x, y, width, height, border
  ) in
  set_std_prop (wid, wname, iname, x, y, width, height, flag lor USSize);
  set_wm_and_class_hints wid;
  wid
;;

let rt_create_transient_widget(pwid, wname, wdesc) =
  let xd = pwid.wid_xd in
  let (width, height, border) = wdesc.wsize xd in
  let wid = wdesc.wcreate(
    xd, xd.rootw, wdesc, 0, 0, width, height, border
  ) in
  let xswa = xswa() in
  set_XSetWindowAttributes_save_under(1, xswa);
  XChangeWindowAttributes(xd.dpy, wid.win, CWSaveUnder, xswa);
  XSetTransientForHint(xd.dpy, wid.win, pwid.win);
  set_std_prop (
    wid, wname, "<transient>", 0, 0, width, height, USPosition lor USSize
  );
  set_wm_and_class_hints wid;
  wid
;;

let rt_map_transient_widget =
  let xsh = mallocated_var alloc_XSizeHints (ref None) in
function (wid, x, y) ->
  let xd = wid.wid_xd in
  let x = min (xd.root_width-wid.width) (max 0 x)
  and y = min (xd.root_height-wid.height) (max 0 y) in
  wid.is_mapped <- true;
  XUnmapWindow(xd.dpy, wid.win);
  XMoveWindow(xd.dpy, wid.win, x, y);
  let xsh = xsh() in
  set_XSizeHints_x(x, xsh);
  set_XSizeHints_y(y, xsh);
  set_XSizeHints_width (wid.width, xsh);
  set_XSizeHints_height (wid.height, xsh);
  set_XSizeHints_flags(USPosition lor USSize, xsh);
  XSetNormalHints(xd.dpy, wid.win, xsh);
  XMapRaised(xd.dpy, wid.win)
;;

let popup_border = ref 2
;;

let rt_create_popup_widget(xd, wdesc) =
  let (width, height, border) = wdesc.wsize xd in
  let wid = wdesc.wcreate(
    xd, xd.rootw, wdesc, 0, 0, width, height, !popup_border
  ) in
  let xswa = xswa() in
  set_XSetWindowAttributes_save_under(1, xswa);
  set_XSetWindowAttributes_override_redirect(1, xswa);
  XChangeWindowAttributes(
    xd.dpy, wid.win, CWSaveUnder lor CWOverrideRedirect, xswa
  );
  wid
;;

let rt_map_popup_widget(wid, x, y, lev) =
  let xd = wid.wid_xd in
  let x = min (xd.root_width-wid.width) (max 0 x)
  and y = min (xd.root_height-wid.height) (max 0 y) in
  let rec unmap n x =
    if n <= 0 then x
    else
      unmap (n-1) (
        match x with
          win::winl ->
            XUnmapWindow(xd.dpy, win);
            winl
        | [] -> []
      )
  in
  xd.popped_up <- unmap (list_length xd.popped_up - lev) xd.popped_up;
  xd.popped_up <- wid.win::xd.popped_up;
  XMoveWindow(xd.dpy, wid.win, x, y);
  XMapRaised(xd.dpy, wid.win)
;;

let rt_sync xd =
  XSync (xd.dpy, 1)
;;

let rt_query_pointer wid =
  let root = ref XNone
  and child = ref XNone
  and root_x = ref 0 and root_y = ref 0
  and win_x = ref 0 and win_y = ref 0
  and keys_buttons = ref 0 in
  let xd = wid.wid_xd in
  let r = XQueryPointer (
    xd.dpy, wid.win, root, child, root_x, root_y,
    win_x, win_y, keys_buttons
  )
  in
  let kb = !keys_buttons in
  let buttl = it_list (fun l (i, m) ->
    l @ (if kb land m <> 0 then [i] else [])
  ) [] [
    (1, Button1Mask); (2, Button2Mask); (3, Button3Mask);
    (4, Button4Mask); (5, Button5Mask)
  ] in
  if r <> 0 then (!win_x, !win_y, buttl)
  else (-1, -1, buttl)
;;

let rt_get_bell_params =
  let xks = mallocated_var alloc_XKeyboardState (ref None) in
function xd ->
  let xks = xks () in
  XGetKeyboardControl (xd.dpy, xks);
  (
    XKeyboardState_bell_percent xks,
    XKeyboardState_bell_pitch xks,
    XKeyboardState_bell_duration xks
  )
;;

let rt_set_bell_params =
  let xkc = mallocated_var alloc_XKeyboardControl (ref None) in
function (xd, percent, pitch, duration) ->
  let xkc = xkc () in
  set_XKeyboardControl_bell_percent (percent, xkc);
  set_XKeyboardControl_bell_pitch (pitch, xkc);
  set_XKeyboardControl_bell_duration (duration, xkc);
  XChangeKeyboardControl (
    xd.dpy,
    KBBellPercent lor KBBellPitch lor KBBellDuration,
    xkc
  )
;;

let rt_bell (xd, percent) =
  XBell (xd.dpy, percent)
;;
