(* $Id: c_title.ml,v 1.9 92/07/28 10:43:30 ddr Exp $
 *
 * Rogloglo Toolkit: title widget class
 *)

#open "xlib";;
#open "std";;
#open "rtdef";;
#open "font";;
#open "util";;

type title_global_info = {
  tfs       : font;
  gc_title  : GC
}
;;

let title_global_info, get_title_global_info = dynamo_global_info
  "title_global_info" (ref None: title_global_info option ref)
;;

let title_border = ref 1
and title_band = ref 2
and title_font = ref "*-helvetica-bold-o-*--14-*"
;;

let TitleA attr txt =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = (function xd ->
    let make_global_info xd =
      let fs = rt_load_query_font(xd, !title_font) in
      let mask = GCForeground lor GCBackground lor fs.gc_mask
      and gstr = gstr() in
      set_XGCValues_font(fs.fid, gstr.xgcv);
      set_XGCValues_foreground(xd.black, gstr.xgcv);
      set_XGCValues_background(xd.white, gstr.xgcv);
      let gc_title = XCreateGC(xd.dpy, xd.rootw, mask, gstr.xgcv) in
      xd.end_func <- (function () ->
        let gi = get_title_global_info(ginfo xd "title") in
        XFreeGC(xd.dpy, gi.gc_title);
        remove_ginfo xd "title"
      ) :: xd.end_func;
      add_ginfo xd "title" title_global_info {
        tfs = fs;
        gc_title = gc_title
      }
    in
    let gi =
      try get_title_global_info(ginfo xd "title")
      with _ -> make_global_info xd in
    let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
      2*!title_band+
        (XTextWidth(gi.tfs.fs, txt, (string_length txt)))
      )
    and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
      2*!title_band+gi.tfs.fheight)
    and b = match szh with (_,_,Some v) -> v | _ -> !title_border
    in (w,h,b)
  )
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let win = create_window(
      xd, pwin, x, y, width, height, border, attr,
      it_list (prefix lor) 0 [
        ExposureMask; StructureNotifyMask
      ]
    ) in
    add_widget attr win {
      wid_xd = xd; win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = no_info; user_info = no_user_info;
      children = []
    }
  )
;
  wdestroy = (function wid ->
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev) ->
    let xd = wid.wid_xd in
    let gi = get_title_global_info(ginfo xd "title")
    and t = XEvent_type xev in
    if t = Expose then (
      XClearWindow(xd.dpy, wid.win);
      let len = string_length txt in
      XDrawString(xd.dpy, wid.win, gi.gc_title,
        ((wid.width-(XTextWidth(gi.tfs.fs, txt, len)))
          / 2),
        ((wid.height+gi.tfs.ascent-gi.tfs.descent) / 2),
        txt, len
      )
    ) else if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      wid.width <- (XConfigureEvent_width xev);
      wid.height <- (XConfigureEvent_height xev)
    )
  )
;
  filler = mem C'FillerAtt attr
}
;;

let TitleD = TitleA []
;;
