(* $Id: c_text.ml,v 1.10 92/10/08 16:44:46 ddr Exp $
 *
 * Rogloglo Toolkit: text widget class
 *)

#open "xlib";;
#open "std";;
#open "rtdef";;
#open "font";;
#open "util";;

type text_global_info = {
  tfs     : font;
  gc_text : GC
}

and text_local_info = {
  text_gi         : text_global_info;
  swin            : Window;
  mutable xb      : int;
  mutable yb      : int;
  mutable y_shift : int;
  mutable y_shmax : int;
  mutable txt     : string vect;
  mutable vlin    : int;
  mutable nlin    : int;
  mutable ncol    : int;
  mutable clin    : int;
  mutable ccol    : int;
  mutable mrk_lin : int;
  mutable mrk_col : int;
  mutable nscroll : int
};;

let text_global_info, get_text_global_info = dynamo_global_info
  "text_global_info" (ref None: text_global_info option ref)
and text_local_info, get_text_local_info = dynamo_local_info
  "text_local_info" (ref None: text_local_info option ref)
;;

let text_border = ref 1
and text_band = ref 2
and text_font = ref "*-courier-medium-r-*-14-*"
;;

let lin_col_of_xy li x y =
  let gi = li.text_gi in
  let lin = (y-li.yb) / gi.tfs.fheight
  and col = (x-li.xb) / gi.tfs.fwidth in
  (max 0 (min (li.nlin-1) lin), max 0 (min (li.ncol-1) col))

and text_expose_lines =
  let i = ref 0 and j = ref 0
  and buff = ref "" in
fun wid imin imax jmin jmax ->
  let xd = wid.wid_xd
  and li = get_text_local_info wid.info in
  let gi = li.text_gi in
  if jmin > 0 & string_length !buff < jmax-jmin then (
    buff := make_string (jmax-jmin) ` `
  );
  i := imin;
  while !i < imax do
    let txt =
      if jmin = 0 then li.txt.(!i)
      else (
        j := jmin;
        while !j < jmax do
          set_nth_char !buff (!j-jmin) (nth_char li.txt.(!i) !j);
          incr j
        done;
        !buff
      ) in
    XDrawImageString(xd.dpy, li.swin, gi.gc_text,
      (li.xb+jmin*gi.tfs.fwidth),
      (li.yb+gi.tfs.fheight*!i+gi.tfs.ascent),
      txt, (jmax-jmin)
    );
    incr i
  done
;;

let text_scroll =
  let i = ref 0 and j = ref 0 in
fun wid ->
  let xd = wid.wid_xd
  and li = get_text_local_info wid.info in
  let gi = li.text_gi
  and b =  li.xb in
  XCopyArea(xd.dpy, li.swin, li.swin, gi.gc_text,
    b, (li.y_shift+gi.tfs.fheight),
    (li.ncol*gi.tfs.fwidth),
    (wid.height-gi.tfs.fheight),
    b, li.y_shift
  );
  li.nscroll <- li.nscroll+1;
  XClearArea(xd.dpy, li.swin,
    b, (li.y_shift+wid.height-gi.tfs.fheight),
    0, 0, 0
  );
  for i = 0 to li.nlin - 2 do
    replace_string li.txt.(i) li.txt.(i+1) 0
  done;
  let ll = li.txt.(li.nlin-1) in
  j := 0;
  while !j < li.ncol do set_nth_char ll !j ` `; incr j done;
  let (imin, _) = lin_col_of_xy li 0 (li.y_shift+wid.height-gi.tfs.fheight)
  and (imax, _) = lin_col_of_xy li 0 (li.y_shift+wid.height-1) in
  let imax = min (imax+1) (li.nlin-1) in
  let imin = min imin imax in
  text_expose_lines wid imin imax 0 li.ncol
;;

let text_home wid =
  let li = get_text_local_info wid.info in
  li.clin <- li.nlin - li.vlin;
  li.ccol <- 0

and text_current_line wid =
  let li = get_text_local_info wid.info in
  li.clin

and text_current_column wid =
  let li = get_text_local_info wid.info in
  li.ccol

and text_goto(wid, lin, col) =
  let li = get_text_local_info wid.info in
  li.clin <- max 0 (min (li.nlin-1) lin);
  li.ccol <- max 0 (min (li.ncol-1) col)

and text_set_mark(wid, lin, col) =
  let li = get_text_local_info wid.info in
  li.mrk_lin <- max 0 (min (li.nlin-1) lin);
  li.mrk_col <- max 0 (min (li.ncol-1) col)

and text_clear =
  let i = ref 0 and j = ref 0 in
fun wid ->
  let xd = wid.wid_xd
  and li = get_text_local_info wid.info in
  li.ccol <- 0; li.clin <- 0;
  XClearWindow(xd.dpy, li.swin);
  i := 0;
  while !i < li.nlin do
    let str = li.txt.(!i) in
    j := 0;
    while !j < li.ncol do set_nth_char str !j ` `; incr j done;
    incr i
  done

and text_shift(wid, val) =
  let xd = wid.wid_xd
  and li = get_text_local_info wid.info in
  li.y_shift <- max 0 (min li.y_shmax (val*li.text_gi.tfs.fheight));
  XMoveWindow(xd.dpy, li.swin, 0, (-li.y_shift))

and text_shift_value wid =
  let li = get_text_local_info wid.info in
  li.y_shift / li.text_gi.tfs.fheight

and text_get_text(wid, lin, col) =
  let li = get_text_local_info wid.info in
  let lin1 = li.mrk_lin and col1 = li.mrk_col in
  if lin = lin1 & 0 <= col1 & col1 <= col & col < li.ncol then
    sub_string li.txt.(lin) col1 (col-col1+1)
  else ""
;;

let text_send_string =
  let i = ref 0
  and buff = make_string 80 ` `
  and beglin = ref 0
  and begcol = ref 0
  and bufflen = ref 0 in
  let draw_char draw_flush lin col c =
    if lin <> !beglin or col <> !begcol+!bufflen
    or !bufflen = string_length buff then (
      draw_flush(); beglin := lin; begcol := col
    );
    set_nth_char buff !bufflen c;
    incr bufflen
  in
fun (wid, text) ->
  let xd = wid.wid_xd
  and li = get_text_local_info wid.info in
  let gi = li.text_gi in
  let len = string_length text in
  let draw_flush() =
    if !bufflen > 0 then (
      XDrawImageString(xd.dpy, li.swin, gi.gc_text,
        (li.xb+!begcol*gi.tfs.fwidth),
        (li.yb+gi.tfs.fheight*!beglin+gi.tfs.ascent),
        buff, !bufflen
      );
      bufflen := 0
    )
  in
  let newline() =
    li.ccol <- 0;
    li.clin <- li.clin+1;
    if li.clin = li.nlin then (
      draw_flush();
      text_scroll wid;
      li.clin <- li.clin-1
    )
  in
  i := 0; beglin := 0; begcol := 0; bufflen := 0;
  while !i < len do
    begin match nth_char text !i with
      `\b` ->
        if li.ccol > 0 then (li.ccol <- li.ccol-1)
        else if li.clin > 0 then (
            li.ccol <- li.ncol-1;
            li.clin <- li.clin-1
        )
    | `\n` ->
        newline()
    | `\t` ->
        li.ccol <- min li.ncol (((li.ccol+8) / 8) * 8)
    | c ->
        if li.ccol = li.ncol then newline();
        set_nth_char li.txt.(li.clin) li.ccol c;
        draw_char draw_flush li.clin li.ccol c;
        li.ccol <- li.ccol+1
    end;
    incr i
  done;
  draw_flush()
;;

let rt_set_cut_buffer(xd, str) =
  XSetSelectionOwner(xd.dpy, XA_PRIMARY, XNone, CurrentTime);
  XStoreBytes(xd.dpy, str, string_length str)

and rt_get_cut_buffer xd =
  let i = ref 0 in
  let b = XFetchBytes(xd.dpy, i) in
  let r = string_of_C_String(b, !i) in
  if !i > 0 then free_C_String b;
  r
;;

exception not_ascii;;

let TextA attr (vlin, vcol, nlin, key_callback, button_callback) =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = (function xd ->
    let make_global_info xd =
      let fs = rt_load_query_font(xd, !text_font) in
      let mask = GCForeground lor GCBackground lor fs.gc_mask
      and gstr = gstr() in
      set_XGCValues_font(fs.fid, gstr.xgcv);
      set_XGCValues_foreground(xd.black, gstr.xgcv);
      set_XGCValues_background(xd.white, gstr.xgcv);
      let gc_text = XCreateGC(xd.dpy, xd.rootw, mask, gstr.xgcv) in
      xd.end_func <- (function () ->
        let gi = get_text_global_info(ginfo xd "text") in
        XFreeGC(xd.dpy, gi.gc_text);
        remove_ginfo xd "text"
      ) :: xd.end_func;
      add_ginfo xd "text" text_global_info {
        tfs = fs;
        gc_text = gc_text
      }
    in
    let gi =
      try get_text_global_info(ginfo xd "text")
      with _ -> make_global_info xd
    and (vlin, vcol) = (max 1 vlin, max 1 vcol) in
    let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
      2*!text_band+gi.tfs.fwidth*vcol)
    and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
      2*!text_band+gi.tfs.fheight*vlin)
    and b = match szh with (_,_,Some v) -> v | _ -> !text_border
    in (w,h,b)
  )
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let width = max width 1 and height = max height 1 in
    let win = XCreateSimpleWindow(xd.dpy, pwin,
      x, y,
      width, height, (max border 0),
      xd.black, xd.black
    ) in
    XSelectInput(xd.dpy, win, StructureNotifyMask);
    let gi = get_text_global_info(ginfo xd "text") in

    let vlin = max 1 ((height-2*!text_band) / gi.tfs.fheight)
    and vcol = max 1 ((width-2*!text_band) / gi.tfs.fwidth) in
    let ncol = vcol in
    let nlin = max vlin nlin in
    let xb = (width-vcol*gi.tfs.fwidth) / 2
    and yb = (height-vlin*gi.tfs.fheight) / 2 in
    let sheight = height+(nlin-vlin)*gi.tfs.fheight in
    let y_shmax = sheight - height in
    let txt = make_vect nlin "" in
    for i = 0 to vect_length txt - 1 do
      txt.(i) <- make_string ncol ` `
    done;

    let swin = XCreateSimpleWindow(xd.dpy, win,
      0, 0,
      width, sheight,
      0, xd.black, xd.white
    ) in
    XSelectInput(xd.dpy, swin, it_list (prefix lor) 0 [
      ExposureMask; KeyPressMask;
      ButtonPressMask; ButtonReleaseMask
    ]);
    XMapWindow(xd.dpy, swin);
    let info = text_local_info {
      text_gi = gi; swin = swin;
      xb = xb; yb = yb;
      y_shift = 0; y_shmax = y_shmax;
      txt = txt;
      vlin = vlin; nlin = nlin; ncol = ncol;
      clin = 0; ccol = 0; mrk_lin = 0; mrk_col = 0;
      nscroll = 0
    } in
    add_widget [] swin (add_widget attr win {
      wid_xd = xd;
      win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = info; user_info = no_user_info;
      children = []
    })
  )
;
  wdestroy = (function wid ->
    let li = get_text_local_info wid.info in
    remove_widget [] li.swin wid;
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev) ->
    let xd = wid.wid_xd in
    let t = XEvent_type xev in
    if t = GraphicsExpose then (
      let xev = XEvent_xgraphicsexpose xev
      and li = get_text_local_info wid.info in
      let x = (XGraphicsExposeEvent_x xev)
      and y = (XGraphicsExposeEvent_y xev)
      and width = (XGraphicsExposeEvent_width xev)
      and height = (XGraphicsExposeEvent_height xev)
      and count = (XGraphicsExposeEvent_count xev) in
      if count = 0 then (li.nscroll <- li.nscroll-1);
      let (imin, jmin) = lin_col_of_xy li x y
      and (imax, jmax) = lin_col_of_xy li (x+width) (y+height) in
      let imin = max 0 (imin - li.nscroll)
      and imax = max 0 (imax - li.nscroll) in
      text_expose_lines wid imin (imax+1) jmin (jmax+1)
    )
    else if t = NoExpose then (
      let li = get_text_local_info wid.info in
      li.nscroll <- li.nscroll-1
    )
    else if t = Expose then (
      let xev = XEvent_xexpose xev
      and li = get_text_local_info wid.info in
      let x = (XExposeEvent_x xev)
      and y = (XExposeEvent_y xev)
      and width = (XExposeEvent_width xev)
      and height = (XExposeEvent_height xev) in
      let (imin, jmin) = lin_col_of_xy li x y
      and (imax, jmax) = lin_col_of_xy li (x+width) (y+height) in
      text_expose_lines wid imin (imax+1) jmin (jmax+1)
    )
    else if t = KeyPress then (
      let xev = XEvent_xkey xev in
      let state = (XKeyEvent_state xev) in
      let c = XKeycodeToKeysym(xd.dpy, XKeyEvent_keycode xev,
        (if state land ShiftMask != 0 then 1 else 0)
      ) in
      let cc =  c in
      let cc = if state land ControlMask != 0 then cc land 31 else cc in
      try
        let a =
          match cc with
            65288 | 65535 -> "\b" | 65293 -> "\n"
          | 65361 -> "Left" | 65362 -> "Up"  | 65363 -> "Right"
          | 65364 -> "Down" | 65379 -> "Ins"
          | 65456 -> "K0" | 65457 -> "K1" | 65458 -> "K2" | 65459 -> "K3"
          | 65460 -> "K4" | 65461 -> "K5" | 65462 -> "K6" | 65463 -> "K7"
          | 65464 -> "K8" | 65465 -> "K9"
          | 65496 -> "R7"  | 65498 -> "R9"  | 65500 -> "R11"
          | 65502 -> "R13" | 65504 -> "R15"
          | _ -> try make_string 1 (char_of_int cc) with _ -> raise not_ascii
        in
        key_callback(wid, a)
      with not_ascii -> ()
    )
    else if t = ButtonPress then (
      let xev = XEvent_xbutton xev
      and li = get_text_local_info wid.info in
      let b = (XButtonEvent_button xev)
      and x = (XButtonEvent_x xev)
      and y = (XButtonEvent_y xev) in
      let i,j = lin_col_of_xy li x y in
      button_callback(wid, b, i, j)
    )
    else if t = ButtonRelease then (
      if xd.win_but == C'WB_None then (
        let xev = XEvent_xbutton xev
        and li = get_text_local_info wid.info in
        let b = (XButtonEvent_button xev)
        and x = (XButtonEvent_x xev)
        and y = (XButtonEvent_y xev) in
        let i,j = lin_col_of_xy li x y in
        button_callback(wid, -b, i, j)
      )
    )
    else if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      let width = (XConfigureEvent_width xev)
      and height = (XConfigureEvent_height xev) in
      if width <> wid.width or height <> wid.height then (
        wid.width <- width; wid.height <- height;
        let li = get_text_local_info wid.info in
        let gi = li.text_gi
        and wdesc = wid.wdesc in

    let vlin = max 1 ((height-2*!text_band) / gi.tfs.fheight)
    and vcol = max 1 ((width-2*!text_band) / gi.tfs.fwidth) in
    let ncol = vcol in
    let nlin = max vlin nlin in
    let xb = (width-vcol*gi.tfs.fwidth) / 2
    and yb = (height-vlin*gi.tfs.fheight) / 2 in
    let sheight = height+(nlin-vlin)*gi.tfs.fheight in
    let y_shmax = sheight - height in
    let txt = make_vect nlin "" in
    for i = 0 to vect_length txt - 1 do
      txt.(i) <- make_string ncol ` `
    done;

        for i = 0 to vect_length txt - 1 do
          if i < li.nlin then (
            for j = 0 to string_length txt.(i) - 1 do
              if j < string_length li.txt.(i) then
                set_nth_char txt.(i) j (nth_char li.txt.(i) j)
            done
          )
        done;
        let y_shift = min li.y_shift y_shmax in
        XMoveResizeWindow(xd.dpy, li.swin,
          0, (-y_shift),
          width, sheight
        );
        li.xb <- xb; li.yb <- yb;
        if li.clin >= nlin or li.ccol >= ncol then (li.ccol <- ncol);
        if li.clin >= nlin then (li.clin <- nlin-1);
        li.vlin <- vlin; li.nlin <- nlin; li.ncol <- ncol;
        li.txt <- txt;
        li.y_shmax <- y_shmax; li.y_shift <- y_shift
      )
    )
  )
;
  filler = mem C'FillerAtt attr
}
;;

let TextD = TextA []
;;
