(* $Id: c_scroll.ml,v 1.8 92/07/26 21:03:18 ddr Exp $
 *
 * Rogloglo Toolkit: scroll bar widget class
 *)

#open "xlib";;
#open "cursorfont";;
#open "std";;
#open "rtdef";;
#open "util";;

type scroll_global_info = {
  h_cursor  : Cursor vect;
  v_cursor  : Cursor vect;
  gray_pixm : Pixmap
}

and scroll_local_info = {
  orient        : orientation;
  vmin          : int;
  vmax          : int;
  bsize         : int;
  mutable vcur  : int;
  cursor        : Cursor vect;
  swin          : Window
}
;;

let scroll_global_info, get_scroll_global_info = dynamo_global_info
  "scroll_global_info" (ref (None: scroll_global_info option))
and scroll_local_info, get_scroll_local_info = dynamo_local_info
  "scroll_local_info" (ref (None: scroll_local_info option))
;;

let scroll_border = ref 1
and scroll_width = ref 14
and scroll_band = ref 1
;;

let nround x y =
  if x>0 & y> 0 or x<0 & y<0 then (2*x+y) / (2*y)
  else (2*x-y) / (2*y)
;;

let scr_callb wid args xev but f_x f_y =
  let (orient, vmin, vmax, _, callb) = args in
  let val = vmin + (
    if orient = C'Vertical then (
      nround (((f_y xev))*(vmax-vmin)) wid.height
    ) else (
      nround (((f_x xev))*(vmax-vmin)) wid.width
    )
  ) in
  callb(wid, but, max vmin (min val vmax))

and scroll_set(wid, val) =
  let li = get_scroll_local_info wid.info
  and xd = wid.wid_xd in
  let gbsize = (if li.bsize = 0 then li.vmax-li.vmin else li.bsize) in
  let val = max li.vmin (min val (li.vmax+gbsize)) in
  li.vcur <- val;
  let x = (if li.orient = C'Vertical then !scroll_band
    else nround (wid.width*(val-li.vmin-gbsize)) (li.vmax-li.vmin)
  )
  and y = (if li.orient = C'Horizontal then !scroll_band
    else nround (wid.height*(val-li.vmin-gbsize)) (li.vmax-li.vmin)
  ) in
  XMoveWindow(xd.dpy, li.swin, x, y)

and scroll_val wid =
  (get_scroll_local_info wid.info).vcur
;;

let get_or_make_scroll_global_info xd =
  try get_scroll_global_info(ginfo xd "scroll")
  with _ ->
    let h_cursor = [|
      XCreateFontCursor(xd.dpy, XC_sb_h_double_arrow);
      XCreateFontCursor(xd.dpy, XC_sb_left_arrow);
      XCreateFontCursor(xd.dpy, XC_sb_up_arrow);
      XCreateFontCursor(xd.dpy, XC_sb_right_arrow)
    |] in
    let v_cursor = [|
      XCreateFontCursor(xd.dpy, XC_sb_v_double_arrow);
      h_cursor.(2);
      h_cursor.(3);
      XCreateFontCursor(xd.dpy, XC_sb_down_arrow)
    |]
    and pix = XCreatePixmapFromBitmapData(
      xd.dpy, xd.rootw, implode_ascii[85; 170],
      8, 2,
      xd.black, xd.white, xd.depth
    ) in
    xd.end_func <- (function () ->
      let gi = get_scroll_global_info(ginfo xd "scroll") in
      XFreePixmap(xd.dpy, gi.gray_pixm);
      XFreeCursor(xd.dpy, gi.v_cursor.(3));
      XFreeCursor(xd.dpy, gi.v_cursor.(0));
      XFreeCursor(xd.dpy, gi.h_cursor.(3));
      XFreeCursor(xd.dpy, gi.h_cursor.(2));
      XFreeCursor(xd.dpy, gi.h_cursor.(1));
      XFreeCursor(xd.dpy, gi.h_cursor.(0));
      remove_ginfo xd "scroll"
    ) :: xd.end_func;
    add_ginfo xd "scroll" scroll_global_info {
      h_cursor = h_cursor;
      v_cursor = v_cursor;
      gray_pixm = pix
    }
;;

let ScrollA attr (orient, vmin, vmax, bsize, _ as args) =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = (function xd ->
    let gi = get_or_make_scroll_global_info xd in
    let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
      (if orient = C'Vertical then !scroll_width+2*!scroll_band else 1))
    and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
      (if orient = C'Vertical then 1 else !scroll_width+2*!scroll_band))
    and b = match szh with (_,_,Some v) -> v | _ -> !scroll_border
    in (w,h,b)
  )
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let width = max 1 width
    and height = max 1 height in
    let win = create_window(
      xd, pwin, x, y, width, height, border, attr,
      it_list (prefix lor) 0 [
        ButtonPressMask; ButtonReleaseMask;
        Button2MotionMask; StructureNotifyMask
      ]
    ) in
    let gbsize = (if bsize = 0 then vmax-vmin else bsize) in
    let ebsize = nround
      ((if orient = C'Vertical then height else width) * gbsize)
      (vmax - vmin)
    in
    let sx = if orient = C'Vertical then !scroll_band else 0
    and sy = if orient = C'Vertical then 0 else !scroll_band
    and sw = if orient = C'Vertical then width-2*!scroll_band else ebsize
    and sh = if orient = C'Vertical then ebsize else height-2*!scroll_band in
    let swin = XCreateSimpleWindow(
      xd.dpy, win, sx, sy,
      sw, sh, 0, xd.black, xd.white
    )
    and ginfo = ginfo xd "scroll" in
    let gi = get_scroll_global_info ginfo in
    XSetWindowBackgroundPixmap(xd.dpy, swin, gi.gray_pixm);
    XMapSubwindows(xd.dpy, win);
    let cursor = if orient = C'Vertical then gi.v_cursor else gi.h_cursor in
    XDefineCursor(xd.dpy, win, cursor.(0));
    let info = scroll_local_info {
      orient = orient; vmin = vmin; vmax = vmax; bsize = bsize;
      vcur = vmin+gbsize; cursor = cursor;
      swin = swin
    } in
    add_widget attr win {
      wid_xd = xd; win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = info; user_info = no_user_info;
      children = []
    }
  )
;
  wdestroy = (function wid ->
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev) ->
    let t = XEvent_type xev
    and xd = wid.wid_xd in
    if t = ButtonPress then (
      let xev = XEvent_xbutton xev
      and li = get_scroll_local_info wid.info in
      let but = (XButtonEvent_button xev) in
      XDefineCursor(xd.dpy, wid.win, li.cursor.(but));
      scr_callb wid args xev but XButtonEvent_x XButtonEvent_y
    )
    else if t = ButtonRelease then (
      let li = get_scroll_local_info wid.info in
      XDefineCursor(xd.dpy, wid.win, li.cursor.(0))
    )
    else if t = MotionNotify then (
      let a = (xd.dpy, ButtonMotionMask, xev) in
      while XCheckMaskEvent a <> 0 do () done;
      scr_callb wid args (XEvent_xmotion xev) 0 XMotionEvent_x XMotionEvent_y
    )
    else if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      let width = (XConfigureEvent_width xev)
      and height = (XConfigureEvent_height xev) in
      if width <> wid.width or height <> wid.height then (
        let li = get_scroll_local_info wid.info in
        wid.width <- width; wid.height <- height;
        let bsize = (if bsize = 0 then vmax-vmin else bsize) in
        let ebsize = nround
          ((if orient = C'Vertical then height else width) * bsize)
          (vmax - vmin)
        in
        let sw = if orient = C'Vertical then width-2*!scroll_band else ebsize
        and sh = if orient = C'Vertical then ebsize else height-2*!scroll_band
        in let sw = max 1 sw and sh = max 1 sh in
        XResizeWindow(xd.dpy, li.swin, sw, sh);
        scroll_set(wid, li.vcur)
      )
    )
  )
;
  filler = mem C'FillerAtt attr
}
;;

let ScrollD = ScrollA []
;;
