(* $Id: c_pack.ml,v 1.7 92/07/26 21:03:14 ddr Exp $
 *
 * Rogloglo Toolkit: pack widget class
 *)

#open "xlib";;
#open "std";;
#open "rtdef";;
#open "util";;

type pack_local_info = {
  widlist : widget list
}
;;

let pack_local_info, get_pack_local_info = dynamo_local_info
  "pack_local_info" (ref (None: pack_local_info option))
;;

let pack_border = ref 1
and pack_band = ref 4
and pack_inter = ref 2
;;

let pack_wsize min_size szh (orient, wlist) xd =
  let (width, height) = it_list (fun (sw,sh) wdesc ->
    let (w,h,b) = wdesc.wsize xd in
    if orient = C'Vertical then
      let nsw = max(w+2*b) sw
      and nsh = sh+h+2*b+!pack_inter in
      (nsw, (if min_size & wdesc.filler then nsh-h+1 else nsh))
    else
      let nsw = sw+w+2*b+!pack_inter
      and nsh = max(h+2*b) sh in
      ((if min_size & wdesc.filler then nsw-w+1 else nsw), nsh)
  ) (0, 0) wlist in
  let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
    width+2*!pack_band-(if orient = C'Vertical then 0 else !pack_inter))
  and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
    height+2*!pack_band-(if orient = C'Vertical then !pack_inter else 0))
  and b = match szh with (_,_,Some v) -> v | _ ->
    !pack_border
  in (w,h,b)
;;

let do_pack_widlist xd szh args width height a_cdr a_list
b_cons b_nil b_make =
  let (FW, FH, _) = pack_wsize false szh args xd
  and (MW, MH, _) = pack_wsize true szh args xd
  and (orient, wdlist) = args in
  let nfill = it_list (fun n wdesc ->
    if wdesc.filler then n+1 else n
  ) 0 wdlist in
  let all_filler = (nfill = 0) in
  let nfill = (if all_filler then list_length wdlist else nfill) in
  let too_small = if orient = C'Vertical then height < MH else width < MW in
  let W = width-2*!pack_band
  and H = height-2*!pack_band in
  if not too_small then (
    let FW = FW-2*!pack_band
    and FH = FH-2*!pack_band in
    let rec action_loop x y err a_list = function
      wdesc::wdl ->
        if wdesc.filler or all_filler then (
          let (fw, fh, b) = wdesc.wsize xd in
          let (ow, oh) =
            if orient = C'Vertical then (W*nfill, err+(fh+2*b)*nfill+H-FH)
            else (err+(fw+2*b)*nfill+W-FW, H*nfill)
          in
          let (dx, dy) =
            if orient = C'Vertical then (0, oh+!pack_inter*nfill)
            else (ow+!pack_inter*nfill, 0)
          in
          let err =
            if orient = C'Vertical then dy mod nfill else dx mod nfill
          in
          let k = b_make wdesc a_list x y
            (max((ow / nfill)-2*b)1) (max((oh / nfill)-2*b)1) b
          in
          b_cons k (
            action_loop (x+(dx / nfill)) (y+(dy / nfill)) err
            (a_cdr a_list) wdl
          )
        )
        else (
          let (fw, fh, b) = wdesc.wsize xd in
          let (ow, oh, dx, dy) =
            if orient = C'Vertical then
              (W, fh+2*b, 0, fh+2*b+!pack_inter)
            else
              (fw+2*b, H, fw+2*b+!pack_inter, 0)
          in
          let k = b_make wdesc a_list x y (max(ow-2*b)1) (max(oh-2*b)1) b in
          b_cons k (
            action_loop (x+dx) (y+dy) err (a_cdr a_list) wdl
          )
        )
    | _ -> b_nil
    in
    action_loop !pack_band !pack_band 0 a_list wdlist
  ) else (
    let FW = (if too_small then MW else FW)-2*!pack_band
    and FH = (if too_small then MH else FH)-2*!pack_band in
    let rec action_loop x y err a_list = function
      wdesc::wdl ->
        let (fw, fh, b) = wdesc.wsize xd in
        let (fw, fh) =
          if too_small & wdesc.filler then
            if orient = C'Vertical then (fw, 1) else (1, fh)
          else (fw, fh)
        in
        let (ow, oh, dx, dy) =
          if orient = C'Vertical then
            (W, (err+(fh+2*b)*H)/ FH, 0, err+(fh+2*b+!pack_inter)*H)
          else
            ((err+(fw+2*b)*W)/ FW, H, err+(fw+2*b+!pack_inter)*W, 0)
        in
        let err =
          if orient = C'Vertical then dy mod FH else dx mod FW
        in
        let k = b_make wdesc a_list x y (max(ow-2*b)1) (max(oh-2*b)1) b in
        b_cons k (
          action_loop (x+(dx / FW)) (y+(dy / FH)) err (a_cdr a_list)
          wdl
        )
    | _ -> b_nil
    in
    action_loop !pack_band !pack_band 0 a_list wdlist
  )
;;

let PackA attr args =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = pack_wsize false szh args
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let win = create_window(
      xd, pwin, x, y, width, height, border, attr, StructureNotifyMask
    ) in
    let widlist = do_pack_widlist xd szh args width height
      (fun x -> x) () (fun x y -> x::y) [] (fun wdesc _ x y w h b ->
        wdesc.wcreate(xd, win, wdesc, x, y, w, h, b)
    ) in
    XMapSubwindows(xd.dpy, win);
    let info = pack_local_info {widlist = widlist} in
    add_widget attr win {
      wid_xd = xd; win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = info; user_info = no_user_info;
      children = widlist
    }
  )
;
  wdestroy = (function wid ->
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev) ->
    let t = XEvent_type xev in
    if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      let x = (XConfigureEvent_x xev)
      and y = (XConfigureEvent_y xev)
      and width = (XConfigureEvent_width xev)
      and height = (XConfigureEvent_height xev) in
      wid.x <- x; wid.y <- y;
      if width <> wid.width or height <> wid.height then (
        wid.width <- width; wid.height <- height;
        let xd = wid.wid_xd
        and li = get_pack_local_info wid.info in
        do_pack_widlist xd szh args width height tl li.widlist
          (fun _ y -> y) () (fun _ widl x y w h b ->
            let wid = hd widl in
            XMoveResizeWindow(xd.dpy, wid.win, x, y, w, h)
        )
      )
    )
  )
;
  filler = mem C'FillerAtt attr
}
;;

let PackD = PackA []
;;
