(* $Id: c_button.ml,v 1.10 92/08/07 16:38:29 ddr Exp $
 *
 * Rogloglo Toolkit: button widget class
 *)

#open "xlib";;
#open "std";;
#open "rtdef";;
#open "font";;
#open "util";;

type button_global_info = {
  bfs                     : font;
  gc_normal               : GC;
  gc_invert               : GC;
  gc_bold                 : GC
}
;;

let button_global_info, get_button_global_info = dynamo_global_info
  "button_global_info" (ref (None: button_global_info option))
;;

let button_border = ref 2
and button_band = ref 1
and button_bold = ref 2
and button_font = ref "*-helvetica-bold-r-*--14-*"
;;

let get_or_make_button_global_info xd =
  try get_button_global_info(ginfo xd "button")
  with _ ->
    let fs = rt_load_query_font(xd, !button_font) in
    let mask =
      GCForeground lor GCBackground lor
      GCLineWidth lor GCCapStyle
    in
    let xgcv = (gstr()).xgcv in
    set_XGCValues_foreground(xd.black, xgcv);
    set_XGCValues_background(xd.white, xgcv);
    set_XGCValues_line_width(!button_bold, xgcv);
    set_XGCValues_cap_style(CapProjecting, xgcv);
    let gc_bold = XCreateGC(xd.dpy, xd.rootw, mask, xgcv) in
    let mask = GCForeground lor GCBackground lor fs.gc_mask in
    set_XGCValues_font(fs.fid, xgcv);
    set_XGCValues_foreground(xd.black, xgcv);
    set_XGCValues_background(xd.white, xgcv);
    let gc_normal = XCreateGC(xd.dpy, xd.rootw, mask, xgcv) in
    set_XGCValues_foreground(xd.white, xgcv);
    set_XGCValues_background(xd.black, xgcv);
    let gc_invert = XCreateGC(xd.dpy, xd.rootw, mask, xgcv) in
    xd.end_func <- (function () ->
      let gi = get_button_global_info(ginfo xd "button") in
      XFreeGC(xd.dpy, gi.gc_bold);
      XFreeGC(xd.dpy, gi.gc_invert);
      XFreeGC(xd.dpy, gi.gc_normal);
      remove_ginfo xd "button"
    ) :: xd.end_func;
    add_ginfo xd "button" button_global_info {
      bfs = fs;
      gc_normal = gc_normal;
      gc_invert = gc_invert;
      gc_bold = gc_bold
    }
;;

let exp_text(xd, wid, txt, comm) =
  let gi = get_button_global_info(ginfo xd "button") in
  XClearWindow(xd.dpy, wid.win);
  let len = (string_length txt) in
  XDrawString(xd.dpy, wid.win, gi.gc_normal,
    (if comm then (!button_bold+!button_band) else
    ((wid.width-(XTextWidth(gi.bfs.fs, txt, len))) / 2)),
    ((wid.height+gi.bfs.ascent-gi.bfs.descent) / 2),
    txt, len)
and exp_high(xd, wid, txt, comm) =
  let gi = get_button_global_info(ginfo xd "button") in
  let T = 0 in
  XDrawRectangle(xd.dpy, wid.win, gi.gc_bold,
    ((!button_bold / 2)+T), ((!button_bold / 2)+T),
    (wid.width-!button_bold-2*T), (wid.height-!button_bold-2*T))
and exp_nohigh(xd, wid, txt, comm) =
  let gi = get_button_global_info(ginfo xd "button") in
  XClearWindow(xd.dpy, wid.win);
  let len = (string_length txt) in
  XDrawString(xd.dpy, wid.win, gi.gc_normal,
    (if comm then (!button_bold+!button_band) else
    ((wid.width-(XTextWidth(gi.bfs.fs, txt, len))) / 2)),
    ((wid.height+gi.bfs.ascent-gi.bfs.descent) / 2),
    txt, len)
and exp_inv(xd, wid, txt, comm) =
  let gi = get_button_global_info(ginfo xd "button") in
  XFillRectangle(
    xd.dpy, wid.win, gi.gc_normal, 0, 0,
    wid.width, wid.height);
  let len = (string_length txt) in
  XDrawString(xd.dpy, wid.win, gi.gc_invert,
    (if comm then (!button_bold+!button_band) else
    ((wid.width-(XTextWidth(gi.bfs.fs, txt, len))) / 2)),
    ((wid.height+gi.bfs.ascent-gi.bfs.descent) / 2),
    txt, len)
;;

let select_mask = it_list (prefix lor) 0 [
  ExposureMask; EnterWindowMask; LeaveWindowMask;
  ButtonPressMask; ButtonReleaseMask; StructureNotifyMask
];;

let GenButtonA popup comm attr (txt, act) =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = (function xd ->
    let gi = get_or_make_button_global_info xd in
    let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
      2*(!button_band+!button_bold)+
        (XTextWidth(gi.bfs.fs, txt, string_length txt)))
    and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
      2*(!button_band+!button_bold)+gi.bfs.fheight)
    and b = match szh with (_,_,Some v) -> v | _ ->
      if comm then 0 else !button_border
    in (w,h,b)
  )
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let win = create_window(xd, pwin, x, y, width, height, border, attr,
      if popup then select_mask lor OwnerGrabButtonMask
      else select_mask
    ) in
    add_widget attr win {
      wid_xd = xd; win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = no_info; user_info = no_user_info;
      children = []
    }
  )
;
  wdestroy = (function wid ->
    let xd = wid.wid_xd in
    (match xd.win_but with
      C'WB_Win win -> if win == wid.win then (xd.win_but <- C'WB_None)
    | _ -> ());
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev) ->
    let xd = wid.wid_xd in
    let t = XEvent_type xev in
    if t = Expose then exp_text(xd, wid, txt, comm)
    else if t = EnterNotify then (
      match xd.win_but with
        C'WB_None -> xd.win_but <- C'WB_Win wid.win;
          exp_high(xd, wid, txt, comm)
      | C'WB_Win win -> ()
      | C'WB_WinBut win -> ()
      | C'WB_WinButExit win ->
          if win = wid.win or comm then (
            xd.win_but <- C'WB_WinBut wid.win;
            exp_inv(xd, wid, txt, comm)
          )
          else if popup then (
            xd.win_but <- C'WB_WinBut wid.win;
            exp_inv(xd, wid, txt, comm);
            let xev = XEvent_xcrossing xev in
            xd.xevent.x_win <- (XCrossingEvent_x xev);
            xd.xevent.y_win <- (XCrossingEvent_y xev);
            xd.xevent.x_root <- (XCrossingEvent_x_root xev);
            xd.xevent.y_root <- (XCrossingEvent_y_root xev);
            act wid
          )
          else (
            xd.win_but <- C'WB_WinButOther win
          )
      | C'WB_WinButOther win -> ()
      | C'WB_But -> xd.win_but <- C'WB_ButWin
      | C'WB_ButWin -> ()
    )
    else if t = LeaveNotify then (
      match xd.win_but with
        C'WB_None -> ()
      | C'WB_Win win -> xd.win_but <- C'WB_None;
          exp_nohigh(xd, wid, txt, comm)
      | C'WB_WinBut win -> xd.win_but <- C'WB_WinButExit win;
          exp_text(xd, wid, txt, comm)
      | C'WB_WinButExit win ->
          do_list (fun win -> XUnmapWindow(xd.dpy, win)) xd.popped_up;
          xd.popped_up <- []; xd.win_but <- C'WB_None
      | C'WB_WinButOther win -> xd.win_but <- C'WB_WinButExit win
      | C'WB_But -> ()
      | C'WB_ButWin -> xd.win_but <- C'WB_But
    )
    else if t = ButtonPress then (
      match xd.win_but with
        C'WB_None -> ()
      | C'WB_Win win -> xd.win_but <- C'WB_WinBut win;
          exp_inv(xd, wid, txt, comm);
          if popup then (
            let xev = XEvent_xbutton xev in
            xd.xevent.x_win <- (XButtonEvent_x xev);
            xd.xevent.y_win <- (XButtonEvent_y xev);
            xd.xevent.x_root <- (XButtonEvent_x_root xev);
            xd.xevent.y_root <- (XButtonEvent_y_root xev);
            xd.xevent.button <- (XButtonEvent_button xev);
            act wid
          )
      | C'WB_WinBut win -> ()
      | C'WB_WinButExit win -> ()
      | C'WB_WinButOther win -> ()
      | C'WB_But -> ()
      | C'WB_ButWin -> ()
    )
    else if t = ButtonRelease then (
      do_list (fun win -> XUnmapWindow(xd.dpy, win)) xd.popped_up;
      xd.popped_up <- [];
      match xd.win_but with
        C'WB_None -> ()
      | C'WB_Win win -> ()
      | C'WB_WinBut win -> xd.win_but <- C'WB_Win win;
          exp_text(xd, wid, txt, comm);
          exp_high(xd, wid, txt, comm);
          if popup then ()
          else (
            let xev = XEvent_xbutton xev in
            xd.xevent.x_win <- (XButtonEvent_x xev);
            xd.xevent.y_win <- (XButtonEvent_y xev);
            xd.xevent.x_root <- (XButtonEvent_x_root xev);
            xd.xevent.y_root <- (XButtonEvent_y_root xev);
            xd.xevent.button <- (XButtonEvent_button xev);
            act wid
          )
      | C'WB_WinButExit win -> xd.win_but <- C'WB_None
      | C'WB_WinButOther win -> xd.win_but <- C'WB_Win wid.win;
          exp_high(xd, wid, txt, comm)
      | C'WB_But -> xd.win_but <- C'WB_None
      | C'WB_ButWin -> xd.win_but <- C'WB_Win wid.win;
          exp_high(xd, wid, txt, comm)
    )
    else if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      wid.width <- (XConfigureEvent_width xev);
      wid.height <- (XConfigureEvent_height xev)
    )
  )
;
  filler = mem C'FillerAtt attr
}
;;

let ButtonA = GenButtonA false false
and PopupA = GenButtonA true false
and CommA = GenButtonA false true
;;

let ButtonD = ButtonA []
and PopupD = PopupA []
and CommD = CommA []
;;
