(* $Id: welltris.ml,v 1.2 91/06/29 17:08:54 ddr Exp $ *)

#open "rt";;
#open "sys";;

let tee (f,g) x = f x,g x
and max x y = if x > y then x else y
and min x y = if x < y then x else y
and prefix quo = prefix /
and add x y = x+y
and B f g x = f (g x)
and seed = ref 7
and maxint = 536870911*2+1
and iterate f = (iterate_f
  where rec iterate_f n x =
    if n > 0 then iterate_f (pred n) (f x))
and it_vect f i v = it_list f i (list_of_vect v)
and prefix mod x y = x mod y + (if x < 0 then y else 0)
;;
let sigma = it_list add 0
and random n =
  seed := 25173*!seed + 13849;
  (if !seed < 0 then - !seed else !seed) / (maxint / n)
and init_random n = seed := n; ()
and implode_ascii l =
  let len = list_length l in
  let s = create_string len in
  iterate (fun (i,hd::tl) ->
    set_nth_char s i (char_of_int hd); (i+1, tl)
  | _ -> failwith "implode_ascii"
  ) len (0, l);
  s
;;

let gloop f p = (looprec where rec
  looprec x = if p x then x else looprec(f x))
;;

type game_data = {
  xd                        : xdata;
  xargs                     : xargs;
  pix                       : pixmap;
  mutable state             : game_state;
  mutable level             : int;
  mutable score             : int;
  mutable nlines            : int;
  mutable explosions_to_do  : bool;
  mutable expl_list         : int list vect;
  mutable condemn           : int vect;
  patt                      : my_patt -> pattern;
  runner	            : runner;
  board                     : square ref vect vect
}

and game_state = NotStarted | Running | Pausing | Ended | Quit

and runner = {
  mutable i : int;
  mutable j : int;
  mutable p : int;
  mutable s : int;
  mutable r : int;
  mutable dl : (int * int) list
}

and square = Square | Empty | Moribund

and pieces = {
  frq       : int;
  dsc       : (int * int) list
}

and my_patt =
  C'WhiteP | C'BlackP | C'DiagonalP
| C'SandP | C'ChBoardP | C'VerticalP | C'GrayP
;;

let welltris_wizard = ref false
;;

let UpperB = 70 and LeftB = 10
and LowerB = 10 and RightB = 10
(*
and XM = 23 and YM = 19 and ZM = 17
and ECH = 40 (* and Z1 = 600/150 *)
*)
and XM = 8 and YM = 8 and ZM = 12
and ECH = 80 (* and Z1 = 520/80 *)
and speed_tab, nlevels = tee (vect_item, vect_length)
  [|1000; 800; 600; 500; 400; 300|]
and change_level = 500
;;
let IM = 2*(XM+YM) and JM = ZM+(max XM YM) and XZM = ZM+XM and YZM = ZM+YM
and L1 = XM and L2 = XM+YM and L3 = XM+YM+XM and L4 = XM+YM+XM+YM
;;
let W = ECH*XM and H = ECH*YM
and CXE = ECH*(XM quo 2) and CYE = ECH*(YM quo 2) and Z1E = 520
;;

let GW = LeftB+W+1+RightB
and GH = UpperB+H+1+LowerB
and CONDEMN_COUNT = 60
;;

let sigma = it_list add 0
;;

let pieces, SUMFRQ = tee
  (vect_item, B (it_list (fun v p -> v+p.frq) 0) list_of_vect)
[|
  {frq=  8; dsc= [(0, 0); (1, 0)]};
  {frq=  8; dsc= [(0, 0); (1, 0); (-1, 0)]};
  {frq=  8; dsc= [(0, 0); (1, 0); (0, 1)]};
  {frq=  8; dsc= [(0, 0); (1, 0); (2, 0); (-1, 0)]}  (* I *);
  {frq=  8; dsc= [(0, 0); (1, 0); (1, 1); (-1, 0)]}  (* L *);
  {frq=  8; dsc= [(0, 0); (1, 0); (1, 1); (0, -1)]}  (* N *);
  {frq=  8; dsc= [(0, 0); (1, 0); (1, 1); (0, 1)]}   (* O *);
  {frq=  8; dsc= [(0, 0); (1, 0); (0, 1); (-1, 0)]}  (* T *);
  {frq=  5; dsc= [(0, 0); (0, 1); (1, 1); (-1, 0); (0, -1)]}  (* F *);
  {frq=  5; dsc= [(0, 0); (0, 1); (0, 2); (0, -1); (0, -1)]}  (* I *);
  {frq=  5; dsc= [(0, 0); (1, 0); (2, 0); (0, 1); (0, 2)]}    (* V *);
  {frq=  5; dsc= [(0, 0); (1, 0); (1, 1); (0, -1); (-1, -1)]} (* W *);
  {frq=  5; dsc= [(0, 0); (1, 0); (-1, 0); (0, 1); (0, -1)]}  (* X *);
  {frq=  5; dsc= [(0, 0); (1, 0); (0, -1); (-1, 0); (-2, 0)]} (* Y *);
  {frq=  5; dsc= [(0, 0); (1, 0); (1, 1); (-1, 0); (-1, -1)]} (* Z *)
|]

and rotmat = vect_item [|
  vect_item
    [| (1, 0, 0, 1); (0, 1, -1, 0); (-1, 0, 0, -1); (0, -1, 1, 0) |];
  vect_item
    [| (1, 0, 0, -1); (0, 1, 1, 0); (-1, 0, 0, 1); (0, -1, -1, 0) |]
|]

and plan_center = vect_item
  [| L1 quo 2; (L1+L2) quo 2; (L2+L3) quo 2; (L3+L4) quo 2 |]

and dijm_of_p = vect_item [|
  (0, L1, ZM+(YM quo 2)); (L1, L2, ZM+(XM quo 2));
  (L2, L3, ZM+(YM quo 2)); (L3, L4, ZM+(XM quo 2))
|]
;;

let proj(x, y, z) =
  LeftB+ CXE+(((x*ECH-CXE)*Z1E)quo(z*ECH+Z1E)),
  UpperB+CYE+(((y*ECH-CYE)*Z1E)quo(z*ECH+Z1E))

and p_of_i(i) =
  if i < L1 then 0
  else if i < L2 then 1
  else if i < L3 then 2
  else 3
;;

let proj_sh(x, y, z, dx, dy) =
  let (X, Y) = proj(x, y, z) in (X+dx, Y+dy)
;;

let condemn_polyg = vect_item [|
  [
    proj_sh(0, 0, 0, 1, 0); proj_sh(0, 0, ZM, 1, -1);
    proj_sh(XM, 0, ZM, -1, -1); proj_sh(XM, 0, 0, -1, 0);
    proj_sh(0, 0, 0, 1, 0)
  ];
  [
    proj_sh(XM, 0, 0, 0, 1); proj_sh(XM, 0, ZM, 1, 1);
    proj_sh(XM, YM, ZM, 1, -1); proj_sh(XM, YM, 0, 0, -1);
    proj_sh(XM, 0, 0, 0, 1)
  ];
  [
    proj_sh(0, YM, 0, 1, 0); proj_sh(0, YM, ZM, 1, 1);
    proj_sh(XM, YM, ZM, -1, 1); proj_sh(XM, YM, 0, -1, 0);
    proj_sh(0, YM, 0, 1, 0)
  ];
  [
    proj_sh(0, 0, 0, 0, 1); proj_sh(0, 0, ZM, -1, 1);
    proj_sh(0, YM, ZM, -1, -1); proj_sh(0, YM, 0, 0, -1);
    proj_sh(0, 0, 0, 0, 1)
  ]
|]
;;

let disp_score(wid, gm) =
  rt_erase_draw_string(wid, LeftB+10, 25,
    if !welltris_wizard then "score <" ^ string_of_int(gm.score) ^ ">   "
    else "score " ^ string_of_int(gm.score) ^ "   "
  )

and disp_nlines(wid, gm) =
  rt_erase_draw_string(wid, LeftB+10, UpperB - 15,
    "nb lines " ^ string_of_int(gm.nlines))

and disp_state(wid, gm) =
  rt_erase_draw_string(wid, GW-RightB-80, UpperB - 15,
    match gm.state with
      Pausing ->   "    <pause>"
    | Ended ->     "<game over>"
    | _ ->          "           "
  )
;;

let possible(board, i, j, c) =
  if i < 0 or j < 0 then true
  else if j = JM or !(board.(i).(j)) = Square then false
  else if i < L1 then (j < YZM) & c.(0) = 0
  else if i < L2 then (j < XZM) & c.(1) = 0
  else if i < L3 then (j < YZM) & c.(2) = 0
  else (j < XZM) & c.(3) = 0

and incr_score(wid, gm, v) =
  for i = 1 to v do
    gm.score <- gm.score+1;
    if (gm.score mod change_level = 0) then (
      if gm.level < nlevels-1 then (
        gm.level <- gm.level+1;
        rt_set_timeout(gm.xargs, rt_current_time gm.xargs + speed_tab gm.level);
        (*disp_level(wid, gm)*)
        print_string "level "; print_int gm.level; print_newline()
      )
    )
  done
;;

let possible_move(gm, i, j, r) =
  let (a, b, c, d) = rotmat gm.runner.s r
  and s = (pieces gm.runner.p).dsc in
  for_all (fun (di, dj) ->
    possible(gm.board, (i+a*di+c*dj) mod IM, j+b*di+d*dj, gm.condemn)
  ) s
;;

let new_runner(wid, gm) =
  let a = it_vect (fun n c -> if c = 0 then n+1 else n) 0 gm.condemn in
  if a = 0 then (
    gm.state <- Ended;
    rt_reset_timeout gm.xargs;
    disp_state (wid, gm)
  ) else (
    let (p, _) = gloop (fun (p, a) ->
      (p+1, if gm.condemn.(p) > 0 then a else a-1)
    ) (fun (p, a) ->
      a = 0 & gm.condemn.(p) = 0
    ) (0, (random( a))) in
    gm.runner.i <- plan_center p;
    gm.runner.j <- -1;
    let (p, _) = gloop (fun (p, a) ->
      (p+1, a-(pieces p).frq)
    ) (fun (_, a) -> a < 0) (0, (random( SUMFRQ))) in
    gm.runner.p <- p-1;
    gm.runner.s <- (random 2);
    gm.runner.r <- (random 4);
    gm.runner.dl <- [];
    if not (possible_move(gm, gm.runner.i, gm.runner.j, gm.runner.r)) then (
      gm.state <- Ended;
      rt_reset_timeout gm.xargs;
      disp_state(wid, gm)
    )
  );
  ()  

and pause(wid, gm) =
  gm.state <- Pausing;
  rt_reset_timeout gm.xargs;
  disp_state(wid, gm)

and restart(wid, gm) =
  gm.state <- Running;
  rt_set_timeout(gm.xargs, rt_current_time gm.xargs + speed_tab gm.level);
  disp_state(wid, gm)
;;

let disp_square, clear_square, mark_square =

  let disp_or_clear_square (wid, gm, i, j, redraw_lines) =
    if j >= 0 then (
      let u = max(j-ZM)0 and v = (if j<ZM then 0 else 1) in
      let (x, y, dx, dy) =
        if i < L1 then (i, u, 1, v)
        else if i < L2 then (XM-u, i-L1, -v, 1)
        else if i < L3 then (L3-i, YM-u, -1, -v)
        else (u, L4-i, v, -1)
      and (z, dz) = (min j ZM, 1-v) in
      let p1 = proj(x, y, z)
      and p2 = proj(x+(if dx =  dy then dx else 0),
                    y+(if dx = -dy then dy else 0),
                    z+dz)
      and p3 = proj(x+dx, y+dy, z+dz)
      and p4 = proj(x+(if dx =  dy then 0 else dx),
                    y+(if dx = -dy then 0 else dy),
                    z)
      in
      rt_fill_polygon(wid, [p1; p2; p3; p4])
    ) in

(function (wid, gm, i, j, redraw_lines as a) ->
  rt_select_pattern(gm.patt C'GrayP, 0, 0);
  disp_or_clear_square a
),
(function (wid, gm, i, j, redraw_lines as a) ->
  rt_select_pixmap gm.pix;
  disp_or_clear_square a
),
(function (wid, gm, i, j, redraw_lines as a) ->
  rt_select_pattern(gm.patt C'DiagonalP, 0, 0);
  disp_or_clear_square a
)
;;

let disp_runner(wid, gm) =
  let i = gm.runner.i
  and j = gm.runner.j
  and p = gm.runner.p
  and s = gm.runner.s
  and r = gm.runner.r
  and dl = gm.runner.dl in
  let dl2 =
    let (a, b, c, d) = rotmat s r in
    map (fun (di, dj) ->
      ((i+a*di+c*dj) mod IM, j+b*di+d*dj)
    ) (pieces p).dsc in
  do_list (fun (i, j) ->
    clear_square(wid, gm, i, j, true)
  ) (subtract dl dl2);
  do_list (fun (i, j) ->
    disp_square(wid, gm, i, j, true)
  ) (subtract dl2 dl);
  gm.runner.dl <- dl2;
  ()
;;

let disp_condemn(wid, gm, p) =
  rt_select_pattern(gm.patt C'SandP, 0, 0);
  rt_fill_polygon(wid, condemn_polyg p)
;;

let disp_board(wid, gm) =
(*
  rt_clear_area(wid, 0, 0, GW, GH);
*)
  disp_score(wid, gm);
  disp_nlines(wid, gm);
  disp_state(wid, gm);
  for i = 0 to vect_length gm.board - 1 do
    let v = gm.board.(i) in
    for j = 0 to vect_length v - 1 do
      let s = v.(j) in
      match !s with
        Square -> disp_square(wid, gm, i, j, false)
      | Moribund -> mark_square(wid, gm, i, j, false)
      | Empty -> ()
    done
  done;
  do_list (fun (i, j) -> disp_square(wid, gm, i, j, false)) gm.runner.dl;
  for p = 0 to vect_length gm.condemn - 1 do
    let c = gm.condemn.(p) in
    if c > 0 then disp_condemn(wid, gm, p)
  done
;;

let disp_uncondemn(wid, gm, p) =
  rt_select_pixmap gm.pix;
  rt_fill_polygon(wid, condemn_polyg p);
  disp_board(wid, gm)
;;

let mark_explosions(wid, gm) =

  let mark_plan(i1, i2, jm) =
    let empty_line j =
      gloop succ (fun i -> i = i2 or !(gm.board.(i).(j)) <> Empty) i1 = i2
    and full_line j =
      gloop succ (fun i -> i = i2 or !(gm.board.(i).(j)) = Empty) i1 = i2 in
    let rec mark_loop jm =
      let j = gloop pred (fun j ->
        j < 0 or full_line j or (j<ZM & empty_line j)
      ) jm in
      if j >= 0 & !(gm.board.(i1).(j)) <> Empty then (
        gm.explosions_to_do <- true;
        incr_score(wid, gm, 50);
        gm.nlines <- gm.nlines+1;
        iterate (fun i ->
          if (!(gm.board.(i).(j)) = Square) then (
            gm.board.(i).(j) := Moribund;
            mark_square(wid, gm, i, j, true)
          );
          i+1
        ) (i2-i1) i1;
        j::mark_loop (j-1)
      ) else []
    in mark_loop (jm-1)

  in
  gm.explosions_to_do <- false;
  for p = 0 to vect_length gm.expl_list - 1 do
    gm.expl_list.(p) <-
      if gm.condemn.(p) > 0 then [] else rev(mark_plan(dijm_of_p p))
  done;
  disp_score(wid, gm);
  disp_nlines(wid, gm);
  if gm.explosions_to_do then (
    if !welltris_wizard then pause(wid, gm)
    else rt_set_timeout(gm.xargs, rt_current_time gm.xargs + 300)
  ) else (
    rt_set_timeout(gm.xargs, rt_current_time gm.xargs + speed_tab gm.level);
    new_runner(wid, gm)
  )

and explose(wid, gm) =
  for p = 0 to vect_length gm.expl_list - 1 do
    let el = gm.expl_list.(p) in
    let (i1, i2, _) = dijm_of_p p in
    do_list (fun j ->
      let rec copy_loop j =
        if j > 0 then (
          iterate (fun i ->
            let s = !(gm.board.(i).(j-1))
            and d = !(gm.board.(i).(j)) in
            if s <> d then (
              gm.board.(i).(j) := s;
              (match s with
                Empty -> clear_square
              | Square -> disp_square
              | Moribund -> mark_square
              ) (wid, gm, i, j, true)
            );
            i+1
          ) (i2-i1) i1;
          copy_loop (j-1)
        )
      in
      copy_loop j;
      iterate (fun i ->
        if !(gm.board.(i).(0)) <> Empty then (
          gm.board.(i).(0) := Empty;
          clear_square(wid, gm, i, j, true)
        );
        i+1
      ) (i2-i1) i1
    ) el
  done
;;

let woops_fun(wid, gm)() =
  if gm.state <> Running then failwith "erreur dans woops_fun";
  rt_set_timeout(gm.xargs, rt_current_time gm.xargs + speed_tab gm.level);
  if gm.explosions_to_do then (
    explose(wid, gm);
    mark_explosions(wid, gm)
  ) else (
    for p = 0 to vect_length gm.condemn - 1 do
      let c = gm.condemn.(p) in
      if c > 0 then (
        if (gm.condemn.(p) <- c-1; gm.condemn.(p)) = 0 then
          disp_uncondemn(wid, gm, p)
      )
    done;
    if exists (fun (i, j) ->
      not (possible(gm.board, i, j+1, gm.condemn))
    ) gm.runner.dl then (
      do_list (fun (i, j) ->
        if i >= 0 & j >= 0 then (
          gm.board.(i).(j) := Square;
          incr_score(wid, gm, 1)
        );
        if j < ZM then (
          let p = p_of_i i in
          if gm.condemn.(p) = 0 then disp_condemn(wid, gm, p);
          gm.condemn.(p) <- CONDEMN_COUNT;
          ()
        )
      ) gm.runner.dl;
      mark_explosions(wid, gm)
    ) else (
      gm.runner.j <- gm.runner.j+1;
      disp_runner(wid, gm)
    )
  )
;;

let patterns = [|
  implode_ascii [  0;   0;   0;   0;   0;   0;   0;   0;
                   0;   0;   0;   0;   0;   0;   0;   0;
                   0;   0;   0;   0;   0;   0;   0;   0;
                   0;   0;   0;   0;   0;   0;   0;   0] (* blanc *);
  implode_ascii [255; 255; 255; 255; 255; 255; 255; 255;
                 255; 255; 255; 255; 255; 255; 255; 255;
                 255; 255; 255; 255; 255; 255; 255; 255;
                 255; 255; 255; 255; 255; 255; 255; 255] (* noir *);
  implode_ascii [240; 240; 120; 120;  60;  60;  30;  30;
                  15;  15; 135; 135; 195; 195; 225; 225;
                 240; 240; 120; 120;  60;  60;  30;  30;
                  15;  15; 135; 135; 195; 195; 225; 225] (* oblique *);
  implode_ascii [  1;   1;   8;   8;  64;  64;   4;   4;
                 128; 128;  16;  16;   2;   2;  32;  32;
                   1;   1;   8;   8;  64;  64;   4;   4;
                 128; 128;  16;  16;   2;   2;  32;  32] (* sable *);
  implode_ascii [255;   0; 255;   0; 255;   0; 255;   0;
                 255;   0; 255;   0; 255;   0; 255;   0;
                   0; 255;   0; 255;   0; 255;   0; 255;
                   0; 255;   0; 255;   0; 255;   0; 255] (* damier *);
  implode_ascii [ 60;  60;  60;  60;  60;  60;  60;  60;
                  60;  60;  60;  60;  60;  60;  60;  60;
                  60;  60;  60;  60;  60;  60;  60;  60;
                  60;  60;  60;  60;  60;  60;  60;  60] (* vertical *);
  implode_ascii [ 85;  85; 170; 170; 85;   85; 170; 170;
                  85;  85; 170; 170; 85;   85; 170; 170;
                  85;  85; 170; 170; 85;   85; 170; 170;
                  85;  85; 170; 170; 85;   85; 170; 170] (* gris *)
|]
;;

let make_patt xd =
  let patt_vect = map_vect (fun p -> rt_create_pattern(xd, p, 16, 16)) patterns
  in
function p -> patt_vect.(
  match p with
     C'WhiteP -> 0 | C'BlackP -> 1 | C'DiagonalP -> 2
  | C'SandP -> 3 | C'ChBoardP -> 4 | C'VerticalP -> 5 | C'GrayP -> 6
)
;;
  

let welltris dname =
  let xd = rt_initialize dname in
  let xargs = rt_args [xd] in
  let gm = {
    xd = xd;
    xargs = xargs;
    pix = rt_create_pixmap(xd, GW, GH);
    state = NotStarted;
    level = 0;
    score = 0;
    nlines = 0;
    explosions_to_do = false;
    expl_list = [| []; []; []; [] |];
    condemn = [| 0; 0; 0; 0 |];
    patt = make_patt xd;
    runner = {i=0; j=0; p=0; s=0; r=0; dl = []};
    board = (make_vect IM [| |])
  } in
  let x = ref Empty in
  for p = 0 to vect_length gm.board - 1 do
    gm.board.(p) <- make_vect JM x
  done;
  for i = 0 to vect_length gm.board - 1 do
    let v = gm.board.(i) in
    for j = 0 to vect_length v - 1 do
      let s = v.(j) in
      v.(j) <-
        if j < ZM then ref Empty
        else if i < L1 then
          if j < YZM then ref Empty else s
        else if i < L2 then
          if j < XZM then gm.board.(XZM-1-j).(ZM+i-L1) else s
        else if i < L3 then
          if j < YZM then gm.board.(L3-1-i).(ZM+YZM-1-j) else s
        else
          if j < XZM then gm.board.(j-ZM).(ZM+L4-1-i) else s
    done
  done;

  let expose(wid, _, _, _, _) =
    let drw = WidgetDr wid in
    if gm.state = NotStarted then (
      gm.state <- Running;
      rt_set_timeout(gm.xargs, rt_current_time gm.xargs + speed_tab gm.level);
      new_runner(drw, gm)
    );
    disp_board(drw, gm);
    ()

  and keyp(wid, k) =
    let drw = WidgetDr wid in
    if k = "q" then rt_stop_main_loop gm.xargs
    else if gm.state = Pausing then restart(drw, gm)
    else (
    (match k with
      "Up" | "R8" | "i" | "K8" ->
        if gm.state = Running then ( 
          let i = gm.runner.i in
          let di = (if i >= L1 quo 2 & i < (L2+L3)quo 2 then -1 else 1) in
          let ni = (i+di) mod IM in
          if possible_move(gm, ni, gm.runner.j, gm.runner.r) then (
            gm.runner.i <- ni;
            disp_runner(drw, gm)
          )
        )
    | "Down" | "R14" | "k" | "K2" ->
        if gm.state = Running then (
          let i = gm.runner.i in                                      
          let di = (if i >= L1 quo 2 & i < (L2+L3)quo 2 then 1 else -1) in
          let ni = (i+di) mod IM in
          if possible_move(gm, ni, gm.runner.j, gm.runner.r) then (
            gm.runner.i <- ni;
            disp_runner(drw, gm)
          )
        )
    | "Left" | "R10" | "j" | "K4" ->
        if gm.state = Running then (
          let i = gm.runner.i in
          let di = (if i >= (L1+L2)quo 2 & i < (L3+L4)quo 2 then 1 else -1) in
          let ni = (i+di) mod IM in
          if possible_move(gm, ni, gm.runner.j, gm.runner.r) then (
            gm.runner.i <- ni;
            disp_runner(drw, gm)
          )
        )
    | "Right" | "R12" | "l" | "K6" ->
        if gm.state = Running then (
          let i = gm.runner.i in
          let di = (if i >= (L1+L2)quo 2 & i < (L3+L4)quo 2 then -1 else 1) in
          let ni = (i+di) mod IM in
          if possible_move(gm, ni, gm.runner.j, gm.runner.r) then (
            gm.runner.i <- ni;
            disp_runner(drw, gm)
          )
        )
    | "R11" | "u" | "K5" ->
        if gm.state = Running then (
          let nr = (gm.runner.r-1) mod 4 in
          if possible_move(gm, gm.runner.i, gm.runner.j, nr) then (
            gm.runner.r <- nr;
            disp_runner(drw, gm)
          )
        )
    | "n" ->
        if gm.state = Ended then (
          gm.state <- Running;
          gm.level <- 0;
          gm.score <- 0;
          gm.nlines <- 0;
          gm.explosions_to_do <- false;
          gm.condemn.(0) <- 0; gm.condemn.(1) <- 0;
          gm.condemn.(2) <- 0; gm.condemn.(3) <- 0;
          gm.expl_list.(0) <- []; gm.expl_list.(1) <- [];
          gm.expl_list.(2) <- []; gm.expl_list.(3) <- [];
          do_vect (fun v -> do_vect (fun s -> s := Empty) v) gm.board;
          rt_set_timeout(gm.xargs, rt_current_time gm.xargs + speed_tab 0);
          rt_select_pattern(gm.patt C'WhiteP, 0, 0);
          rt_fill_rectangle(drw, 0, 0, GW, GH);
          new_runner(drw, gm);
          disp_board(drw, gm)
        )
    | "c" ->
        rt_select_pattern(gm.patt C'WhiteP, 0, 0);
        rt_fill_rectangle(drw, 0, 0, GW, GH)
    | "e" ->
        disp_board(drw, gm)
    | " "| "Ins" | "K0" ->
        if gm.state = Running & gm.runner.j >= 1 then (
          let (a, b, c, d) = rotmat gm.runner.s gm.runner.r
          and s = (pieces gm.runner.p).dsc in
          while for_all (fun (di, dj) ->
            let i = (gm.runner.i+a*di+c*dj) mod IM
            and j = gm.runner.j+b*di+d*dj in
            possible(gm.board, i, j+1, gm.condemn)
          ) s do
            for p = 0 to vect_length gm.condemn - 1 do
              let c = gm.condemn.(p) in
              if c > 0 then (
                if (gm.condemn.(p) <- c-1; gm.condemn.(p)) = 0 then
                  disp_uncondemn(drw, gm, p)
              )
            done;
            gm.runner.j <- gm.runner.j+1;
            incr_score(drw, gm, 1)
          done;
          disp_runner(drw, gm);
          do_list (fun (i, j) ->
            if i >= 0 & j >= 0 then (
              gm.board.(i).(j) := Square;
              incr_score(drw, gm, 1)
            );
            if j < ZM then (
              let p = p_of_i i in
              if gm.condemn.(p) = 0 then disp_condemn(drw, gm, p);
              gm.condemn.(p) <- CONDEMN_COUNT;
              ()
            )
          ) gm.runner.dl;
          mark_explosions(drw, gm)
        )
    | "p" ->
        if gm.state = Running then pause(drw, gm)
    | _ ->
       print_string k; print_string "-"; flush std_out
    )
    );
    ()
  in

  (try
    let draw = PixmapDr gm.pix in
    rt_select_pattern(gm.patt C'WhiteP, 0, 0);
    rt_fill_rectangle(draw, 0, 0, GW, GH);
    rt_select_pattern(gm.patt C'BlackP, 0, 0);
    iterate (fun z ->
      let (x1, y1) = proj(0, 0, z)
      and (x2, y2) = proj(XM, YM, z) in
      rt_draw_rectangle(draw, x1, y1, x2-x1, y2-y1);
      z+1
    ) (ZM+1) 0;
    iterate (fun x ->
      let p1 = proj(x,  0, 0) and p2 = proj(x,  0, ZM)
      and p3 = proj(x, YM, ZM) and p4 = proj(x, YM, 0) in
      rt_draw_lines(draw, [p1; p2; p3; p4]);
      x+1
    ) (XM+1) 0;
    iterate (fun y ->
      let p1 = proj( 0, y,  0) and p2 = proj( 0, y, ZM)
      and p3 = proj(XM, y, ZM) and p4 = proj(XM, y,  0) in
      rt_draw_lines(draw, [p1; p2; p3; p4]);
      y+1
    ) (YM+1) 0;
    let wid = rt_create_widget(xd, "welltris", "welltris",
      RawA [BackgroundAtt(PixmapBg gm.pix)]
        (GW, GH, 0, [ExposePr expose; KeyPr keyp])
    ) in
    rt_map_widget wid;
    rt_set_timeout_fun(gm.xargs, woops_fun(WidgetDr wid, gm));
    rt_main_loop gm.xargs
  with x ->
    rt_end xd; raise x
  );
  rt_end xd
;;

let dname = ref ""
;;

let i = ref 1 in
while !i  < vect_length command_line do
  (match command_line.(!i) with
    "-d" -> incr i
  | s -> dname := s);
  incr i
done;;

try welltris !dname with
  Failure s -> print_string "Failure: "; print_string s; print_newline()
| Invalid_argument s ->
    print_string "Invalid_argument: "; print_string s; print_newline()
| x -> print_string "Unknown exception in welltris"; print_newline(); raise x
;;
