(* $Id: sht.ml,v 1.4 91/07/05 18:11:36 ddr Exp $
 *
 * Seahaven towers
 *)

#open "rt";;
#open "sys";;
#open "time";;

let line_buff = create_string 200
and seed = ref 0
;;
let input_line ic = sub_string line_buff 0 (input_rec 0
  where rec input_rec i =
    match (input_char ic) with
      `\n` -> i
    | c -> set_nth_char line_buff i c; input_rec (succ i))
and modify_vect f v =
  let len = vect_length v in (modify_rec 0
    where rec modify_rec i =
      if i < len then (v.(i) <- f v.(i); modify_rec (succ i)))
and prefix quo x y = x / y
and max x y = if x > y then x else y
and min x y = if x < y then x else y
and B f g x = f(g x)
and it_vect f a v = (it_vect_f a 0
  where rec it_vect_f a n =
    if n >= vect_length v then a
    else it_vect_f (f a v.(n)) (succ n))
and hd = function x::l -> x | _ -> failwith "hd"
and tl = function x::l -> l | _ -> failwith "tl"
and item = (item_rec
  where rec item_rec = function
    (a::l) -> (function 0 -> a | i -> item_rec l (i-1))
  | _ -> failwith "item")
and sqrt x = int_of_float(sqrt(float_of_int x))
and eq(x,y) = x == y
and random n =
  seed := 25173*!seed + 13849;
  (if !seed < 0 then - !seed else !seed) mod n
and init_random n = seed := n; ()
and chop_list n l = (chop_aux n ([],l)
  where rec chop_aux = fun
    0 (l1,l2) -> rev l1,l2
  | n (l1,x::l2) -> chop_aux (pred n) (x::l1,l2)
  | _ _ -> failwith "chop_list")
and null = function [] -> true | _ -> false
and do_list_i f = (do_list_f
  where rec do_list_f i = function
    [] -> () | x::l -> f i x; do_list_f (succ i) l)
and do_list f = (do_list_f
  where rec do_list_f = function
    [] -> () | x::l -> f x; do_list_f l)
and message s = print_string s; print_newline()
and gc_alarm _ = ()
and implode_ascii l =
  let len = list_length l in
  let s = make_string len ` ` in (implode_rec l 0
  where rec implode_rec = fun
    (x::l) i -> set_nth_char s i (char_of_int x); implode_rec l (succ i)
  | _ _ -> s)
and iterate f = (iterate_f
  where rec iterate_f n x =
    if n > 0 then iterate_f (pred n) (f x))
and prim_rec f = (prim_loop
  where rec prim_loop init = function
    0 -> init | n -> prim_loop (f init n) (n-1))
;;

let gloop f p = (looprec where rec
  looprec x = if p x then x else looprec(f x))
;;

let vect_exists p v = exists_p 0
  where rec exists_p i =
    if i = vect_length v then false
    else p (v.(i)) or exists_p(i+1)
;;    

type suit = C'Clubs | C'Diamonds | C'Hearts | C'Spades

and val =
  C'Ace | C'King | C'Queen | C'Jack | C'Plain of int
;;

type card = {
  suit              : suit;
  val               : val;
  gm                : game;
  mutable position  : position
}

and game_state =
  NotStarted
| NormalMove
| AutoMove of widget * position * int * int

and game = {
  xargs         : xargs;
  towers        : widget list vect;
  buffers       : widget list vect;
  columns       : widget list vect;
  wid_list      : widget list;
  mover         : widget;
  mutable state : game_state;
  mutable undo  : (widget * position) list;
  mutable redo  : (widget * position) list;
  mutable bp    : bool;
  mutable mwid  : widget list;
  mutable xg    : int;
  mutable yg    : int
}

and position = Column of int | Buff of int | Tower of int | Nowhere
;;

let bw = ref false
and dname = ref ""
and cards_file = ref "./cards"
;;

let i = ref 1 in
while !i < vect_length command_line do
  (match command_line.(!i) with
    "-bw" -> bw := true
  | "-c" -> incr i; cards_file := command_line.(!i)
  | "-d" -> incr i; cards_file := command_line.(!i) ^ "/cards"
  | s -> dname := s);
  incr i
done
;;

let buff = create_string 1024;;
let WID, HEI, data =
  let ic = open_in !cards_file in
  let x =
    (try
      let width = int_of_string (input_line ic)
      and height = int_of_string (input_line ic)
      and data = make_vect 52 "" in
      modify_vect (fun _ ->
        let len = ((width+7) quo 8) * height in
        really_input ic buff 0 len;
        sub_string buff 0 len
      ) data;
      width, height, data
    with x -> close_in ic; raise x) in
  close_in ic;
  x
and BD = 0
and LeftB = 20 and RightB = 20 and UpperB = 20 and LowerB = 20
and IC = 20 and SC = 50 and RC = 25
;;

let OW = BD+WID+BD and OH = BD+HEI+BD
and SPEED = 500 (* pix / s *) and PERIOD = 60 (* ms *)
;;

let GW = LeftB+10*(OW+IC)-IC+RightB
and GH = UpperB+OH+SC+17*RC+OH+LowerB
and DELTA = max 1 ((SPEED * PERIOD) quo 1000)
;;

let card, get_card =
  (user_info "card" (ref UNone) : card user_info_func)
and game, get_game =
  (user_info "game" (ref UNone) : game user_info_func)
;;

let item_suit = vect_item [| C'Hearts; C'Diamonds; C'Clubs; C'Spades |]
and item_val = vect_item [|
  C'Ace; C'Plain 2; C'Plain 3; C'Plain 4; C'Plain 5; C'Plain 6; C'Plain 7;
  C'Plain 8; C'Plain 9; C'Plain 10; C'Jack; C'Queen; C'King
|]
;;

let suit_item = function
   C'Hearts -> 0 | C'Diamonds -> 1 | C'Clubs -> 2 | C'Spades -> 3

and val_item = function
  C'Ace -> 0 | C'King -> 12 | C'Queen -> 11 | C'Jack -> 10 | C'Plain n -> n-1
;;

let suit_txt = B (vect_item [|
  "C"; "K"; "T"; "P"
|]) suit_item

and val_txt = B (vect_item [|
  "A"; "2"; "3"; "4"; "5"; "6"; "7"; "8"; "9"; "0"; "V"; "D"; "R"
|]) val_item
;;

let col_x n =
  LeftB+n*(OW+IC)
;;

let tower_x = vect_item [| col_x 0; col_x 1; col_x 8; col_x 9 |]
and tower_y _ = UpperB
and buff_x = vect_item [| col_x 3; col_x 4; col_x 5; col_x 6 |]
and buff_y _ = UpperB
and column_x = col_x
and column_y gm nb = UpperB+OH+SC+RC*(list_length gm.columns.(nb))
;;

let nth_free_buff gm = (rec_nth 0
where rec rec_nth i j =
  if gm.buffers.(j) = [] then
  if i = 0 then (Buff j)
  else rec_nth (i-1) (j+1)
  else rec_nth i (j+1)
)

and nb_free_buffers gm =
  it_vect (fun i widl ->
    i+(if widl = [] then 1 else 0)
  ) 0 gm.buffers
;;

let move_to pos wid =
  let card = get_card wid in
  let gm = card.gm in
  (match card.position with
    Tower nb -> gm.towers.(nb) <- tl gm.towers.(nb); ()
  | Buff nb -> gm.buffers.(nb) <- tl gm.buffers.(nb); ()
  | Column nb -> gm.columns.(nb) <- tl gm.columns.(nb); ()
  | Nowhere -> ()
  );
  (match pos with
    Tower nb ->
      rt_move_widget (wid, tower_x nb, tower_y nb);
      gm.towers.(nb) <- wid::gm.towers.(nb); ()
  | Buff nb ->
      rt_move_widget (wid, buff_x nb, buff_y nb);
      gm.buffers.(nb) <- wid::gm.buffers.(nb); ()
  | Column nb ->
      rt_move_widget (wid, column_x nb, column_y gm nb);
      gm.columns.(nb) <- wid::gm.columns.(nb); ()
  | Nowhere -> ()
  );
  card.position <- pos;
  ()
;;

let dist x1 y1 x2 y2 =
  let dx = x1-x2 and dy = y1-y2 in
  sqrt(dx*dx+dy*dy)
;;

let auto_move gm =
  let what dest card = what_rec 0 where rec what_rec i =
    if i = vect_length dest then -1
    else match dest.(i) with
      swid::_ ->
        let scard = get_card swid in
        if card.suit = scard.suit then
          if eq(dest, gm.towers) then
            if val_item card.val - val_item scard.val = 1 then i
            else -1
          else
            if val_item card.val - val_item scard.val = -1 then i
            else what_rec (i+1)
        else what_rec (i+1)
    | _ ->
        if eq(dest, gm.towers) & card.val = C'Ace then i
        else what_rec (i+1)
  in
  let test_push_on dest = function
    wid::_ ->
      let card = get_card wid in
      let nb = what dest card in
      if nb <> -1 then (
        gm.state <-
          if eq(dest, gm.towers) then
            AutoMove(wid, Tower nb, tower_x nb, tower_y nb)
          else
            AutoMove(wid, Column nb, column_x nb, column_y gm nb);
        rt_set_timeout(gm.xargs, rt_current_time gm.xargs + PERIOD);
        true
      ) else false
  | _ ->
      false
  in
  gm.state <- NormalMove;
  let _ =
  vect_exists (test_push_on gm.towers) gm.buffers or
  vect_exists (test_push_on gm.towers) gm.columns or
  vect_exists (test_push_on gm.columns) gm.buffers in
  ()
;;

let woops gm _ =
  match gm.state with
    AutoMove(wid, dest, dx, dy) ->
      let card = get_card wid in
      let xw = widget_x wid and yw = widget_y wid in
      let d = dist dx dy xw yw in
      let nxw = xw + ((DELTA * (dx-xw)) quo d)
      and nyw = yw + ((DELTA * (dy-yw)) quo d) in
      let nd = dist dx dy nxw nyw in
      if nd < (DELTA quo 2) or nd >= d then (
        gm.undo <- (wid, card.position)::gm.undo;
        gm.redo <- [];
        move_to dest wid;
        auto_move gm
      ) else (
        rt_set_timeout(gm.xargs, rt_current_time gm.xargs + PERIOD);
        rt_move_widget(wid, nxw, nyw)
      )
  | _ -> failwith "woops"
;;

let reset_game gm =
  modify_vect (fun _ -> []) gm.towers;
  modify_vect (fun _ -> []) gm.buffers;
  modify_vect (fun _ -> []) gm.columns;
  gm.state <- NormalMove;
  rt_reset_timeout gm.xargs;
  gm.undo <- []; gm.redo <- [];
  do_list (fun wid -> (get_card wid).position <- Nowhere) gm.wid_list
;;

let new_game gm =
  reset_game gm;
  let _ = gloop (fun wl ->
    let len = list_length wl in
    let r = random len in
    let b, e = chop_list r wl in
    let wid = hd e in
    if len <= 2 then move_to (Buff len) wid
    else move_to (Column((52-len) quo 5)) wid;
    b @ (tl e)
  ) null gm.wid_list in
  auto_move gm;
  ()

and save_game gm =
  print_string "tours:"; print_newline();
  let _ = it_vect (function sep -> function
    wid::_ ->
      let card = get_card wid in
      print_string sep;
      print_string (val_txt card.val);
      print_string (suit_txt card.suit);
      " "
  | _ -> sep
  ) "" gm.towers in
  print_string ";"; print_newline();
  print_string "buffers:"; print_newline();
  let _ = it_vect (function sep -> function
    wid::_ ->
      let card = get_card wid in
      print_string sep;
      print_string (val_txt card.val);
      print_string (suit_txt card.suit);
      " "
  | _ -> sep
  ) "" gm.buffers in
  print_string ";"; print_newline();
  print_string "colonnes:"; print_newline();
  let _ = it_vect (fun sep widl ->
    print_string sep;
    let _ = it_list (fun sep wid ->
      let card = get_card wid in
      print_string sep;
      print_string (val_txt card.val);
      print_string (suit_txt card.suit);
      " "
    ) "" widl in
    ",\n"
  ) "" gm.columns in
  print_string ";"; print_newline()

and find_game gm =
  let is_space c = c = " " or c = "\n" or c = "\r" or c = ""
  and is_sep c = c = "," or c = ";" or c = ":" in
  let rec first_no_space c =
    if is_space c then first_no_space(make_string 1 (input_char std_in))
    else c in
  let next_tok c =
    let rec read_tok c =
      let nc = make_string 1 (input_char std_in) in
      if is_sep nc or is_space nc then c,nc
      else let t,nc = read_tok nc in c^t,nc
    in
    let c = first_no_space c in
    if is_sep c then c,""
    else read_tok c
  and wid_of_card (v,s) =
    item gm.wid_list ((suit_item s)*13+val_item v)
  in
  let rec read_cards(t,c) =
    try
      let v = match nth_char t 0 with
        `A` -> C'Ace | `R` -> C'King | `D` -> C'Queen | `V` -> C'Jack
      | `X` | `0` -> C'Plain 10 | s -> C'Plain (int_of_char s)
      and s = match nth_char t 1 with
        `P` -> C'Spades | `C` -> C'Hearts | `K` -> C'Diamonds | `T` -> C'Clubs
      | _ -> failwith "read cards" in
      (v,s)::read_cards(next_tok c)
    with _ -> []
  in
  let rec read_cols(t,c) n =
    let col = read_cards(t,c) in
    if n = 1 then [col] else
    col::(read_cols(next_tok c)(n-1))
  in
  let skip_header txt =
    let t,c = next_tok "" in
    let t,c = if t = txt then next_tok c else t,c in
    if t = ":" then next_tok c else t,c
  in
  try
    let towers = read_cards(skip_header "tours")
    and buffers = read_cards(skip_header "buffers")
    and columns = read_cols(skip_header "colonnes") 10 in
    reset_game gm;
    do_list_i (fun i col ->
      do_list (fun c ->
        let wid = wid_of_card c in
        move_to (Column i) wid
      ) (rev col)
    ) 0 columns;
    do_list_i (fun i c ->
      let wid = wid_of_card c in
      move_to (Buff i) wid
    ) 0 buffers;
    do_list_i (fun i c ->
      let wid = wid_of_card c in
      move_to (Tower i) wid
    ) 0 towers;
    ()
  with _ ->
    message "incorrect"
;;

exception bad_position;;
let motion(wid, x, y) =
  let gm = get_game wid in
  if gm.state = NormalMove then (
    match gm.mwid with
      mwid::_ ->
        let nx = (widget_x gm.mover)+x-gm.xg
        and ny = (widget_y gm.mover)+y-gm.yg
        and w = widget_width gm.mover
        and h = widget_height gm.mover in
        let new_x = max 0 (min ((widget_width wid)-w) nx)
        and new_y = max 0 (min ((widget_height wid)-h) ny) in
        rt_move_widget(gm.mover, new_x, new_y);
        gm.xg <- x + new_x - nx;
        gm.yg <- y + new_y - ny;
        ()
    | [] ->
        ()
  )

and enter_wind wid =
  let _ = random 1 in
  let gm = (get_card wid).gm in
  if not gm.bp then (gm.mwid <- [wid]; ())

and leave_wind(wid) =
  let gm = (get_card wid).gm in
  if not gm.bp then (gm.mwid <- []; ())

and bpressed(wid, b, x, y) =
  let gm = get_game wid in
  let tagada twid = tagada_rec
  where rec tagada_rec = function
    (wid::widl as widls) ->
      if eq(wid, twid) then widls
      else tagada_rec widl
  | _ -> failwith "tagada" in
  if gm.state = NormalMove then (
    gm.bp <- true;
    (match gm.mwid with
      twid::_ ->
        let card = get_card twid in
        if match card.position with
          Tower nb -> false
        | Buff nb ->
            gm.mwid <- tagada twid (rev gm.buffers.(nb));
            true
        | Column nb ->
            gm.mwid <- tagada twid (rev gm.columns.(nb));
            true
        | _ -> false then (
          rt_move_resize_widget(
            gm.mover, widget_x twid, widget_y twid,
            WID, HEI+((list_length gm.mwid)-1)*RC
          );
          rt_map_widget gm.mover;
          gm.xg <- x; gm.yg <- y;
          do_list (fun wid -> rt_unmap_widget wid) gm.mwid
        ) else (gm.mwid <- []; ())
    | [] ->
        ()
    );
    ()
  )

and breleased pwid =
  let gm = get_game pwid in
  if gm.state = NormalMove then (
    gm.bp <- false;
    begin match gm.mwid with
      (mwid::widl as widls) ->
        let card = get_card mwid in
        begin try
          let nb = (gm.xg-LeftB+(IC quo 2))quo(OW+IC) in
          if nb < 0 or nb > 9 then raise bad_position;
          if gm.yg >= UpperB+OH+(SC quo 2) then (
            (* column *)
            do_list_i (fun i swid ->
              let scard = get_card swid in
              if scard.suit <> card.suit or
              val_item scard.val <> (val_item card.val)-i then
                raise bad_position
            ) 1 widl;
            let len = list_length widl in
            if len >= 1 & len > nb_free_buffers gm then raise bad_position;
            begin match gm.columns.(nb) with
              wid::_ ->
                let scard = get_card wid in
                if card.suit <> scard.suit or
                val_item card.val <> (val_item scard.val)-1 then
                  raise bad_position
            | _ ->
                if card.val <> C'King then raise bad_position
            end;
            do_list (fun wid ->
              let card = get_card wid in
              gm.undo <- (wid, card.position)::gm.undo
            ) (rev widl);
            gm.undo <- (mwid, card.position)::gm.undo;
            gm.redo <- [];
            move_to (Column nb) mwid;
            do_list_i (fun i wid ->
              gm.undo <- (wid, nth_free_buff gm i)::gm.undo;
              move_to (Column nb) wid
            ) 0 widl
          ) else if nb >= 3 & nb <= 6 then (
            (* buffer *)
            let len = list_length widl in
            if len >= nb_free_buffers gm then raise bad_position;
            let nb = nb-3 in
            do_list_i (fun i wid ->
              gm.undo <- (wid, card.position)::gm.undo;
              gm.redo <- [];
              move_to (
                if i = 0 & gm.buffers.(nb) = [] then Buff nb
                else nth_free_buff gm 0
              ) wid
            ) 0 (rev widls)
          ) else if nb <= 1 or nb >= 8 then (
            (* tower *)
            if widl <> [] then raise bad_position;
            let nb = (if nb <= 1 then nb else nb-6) in
            begin match gm.towers.(nb) with
              wid::_ ->
                let scard = get_card wid in
                if card.suit <> scard.suit or
                val_item card.val <> (val_item scard.val)+1 then
                  raise bad_position
            | _ ->
                if card.val <> C'Ace then raise bad_position
            end;
            gm.undo <- (mwid, card.position)::gm.undo;
            gm.redo <- [];
            move_to (Tower nb) mwid
          ) else raise bad_position
        with bad_position -> ()
        end;
        rt_unmap_widget gm.mover;
        do_list (fun wid -> rt_map_widget wid) widls;
        auto_move gm
    | [] ->
        ()
    end;
    gm.mwid <- [];
    ()
  )
;;
(*
let expose(wid, x, y, width, height) =
  let card = get_card wid in
  rt_put_image(WidgetDr wid, card.image, x, y, x, y, width, height)
*)
let expose_tower(wid, _, _, _, _) =
  let gm = get_game wid in
  if gm.state = NotStarted then new_game gm;
  let draw = WidgetDr wid in
  rt_draw_line(draw, 0, 0, OW-2, OH-2);
  rt_draw_line(draw, 0, OH-2, OW-2, 0)

and expose_buffer(wid, _, _, _, _) =
  let draw = WidgetDr wid in
  rt_draw_line(draw, 0, 0, OW-2, OH-2);
  rt_draw_line(draw, 0, OH-2, OW-2, 0)
;;

let keyp xargs (wid, s) =
  let gm = get_game wid in
  match s with
    "q" -> rt_stop_main_loop xargs
  | "z" ->
      if gm.state = NormalMove then (
        begin match gm.undo with
          (wid, pos)::m ->
            let card = get_card wid in
            gm.redo <- (wid, card.position)::gm.redo;
            move_to pos wid;
            gm.undo <- m; ()
        | _ -> ()
        end;
        ()
      )
  | "r" ->
      if gm.state = NormalMove then (
        begin match gm.redo with
          (wid, pos)::m ->
            let card = get_card wid in
            gm.undo <- (wid, card.position)::gm.undo;
            move_to pos wid;
            gm.redo <- m; ()
        | _ -> ()
        end;
        ()
      )
  | "f" ->
      find_game gm
  | "s" ->
      save_game gm
  | "n" ->
      new_game gm
  | _ ->
      ()
;;

let create_tower =
  let pl = [ ExposePr expose_tower ] in
fun col (pwid, nb) ->
  let x = tower_x nb and y = tower_y nb in
  let wid = rt_create_subwidget(pwid, x, y,
    RawA [BackgroundAtt(ColorBg col)] (OW-2, OH-2, 1, pl)
  ) in
  rt_map_widget wid;
  wid

and create_buff =
  let pl = [ ExposePr expose_buffer ] in
fun col (pwid, nb) ->
  let x = buff_x nb and y = buff_y nb in
  let wid = rt_create_subwidget(pwid, x, y,
    RawA [BackgroundAtt(ColorBg col)] (OW-2, OH-2, 1, pl)
  ) in
  rt_map_widget wid;
  wid

and create_card =
  let pl = [
    EnterWindowPr enter_wind; LeaveWindowPr leave_wind
  ] in
fun xd pwid data ->
  let pixmap = rt_create_pixmap(xd, WID, HEI)
  and image = rt_create_image(xd, data, WID, HEI, 1) in
  rt_put_image(PixmapDr pixmap, image, 0, 0, 0, 0, WID, HEI);
  let wid = rt_create_subwidget(pwid, -2*OW, -2*OH,
    RawA [BackgroundAtt(PixmapBg pixmap)] (WID, HEI, BD, pl)
  ) in
  rt_map_widget wid;
  wid
;;

let gen_sht bw dname =
  init_random (ftime()).time;
  let xd = rt_initialize dname in (try
    gc_alarm true;
    let bg =
      if is_colored xd & not bw then
        ColorBg(rt_create_color(xd, 153, 256, 153))
      else (
        let image = rt_create_image(xd,
          implode_ascii[1;8;64;4;128;16;2;32], 8, 8, 1
        )
        and pixm = rt_create_pixmap(xd, 8, 8) in
        rt_put_image(PixmapDr pixm, image, 0, 0, 0, 0, 8, 8);
        PixmapBg pixm
      )
    in
    let xargs = rt_args[xd] in

    let main_wid = rt_create_widget(xd, "Seahaven towers", "sht",
      RawA [BackgroundAtt bg] (
        GW, GH, 0, [
          KeyPr (keyp xargs);
          ButtonPressedPr bpressed;
          ButtonReleasedPr breleased;
          ButtonMotionPr motion
        ]
      )
    ) in

    let mover = rt_create_subwidget(
      main_wid, 0, 0, RawA [BackgroundAtt NoneBg] (0, 0, 0, [])
    ) in

    let black_col = rt_black_color xd in
    let red_col =
      if is_colored xd & not bw then
        rt_create_color(xd, 190, 38, 0)
      else black_col in
    let wid_list = prim_rec (fun widl i ->
      if i = 26 then rt_select_color red_col;
      create_card xd main_wid data.(i-1)::widl
    ) [] 52 in
    rt_select_color black_col;
    let gm = {
      xargs = xargs;
      towers = (make_vect 4 []);
      buffers = (make_vect 4 []);
      columns = (make_vect 10 []);
      wid_list = wid_list; mover = mover;
      state = NotStarted;
      undo = []; redo = [];
      bp = false; mwid = []; xg = 0; yg = 0
    } in
    rt_set_user_info(main_wid, game gm);
    do_list_i (fun i wid ->
      let suit = item_suit(i quo 13)
      and val = item_val(i mod 13) in
      rt_set_user_info(wid,
        card{
          val=val; suit=suit; position=Nowhere;
          gm=gm
        }
      )
    ) 0 gm.wid_list;

    let create_tower = create_tower(
      if is_colored xd & not bw then
        rt_create_color(xd, 0, 255, 255)
      else rt_white_color xd
    ) in
    iterate (fun x ->
      let wid = create_tower(main_wid, x) in
      rt_set_user_info(wid, game gm);
      x+1
    ) 4 0;
    let create_buff = create_buff(
      if is_colored xd & not bw then rt_create_color(xd, 255, 143, 76)
      else rt_white_color xd
    ) in
    iterate (fun x ->
      let _ = create_buff(main_wid, x) in x+1
    ) 4 0;

    rt_set_timeout_fun(gm.xargs, woops gm);
    rt_map_widget main_wid;
    rt_main_loop gm.xargs
  with x -> gc_alarm false; rt_end xd; raise x);
  gc_alarm false; rt_end xd
;;

try
  gen_sht !bw !dname
with
  Failure s -> print_string "failure: "; print_string s; print_newline()
| Invalid_argument s ->
    print_string "Invalid_argument: "; print_string s; print_newline()
| x -> print_string "Unknown exception in ssht"; print_newline(); raise x
;;
