(* $Id: xsht.ml,v 1.2 91/06/29 17:07:54 ddr Exp $
 *
 * Seahaven Towers en couleurs. Partie X.
 *)

#open "rt";;
#open "sys";;
#open "time";;
#open "card_pix";;

type num == int;;
let seed = ref 0;;
let max x y = if x > y then x else y
and min x y = if x < y then x else y
and prefix quo x y = x / y
and iterate f = (iterate_f
  where rec iterate_f n x =
    if n > 0 then iterate_f (pred n) (f x))
and modify_vect f v =
  let len = vect_length v in (
  modify_rec 0 where rec modify_rec i =
    if i < len then (v.(i) <- f v.(i); modify_rec (succ i)))
and modify_vect_i f v =
  let len = vect_length v in (
  modify_rec 0 where rec modify_rec i =
    if i < len then (v.(i) <- f i v.(i); modify_rec (succ i)))
and string_of_num = string_of_int
and num_of_string = int_of_string
and display_num = print_int
and display_string = print_string
and display_newline = print_newline
and display_flush() = flush std_out
and random n =
  seed := 25173*!seed + 13849;
  (if !seed < 0 then - !seed else !seed) mod n
and init_random n = seed := n; ()
and B f g x = f(g x)
and ascii_code s = int_of_char(nth_char s 0)
and sqrt x = int_of_float(sqrt(float_of_int x))
and length_string = string_length
and extract_string s f t = sub_string s f (t-f+1)
and gc_alarm _ = ()
;;

let WID = 47 and HEI = 67
;;

let BD = 0
and LeftB = 20 and RightB = 20 and UpperB = 20 and LowerB = 20
and IC = 20 and SC = 50 and RC = 25
;;

let OW = BD+WID+BD and OH = BD+HEI+BD
and SPEED = 500 (* pix / s *) and PERIOD = 60 (* ms *)
;;

let GW = LeftB+10*(OW+IC)-IC+RightB
and GH = UpperB+OH+SC+17*RC+OH+LowerB
and DELTA = max 1 ((SPEED * PERIOD) quo 1000)
;;

type xsht_info = {
  xd                    : xdata;
  args                  : xargs;
  named                 : string -> widget;
  mutable out_len       : num;
  mutable buff          : string;
  mutable col           : num vect;
  mutable bp            : bool;
  mutable scard         : string list;
  mutable xg            : num;
  mutable yg            : num;
  mutable mover_mapped  : bool;
  mutable state         : move_state;
  mutable fast_move     : bool
}

and move_state =
  C'NormalMove
| C'AutoMove of widget * string * num * string * num
;;

let string_of_val = vect_item [|
  "A"; "2"; "3"; "4"; "5"; "6"; "7"; "8"; "9"; "0"; "V"; "D"; "R"
|]

and string_of_suit = vect_item [| "P"; "C"; "K"; "T" |]
;;

let string_of_card =
  let card_string = make_vect 52 "" in
  modify_vect_i (fun card _ ->
    (string_of_val (card mod 13)) ^ (string_of_suit (card quo 13))
  ) card_string;
function card -> card_string.(card)
;;

let col_x n =
  LeftB+n*(OW+IC)
;;

let tower_x = vect_item [| col_x 0; col_x 1; col_x 8; col_x 9 |]
and tower_y _ = UpperB
and buff_x = vect_item [| col_x 3; col_x 4; col_x 5; col_x 6 |]
and buff_y _ = UpperB
and column_x = col_x
and column_y xi nb = UpperB+OH+SC+RC*xi.col.(nb)
and heap_x = (GW-OW) quo 2
and heap_y = (GH-LowerB-OH)
;;

let position_of_coord x y =
  let nb = (x-LeftB+(IC quo 2))quo(OW+IC) in
  if y >= UpperB+OH+(SC quo 2) then (
    "C" ^ (string_of_num nb)
  ) else if nb >= 3 & nb <= 6 then (
    let nb = nb-3 in "B" ^ (string_of_num nb)
  ) else if nb <= 1 or nb >= 8 then (
    let nb = (if nb <= 1 then nb else nb-6) in "T" ^ (string_of_num nb)
  ) else (
    ""
  )
;;

let motion xi (wid, x, y) =
  if xi.scard <> [] then (
    let mover = xi.named "mover" in
    let nx = (widget_x mover)+x-xi.xg
    and ny = (widget_y mover)+y-xi.yg
    and w = widget_width mover
    and h = widget_height mover in
    let new_x = max 0 (min ((widget_width wid)-w) nx)
    and new_y = max 0 (min ((widget_height wid)-h) ny) in
    rt_move_widget(mover, new_x, new_y);
    xi.xg <- x + new_x - nx;
    xi.yg <- y + new_y - ny;
    ()
  )

and enter_wind xi card wid =
  let _ = random 1 in
  if not xi.bp then (xi.scard <- [ card ]; ())

and leave_wind xi wid =
  if not xi.bp then (xi.scard <- []; ())

and bpressed xi (wid, b, x, y) =
  xi.bp <- true;
  if xi.state = C'NormalMove then (
    match xi.scard with [] -> ()
    | scard::_ ->
        xi.xg <- x; xi.yg <- y;
        let swid = xi.named scard in
        let x = widget_x swid + (OW quo 2)
        and y = widget_y swid + (OH quo 2) in
        xi.scard <- [];
        xi.buff <- "S" ^ scard ^ (position_of_coord x y);
        ()
  )

and breleased xi pwid =
  xi.bp <- false;
  if xi.state = C'NormalMove then (
    match xi.scard with [] -> ()
    | scard::_ ->
        let dst = position_of_coord xi.xg xi.yg
        and swid = xi.named scard in
        xi.scard <- [];
        if dst <> "" then (
          let x = widget_x swid + (OW quo 2)
          and y = widget_y swid + (OH quo 2) in
          let src = position_of_coord x y in
          xi.buff <- "M" ^ scard ^ src ^ dst;
          ()
        ) else (
          let mover = xi.named "mover" in
          xi.mover_mapped <- false;
          rt_map_widget swid;
          rt_unmap_widget mover
        )
  )
;;

let redb = ref 34 and greenb = ref 139 and blueb = ref 34;;
let glop = B (min 255) (max 0);;

let keyp xi col (wid, s) =
  match s with
    "1" | "2" | "3" | "4" | "5" | "6" | "0" ->
      let v = ascii_code s - ascii_code "1" in
      if v < 0 then (
        display_num !redb; display_string " ";
        display_num !greenb; display_string " ";
        display_num !blueb; display_newline()
      )
      else if v <= 1 then (redb := glop (!redb + 2*v-1); ())
      else if v <= 3 then (greenb := glop(!greenb + 2*v-5); ())
      else if v <= 5 then (blueb := glop(!blueb + 2*v-9); ());
      rt_change_color(col, !redb, !greenb, !blueb)
  | _ ->
      xi.buff <- "C"^s;
      ()
;;

let exact_move xi wid src si dst di =
  if src = "C" then (
    xi.col.(si) <- xi.col.(si)-1; ()
  );
  let x, y = match dst with
    "C" ->
      let x, y = column_x di, column_y xi di in
      xi.col.(di) <- xi.col.(di)+1;
      x, y
  | "B" -> buff_x di, buff_y di
  | "T" -> tower_x di, tower_y di
  | _ -> heap_x, heap_y in
  rt_move_widget(wid, x, y)
;;

let dist x1 y1 x2 y2 =
  let dx = x1-x2 and dy = y1-y2 in
  sqrt(dx*dx+dy*dy)
;;

let woops xi _ =
  match xi.state with
    C'AutoMove(wid, src, si, dst, di) ->
      let x1 = widget_x wid and y1 = widget_y wid
      and x2, y2 = match dst with
        "B" -> buff_x di, buff_y di
      | "C" -> column_x di, column_y xi di
      | "T" -> tower_x di, tower_y di
      | _ -> failwith "woops 1" in
      let d = dist x1 y1 x2 y2 in
      let xw = x1 + ((DELTA * (x2-x1)) quo d)
      and yw = y1 + ((DELTA * (y2-y1)) quo d) in
      let nd = dist xw yw x2 y2 in
      if xi.fast_move or nd < (DELTA quo 2) or nd >= d then (
        exact_move xi wid src si dst di;
        xi.buff <- "E";
        xi.state <- C'NormalMove;
        ()
      ) else (
        rt_move_widget(wid, xw, yw);
        rt_set_timeout(xi.args, rt_current_time xi.args + PERIOD)
      )
  | _ -> failwith "woops"
;;

let no_fun _ = ()
;;

let create_tower pwid nb attr =
  let x = tower_x nb and y = tower_y nb in
  let wid = rt_create_subwidget(pwid, x, y, RawA attr (WID, HEI, BD, [])) in
  rt_map_widget wid;
  wid

and create_buff pwid nb attr =
  let x = buff_x nb and y = buff_y nb in
  let wid = rt_create_subwidget(pwid, x, y, RawA attr (WID, HEI, BD, [])) in
  rt_map_widget wid;
  wid

and create_card xi pwid val suit attr =
  let card = 13*suit+val in
  let pl = [
    EnterWindowPr(enter_wind xi (string_of_card card));
    LeaveWindowPr(leave_wind xi)
  ] in
  let wid = rt_create_subwidget(pwid, heap_x, heap_y,
    RawA (NameAtt(string_of_card card)::attr) (WID, HEI, BD, pl)
  ) in
  rt_map_widget wid;
  wid
;;

let xread xi buff =
  xi.buff <- "";
  while xi.buff = "" do rt_treat_one_event xi.args done;
  replace_string buff xi.buff 0;
  length_string xi.buff
;;

let xaction xi txt =
  try
  let comm = sub_string txt 0 1 in
  match comm with
    "U" ->
      let card = extract_string txt 1 2 in
      let swid = xi.named card
      and mover = xi.named "mover" in
      xi.mover_mapped <- false;
      rt_map_widget swid;
      rt_unmap_widget mover
  | "M" ->
      let card = extract_string txt 1 2
      and src = extract_string txt 3 3
      and si = num_of_string(extract_string txt 4 4)
      and dst = extract_string txt 5 5
      and di = num_of_string (extract_string txt 6 6) in
      let swid = xi.named card in
      exact_move xi swid src si dst di;
      if xi.mover_mapped then (
        let mover = xi.named "mover" in
        xi.mover_mapped <- false;
        rt_map_widget swid;
        rt_unmap_widget mover
      )
  | "W" ->
      let card = extract_string txt 1 2
      and src = extract_string txt 3 3
      and si = num_of_string(extract_string txt 4 4)
      and dst = extract_string txt 5 5
      and di = num_of_string (extract_string txt 6 6) in
      let swid = xi.named card in
      xi.state <- C'AutoMove(swid, src, si, dst, di);
      rt_set_timeout(xi.args, rt_current_time xi.args + PERIOD)
  | "S" ->
      let scard = extract_string txt 1 2 in
      let swid = xi.named scard
      and mover = xi.named "mover" in
      xi.scard <- [scard];
      rt_move_resize_widget(mover,
        widget_x swid, widget_y swid, OW, OH
      );
      xi.mover_mapped <- true;
      rt_map_widget mover;
      rt_unmap_widget swid
  | "A" -> select_raise(xi.named "auto/manual", 1)
  | "B" -> select_raise(xi.named "auto/manual", 0)
  | "Z" ->
      xi.state <- C'NormalMove;
      rt_reset_timeout xi.args;
      modify_vect (fun _ -> 0) xi.col
  | "Q" ->
      ()
  | _ ->
      display_string "X: "; display_string txt;
      display_string "... "; display_flush()
  with _ ->
      display_string "X bad command: "; display_string txt;
      display_newline()
;;

let line = create_string 20;;
let xwrite xi txt len =
  iterate (fun i ->
    let c = nth_char txt i in
    if c = `\n` then (
      xaction xi (sub_string line 0 xi.out_len);      
      xi.out_len <- 0
    ) else (
      set_nth_char line xi.out_len c;
      xi.out_len <- xi.out_len + 1
    );
    i+1
  ) len 0;
  ()
;;

let xclose xi _ =
  gc_alarm false;
  rt_end xi.xd
;;

let xselect_fd xi fd fd_fun =
  rt_select_file(xi.args, fd, fd_fun)
;;

let dir = ref "../cards/pixmap"
and dname = ref ""
;;

let xdriver _ _ =
  let i = ref 1 in
  while !i < vect_length command_line do
    (match command_line.(!i) with
      "-c" -> incr i; dir := command_line.(!i)
    | "-d" -> incr i; dir := command_line.(!i) ^ "/../cards/pixmap"
    | s -> dname := s);
    incr i
  done;
  let tabf = !dir ^ "/colors-6x256"
  and pixf = !dir ^ "/cards" in
  init_random ((ftime()).time);
  let xd = rt_initialize !dname in try
    gc_alarm true;

    let card_pixmap, tower_pixmap, buff_pixmap = make_pixmap xd tabf pixf in

    let xi = {
      xd = xd; named = widget_named xd;
      args = rt_args[xd];
      out_len = 0;
      buff = ""; col = (make_vect 10 0);
      bp = false; scard = []; xg = 0; yg = 0;
      mover_mapped = false; state = C'NormalMove;
      fast_move = false
    } in
    let col = rt_create_color(xd, !redb, !greenb, !blueb) in
    let attr = [BackgroundAtt(ColorBg col)]
    and attr1 = [
      BackgroundAtt(ColorBg(rt_create_color(xd, 173, 255, 47)));
      BorderBackgAtt(ColorBg(rt_create_color(xd, 95, 158, 160)))
    ]
    and attr2 = [
      BackgroundAtt(ColorBg(rt_create_color(xd, 205, 92, 92)));
      BorderBackgAtt(ColorBg(rt_create_color(xd, 95, 158, 160)))
    ] in
    let main_wid = rt_create_widget(xd, "Seahaven towers", "sht",
      PackA attr (Vertical, [
        RawA (NameAtt "game"::attr) (GW, GH, 0, [
          KeyPr(keyp xi col);
          ButtonPressedPr(bpressed xi);
          ButtonReleasedPr(breleased xi);
          ButtonMotionPr(motion xi)
        ]);
        PackA (BorderAtt 0::attr) (Horizontal, [
          SelectA [NameAtt "fast/slow"; WidthAtt 80] [
            ButtonA attr1 ("fast", fun _ ->
              xi.fast_move <- true;
              select_raise(xi.named "fast/slow", 1)
            );
            ButtonA attr1 ("slow", fun _ ->
              xi.fast_move <- false;
              select_raise(xi.named "fast/slow", 0)
            )
          ];
          SelectA [NameAtt "auto/manual"; WidthAtt 80] [
            ButtonA attr1 ("auto", fun _ -> xi.buff <- "Ca"; ());
            ButtonA attr1 ("manual", fun _ -> xi.buff <- "Cm"; ())
          ];
          ButtonA (WidthAtt 80::attr1)
            ("undo", (fun _ -> xi.buff <- "Cz"; ()));
          ButtonA (WidthAtt 80::attr1)
            ("redo", (fun _ -> xi.buff <- "Cr"; ()));
          ButtonA (WidthAtt 80::attr1)
            ("new game", (fun _ -> xi.buff <- "Cn"; ()));
          ButtonA (WidthAtt 80::attr2)
            ("quit", (fun _ -> xi.buff <- "Cq"; ()))
        ])
      ])
    ) in
    let game_wid = xi.named "game" in

    let _ = rt_create_subwidget(
      game_wid, 0, 0,
      RawA [NameAtt "mover"; BackgroundAtt NoneBg] (0, 0, 0, [])
    ) in

    iterate (fun i ->
      let attr = [BackgroundAtt(PixmapBg tower_pixmap.(i))] in
      let _ = create_tower game_wid i attr in
      i+1
    ) 4 0;
    let attr = [BackgroundAtt(PixmapBg buff_pixmap)] in
    iterate (fun i ->
      let _ = create_buff game_wid i attr in
      i+1
    ) 4 0;

    iterate (fun suit ->
      iterate (fun val ->
        let attr = [BackgroundAtt(PixmapBg card_pixmap.(suit).(val))] in
        let _ = create_card xi game_wid val suit attr in
        val+1
      ) 13 0;
      suit+1
    ) 4 0;

    rt_set_timeout_fun(xi.args, woops xi);
    rt_map_widget main_wid;
    xread xi, xwrite xi, xclose xi, xselect_fd xi
  with x -> gc_alarm false; rt_end xd; raise x
;;
