(* $Id: csht.ml,v 1.2 91/06/29 17:07:49 ddr Exp $
 *
 * Seahaven Towers. Algorithme.
 *)

#open "xsht";;
#open "sys";;

let iterate f = (iterate_f
  where rec iterate_f n x =
    if n > 0 then iterate_f (pred n) (f x))
and modify_vect f v =
  let len = vect_length v in (modify_rec 0
    where rec modify_rec i =
      if i < len then (v.(i) <- f v.(i); modify_rec (succ i)))
and modify_vect_i f v =
  let len = vect_length v in (modify_rec 0
    where rec modify_rec i =
      if i < len then (v.(i) <- f i v.(i); modify_rec (succ i)))
and prefix quo x y = x / y
and gc_alarm _ = ()
and hd = function x::l -> x | _ -> failwith "hd"
and tl = function x::l -> l | _ -> failwith "tl"
and uncons = function x::l -> x,l | _ -> failwith "uncons"
and item = (item_rec
  where rec item_rec = function
    (a::l) -> (function 0 -> a | i -> item_rec l (i-1))
  | _ -> failwith "item")
and prim_rec f = (prim_loop
  where rec prim_loop init = function
    0 -> init | n -> prim_loop (f init n) (n-1))
and chop_list n l = (chop_aux n ([],l)
  where rec chop_aux = fun
    0 (l1,l2) -> rev l1,l2
  | n (l1,x::l2) -> chop_aux (pred n) (x::l1,l2)
  | _ _ -> failwith "chop_list")
and make_string_c len s = make_string len (nth_char s 0)
and nth_char_c n s = make_string 1 (nth_char s n)
and display_string = print_string
and display_flush() = flush std_out
and length_string = string_length
and message s = print_string s; print_newline()
;;

let vect_exists_i p v = exists_p 0
  where rec exists_p i =
    if i = vect_length v then false
    else p i v.(i) or exists_p(i+1)
;;    

type game = {
  xwrite            : string -> unit;
  xread             : string -> int;
  mutable in_cur    : int;
  mutable in_len    : int;
  tower             : int vect;
  buffer            : (int * int) list vect;
  column            : (int * int) list vect;
  mutable undo      : (bool * int * int * string * int * string * int) list;
  mutable redo      : (bool * int * int * string * int * string * int) list;
  mutable auto      : bool;
  mutable wait_move : bool
}
;;

let string_of_val = vect_item [|
  "A"; "2"; "3"; "4"; "5"; "6"; "7"; "8"; "9"; "0"; "V"; "D"; "R"
|]

and string_of_suit = vect_item [| "P"; "C"; "K"; "T" |]

and val_of_string = function
  "A" -> 0 | "0" -> 9 | "V" -> 10 | "D" -> 11 | "R" -> 12
| n -> (int_of_string n)-1

and suit_of_string = function
  "P" -> 0 | "C" -> 1 | "K" -> 2 | "T" -> 3 | _ -> failwith "suit_of_string"
;;

let string_of_card =
  let card_string = make_vect 52 "" in
  modify_vect_i (fun card _ ->
    (string_of_val (card mod 13)) ^ (string_of_suit (card quo 13))
  ) card_string;
function card -> card_string.(card)
;;

let record_move gm x =
  gm.undo <- x::gm.undo;
  gm.redo <- [];
  ()
;;

let move comm gm val suit src si dst di =
  (match src with
    "T" -> gm.tower.(si) <- gm.tower.(si)-1; ()
  | "B" -> gm.buffer.(si) <- []; ()
  | "C" -> gm.column.(si) <- tl gm.column.(si); ()
  | _ -> failwith "move 1");
  (match dst with
    "T" -> gm.tower.(di) <- gm.tower.(di)+1; ()
  | "B" -> gm.buffer.(di) <- (val, suit)::[]; ()
  | "C" -> gm.column.(di) <- (val, suit)::gm.column.(di); ()
  | _ -> failwith "move 2");
  gm.xwrite (comm ^ string_of_val val ^ string_of_suit suit ^
    src ^ string_of_int si ^ dst ^ string_of_int di
  )
;;

let auto_move gm =
  let from_buffer_or_column_to_tower name =
    vect_exists_i (function i -> function
      [] -> false
    | (val,suit)::_ ->
        if gm.tower.(suit) = val then (
          record_move gm (true, val, suit, "T", suit, name, i);
          move "W" gm val suit name i "T" suit;
          true
        ) else false
    )
  and from_buffer_to_column() =
    false
(*
    vect_exists_i (function i -> function
      [] -> false
    | (val,suit)::_ ->
        vect_exists_i (function j -> function
          [] -> false
        | (v, s)::_ ->
            if s = suit & v = val+1 then (
              record_move gm (true, val, suit, "C", j, "B", i);
              move "W" gm val suit "B" i "C" j;
              true
            ) else false
        ) gm.column
    ) gm.buffer
*)
  in
  if gm.wait_move or
  from_buffer_or_column_to_tower "B" gm.buffer or
  from_buffer_or_column_to_tower "C" gm.column or
  from_buffer_to_column() then (
    gm.wait_move <- true;
    ()
  )
;;

let manual gm =
  gm.auto <- false;
  gm.xwrite "B"

and auto gm =
  gm.auto <- true;
  gm.xwrite "A";
  auto_move gm
;;

let new_game gm =
  let card_list = prim_rec (fun cl i -> i-1::cl) [] 52 in
  let mixed_cards, _ = prim_rec (fun (mc, cl) len ->
    let r = random len in
    let b, e = chop_list r cl in
    (hd e::mc, b @ tl e)
  ) ([], card_list) 52 in
  gm.xwrite "Z"; gm.wait_move <- false;
  modify_vect (fun _ -> 0) gm.tower;
  modify_vect (fun _ -> []) gm.column;
  gm.buffer.(0) <- [];
  gm.buffer.(3) <- [];
  let card = item mixed_cards 0 in
  gm.xwrite ("M" ^ (string_of_card card) ^ "H0B1");
  gm.buffer.(1) <- (card mod 13, card quo 13)::[];
  let card = item mixed_cards 1 in
  gm.xwrite ("M" ^ (string_of_card card) ^ "H0B2");
  gm.buffer.(2) <- (card mod 13, card quo 13)::[];
  let _ = prim_rec (fun cl i ->
    let c, cl = uncons cl in
    let n = (50-i) mod 10 in
    gm.xwrite ("M" ^ (string_of_card c) ^ "H0C" ^ (string_of_int n));
    gm.column.(n) <- (c mod 13, c quo 13)::gm.column.(n);
    cl
  ) (tl(tl mixed_cards)) 50 in
  auto gm;
  gm.undo <- []; gm.redo <- [];
  ()
;;

let unmove comm gm val suit src si dst di =
  gm.xwrite (comm ^ string_of_val val ^ string_of_suit suit ^
    src ^ string_of_int si ^ dst ^ string_of_int di
  )
;;

let rec undo gm =
  match gm.undo with [] -> ()
  | (auto, val, suit, src, si, dst, di)::rest ->
      move "M" gm val suit src si dst di;
      gm.redo <- (auto, val, suit, dst, di, src, si)::gm.redo;
      gm.undo <- rest;
      if auto & gm.auto then undo gm;
      ()
;;

let redo gm = redo_rec true
where rec redo_rec first =
  match gm.redo with [] -> ()
  | (auto, val, suit, src, si, dst, di)::rest ->
      if first or auto & gm.auto then (
        move "M" gm val suit src si dst di;
        gm.undo <- (auto, val, suit, dst, di, src, si)::gm.undo;
        gm.redo <- rest;
        redo_rec false
      )
;;

exception eof;;
let buff = make_string_c 20 " ";;
let get_char gm =
  if gm.in_cur = gm.in_len then (
    gm.in_len <- gm.xread buff;
    if gm.in_len = 0 then raise eof else (gm.in_cur <- 0; ())
  );
  let c = nth_char_c gm.in_cur buff in
  gm.in_cur <- gm.in_cur + 1;
  c
;;

let move_it gm =
  let val = val_of_string(get_char gm) in
  let suit = suit_of_string(get_char gm) in
  let src = get_char gm in
  let si = int_of_string(get_char gm) in
  let dst = get_char gm in
  let di = int_of_string(get_char gm) in
  (match dst with
    "T" ->
      if di <> suit then unmove "U"
      else if gm.tower.(di) <> val then unmove "U"
      else (
        record_move gm (false, val, suit, dst, di, src, si);
        move "M"
      )
  | "B" ->
      if gm.buffer.(di) <> [] then unmove "U"
      else (
        record_move gm (false, val, suit, dst, di, src, si);
        move "M"
      )
  | "C" ->
      if match gm.column.(di) with
        [] -> val <> 12
      | (v,s)::_ -> suit <> s or val <> v-1 then unmove "U"
      else (
        record_move gm (false, val, suit, dst, di, src, si);
        move "M"
      )
  | _ ->
      failwith "move_it")
  gm val suit src si dst di
;;

let select_it gm =
  let val = get_char gm in
  let suit = get_char gm in
  let src = get_char gm in
  let si = get_char gm in
  match src with
    "T" | "B" ->
      gm.xwrite ("S"^val^suit^src^si)
  | "C" ->
      if match gm.column.(int_of_string si) with
        [] -> failwith "select_it 1"
      | (v,s)::_ -> suit_of_string suit = s & val_of_string val = v then (
        gm.xwrite ("S"^val^suit^src^si)
      )
  | _ -> failwith "select_it 2"
;;

let stdin_fun _ =
(*
  let str = " " in
  read 0 str 0 1;
*)
  let str = read_line () in
  display_string "<stdin:";
  display_string str;
  display_string ">";
  display_flush()
;;

exception end_prog;;
let xclient machine port =
  let xread, xwrite, xclose, xselect_fd = xdriver machine port in (try
    gc_alarm true;
    xselect_fd 0 stdin_fun;
    let gm = {
      xwrite = (fun s -> xwrite (s^"\n") ((length_string s)+1));
      xread = xread; in_cur = 0; in_len = 0;
      tower = (make_vect 4 0);
      buffer = (make_vect 4  []);
      column = (make_vect 10 []);
      undo = []; redo = [];
      auto = false; wait_move = false
    } in
    new_game gm;
    while true do try
      match get_char gm with
        "C" ->
          (match get_char gm with
            "q" -> raise end_prog
          | "n" -> new_game gm
          | "z" | "u" -> undo gm
          | "r" -> redo gm
          | "h" ->
              iterate (fun i ->
                gm.xwrite ("M" ^ (string_of_card i) ^ "H0H0");
                i+1
              ) 52 0; ()
          | "m" -> manual gm
          | "a" -> auto gm
          | _ -> ())
      | "M" ->
          move_it gm;
          if gm.auto then auto_move gm
      | "S" ->
          select_it gm
      | "E" ->
          gm.wait_move <- false;
          if gm.auto then auto_move gm
      | c ->
          display_string "A:"; display_string c;
          display_string "... "; display_flush()
      with end_prog -> raise end_prog | eof -> raise eof | _ ->
        message "csht: error in command"
    done
  with end_prog -> () | eof -> () | x -> gc_alarm false; xclose(); raise x);
  gc_alarm false; xclose()
;;

let csht() = xclient "" 0
;;

try csht() with
  Failure s -> print_string "failure: "; print_string s; print_newline()
| Invalid_argument s ->
    print_string "Invalid_argument: "; print_string s; print_newline()
| x -> print_string "Unknown exception in csht"; print_newline(); raise x
;;
