(* $Id: column.ml,v 1.8 91/07/05 18:11:28 ddr Exp $
 *
 * xcolumn
 *)

#open "rt";;
#open "sys";;
#open "time";;

type num == int
and 'a seg == 'a vect
and 'a option = None | Some of 'a;;
let mkw w =
  let s = create_string (list_length w) in (mkw_rec 0 w
    where rec mkw_rec i = function
      [] -> s
    | c::w -> set_nth_char s i c; mkw_rec (succ i) w)
and line_buff = create_string 200
;;
  
let iterate f = (iterate_f
  where rec iterate_f n x =
    if n > 0 then iterate_f (pred n) (f x))
and break_string sep s = (break_rec [] 0
  where rec break_rec w i =
    if i = string_length s then if w = [] then [] else [mkw(rev w)]
    else let c = nth_char s i in
      if c = nth_char sep 0 then mkw(rev w)::break_rec [] (succ i)
      else break_rec (c::w) (succ i))
and input_line ic = sub_string line_buff 0 (input_rec 0
  where rec input_rec i =
    match (input_char ic) with
      `\n` -> i
    | c -> set_nth_char line_buff i c; input_rec (succ i))
and nth = (nth_rec
  where rec nth_rec = fun
    (x::l) 1 -> x
  | (x::l) i -> nth_rec l (pred i)
  | _ _ -> failwith "nth")
and do_list_i f = (do_list_f
  where rec do_list_f i = function
    [] -> () | x::l -> f i x; do_list_f (succ i) l)
and output_line ch s =
  output ch s 0 (string_length s);
  output ch "\n" 0 1
;;
let except_assq e = (except_e
  where rec except_e = function
    (x, _ as y)::l -> if x == e then l else y::except_e l
  | _ -> [])
and hd = function x::l -> x | _ -> failwith "hd"
and tl = function x::l -> l | _ -> failwith "tl"
and modify_vect f v =
  iterate (fun i -> v.(i) <- f v.(i); i+1) (vect_length v) 0
and modify_vect_i f v =
  iterate (fun i -> v.(i) <- f i v.(i); i+1) (vect_length v) 0
and do_vect_i f v =
  iterate (fun i -> f i v.(i); i+1) (vect_length v) 0
and tee (f, g) x = f x, g x
and curry f x y = f(x, y)
and seg_item(a,b) = vect_item a b
and seg_length = vect_length
and I x = x
and B f g x = f( g x)
and int_of_num (x : num) = x
and prefix quo x y = x / y
and string_of_num = string_of_int
and length_string = string_length
and display_string = print_string
and display_num = print_int
and display_newline = print_newline
and display_flush() = flush std_out
and min x y = if x < y then x else y
and num_of_int (x: int) = x
and read ic len =
  let s = make_string len ` ` in
  let _ = input ic s 0 len in s
and gc_alarm (b: bool) = ()
and length = list_length
and rtime t2 = (ftime()).time - t2
;;
let line_buff = create_string 200;;
let read_line ic = sub_string line_buff 0 (input_rec 0)
  where rec input_rec i =
    match (input_char ic) with
      `\n` -> i
    | c -> set_nth_char line_buff i c; input_rec (succ i)
;;
let seed = ref 7
and maxint = 536870911*2+1;;
let random n =
  seed := 25173*!seed + 13849;
  (if !seed < 0 then - !seed else !seed) / (maxint / n)
and init_random n = seed := n; ()
;;
let implode_ascii l =
  let len = list_length l in
  let s = make_string len ` ` in
  iterate (fun (i,l) ->
    set_nth_char s i (char_of_int (hd l));
    (i+1, tl l)
  ) len (0, l);
  s
;;

(*let column_version = nth (words "$Revision: 1.8 $") 2;;*)

let FillerD = PackA [FillerAtt; BorderAtt 0] (Vertical, []);;

type 'a stream = {Hd : 'a; mutable Tl : 'a stream option}
;;

type game_data = {
  xd                              : xdata;
  dname                           : string;
  pname                           : string;
  pnum                            : num;
  patt                            : glop;
  mutable game_type               : game_type;
  mutable score                   : num;
  mutable time_beginning_of_game  : num;
  mutable time_beginning_of_level : num;
  mutable falling_height          : num;
  mutable state                   : game_state;
  mutable ask                     : ask;
  mutable level                   : num;
  mutable min_level               : num;
  mutable runner_state            : runner_state;
  mutable timeout                 : num option;
  col                             : fall_col;
  board                           : num vect vect;
  mark                            : bool vect vect
}

and glop = C'Pattern of pattern vect | C'Color of color vect

and global_data = {
  xargs             : xargs;
  init_time         : num;
  mutable gwl       : (game_data * drawable) list
}

and fall_col = {
  mutable x     : num;
  mutable y     : num;
  squ           : num vect
}

and game_type =
  C'Normal
| C'Tournament of (num * num seg) stream

and game_state =
  C'NotStarted | C'Running | C'Pausing | C'Ended

and runner_state = C'TouchedDown | C'Falling | C'Exploding

and ask = C'AskForAbort | C'AskForQuit | C'NoAsk

and score_tab = {
  mutable player    : string;
  mutable bscore    : num
}

and my_patterns =
  C'WhiteP | C'BlackP | C'DiagonalP
| C'SandP | C'ChBoardP | C'VerticalP | C'GrayP
;;

let patterns = [|
  implode_ascii [  0;   0;   0;   0;   0;   0;   0;   0;
                   0;   0;   0;   0;   0;   0;   0;   0;
                   0;   0;   0;   0;   0;   0;   0;   0;
                   0;   0;   0;   0;   0;   0;   0;   0] (* blanc *);
  implode_ascii [255; 255; 255; 255; 255; 255; 255; 255;
                 255; 255; 255; 255; 255; 255; 255; 255;
                 255; 255; 255; 255; 255; 255; 255; 255;
                 255; 255; 255; 255; 255; 255; 255; 255] (* noir *);
  implode_ascii [240; 240; 120; 120;  60;  60;  30;  30;
                  15;  15; 135; 135; 195; 195; 225; 225;
                 240; 240; 120; 120;  60;  60;  30;  30;
                  15;  15; 135; 135; 195; 195; 225; 225] (* oblique *);
  implode_ascii [  1;   1;   8;   8;  64;  64;   4;   4;
                 128; 128;  16;  16;   2;   2;  32;  32;
                   1;   1;   8;   8;  64;  64;   4;   4;
                 128; 128;  16;  16;   2;   2;  32;  32] (* sable *);
  implode_ascii [255;   0; 255;   0; 255;   0; 255;   0;
                 255;   0; 255;   0; 255;   0; 255;   0;
                   0; 255;   0; 255;   0; 255;   0; 255;
                   0; 255;   0; 255;   0; 255;   0; 255] (* damier *);
  implode_ascii [ 60;  60;  60;  60;  60;  60;  60;  60;
                  60;  60;  60;  60;  60;  60;  60;  60;
                  60;  60;  60;  60;  60;  60;  60;  60;
                  60;  60;  60;  60;  60;  60;  60;  60] (* vertical *);
  implode_ascii [ 85;  85; 170; 170; 85;   85; 170; 170;
                  85;  85; 170; 170; 85;   85; 170; 170;
                  85;  85; 170; 170; 85;   85; 170; 170;
                  85;  85; 170; 170; 85;   85; 170; 170] (* gris *)
|]

and colors = [|
  (255, 255, 255);
  (0, 0, 0);
  (203, 46, 201);
  (135, 206, 255);
  (255, 255, 135);
  (34, 139, 24);
  (255, 143, 52)
|]
;;

let select_pattern(gm, n) =
  let i = match n with C'WhiteP -> 0 | C'BlackP -> 1 | C'DiagonalP -> 2
  | C'SandP -> 3 | C'ChBoardP -> 4 | C'VerticalP -> 5 | C'GrayP -> 6 in
  match gm.patt with
    C'Pattern patt -> rt_select_pattern (patt.(i), 0, 0)
  | C'Color col -> rt_select_color col.(i)
;;

let patt_init xd =
  let SZ = 16 in
  let pat0 = rt_create_pattern(xd, patterns.(0), 16, 16) in
  let patt = make_vect (vect_length patterns) pat0 in
  modify_vect_i (fun i p ->
    if i = 0 then pat0 else rt_create_pattern(xd, patterns.(i), 16, 16)
  ) patt;
  C'Pattern patt
;;

let col_init xd =
  let (r,g,b) = colors.(0) in
  let pat0 = rt_create_color(xd, r, g, b) in
  let patt = make_vect (vect_length colors) pat0 in
  modify_vect_i (fun i p ->
    if i = 0 then pat0
    else let (r,g,b) = colors.(i) in rt_create_color(xd, r, g, b)
  ) patt;
  C'Color patt
;;

let column_wizard = ref false
and trace_games = ref true
and trace_gc = ref true
;;

let SZ = 32
and W = 6
and H = 20
and UpperB = 64 and LeftB = 8
and LowerB = 8 and RightB = 150
and speed_tab, nlevels = tee (curry seg_item, seg_length) [|
  1000; 800; 600; 500; 440 ; 400; 360; 320; 280; 240;
  200; 180; 160; 140; 120; 100; 80; 60; 40; 20
|]
and change_level = (curry seg_item) [|
  10; 20; 30; 40; 50; 60; 70; 80; 90; 100;
  110; 120; 130; 140; 150; 160; 170; 180; 190; 200;
  210; 220; 230; 240; 250; 260; 270; 280; 290; 300
|]
and SCORE_FILE = "./column.sc"
and tab_colors, ncolors = tee (curry seg_item, B int_of_num seg_length)
  [| C'BlackP; C'DiagonalP; C'SandP; C'ChBoardP; C'VerticalP; C'GrayP |]
;;

let GW = LeftB+W*SZ+RightB
and GH = UpperB+H*SZ+LowerB
;;

let clear_square gm drw x y =
  select_pattern(gm, C'WhiteP);
  rt_fill_rectangle(drw, x*SZ+LeftB, y*SZ+UpperB, SZ, SZ)

and draw_square gm drw x y k =
  select_pattern(gm, tab_colors(k-1));
  rt_fill_rectangle(drw, x*SZ+LeftB+1, y*SZ+UpperB+1, SZ-1, SZ-1);
  select_pattern(gm, C'BlackP);
  rt_draw_rectangle(drw, x*SZ+LeftB, y*SZ+UpperB, SZ-1, SZ-1)

and draw_score gm drw =
  select_pattern(gm, C'BlackP);
  rt_erase_draw_string(drw, LeftB+(SZ quo 3), 25,
    if !column_wizard then "score <" ^ string_of_num(gm.score) ^ ">   "
    else "score " ^ string_of_num(gm.score) ^ "   "
  )

and draw_level gm drw =
  select_pattern(gm, C'BlackP);
  rt_erase_draw_string(drw, GW-RightB-(SZ quo 3)-50, 25,
      "level " ^
      (let lev = gm.level in
       (if lev < 10 then " " else "") ^
       string_of_num lev))

and draw_state gm drw =
  select_pattern(gm, C'BlackP);
  rt_erase_draw_string(drw, GW-RightB-(SZ quo 3)-70, UpperB - 15,
    match gm.state with
      C'Pausing ->   "    <pause>"
    | C'Ended ->     "<game over>"
    | _ ->          "           "
  )

and draw_player_score gm1 gm drw =
  let s = "     " ^ string_of_num gm1.score in
  let s = sub_string s ((length_string s)-6) 6 in
  select_pattern(gm, C'BlackP);
  rt_erase_draw_string(drw, GW-40, UpperB+20+gm1.pnum*20, s)

and display_player gm =
  display_string "*** player "; display_num gm.pnum;
  display_string " ("; display_string gm.pname;
  display_string "): "

and display_closing_display dname =
  display_string "*** closing display \"";
  display_string dname;
  display_string "\"";
  display_newline()
;;

let gen_timeout gd =
  match it_list (fun v (gm, _) ->
    match v with
      None -> gm.timeout
    | Some tm1 ->
        match gm.timeout with
          None -> Some tm1
        | Some tm2 -> Some (min tm1 tm2)
  ) None gd.gwl with
    None ->
       rt_reset_timeout gd.xargs
  | Some wt ->
       rt_set_timeout(gd.xargs, wt)
;;

let set_gm_timeout gd gm wait_time =
  gm.timeout <- Some (rt_current_time gd.xargs + wait_time);
  gen_timeout gd
and reset_gm_timeout gd gm =
  gm.timeout <- None;
  gen_timeout gd
;;

let draw_players gd gm drw =
  select_pattern(gm, C'WhiteP);
  rt_fill_rectangle(drw, GW-RightB+10, UpperB, 0, 0);
  do_list (fun(gm1, _) ->
    select_pattern(gm, C'BlackP);
    rt_erase_draw_string(drw, GW-RightB+15, UpperB+20+gm1.pnum*20,
      (string_of_num gm1.pnum) ^ ": " ^ gm1.pname);
    draw_player_score gm1 gm drw
  ) gd.gwl
;;

let draw_column gm drw =
  let draw = draw_square gm drw gm.col.x in
  do_vect_i (fun i c -> draw (gm.col.y+i) c) gm.col.squ

and clear_column gm drw =
  let clear = clear_square gm drw gm.col.x in
  do_vect_i (fun i c -> clear (gm.col.y+i)) gm.col.squ

and explose gm drw =
  let draw = draw_square gm drw
  and clear = clear_square gm drw in
  do_vect_i (fun x c ->
    let draw = draw x and clear = clear x in
    do_vect_i (fun y m ->
      if m then (
        let c = gm.board.(x) 
        and i = ref y in
	  while ((!i  > 0) & (c.(!i-1) <> 0)) do
	     c.(!i) <- c.(!i-1); draw !i c.(!i); decr i
	  done;
	  c.(!i) <- 0;
          clear !i
      )
    ) c
  ) gm.mark;
  gm.runner_state <- C'TouchedDown;
  ()

and explosive gm drw x y =
  gm.mark.(x).(y) <- true;
  let A = SZ quo 3 in
  select_pattern(gm, C'WhiteP);
  rt_fill_rectangle(drw, x*SZ+LeftB+A, y*SZ+UpperB+A, SZ-2*A, SZ-2*A)

and incr_score gd gm drw =
  gm.score <- gm.score+(9+gm.falling_height)*(succ gm.min_level);
  let t = rtime gd.init_time in
  if (t - gm.time_beginning_of_level >= change_level gm.level) then (
    if gm.level < nlevels-1 then (
      gm.time_beginning_of_level <- t;
      gm.level <- gm.level+1;
      set_gm_timeout gd gm (speed_tab gm.level);
      draw_level gm drw
    )
  )

and rand_gen() = {
  Hd = (
    num_of_int(random(int_of_num W)), [|
      num_of_int(random ncolors) + 1;
      num_of_int(random ncolors) + 1;
      num_of_int(random ncolors) + 1
    |]
  );
  Tl = None
}
;;

let new_column gm drw =
  gm.min_level <- gm.level;
  (match gm.game_type with
    C'Normal ->
      gm.col.x <- num_of_int(random(int_of_num W));
      modify_vect (fun _ -> num_of_int(random ncolors) + 1) gm.col.squ
  | C'Tournament c ->
      let nc = match c.Tl with
        Some nc -> nc
      | None -> let nc = rand_gen() in c.Tl <- Some nc; nc in
      gm.game_type <- C'Tournament nc;
      let x, y = c.Hd in
      gm.col.x <- x;
      modify_vect_i (fun i _ -> seg_item(y, i)) gm.col.squ
  );
  gm.col.y <- 0;
  draw_column gm drw

and pause gd gm drw =
  gm.state <- C'Pausing;
  let t = rtime gd.init_time in
  gm.time_beginning_of_game <- gm.time_beginning_of_game - t;
  gm.time_beginning_of_level <- gm.time_beginning_of_level - t;
  reset_gm_timeout gd gm;
  if not !column_wizard then (
    select_pattern(gm, C'GrayP);
    rt_fill_rectangle(drw, LeftB, UpperB, W*SZ, H*SZ);
    if !trace_games then (
      display_player gm; display_string "pausing"; display_newline()
    )
  );
  draw_state gm drw

and restart gd gm drw =
  gm.state <- C'Running;
  let t = rtime gd.init_time in
  gm.time_beginning_of_game <- gm.time_beginning_of_game + t;
  gm.time_beginning_of_level <- gm.time_beginning_of_level + t;
  set_gm_timeout gd gm (speed_tab gm.level);
  if not !column_wizard then (
    select_pattern(gm, C'WhiteP);
    rt_fill_rectangle(drw, LeftB, UpperB, W*SZ, H*SZ);
    let draw = draw_square gm drw in
    do_vect_i (fun x c ->
      let draw = draw x in
      do_vect_i (fun y e -> if e <> 0 then draw y e) c
    ) gm.board;
    draw_column gm drw;
    if !trace_games then (
      display_player gm; display_string "restarting"; display_newline()
    )
  );
  draw_state gm drw
;;

let start_new_game gd gm drw =
  do_vect (fun x -> modify_vect (fun _ -> 0) x) gm.board;
  select_pattern(gm, C'WhiteP);
  rt_fill_rectangle(drw, LeftB, UpperB, SZ*W, SZ*H);
  gm.level <- 0;
  gm.min_level <- 0;
  gm.score <- 0;
  gm.falling_height <- 0;
  gm.time_beginning_of_game <- rtime gd.init_time;
  gm.time_beginning_of_level <- gm.time_beginning_of_game;
  gm.state <- C'Running;
  draw_score gm drw;
  draw_state gm drw;
  draw_level gm drw;
  new_column gm drw;
  set_gm_timeout gd gm (speed_tab 0)

and mark_explosions gd gm drw =
  do_vect (fun x -> modify_vect (fun _ -> false) x) gm.mark;
  let explosive = explosive gm drw
  and incr_score = incr_score gd gm in
  do_vect_i (fun x c ->
    do_vect_i (fun y e ->
      if e <> 0 then (
        (* right direction *)
        if x+2 < W & e = gm.board.(x+1).(y) & e = gm.board.(x+2).(y) then (
          iterate (fun i -> explosive (x+i) y; i+1) 3 0;
          incr_score drw;
          gm.runner_state <- C'Exploding; ()
        );
        (* down direction *)
        if y+2 < H & e = c.(y+1) & e = c.(y+2) then (
          iterate (fun i -> explosive x (y+i); i+1) 3 0;
          incr_score drw;
          gm.runner_state <- C'Exploding; ()
        );
        (* right down *)
        if x+2 < W & y+2 < H & e = gm.board.(x+1).(y+1) &
            e = gm.board.(x+2).(y+2) then (
          iterate (fun i -> explosive (x+i) (y+i); i+1) 3 0;
          incr_score drw;
          gm.runner_state <- C'Exploding; ()
        );
        (* left down *)
        if x-2 >= 0 & y+2 < H & e = gm.board.(x-1).(y+1) &
            e = gm.board.(x-2).(y+2) then (
          iterate (fun i -> explosive (x-i) (y+i); i+1) 3 0;
          incr_score drw;
          gm.runner_state <- C'Exploding; ()
        )
      )
    ) c
  ) gm.board;
  if gm.runner_state = C'Exploding then (
    draw_score gm drw;
    let draw = draw_player_score gm in
    do_list (fun(gm1, drw1) -> draw gm1 drw1) gd.gwl;
    if !column_wizard then pause gd gm drw
    else set_gm_timeout gd gm 300
  ) else (
    if gm.level < 10 then set_gm_timeout gd gm 1000
    else if gm.level < 20 then set_gm_timeout gd gm 500
    else if gm.level < 25 then set_gm_timeout gd gm 250
    else set_gm_timeout gd gm 125
  )
;;

let record_if_high_score gm =
  let rec score_list ch =
    try
      let il = break_string "@" (input_line ch) in
      let scv = int_of_string(nth il 2) in
      {player= hd il; bscore= scv}::(score_list ch)
    with _ -> []

  and add_score = function
    sh::st as ls ->
      if gm.score > sh.bscore then {player=gm.pname; bscore=-gm.score}::ls
      else sh::add_score st
  | _ ->
      {player=gm.pname; bscore=-gm.score}::[]

  in
  let ls = add_score(
    try
      let ch = open_in SCORE_FILE in
      let ls = score_list ch in
      close_in ch;
      ls
    with _ -> []
  ) in
  let twid = widget_named gm.xd "high scores" in
  let display_string s = text_send_string(twid,s) in
  let display_num = B display_string string_of_int
  and display_newline() = display_string "\n" in
  text_clear twid;
  display_string "*** best scores:\n\n";
  do_list_i (fun i b ->
    if b.bscore <= 0 then (
      b.bscore <- -b.bscore;
      display_string " -> "
    ) else display_string "    ";
    display_num i; display_string "\t";
    display_string b.player; display_string "\t";
    display_num b.bscore; display_newline()
  ) 1 ls;
 (try
    let ch = open_out SCORE_FILE in (try
      do_list_i (fun i b ->
        if i <= 19 then
          output_line ch (b.player ^ "@" ^ (string_of_num b.bscore) ^ "@")
      ) 1 ls
    with _ -> ());
    close_out ch
  with _ ->
    display_string "Can't write score file"; display_newline());
  select_raise(widget_named gm.xd "woops", 1)
;;

let touch_down gd gm drw =
  gm.runner_state <- C'TouchedDown;
  if gm.col.y < 1 then (
    gm.state <- C'Ended;
    reset_gm_timeout gd gm;
    draw_state gm drw;
    if !trace_games then (
      display_player gm; display_string "game over, score: ";
      display_num gm.score;
      display_newline()
    );
    if not !column_wizard then record_if_high_score gm
  ) else (
    let c = gm.board.(gm.col.x) in
    do_vect_i (fun i e ->
      if gm.col.y+i >= 0 then (c.(gm.col.y+i) <- e; ())
    ) gm.col.squ;
    mark_explosions gd gm drw;
    new_column gm drw
  )
;;

let quit_game gd gm =
  if !trace_games then (
    display_player gm; display_string "quit";
    if gm.state <> C'Ended then (
      display_string ", score: "; display_num gm.score
    );
    display_newline()
  );
  if gm.state <> C'Ended & not !column_wizard then
    record_if_high_score gm;
  gd.gwl <- except_assq gm gd.gwl;
  do_list (fun (gm, drw) -> draw_players gd gm drw) gd.gwl;
  if !trace_games then display_closing_display gm.dname;
  rt_end gm.xd;
  if gd.gwl = [] then rt_stop_main_loop gd.xargs
  else rt_unselect_xdata(gd.xargs, gm.xd)
;;

let woops_fun gd _ =
  do_list (fun (gm, drw) ->
    match gm.timeout with None -> () | Some timeout ->
    if rt_current_time gd.xargs >= timeout  then (
      if gm.state <> C'Running then failwith "erreur dans woops_fun";
      if gm.runner_state = C'Exploding then (
        explose gm drw;
        mark_explosions gd gm drw
      ) else (
        if gm.runner_state = C'TouchedDown then (
          gm.runner_state <- C'Falling; ()
        );
        set_gm_timeout gd gm (speed_tab gm.level);
        if gm.col.y+3 = H or gm.board.(gm.col.x).(gm.col.y+3) <> 0 then (
          gm.falling_height <- 0;
          touch_down gd gm drw
        ) else (
          clear_square gm drw gm.col.x gm.col.y;
          gm.col.y <- gm.col.y + 1;
          draw_column gm drw
        )
      )
    )
  ) gd.gwl
;;

let expose gd gm (wid, _, _, _, _) =
  let drw = WidgetDr wid in
  select_pattern(gm, C'BlackP);
  rt_draw_rectangle(drw, LeftB-1, UpperB-1, W*SZ+1, H*SZ+1);
  draw_score gm drw;
  draw_state gm drw;
  draw_level gm drw;
  draw_players gd gm drw;
  if gm.state = C'NotStarted then (
    gm.state <- C'Running;
    new_column gm drw;
    set_gm_timeout gd gm (speed_tab 0);
    if !trace_games then (
      display_player gm; display_string "starting"; display_newline()
    )
  ) else if gm.state = C'Pausing & not !column_wizard then (
    select_pattern(gm, C'GrayP);
    rt_fill_rectangle(drw, LeftB, UpperB, W*SZ, H*SZ)
  ) else (
    let draw = draw_square gm drw in
    do_vect_i (fun x c ->
      let draw = draw x in
      do_vect_i (fun y e -> if e <> 0 then draw y e) c
    ) gm.board;
    draw_column gm drw
  )
;;

let wid_q gd gm wid =
  let wid_named = widget_named gm.xd in
  try
    wid_named "question"
  with _ ->
    let drw = WidgetDr wid in
    let answer b _ =
      rt_unmap_widget(wid_named "question");
      if b then (
        match gm.ask with
          C'AskForQuit -> quit_game gd gm
        | C'AskForAbort ->
            gm.state <- C'Ended;
            draw_state gm drw;
            if !trace_games then (
              display_player gm; display_string "game aborted, score: ";
              display_num gm.score; display_newline()
            );
            if not !column_wizard then record_if_high_score gm
        | _ -> ()
      ) else (
        restart gd gm drw
      );
      gm.ask <- C'NoAsk;
      () in
    rt_create_subwidget(wid, 10, 10,
      PackA [NameAtt "question"] (Vertical, [
        TitleD("Abort game ?");
        PackD(Horizontal, [
          ButtonD("yes", answer true);
          ButtonD("cancel", answer false)
        ])
      ])
    )
;;

let keyp gd gm (wid, k) =
  if gm.ask = C'NoAsk then
  let drw = WidgetDr wid in
  match k with
    "q" ->
      if gm.state = C'Ended then quit_game gd gm
      else (
        gm.ask <- C'AskForQuit;
        pause gd gm drw;
        let wid_q = wid_q gd gm wid in
        let w = widget_width wid and h = widget_height wid
        and sw = widget_width wid_q and sh = widget_height wid_q in
        rt_move_widget(wid_q, (w-sw)quo 2, (h-sh)quo 2);
        rt_map_widget wid_q
      )
  | "p" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Running then pause gd gm drw
  | "a" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Running then (
        gm.ask <- C'AskForAbort;
        pause gd gm drw;
        let wid_q = wid_q gd gm wid in
        let w = widget_width wid and h = widget_height wid
        and sw = widget_width wid_q and sh = widget_height wid_q in
        rt_move_widget(wid_q, (w-sw)quo 2, (h-sh)quo 2);
        rt_map_widget wid_q
      )

  | "n" | "w" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Ended & (k = "n" or !column_wizard) then (
        if k = "n" then (column_wizard := false; ());
        gm.game_type <- C'Normal;
        start_new_game gd gm drw;
        draw_players gd gm drw;
        if !trace_games then (
          display_player gm; display_string "starting a new game";
          display_newline()
        )
      )
  | "u" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.level < nlevels-1 then (
      gm.time_beginning_of_level <- rtime gd.init_time;
      gm.level <- gm.level+1;
      set_gm_timeout gd gm (speed_tab gm.level);
      draw_level gm drw
      )
  | "d" ->
      if gm.state = C'Pausing then restart gd gm drw
      else 
      let pred_level =  gm.level - 1 in
      if pred_level >= 0 then (
         let t = ref(rtime gd.init_time - gm.time_beginning_of_game)
         and i = ref 0 in
             while (!t > 0) do
                t := !t - change_level !i;
                incr i
             done;
             gm.level <- !i;
             gm.min_level <- !i;
             set_gm_timeout gd gm (speed_tab gm.level);
             draw_level gm drw
       )
  | "Left" | "R10" | "j" | "K4" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Running & gm.col.x > 0 & gm.col.y+2 >= 0 &
          gm.board.(gm.col.x-1).(gm.col.y+2) = 0 then (
        clear_column gm drw;
        gm.col.x <- gm.col.x - 1;
        draw_column gm drw
      )
  | "R11" | "K5" | "k" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Running then (
        let f = gm.col.squ.(2) in
        gm.col.squ.(2) <- gm.col.squ.(1);
        gm.col.squ.(1) <- gm.col.squ.(0);
        gm.col.squ.(0) <- f;
        draw_column gm drw
      )
  | "Right" | "R12" | "l" | "K6" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Running & gm.col.x < W - 1 & gm.col.y+2 >= 0 &
          gm.board.(gm.col.x+1).(gm.col.y+2) = 0 then (
        clear_column gm drw;
        gm.col.x <- gm.col.x + 1;
        draw_column gm drw
      )
  | " "| "Ins" | "K0" ->
      if gm.state = C'Pausing then restart gd gm drw
      else if gm.state = C'Running & gm.col.y+2 >= 0 then (
        clear_column gm drw;
        gm.falling_height <- 0;
        while gm.col.y+3 < H & gm.board.(gm.col.x).(gm.col.y+3) = 0 do
          gm.col.y <- gm.col.y + 1;
          gm.falling_height <- gm.falling_height + 1
        done;
        draw_column gm drw;
        touch_down gd gm drw
      )
  | _   ->
      if gm.state = C'Pausing then restart gd gm drw
;;

let BW = ref false
;;

let one_column(dname, pnum, gd) =
  let xd = rt_initialize dname
  and dname = rt_display_name dname in
  try
    display_string "name of player "; display_num pnum;
    display_string " on "; display_string dname;
    display_string " (";
    let default_name = try getenv "USER" with _ -> dname in
    display_string default_name;
    display_string "): "; display_flush();
    let pname = read_line std_in in
    let pname = if pname = "" then default_name else pname in
    let gm = {
      xd                      = xd;
      dname                   = dname;
      pname                   = pname;
      pnum                    = pnum;
      patt                    = if is_colored xd & not !BW then col_init xd
                                else patt_init xd;
      game_type               = C'Normal;
      score                   = 0;
      falling_height          = 0;
      state                   = C'NotStarted;
      ask                     = C'NoAsk;
      level                   = 0;
      min_level               = 0;
      time_beginning_of_game  = 0;
      time_beginning_of_level = 0;
      runner_state            = C'TouchedDown;
      timeout                 = None;
      col                     = {x = 0; y = 0; squ = make_vect 3 0};
      board                   = make_vect W [| |];
      mark                    = make_vect W [| |]
    } in
    modify_vect (fun _ -> make_vect H 0) gm.board;
    modify_vect (fun _ -> make_vect H false) gm.mark;
    let wdesc = 
      SelectA [NameAtt "woops"] [
        RawA [NameAtt "raw"] (
            GW, GH, 0, [ExposePr(expose gd gm); KeyPr(keyp gd gm)]
        );
        PackA [] (Vertical, [
          TitleD "high scores";
          FillerD;
          TextA [NameAtt "high scores"] (25, 1, 0, (fun _ -> ()), fun _ -> ());
          FillerD;
          ButtonD("ok", (fun _ ->
            select_raise(widget_named xd "woops", 0)
          ))
        ])
      ]
    in
    let (width, height) = widget_size(xd, wdesc) in
    let wid = rt_create_located_widget(xd,
      "player " ^ string_of_num pnum, "column",
      UserPosition(
        (screen_width xd - width) quo 2,
        (screen_height xd - height) quo 2
      ),
      wdesc
    ) in
    rt_map_widget wid;
    rt_select_xdata(gd.xargs, xd);
    gd.gwl <- (gm, WidgetDr(widget_named xd "raw")):: gd.gwl;
    ()
  with x ->
    display_string "*** error while creating game on display \"";
    display_string dname; display_string "\""; display_newline();
    display_closing_display dname;
    rt_end xd; raise x
;;

let stdin_act gd c =
  match c with
    "c" ->
      display_string "*** available commands in game:"; display_newline();
      display_string "u: one level up"; display_newline();
      display_string "d: one level down"; display_newline();
      display_string "q: quit game"; display_newline();
      display_string "p: pause (any key to restart)"; display_newline();
      display_string "a: abort game"; display_newline();
      display_string "n: new game (if game over)"; display_newline();
      display_string "j: column left"; display_newline();
      display_string "k: column rotate"; display_newline();
      display_string "l; column right"; display_newline();
      display_string "<space>: column touch down"; display_newline()
  | "n" ->
      display_string "display name: "; display_flush();
      let dname = read_line std_in in
      (try
        one_column(dname, (fst (hd gd.gwl)).pnum + 1, gd);
        do_list (fun (gm, drw) -> draw_players gd gm drw) gd.gwl
      with _ ->
        display_string "*** Can't open display \"";
        display_string dname; display_string "\""; display_newline()
      )
  | "g" ->
      trace_gc := not !trace_gc;
      gc_alarm !trace_gc;
      if not !trace_gc then display_string "un";
      display_string "tracing gcs"; display_newline()
  | "q" ->
      rt_stop_main_loop gd.xargs
  | "s" ->
      do_list (fun (gm, _) ->
        display_player gm;
        display_string (match gm.state with
          C'NotStarted -> "not started"
        | C'Running -> "running"
        | C'Pausing -> "pause"
        | C'Ended -> "game over"
        );
        display_string ", score: "; display_num gm.score;
        display_newline()
      ) gd.gwl
  | "t" ->
      let g = C'Tournament(rand_gen()) in
      do_list (fun (gm, drw) ->
        gm.game_type <- g;
        start_new_game gd gm drw
      ) gd.gwl;
      do_list (fun (gm, drw) -> draw_players gd gm drw) gd.gwl;
      display_string "tournament"; display_newline()
  | "v" ->
      trace_games := not !trace_games;
      if not !trace_games then display_string "un";
      display_string "tracing games"; display_newline()
  | "\n" ->
      ()
  | _ ->
      display_string "*** available commands:"; display_newline();
      display_string "s: status of games"; display_newline();
      display_string "t: tournament"; display_newline();
      display_string "n: new player"; display_newline();
      display_string "c: game commands"; display_newline();
      display_string "g: "; if !trace_gc then display_string "un";
      display_string "trace gcs"; display_newline();
      display_string "v: "; if !trace_games then display_string "un";
      display_string "trace games"; display_newline();
      display_string "q: general quit"; display_newline()
;;

let buff = create_string 1024;;
let stdin_fun gd _ =
  let len = input std_in buff 0 (string_length buff) in
  iterate (fun i ->
    stdin_act gd (make_string 1 (nth_char buff i));
    i+1
  ) len 0;
  ()
;;

let columns dnamel =

  let rec start_columns gd = function
    dname::dnamel ->
      start_columns gd dnamel;
      (try
        one_column(dname, succ(length gd.gwl), gd)
      with _ ->
        display_string "*** Can't open display \"";
        display_string dname; display_string "\""; display_newline()
      )
  | _ ->
    ()

  and exec_columns gd =
    if gd.gwl <> [] then try
      rt_main_loop gd.xargs;
      do_list (fun (gm, _) ->
        if !trace_games then display_closing_display gm.dname;
        rt_end gm.xd
      ) gd.gwl
    with x ->
      display_newline();
      display_string "*** error in execution"; display_newline();
      do_list (fun (gm, _) ->
        display_closing_display gm.dname;
        try rt_end gm.xd with _ -> ()
      ) gd.gwl;
      gc_alarm false; raise x

  in
  init_random (((ftime()).time) mod 32768);
  trace_games := true;
  trace_gc := true; gc_alarm true;
  let gd = {
    xargs = rt_args[]; init_time = (ftime()).time; gwl = []
  } in
  start_columns gd dnamel;
  rt_select_file(gd.xargs, 0, stdin_fun gd);
  rt_set_timeout_fun(gd.xargs, woops_fun gd);
  exec_columns gd;
  gc_alarm false;
  ()
;;

let column dname = columns [dname]
;;

let dname = ref ([] : string list)
;;

let i = ref 1 in
while !i < vect_length command_line do
  (match command_line.(!i) with
    "-bw" -> BW := true
  | "-d" -> incr i
  | s -> dname := s::!dname);
  incr i
done;;

if !dname = [] then (dname := [""]; ())
;;

try columns !dname with
  Failure s -> print_string "failure: "; print_string s; print_newline()
| Invalid_argument s ->
    print_string "Invalid_argument: "; print_string s; print_newline()
| x -> print_string "Unknown exception in column"; print_newline(); raise x
;;
