(* $Id: chol.ml,v 1.6 91/07/16 11:48:14 ddr Exp $ *)

#open "rt";;
#open "sys";;

pack_border := 0;;
pack_band := 1;;
pack_inter := 0;;
button_border := 1;;

let max x y = if x > y then x else y
and min x y = if x < y then x else y
and nround x y = (2*x+y)/(2*y)
and string_of_num = string_of_int
and last_n_string n s =
  let len = string_length s in sub_string s (len-n) n
and K x y = x
and gc_alarm _ = ();;

type glop = {
  wid_named : string -> widget;
  col       : color;
  red       : color vect;
  green     : color vect;
  blue      : color vect;
  hue       : color vect;
  sat       : color vect;
  bri       : color vect
};;

let MX = 255;;
let VL = 30;;

let hsb_of_rgb(R, G, B) =
  let V = max R (max G B)
  and X = min R (min G B) in
  (
    (if V = 0 or V = X then 0 else
      let r = (V-R)*MX/(V-X)
      and g = (V-G)*MX/(V-X)
      and b = (V-B)*MX/(V-X) in
      let h =
        if R = V then if G = X then 5*MX+b else MX-g
        else if G = V then if B = X then MX+r else 3*MX-b
        else if R = X then 3*MX+g else 5*MX-r in
      h / 6)
  ,
    (if V = 0 then 0 else nround (MX*(V-X)) V)
  ,
    V
  )
;;

let rgb_of_hsb(H, S, V) =
  let H = H*6 in
  let I = H/MX*MX in
  let F = H-I in
  let M = V*(MX-S)/MX and N = V*(MX-S*F/MX)/MX
  and K = V*(MX-S*(MX-F)/MX)/MX in
  (
    nround (MX*(
      match I/MX with
        0 | 6 -> V | 1 -> N | 2 -> M | 3 -> M | 4 -> K | 5 -> V
      | _ -> failwith "red_of_hsb"
    )) MX,
    nround (MX*(
      match I/MX with
        0 | 6 -> K | 1 -> V | 2 -> V | 3 -> N | 4 -> M | 5 -> M
      | _ -> failwith "green_of_hsb"
    )) MX,
    nround (MX*(
      match I/MX with
        0 | 6 -> M | 1 -> M | 2 -> K | 3 -> V | 4 -> V | 5 -> N
      | _ -> failwith "blue_of_hsb"
    )) MX
  )
;;

let red_val = ref 0 and green_val = ref 0 and blue_val = ref 0;;
let (hue, sat, bri) = hsb_of_rgb(!red_val, !green_val, !blue_val);;
let hue_val = ref hue and sat_val = ref sat and bri_val = ref bri;;

let ctxt n =
  let t = string_of_num n in
  (last_n_string 5 ("     " ^ t)) ^ "\b\b\b\b\b"
;;

let update_colors g =
  let f i = (2*i+1)*MX/(2*VL) in
  rt_change_color(g.col, !red_val, !green_val, !blue_val);
  for i = 0 to VL-1 do
    rt_change_color(g.red.(i), f i, !green_val, !blue_val)
  done;
  for i = 0 to VL-1 do
    rt_change_color(g.green.(i), !red_val, f i, !blue_val)
  done;
  for i = 0 to VL-1 do
    rt_change_color(g.blue.(i), !red_val, !green_val, f i)
  done;
  for i = 0 to VL-1 do
    let (R, G, B) = rgb_of_hsb(f i, !sat_val, !bri_val) in
    rt_change_color(g.hue.(i), R, G, B)
  done;
  for i = 0 to VL-1 do
    let (R, G, B) = rgb_of_hsb(!hue_val, f i, !bri_val) in
    rt_change_color(g.sat.(i), R, G, B)
  done;
  for i = 0 to VL-1 do
    let (R, G, B) = rgb_of_hsb(!hue_val, !sat_val, f i) in
    rt_change_color(g.bri.(i), R, G, B)
  done
;;

let my_scroll_set vmin vmax shift act (wid, but, val) =
  let oval = (scroll_val wid)-shift in
  let val =
    if but = 1 then oval-1
    else if but = 3 then oval+1
    else val in
  let val = max vmin (min vmax val) in
  if val <> oval then (
    scroll_set(wid, val+shift);
    act val
  )
;;

let RGB_scr g wname rgb_val =
  my_scroll_set 0 MX 0 (fun sval ->
    rgb_val := sval;
    let (hue, sat, bri) = hsb_of_rgb(!red_val, !green_val, !blue_val) in
    hue_val := hue; sat_val := sat; bri_val := bri;
    text_send_string(g.wid_named wname, ctxt !rgb_val);
    scroll_set(g.wid_named "hue_scr", !hue_val);
    scroll_set(g.wid_named "sat_scr", !sat_val);
    scroll_set(g.wid_named "bri_scr", !bri_val);
    text_send_string(g.wid_named "hue", ctxt !hue_val);
    text_send_string(g.wid_named "sat", ctxt !sat_val);
    text_send_string(g.wid_named "bri", ctxt !bri_val);
    update_colors g
  )

and HSB_scr g wname hsb_val =
  my_scroll_set 0 MX 0 (fun sval ->
    hsb_val := sval;
    let (red, green, blue) = rgb_of_hsb(!hue_val, !sat_val, !bri_val) in
    red_val := red; green_val := green; blue_val := blue;
    text_send_string(g.wid_named wname, ctxt !hsb_val);
    scroll_set(g.wid_named "red_scr", !red_val);
    scroll_set(g.wid_named "green_scr", !green_val);
    scroll_set(g.wid_named "blue_scr", !blue_val);
    text_send_string(g.wid_named "red", ctxt !red_val);
    text_send_string(g.wid_named "green", ctxt !green_val);
    text_send_string(g.wid_named "blue", ctxt !blue_val);
    update_colors g
  )
;;

let glop f m = (glop_rec 0
  where rec glop_rec i =
  if i >= m then []
  else f i::glop_rec(succ i))
;;

let pack_scroll_text name val gen_callb =
  PackD(Vertical, [
    ScrollA [NameAtt(name^"_scr")] (Horizontal, 0, MX, 0, gen_callb name val);
    TextA [NameAtt name; BorderAtt 0] (1, 12, 1, (fun _ -> ()), (fun _ -> ()))
  ])

and pack_raw col =
  PackA [FillerAtt] (Horizontal,
    glop (fun i ->
      RawA [BackgroundAtt(ColorBg col.(i))] (100/VL, 30, 0, [])
    ) VL
  )
;;

let chol dname =
  let xd = rt_initialize dname in (try
    gc_alarm true;
    if not (is_colored xd) then failwith "works only on color displays";
    let xargs = rt_args[xd] in
    let wid_named = widget_named xd in
    let g = {
      wid_named = wid_named;
      col = rt_create_color(xd, !red_val, !green_val, !blue_val);
      red = make_vect VL (rt_black_color xd);
      green = make_vect VL (rt_black_color xd);
      blue = make_vect VL (rt_black_color xd);
      hue = make_vect VL (rt_black_color xd);
      sat = make_vect VL (rt_black_color xd);
      bri = make_vect VL (rt_black_color xd)
    } in
    for i = 0 to VL-1 do
      g.red.(i) <- rt_create_color(xd, 0, 0, 0);
      g.green.(i) <- rt_create_color(xd, 0, 0, 0);
      g.blue.(i) <- rt_create_color(xd, 0, 0, 0);
      g.hue.(i) <- rt_create_color(xd, 0, 0, 0);
      g.sat.(i) <- rt_create_color(xd, 0, 0, 0);
      g.bri.(i) <- rt_create_color(xd, 0, 0, 0)
    done;
    update_colors g;
    let widr = rt_create_widget(xd, "pattern", "col pattern",
      RawA [NameAtt"raw"; BackgroundAtt(ColorBg g.col)] (100, 100, 1, [])
    ) in rt_map_widget widr;
    let wid = rt_create_widget(xd, "color select", "col select",
      PackD(Vertical, [
        PackA [FillerAtt] (Horizontal, [
          PackD(Vertical, [
            pack_raw g.red;
            pack_scroll_text "red" red_val (RGB_scr g);
            pack_raw g.hue;
            pack_scroll_text "hue" hue_val (HSB_scr g)
          ]);
          PackD(Vertical, [
            pack_raw g.green;
            pack_scroll_text "green" green_val (RGB_scr g);
            pack_raw g.sat;
            pack_scroll_text "sat" sat_val (HSB_scr g)
          ]);
          PackD(Vertical, [
            pack_raw g.blue;
            pack_scroll_text "blue" blue_val (RGB_scr g);
            pack_raw g.bri;
            pack_scroll_text "bri" bri_val (HSB_scr g)
          ])
        ]);
        ButtonD("quit", fun _ -> rt_stop_main_loop xargs)
      ])
    ) in
    scroll_set(wid_named "red_scr", !red_val);
    scroll_set(wid_named "green_scr", !green_val);
    scroll_set(wid_named "blue_scr", !blue_val);
    scroll_set(wid_named "hue_scr", !hue_val);
    scroll_set(wid_named "sat_scr", !sat_val);
    scroll_set(wid_named "bri_scr", !bri_val);
    text_send_string(wid_named "red", "red    " ^ (ctxt !red_val));
    text_send_string(wid_named "green", "green  " ^ (ctxt !green_val));
    text_send_string(wid_named "blue", "blue   " ^ (ctxt !blue_val));
    text_send_string(wid_named "hue", "hue    " ^ (ctxt !hue_val));
    text_send_string(wid_named "sat", "sat    " ^ (ctxt !sat_val));
    text_send_string(wid_named "bri", "bri    " ^ (ctxt !bri_val));
    rt_map_widget wid;
    rt_main_loop xargs
  with x ->
    gc_alarm false; rt_end xd; raise x);
  gc_alarm false; rt_end xd
;;

let dname = ref "";;

let i = ref 1 in
while !i < vect_length command_line do
  (match command_line.(!i) with
    "-d" -> incr i
  | s -> dname := s);
  incr i
done;;

try chol !dname with
  Failure s -> print_string "failure: "; print_string s; print_newline()
| Invalid_argument s ->
    print_string "Invalid_argument: "; print_string s; print_newline()
| x -> print_string "Unknown exception in chol"; print_newline(); raise x
;;

