(* trick.ml,v 1.1 1993/02/12 16:20:18 murthy Exp *)

#open "rt";;
#open "card_pix";;
#open "sys";;
(* #open "time";; *)

type num == int;;
let interval n m = (interval_n ([],m)
  where rec interval_n (l,m) =
    if n > m then l else interval_n (m::l,pred m))
and modify_vect_i f v =
  let len = vect_length v in (modify_rec 0
    where rec modify_rec i =
      if i < len then (v.(i) <- f i v.(i); modify_rec (succ i)))
and do_list_i f = (do_list_f
  where rec do_list_f i = function
    x::l -> f i x; do_list_f (succ i) l | _ -> ())
and iterate f = (iterate_f
  where rec iterate_f n x =
    if n > 0 then iterate_f (pred n) (f x))
and hd = function x::l -> x | _ -> failwith "hd"
and tl = function x::l -> l | _ -> failwith "tl"
;;
#infix "o";;
let seed = ref 7;;
let prefix o f g x = f(g x)
and random n =
  seed := 25173*!seed + 13849;
  (if !seed < 0 then - !seed else !seed) mod n
and init_random n = seed := n; ()
;;
let implode_ascii l =
  let len = list_length l in
  let s = create_string len in
  iterate (fun (i,hd::tl) ->
    set_nth_char s i (char_of_int hd);
    (i+1, tl) | _ -> failwith "implode_ascii: system error"
  ) len (0, l);
  s
;;

(* The cards trick interface *)

type suit = Heart | Spade | Club | Diamond;;
type card = Card of suit * num;;
type deck == card list;;

let num_of_suit = function
  Heart -> 1 | Spade -> 0 | Club -> 3 | Diamond ->2;;

let Hearts = map (fun n -> Card(Heart,n)) (interval 1 13)
and Spades = map (fun n -> Card(Spade,n)) (interval 1 13);;

let rand = random;;

let cut l =
   let m = rand(list_length(l)+1) in cut_rec m [] l
   where rec cut_rec k l1 l2 =
        if k=0 then (rev l1,l2)
        else cut_rec (k-1) (hd l2::l1) (tl l2);;

let rotate = function
        [] -> []
    | c::l -> l@[c];;

(* pick n l returns the n+1st card from l, with the rest of l *)
let pick n l = pick_rec n l [] 
  where rec pick_rec n (c::l) l' =
        if n=0 then (c,(rev l')@l)
        else pick_rec (n-1) l (c::l');;

let randomize l = random_rec (list_length l,l,[]) 
  where rec random_rec (n,l,res) = 
        if n=0 then res
        else let k=rand(n) in
             let (c,rest) = pick k l in
             random_rec(n-1,rest,c::res);;

let rec alternate = function
   [],[] -> []
 | (c1::l1,c2::l2) -> c1::c2::alternate(l1,l2)
 | _ -> failwith "alternate";;

let shuffle l1 l2 = shuffle_rec l1 l2 (list_length l1) (list_length l2)
  where rec shuffle_rec l1 l2 len1 len2 =
    if len1=0 then l2
    else if len2=0 then l1
    else if rand(len1+len2) < len1 then
      hd l1::shuffle_rec (tl l1) l2 (len1-1) len2
    else
      hd l2::shuffle_rec l1 (tl l2) len1 (len2-1);;

(* shuffle Hearts Spades;; *)

let R() = randomize (Hearts @ Spades)
and A() = alternate(randomize Hearts,randomize Spades);;

(* display *)

let WID = 47 and HEI = 67
;;

type trick_info = {
  win_backg         : background;
  comm_backg        : background;
  quit_backg        : background;
  mutable deck      : deck;
  mutable deck1     : deck;
  mutable deck2     : deck
};;

let Border = 10 and Card_shift = WID*6 / 10
and Inter_line = 5 and Half_shift = 15;;

let string_of_val = vect_item [|
  "A"; "2"; "3"; "4"; "5"; "6"; "7"; "8"; "9"; "0"; "V"; "D"; "R"
|]

and string_of_suit = vect_item [| "P"; "C"; "K"; "T" |]
;;

let string_of_card =
  let card_string = make_vect 52 "" in
  modify_vect_i (fun card _ ->
    (string_of_val (card mod 13)) ^ (string_of_suit (card / 13))
  ) card_string;
function card -> card_string.(card)
;;

let create_card xd pwid n val suit attr x y =
  let card = 13*suit+val in
  let wid = rt_create_subwidget(pwid, x, y,
    RawA (NameAtt(string_of_card card ^ n)::attr) (WID, HEI, 0, [])
  ) in
  rt_map_widget wid;
  wid
;;

let move_widgets xd n card_list y =
  do_list_i (function i -> function (Card(suit, val)) ->
    let card = (num_of_suit suit)*13+val-1 in
    let wid = widget_named xd (string_of_card card ^ n) in
    let x = Border + Card_shift * i in
    rt_move_widget(wid, x, y)
  ) 0 card_list
;;

let forward name = fun _ -> failwith ("forward " ^ name);;

let shuffle_fun xd ti _ =
  let wid = widget_named xd "widget 1" in
  select_raise(widget_named xd "pair/right", 0);
  ti.deck <- shuffle ti.deck1 ti.deck2;
  move_widgets xd "1" ti.deck (Border+Half_shift);
  rt_map_widget wid

and rotate_fun xd ti _ =
  ti.deck <- rotate ti.deck;
  move_widgets xd "1" ti.deck (Border+Half_shift)

and cut_fun = ref (
  forward "cut_fun" : xdata -> pixmap vect vect -> trick_info -> widget -> unit
)

and pair_fun xd ti _ =
  select_raise(widget_named xd "pair/right", 1);
  do_list_i (function i -> function (Card(suit, val)) ->
    let card = (num_of_suit suit)*13+val-1 in
    let wid = widget_named xd (string_of_card card ^ "1") in
    let x = Border + Card_shift * i
    and y = Border + (if i mod 4 <= 1 then 0 else 2*Half_shift) in
    rt_move_widget(wid, x, y)
  ) 0 ti.deck

and right_fun xd ti _ =
  select_raise(widget_named xd "pair/right", 0);
  move_widgets xd "1" ti.deck (Border+Half_shift)

and flush_1_fun xd _ =
  rt_unmap_widget (widget_named xd "widget 1")

and flush_2_fun xd _ =
  rt_unmap_widget (widget_named xd "widget 2")
;;

let widget_1 =
  let width = 2*Border+25*Card_shift+WID
  and height = 2*Border+HEI+2*Half_shift in
fun xd card_pixmap ti ->
  try widget_named xd "widget 1"
  with _ ->
    let attr = [BackgroundAtt ti.win_backg]
    and attr1 = [BackgroundAtt ti.comm_backg]
    and attr2 = [BackgroundAtt ti.quit_backg] in
    let wid = rt_create_widget(xd, "1", "1",
      PackA (NameAtt "widget 1"::attr) (Horizontal, [
        RawA (NameAtt "raw 1"::FillerAtt::attr) (width, height, 0, []);
        SelectA [NameAtt "pair/right"] [
          PackA (BorderAtt 0::attr) (Horizontal, [
            ButtonA attr1 ("Rotate", rotate_fun xd ti);
            ButtonA attr1 ("Cut", !cut_fun xd card_pixmap ti);
            ButtonA attr1 ("Pair", pair_fun xd ti)
          ]);
          ButtonA attr1 ("Right", right_fun xd ti)
        ];
        ButtonA attr2 ("Flush", flush_1_fun xd)
      ])
    ) in
    let raw1 = widget_named xd "raw 1" in
    iterate (fun suit ->
      iterate (fun val ->
        let attr = [BackgroundAtt(PixmapBg card_pixmap.(suit).(val))]
        and x = Border+Card_shift*(suit*13+val) in
        let _ = create_card xd raw1 "1" val suit attr x (Border+Half_shift) in
        val+1
      ) 13 0;
      suit+1
    ) 2 0;
    wid

and widget_2 =
  let width = 2*Border+25*Card_shift+WID
  and height = 2*Border+2*HEI+Inter_line in
fun xd card_pixmap ti ->
  try widget_named xd "widget 2"
  with _ ->
    let attr = [BackgroundAtt ti.win_backg]
    and attr1 = [BackgroundAtt ti.comm_backg]
    and attr2 = [BackgroundAtt ti.quit_backg] in
    let wid = rt_create_widget(xd, "2", "2",
      PackA (NameAtt "widget 2"::attr) (Horizontal, [
        RawA (NameAtt "raw 2"::FillerAtt::attr) (width, height, 0, []);
        ButtonA attr1 ("Shuffle", shuffle_fun xd ti);
        ButtonA attr2 ("Flush", flush_2_fun xd)
      ])
    ) in
    let raw2 = widget_named xd "raw 2" in
    iterate (fun suit ->
      iterate (fun val ->
        let attr = [BackgroundAtt(PixmapBg card_pixmap.(suit).(val))]
        and x = Border+Card_shift*val
        and y = Border+(HEI+Inter_line)*suit in
        let _ = create_card xd raw2 "2" val suit attr x y in
        val+1
      ) 13 0;
      suit+1
    ) 2 0;
    wid
;;

cut_fun := fun xd card_pixmap ti _ ->
  let wid = widget_2 xd card_pixmap ti in
  let deck1, deck2 = cut ti.deck in
  move_widgets xd "2" deck1 Border;
  move_widgets xd "2" deck2 (Border+HEI+Inter_line);
  ti.deck1 <- deck1; ti.deck2 <- deck2;
  rt_map_widget wid
;;

let sorted_fun xd card_pixmap ti _ =
  let wid = widget_1 xd card_pixmap ti in
  select_raise(widget_named xd "pair/right", 0);
  ti.deck <- rev(Hearts @ Spades);
  move_widgets xd "1" ti.deck (Border+Half_shift);
  rt_map_widget wid
and random_fun xd card_pixmap ti _ =
  let wid = widget_1 xd card_pixmap ti in
  select_raise(widget_named xd "pair/right", 0);
  ti.deck <- R();
  move_widgets xd "1" ti.deck (Border+Half_shift);
  rt_map_widget wid
and alternate_fun xd card_pixmap ti _ =
  let wid = widget_1 xd card_pixmap ti in
  select_raise(widget_named xd "pair/right", 0);
  ti.deck <- A();
  move_widgets xd "1" ti.deck (Border+Half_shift);
  rt_map_widget wid
;;

let dir = ref "../cards/pixmap"
and dname = ref ""
and colored = ref true
;;

let trick _ =
  let i = ref 1 in
  while !i < vect_length command_line do
    (match command_line.(!i) with
      "-c" -> incr i; dir := command_line.(!i); ()
    | "-d" -> incr i; dir := command_line.(!i) ^ "/../cards/pixmap"
    | "-bw" -> colored := false; ()
    | s -> dname := s; ());
    incr i
  done;
  let tabf = !dir ^ "/colors-6x256"
  and pixf = !dir ^ "/cards" in
  init_random  0; (* ((ftime()).time); *)
  rt_run !dname (fun xd ->
    let xa = rt_args[xd] in
(*
    let card_pixmap, _, _ = (if colored then make_pixmap else make_bitmap) xd
*)
    let card_pixmap, _, _ = make_pixmap xd tabf pixf
    and ti = {
      win_backg =
        if !colored then ColorBg(rt_create_color(xd, 34, 139, 34))
        else (
          let image = rt_create_image(xd,
            implode_ascii[1;8;64;4;128;16;2;32], 8, 8, 1)
          and pixm = rt_create_pixmap(xd, 8, 8) in
          rt_put_image(PixmapDr pixm, image, 0, 0, 0, 0, 8, 8);
          PixmapBg pixm
        );
      comm_backg = ColorBg(
        if !colored then rt_create_color(xd, 135, 206, 255)
        else rt_white_color xd);
      quit_backg = ColorBg(
        if !colored then rt_create_color(xd, 255, 80, 80)
        else rt_white_color xd);
      deck = []; deck1 = []; deck2 = []
    } in
    let attr = [BackgroundAtt ti.win_backg]
    and attr1 = [BackgroundAtt ti.comm_backg]
    and attr2 = [BackgroundAtt ti.quit_backg] in
    let init_wid = rt_create_located_widget(xd, "Deck", "Deck",
      UserPosition(0, 0),
      PackA attr (Vertical, [
        ButtonA attr1 ("Sorted", sorted_fun xd card_pixmap ti);
        ButtonA attr1 ("Random", random_fun xd card_pixmap ti);
        ButtonA attr1 ("Random alternate", alternate_fun xd card_pixmap ti);
        ButtonA attr2 ("Exit", fun _ -> rt_stop_main_loop xa)
      ])
    ) in
    rt_map_widget init_wid;
    rt_main_loop xa
  )
;;

printexc__f trick ();;
