%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% (C) 1989 - 1993                                         %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- module(tk_tree,[tk_tree/2]).


:- use_module( [ library(flags),
                 library(tk_draw) ]).

:- use_module( library(concat), [ term_atom/2,
	                          term_atom/3,
	                          concat_all/2 ] ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                   %
%  pretty printer of graphic trees  %
%      x version based on           %
%      library(p_tree)              %
%                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                       %
%  the following predicates must be defined elsewhere:  %
% graphic_path(Name,FS,FS2)                             %
% graphic_daughter(Name,Pos,FS,Daught)                  %
% graphic_label(Name,FS,Label)                          %
%                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% cf. documentation with library(p_tree)
% this should also be computed on the basis of the font
node_height(7).         
node_width_times(10).
vert_dist(50).
hor_dist(10).

start(100,60).

tk_tree(Name,FS):-
	user:graphic_path(Name,FS,FS2),
	tree(Name,FS2,Tree),  % builds datastructure
	start(X,Y),
	compute_tk_tree(Tree,X,Y,_,_),
	tk_clear,
	pp_tree(Tree).

node_width(Label0,Label,Len) :-
	term_atom(Label0,Label,Len1),
	node_width_times(T),
	Len is Len1*T.

% prints the tree, with x,y coordinates of each node given.
pp_tree(tree(Label,Ds,X,Y)) :-
	tk_string(X,Y,Label),
	node_height(Ten),
	pp_tree_ds(Ds,X,Y+Ten).

pp_tree(tree(Label,Ds,X,Y),MX,MY) :-
	tk_string(X,Y,Label),
	node_height(Ten),
	tk_line(MX,MY,X,Y-2*Ten),
	pp_tree_ds(Ds,X,Y+Ten).

pp_tree_ds([],_,_).
pp_tree_ds([H|T],X,Y):-
	pp_tree(H,X,Y),
	pp_tree_ds(T,X,Y).

% tk_tree computes x,y coordinates of each node in tree.
% Note: if terminal node p precedes terminal node q, then
% this is always reflected in the picture, so no efficient
% use of paper.. For linguistic applications this can be
% defended... (but real reason is that it is too difficult
% otherwise).
%
% this is messy only because of the problem that
% a nonterminal node may be wider than the width of its
% daughters (usually only for nonbranching nodes).
compute_tk_tree(tree(Label,Ds,X,Y0),X0,Y0,X,Width):-
	hor_dist(Ten),
	vert_dist(Fifty),
	copy_term(Ds,Ds1),
	node_width(Label,_,L),
	tk_ds(Ds1,X0,Y0+Fifty,Xfirst,Xlast,XW),
	(  XW < L
        -> Extra is (L+Ten-XW)/2,
	   tk_ds(Ds,X0+Extra,Y0+Fifty,Xf,Xl,_),
           X is (Xl-Xf)/2 + Xf,
	   Width = L+Ten
	; 
	Width=XW,
	Ds1 = Ds,
	X is (Xlast-Xfirst)/2 + Xfirst
        ).


tk_ds([],X,_,X,X,0).
tk_ds([Tree|T],X,Y,Xf,Xl,W0+W):-
	compute_tk_tree(Tree,X,Y,Xf,W0),
	tk_ds2(T,W0+X,Y,Xf,Xl,W).

tk_ds2([],_,_,Xl,Xl,0).
tk_ds2([Tree|T],X0,Y,_,Xl,W0+W):-
	compute_tk_tree(Tree,X0,Y,X1,W0),
	tk_ds2(T,X0+W0,Y,X1,Xl,W).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                    %
%  tree looks like tree(Label,Daughters,WidthLabel,WidthDaughters)   %
%    where Label is atom                                             %
%          Daughters is list of Daughters (or [])                    %
%          Width is the width of the tree                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


find_graphic_label(Name,FS,Label):-
	(  nonvar(FS),
           user:graphic_label(Name,FS,Label)
        -> true
        ;  Label = ''
        ).

tree(Name,FS,tree(Label,Daughters,_,_)):-
	find_graphic_label(Name,FS,Label),
	find_graphic_daughters(Name,FS,Daughters).

find_graphic_daughters(Name,FS,Daughters):-
	graphic_daughters(Name,1,FS,Daughters2),
	tree_list(Name,Daughters2,Daughters).

tree_list(_Name,[],[]).
tree_list(Name,[H|T],[H2|T2]):-
	tree(Name,H,H2),
	tree_list(Name,T,T2).


graphic_daughters(Name,Pos,FS,[]):-
        nothigherone(Name,Pos,FS),
        !.


graphic_daughters(Name,Pos,FS,[Daughter|Daughters]):-
        user:graphic_daughter(Name,Pos,FS,Daughter),
        !,
        Pos2 is Pos + 1,
        graphic_daughters(Name,Pos2,FS,Daughters).


graphic_daughters(Name,Pos,FS,Ds):-
        Pos2 is Pos + 1,
        graphic_daughters(Name,Pos2,FS,Ds).


nothigherone(_Name,_Pos,Var):-
	var(Var),!.
nothigherone(Name,Pos,FS):-
        user:graphic_daughter(Name,Pos2,FS,_),
        Pos2 >= Pos,
        !,
        fail.


nothigherone(_,_,_).










