:- module(show,[ sh/3 ]).        % available thru commands / X interface


%% This module defines interface between several pretty-printing 
%% libraries. However, since not all of these libraries might be
%% needed, these are not use_module'd here..

%% I don't yet see how to do this both modular and conveniant

%% cf. library(latex)
%%     library(feature)
%%     library(tk_tree)

:- use_module( library(lists),   [ nth/3, 
                                   member/2, 
                                   append/3             ]).
:- use_module( library(concat),  [ concat_all/2         ]).
:- use_module( library(between), [ between/3,
	                           between/4            ]).
:- use_module( library(decons),  [ prolog_conjunction/2 ]).

:- use_module( library(flags)).

% find_user_clause(F/Ar,Hd,Clause)
% for pp of clauses
% 
find_user_clause( F/Ar ,H,Cl) :-
	functor(H,F,Ar),
	!,
	user:user_clause(H,B0),
	(  B0 = [] 
        -> Cl = H 
        ;  prolog_conjunction(B,B0),
	   Cl = (H:-B)
        ).

find_user_clause(F,H,Cl) :-
	user:user_clause(H,B),
	functor(H,F,_),
	(  B0 = [] 
        -> Cl = H 
        ;  prolog_conjunction(B,B0),
	   Cl = (H:-B)
        ).
	
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% nice output for relations etc. %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fc(Exp) :-
	find_user_clause(Exp,_,Clause),
	feature:tty_fs_clause(Clause).

fl(Exp):-
	findall(Cl,find_user_clause(Exp,_,Cl),Clauses),
	feature:tty_fs_listing(Clauses).

cons_lfc(Exp,Cons) :-
	find_user_clause(Exp,Cons,Clause),
	latex:latex_fs_clause(Clause).

lfc(Exp) :-
	find_user_clause(Exp,_,Clause),
	latex:latex_fs_clause(Clause).

lfl(Exp):-
	findall(Cl,find_user_clause(Exp,_,Cl),Clauses),
	latex:latex_fs_listing(Clauses).

ltc(Exp):-
	find_user_clause(Exp,_,Clause),
	latex:latex_term_clause(Clause).

ltl(Exp) :-
	findall(Cl,find_user_clause(Exp,_,Cl),Clauses),
	latex:latex_term_listing(Clauses).

exp(Exp) -->
	fa_spec(user,Exp).

parts(Atom,F/A) :-
	concat_all([F,'/',A],Atom),!.
parts(A,A).

% eval_paths(Obj,Paths,NewObjWithPaths)
% makes version of Obj in which only Paths are instantiated
% if there is more than one path, then we keep all the information
% leading to the specified parts. If there is only one path then
% we `extract' the information...
%
% this all only makes sense in context of library(feature)
eval_paths([],V,V).
eval_paths([H],V0,V) :-
	eval_single(H,V0,V).
eval_paths([H,I|T],V0,V) :-
	eval_paths0([H,I|T],V0,V).

eval_paths0([],_,_).
eval_paths0([H|T],Obj,Val):-
	eval_p(Obj,H,Val),
	eval_paths0(T,Obj,Val).

eval_paths_list([],[],_Paths).
eval_paths_list([H0|T0],[H|T],Paths) :-
	eval_paths_o(H0,H,Paths),
	eval_paths_list(T0,T,Paths).

eval_paths_o(object(_Name,o(FS0,_S,_W)),fs(FS),Paths) :-
	eval_paths(Paths,FS0,FS).

eval_paths_o(value(_Name,FS0),clause(FS),Paths) :-
	eval_paths(Paths,FS0,FS).

eval_paths_o(clause(_Name,H,[]),clause(H),[]).

eval_paths_o(clause(_Name,H,[B0|B]),clause(H,[B0|B]),[]).

eval_p(Obj,path(F),Val):-
	eval_f(F,Obj,Val).

eval_single(path(F),V0,V) :-
	eval_single(F,V0,V).


eval_single([],Obj,Obj).
eval_single([0|T],tree(Obj,_,_),Val) :-
	!,
	eval_single(T,Obj,Val).
eval_single([I|T],Obj,D) :-
	integer(I),
	!,
	Obj = tree(_,_,Ds0),
	nth(I,D0,Ds0),
	eval_single(T,D0,D).
eval_single([F|T],Obj,Val):-
	feature:'<=>'(Obj:F,Obj1),
	!,
	eval_single(T,Obj1,Val).

eval_single([F|T],tree(Obj,_,_),Val) :-
	eval_single([F|T],Obj,Val).

%% 
eval_f([],Obj,Obj).
eval_f([0|T],tree(Obj,_,_),tree(Val,_,_)) :-
	!,
	eval_f(T,Obj,Val).
eval_f([I|T],Obj,Val) :-
	integer(I),
	!,
	Obj = tree(_,_,Ds0),
	Val = tree(_,_,Ds),
	length(Ds0,L),
	length(Ds,L),
	nth(I,D0,Ds0),
	nth(I,D,Ds),
	eval_f(T,D0,D).
eval_f([F|T],Obj,Val):-
	feature:'<=>'(Obj:F,Obj1),
	feature:'<=>'(Val:F,Val1),
	!,
	eval_f(T,Obj1,Val1).

eval_f([F|T],tree(Obj,_,_),tree(Val,_,_)) :-
	eval_f([F|T],Obj,Val).

%%%%%%%%%%%%%%%%%
%               %
% FEATURE-path  %
%               %
%%%%%%%%%%%%%%%%%

paths([Path|Paths])-->
        path(Path),
        paths(Paths).

paths([]) -->
        [].

path(Path) --> [P],
        { is_path(P,Path) }.

is_path(Atom,Path):-
	parse_specials(Atom,List),
        path2(Path,List,[]).

parse_specials(Atom,AtomList) :-
	name(Atom,CharList),
	parse_specials2(CharList,AtomList).

parse_specials2([],[]).
parse_specials2(CharList,[Atom,SpecialAtom|Tail]):-
	append(Prefix,[Special|Rest],CharList),
	special(Special,SpecialAtom),!,
	name(Atom,Prefix),
	parse_specials2(Rest,Tail).
parse_specials2([H|T],[Atom]):-
	name(Atom,[H|T]),
	is_attribute(Atom).

is_attribute(A) :-
	feature:e(A,_,_).
is_attribute(A) :-
	integer(A).

special(46,'.').
special(47,'/').

% een pad bestaat uit een eerste deel dat dochter aanwijst
% en een tweede deel dat ingebedde featurestructuur aanwijst        

path2(path(Feature)) -->
	feature_path(Feature).

feature_path([Att|Tail]) -->
	[Att],
	{is_attribute(Att)},
	feature_path2(Tail).

feature_path2([]) -->
	[].
feature_path2([Att|Tail]) -->
	['.'],
	[Att],
	{is_attribute(Att)},
	feature_path2(Tail).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%% SHOW %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the things to show are
%
% - term
% - clause
% - listing 
%
% the options are
%
% - terms are feature structures
% - terms are trees (different possibilities!)
% - terms are terms (internal)
%
% output goes to
%
% - standard output (ascii)
% - latex & xdvi
% - tk

%% only makes sense if library(command) is loaded
:- user:del_command(show:new_show_command).

new_show_command(show:sh(Type,Output,Thing)) -->
	[s],
	type(Type),
	output(Output),
	item_spec(Thing).

new_show_command(show:sh(Type,Output,Thing)) -->
	[s],
	output(Output),
	type(Type),
	item_spec(Thing).

new_show_command(show:show_help) -->
	[s],
	user:sequence_of_terms(_,_).
	
:- user:add_command(show:new_show_command).

show_help :-
	write('Usage: '),
	write('s [i, j, s, w, f <path>,<tree type>] [x, tk] [l] objects'),nl. 

type(words) -->
	[w].

type(sem) -->
	[s].

type(fs(Paths)) -->
	[f],
	paths(Paths).

type(term(write)) -->
	[i].

type(tree(Kind)) -->
	[Kind],
        { user:graphic_path(Kind,_,_) }.

type(term(print)) -->
	[j].

type(term(print)) -->  % default
	[].

output(tk) -->
	[tk].

output(latex) -->
	[x].

output(user) -->
	[user].

output(user) -->
	[].

%%
%% things: list of:
%% object(Name,o(Fs,Sem,Str))
%% clause(Name,Head,Body)
%% value(Name,Fs)

% for some things you want to be prompted, for others you dont

sh(words,     user, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_words(Thing)).

sh(sem,       user, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_sems(Thing)).

sh(sem,       tk, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_sems(tk,Thing)).

sh(sem,       latex, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_sems(latex,Thing)).

sh(fs(Path),  user, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_fs(Thing,Path)).

sh(term(I),      user, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_term(I,Thing)).

sh(tree(Mode),user, Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_tree(Thing,Mode)).

%% sh(words,     latex,Things)    % could be provided
%% sh(sem,       latex,Things)    % could be provided
sh(fs(Path),latex,Things0) :-
	eval_paths_list(Things0,Things,Path),
	latex:latex_fs(Things).

sh(term(_),      latex,Things0) :-
	eval_paths_list(Things0,Things,[]),
	latex:latex_term(Things).

sh(tree(Mode),latex,Things0):-
	eval_paths_list(Things0,Things,[]),
	prompting(lists:member(Thing,Things),next,show:latex_tree(Mode,Thing)).

sh(tree(Mode),tk,Things) :-
	prompting((lists:member(Thing,Things),arg(1,Thing,Name)),Name,show:sh_tk(Thing,Mode)).

%%% specific calls to different pp routines (in different files/modules)

latex_tree(Mode,fs(Thing)) :-
	latex:latex_tree(Mode,Thing).

sh_words(object(_,o(_,Words,_))) :-
	user:short_phon(Words).

sh_words(value(_,FS)) :-
	user:phonology(FS,Words),
	user:short_phon(Words).

sh_sems(latex,object(_,o(_,_,Sem))) :-
	user:extern_sem(Ext,Sem),
	latex:latex_term(term(Ext)).

sh_sems(latex,value(_,FS)) :-
	user:semantics(FS,Sem),
	user:extern_sem(Ext,Sem),
	latex:latex_term(term(Ext)).

sh_sems(object(_,o(_,_,Sem))) :-
	user:extern_sem(ExtSem,Sem),
	write(ExtSem), nl.

sh_sems(value(_,FS)) :-
	user:semantics(FS,Sem),
	user:extern_sem(Ext,Sem),
	write(Ext), nl.

sh_fs(object(_,o(FS0,_,_)),Paths) :-
	eval_paths(Paths,FS0,FS),
	feature:tty_fs(FS).

sh_fs(value(_,FS),[]) :-
	feature:tty_fs_clause(FS).

sh_fs(clause(_,H,B),[]) :-
	feature:tty_fs_clause((H:-B)).

sh_tree(object(_,o(FS,_,_)),Mode) :-
	p_tree:pretty_graphic(Mode,FS).

sh_tree(value(_,FS),Mode) :-
	p_tree:pretty_graphic(Mode,FS).

sh_tk(object(_,o(FS,_,_)),Mode) :-
	tk_tree:tk_tree(Mode,FS).

sh_tk(value(_,FS),Mode) :-
	tk_tree:tk_tree(Mode,FS).

sh_term(print,object(_,o(FS,_,_))) :-
	print(FS),nl.

sh_term(print,value(_,FS)) :-
	print(FS),nl.

sh_term(write,object(_,o(FS,_,_))) :-
	write(FS),nl.

sh_term(write,value(_,FS)) :-
	write(FS),nl.

sh_term(_,clause(_,H,[])) :-
	call_residue(copy_term(H,H2),_),
	portray_clause(H2).

sh_term(_,clause(_,H,[B0|B])) :-
	prolog_conjunction(Body,[B0|B]),
	call_residue(copy_term((H:-Body),Clause),_),
	portray_clause(Clause).

%%%%%%%%%%%%%%%%%
%               %
% FIND OBJECTS  %
%               %
%%%%%%%%%%%%%%%%%

item_spec(Obj_spec) -->
        obj_spec(Obj_spec).

item_spec(Pred_spec) -->
	definition_spec(Pred_spec).

item_spec(Pred_spec) -->
	value_spec(Pred_spec).

definition_spec(Clauses) -->
	[l],                     % for `listing'
	fa_spec(F,A),
	{ all_defs(F,A,Clauses)}.

fa_spec(F,A) -->
	[Exp],
	{ concat_all([F,'/',A],Exp),
	  functor(Term,F,A),
	  user:current_predicate(F,Term)
        }.

fa_spec(F,A) -->
	[F],
	{ user:current_predicate(F,Term),
	  functor(Term,F,A)
        }.

value_spec(Value) -->
	fa_spec(F,A),
	pos_spec(Arg),
	{ all_val(F,A,Arg,Value) }.

all_val(F,A,Arg,Values) :-
	findall(V,a_val(F,A,Arg,V),Values).

all_defs(F,A,Clauses) :-
	findall(Clause,a_def(F,A,Clause),Clauses).

a_def(F,A,clause(F/A,Term,Body)) :-
	functor(Term,F,A),
	user:user_clause(Term,Body).

a_val(F,A,whole,value(F/A,Term)) :-
	!,
	functor(Term,F,A),
	user:Term.

a_val(F,A,Arg,object(F/A/Arg,o(Term,W,S))) :-
	user:semantics(Term,S),
	user:phonology(Term,W),
	functor(T0,F,A),
	arg(Arg,T0,Term),
	user:T0.

pos_spec(Arg) -->
	[Arg],
	{ integer(Arg) }.

pos_spec(whole) -->
	[].

obj_spec(Objects) -->
        num_specs(Nums),
	{ findall(Obj,find_obj(Nums,Obj),Objects) }.

find_obj([No,to,No2],object(N,Obj)):-
        !,
        between(No,No2,N),
	user:object(N,Obj).

find_obj([No,plus],object(N,Obj)):-
        !,
        user:find_current_no(Max),
        between(No,Max,N),
	user:object(N,Obj).

find_obj([No,minus],object(No2,Obj)):-
        !,
        between(1,No,No2,'-'),
	user:object(No2,Obj).

find_obj([],object(N,O)) :-
        user:object(N,O),
	!.

find_obj([H],object(H,M)) :-
        user:object(H,M),
	!.

find_obj([No|_Nums],object(No,O)) :-
	user:object(No,O).

find_obj([_|Nums],Obj) :-
        find_obj(Nums,Obj).

num_specs1([plus]) -->
        ['+'],
        !.

num_specs1([minus]) -->
        ['-'],
        !.

num_specs1([to,Num]) -->
        [to],
        num_spec(Num),
        !.

num_specs([Num|Nums]) -->
        num_spec(Num),
        num_specs1(Nums).

num_specs1([Num|Nums]) -->
        num_spec(Num),
        num_specs1(Nums).

num_specs1([]) -->
        [].

num_spec(Num) -->
        [Num],
        {integer(Num)}.


prompting(Generator,Name,Action) :-
	flag(ask,Ask),
	Generator,
	read_stop(Ask,Action,Name),
	!.
prompting(_,_,_).

read_stop(off,Action,_) :-
	call(Action),
	fail. 

read_stop(undefined,Action,_) :-
	call(Action),
	fail. 

read_stop(on,Action,Name) :-
	write(Name),
	write(' ? '),
	ttyflush,
	get0(Char),
	read_stop([Char],Action).

read_stop("s",_).
read_stop("n",_) :-
	fail.
read_stop("y",Action) :-
	Action,
	fail.
read_stop([10],Action) :-
	Action,
	fail.

