%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% (C) 1989                                                %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- module(p_tree,[pretty_graphic/2,
                  pretty_graphic/3,
		  tree_def_to_tree/3,
		  add_graphic/3,
		  del_graphic/3
	         ]).

:- use_module( library(concat), [ term_atom/2 ] ).
:- use_module( library(lists),  [ append/3 ]).
:- use_module( library(flags),  [ flag/2 ]).

%%% uses link_clause.pl

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                   %
%  pretty printer of graphic trees  %
%                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                       %
%  the following predicates must be defined elsewhere:  %
% graphic_path(Name,FS,FS2)                             %
% graphic_daughter(Name,Pos,FS,Daught)                  %
% graphic_label(Name,FS,Label)                          %
% graphic_width(Width) (optional)                       %
%                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

/*
As an example, consider the following:

% picks out the part of a term that should be shown as a tree
graphic_path(test,Tree,Tree,_).

% picks out a daughter; Pos must be instantiated..
% enumerator
graphic_daughter(test,Pos,Tree,Arg):-
    member(Pos,[1,2,3,4,5,6,7,8,9,10]),
    arg(Pos,Tree,Arg).

% returns label of a daughter to be printed
graphic_label(test,Tree,Label):-
    functor(Tree,Label,_).

% now the call:

?- pretty_graphic(test,s(np(det,n),vp(np,v))).

will then print:



     s
   _____   
  |     |  
  np   vp  
  ___  ___ 
 |  |  | | 
det n np v 
*/

:- meta_predicate add_graphic/3, del_graphic/3.

add_graphic(Path,Label,Daughter):-
	user:add_linking_clause(Path,graphic_path,3),
	user:add_linking_clause(Label,graphic_label,3),
	user:add_linking_clause(Daughter,graphic_daughter,4).

del_graphic(Path,Label,Daughter):-
	user:del_linking_clause(Path,graphic_path,3),
	user:del_linking_clause(Label,graphic_label,3),
	user:del_linking_clause(Daughter,graphic_daughter,4).

graphic_path(Name,Term0,Term) :-
	user:graphic_path(Name,Term0,Term).

graphic_daughter(Name,Pos,Term,Daughter) :-
	user:graphic_daughter(Name,Pos,Term,Daughter).

graphic_label(Name,Term,Label) :-
	user:graphic_label(Name,Term,Label).

% pretty graphic(+Name,+Term)
pretty_graphic(Name,FS):-
	pretty_graphic(Name,FS,[]).

% pretty graphic(+Name,+Term,?TabExpr)
% TabExpr is a list consisting of tab(I) statements and atoms to be 
% printed with write_list/1
% Name is a name of a graphic mode to allow for different ways to print
% a given structure
% Term is the term to be shown
pretty_graphic(Name,FS,Tab):-
	graphic_path(Name,FS,FS2),
	tree(Name,FS2,Tree,_,_),      % build tree
	find_graphic_width(Tab,Max),  % get width
	truncate_tree(Tree,Max),      % rearrange w.r.t. width
	scalp(Tree,Tab),              % print tree
	!.                            % backtracking gives rubbish on the screen..

find_graphic_width(TabExpr,Max):-
	(  telling(user_output)
	-> graphic_width(Width),   
	   compute_length(TabExpr,0,TabLength),
	   Max is Width - TabLength
        ;  Max = 100000
        ).

graphic_width(W) :-
	flag(width,V),
	(  V=undefined
        -> W=130
        ;  V=W
        ).

compute_length([],X,X).
compute_length([tab(I)|T],I2,O):-
	!,
	I3 is I2 + I,
	compute_length(T,I3,O).
compute_length([Term|Tail],I2,O):-
	term_atom(Term,Atomic),
	atom_length(Atomic,I),
	I3 is I2 + I,
	compute_length(Tail,I3,O).

%  tree looks like tree(Label,Daughters,Width1,Width2,Width3 )
%    where Label is label(Label) or empty
%          Daughters is list of Daughters (or [])
%          Width1 is number of labels
%          Width2 is number of spaces that is needed
%          Width3 is number of spaces that is available
%             (does not take Tab into account...

find_graphic_label(Name,FS,Label):-
	(  nonvar(FS)
        -> graphic_label(Name,FS,Label1),
	   term_atom(Label1,Label)
        ;  Label = ''
        ).

tree_def_to_tree(Name,FS,Tree) :-
	tree(Name,FS,Tree0,_,_),
	!,  % is necc. but don't remember why..
	tree_def_to_tree0(Tree0,[],[Tree]).

% removes unnecc information, and removes `empty' branches
tree_def_to_tree0(tree(label(L),Ds0,_,_,_),T,[tree(L,_,Ds)|T]) :-
	tree_def_to_tree_l(Ds0,Ds).

tree_def_to_tree0(tree(empty,_,_,_,_),T,T).

tree_def_to_tree_l([],[]).
tree_def_to_tree_l([H0|T0],Res) :-
	tree_def_to_tree0(H0,T,Res),
	tree_def_to_tree_l(T0,T).

tree(Name,FS,tree(label(Label),Ds,W1,W2,_W3),W1,W2):-
        find_graphic_label(Name,FS,Label),
        atom_length(Label,Length),
        graphic_daughters(Name,1,FS,Daughters),
        (   Daughters = [] 
        ->  W1 = 1 , Ds2 = []
        ;   tree_list(Daughters,Name,Ds2,0,W1,0,Length_d)
        ),
        max_tree(Ds2,Ds,Length,Length_d,W2).

tree_list([],_,[],W1,W1,W2,W2).
tree_list([H|T],Name,[H2|T2],W10,W1,W20,W2):-
        tree(Name,H,H2,W1H,W2H),
	W11 is W10 + W1H,
	W21 is W20 + W2H,
        tree_list(T,Name,T2,W11,W1,W21,W2).

% max_tree(Ds,NewDs,LabelWidth,DaughtersWidth,NewLabelWidth)
% we add extra space around the daughters if it turns out that
% the current label is wider than the space needed for the daughters.
max_tree([],[],L,L,W2):-
        W2 is L + 1.

max_tree([Hd|Tds],New_ds,Length,Length_d,W2):-
        (  Length>=Length_d
        -> W2 is Length + 1,
           Extra is W2 - Length_d,
           divi(Extra,E1,E2),
           append([tree(empty,[],nobar,E1,_)|[Hd|Tds]],[tree(empty,[],nobar,E2,_)],New_ds)
        ;  New_ds = [Hd|Tds],
           W2 = Length_d
        ).

% truncate_tree(Tree,Max) truncates labels in tree if the tree
% would otherwise be too big.
truncate_tree(tree(Label,Ds,W1,W2,W3),Max):-
        (  Max > W2
        -> w2_is_w3(tree(Label,Ds,W1,W2,W3))
        ;  Middle is Max // W1,
           give_spaces(tree(Label,Ds,W1,W2,W3),Middle)
        ).

w2_is_w3(tree(_,Ds,_,W2,W2)):-
	w2_is_w3(Ds).
w2_is_w3([]).
w2_is_w3([H|T]):-
	w2_is_w3(H),
	w2_is_w3(T).

give_spaces(tree(_Label,Ds,W1,_W2,W3),Middle):-
	(  W1 = nobar
        -> Mult = 1
        ;  Mult = W1
        ),
        W3 is Mult * Middle,
        give_spaces_l(Ds,Middle).

give_spaces_l([],_).
give_spaces_l([H|T],M):-
        give_spaces(H,M),
        give_spaces_l(T,M).

scalp(tree(label(Id),Br,W1,Ds,W3),StTab):-
	atom_length(Id,Idl),
        Tab is (W3-Idl) // 2,
        tab(Tab),
        write(Id),nl,
        print_hor_bars(StTab,[tree(label(Id),Br,W1,Ds,W3)]),
        scalplist(Br,StTab).

scalplist(Ds,Tab):- 
	(  empty(Ds)
        -> true
        ;  print_vert_bars(Ds),
           nl,
	   write_list(Tab),
           print_ids(Ds),
           nl,
           print_hor_bars(Tab,Ds),
           scalp_all(Ds,NewDs),
           scalplist(NewDs,Tab)
        ).

print_vert_bars([]).
print_vert_bars([tree(empty,[],_,_,Br)|Tail]):-
        tab(Br),
        print_vert_bars(Tail).

print_vert_bars([tree(label(_),_Ds,_,_,Br)|Tail]):-
        Br2 is (Br - 1) // 2,
        tab(Br2),
        write('|'),
        tab(Br2),
        tab2(even(Br),1),
        print_vert_bars(Tail).

scalp_all([],[]).
scalp_all([tree(_Label,[],_W1,W2,W3)|Tail],[tree(empty,[],bar,W2,W3)|Res]):-
        scalp_all(Tail,Res).
scalp_all([tree(_,[Hds|Tds],_,_,_)|Tail],Res):-
        scalp_all(Tail,Res1),
        append([Hds|Tds],Res1,Res).

print_ids([]).
print_ids([tree(empty,_,_,_,Br)|Tail]):-
        tab(Br),
        print_ids(Tail).

print_ids([tree(label(Id1),_,_,_,Br)|Tail]):-
        rewrite_id(Id1,Br,Id,L),
        Br3 is Br - L,
        Br2 is Br3 // 2,
        tab(Br2),
        write(Id),
        tab(Br2),
        tab2(oneven(Br3),1),
        print_ids(Tail).

rewrite_id(_X,0,_,_):-
      % msg(['graphic width is too small, sorry',nl]),
	format(user_output,
"Graphic width is too small, giving up~n",[]),
        !,
	raise_exception(restart).

rewrite_id(Id1,Br,Id,L):-
	(  Br = 0
        -> format(user_output,
"Graphic width is too small, giving up~n",[]),
	%  msg(['graphic width is too small, sorry',nl]),
	   fail
        ;  true
        ),
	atom_length(Id1,OldLength),
        (  OldLength < Br 
        -> L = OldLength,
	   Id1 = Id
        ;  remove_id(Id1,Id,Br),
           L is Br-1
        ).

remove_id(OldId,NewId,NewLength) :-
	name(OldId,OldList),
	length(NewList,NewLength),
	append(NewList,_,OldList),
	name(NewId,NewList).

empty_hb([]).
empty_hb([tree(_,[],_,_,_)|T]):-
	empty_hb(T).

no_hor_bars([]).
no_hor_bars([tree(_,[],_,_,_)|T]):-
	no_hor_bars(T).
no_hor_bars([tree(_,[H|C],_,_,_)|T]):-
	atmostone([H|C],in),
	no_hor_bars(T).

atmostone([],_).
atmostone([tree(empty,[],_,_,_)|T],I):-
	!,
	atmostone(T,I).
atmostone([_R|T],in):-
	atmostone(T,out).

print_hor_bars(_,L):-
        empty_hb(L),!.

print_hor_bars(Tab,L):-
	no_hor_bars(L),!,
	write_list(Tab).

print_hor_bars(Tab,L):-
	write_list(Tab),
	print_hor_bars0(L),
	nl,
	write_list(Tab).

print_hor_bars0([]).
print_hor_bars0([tree(_,[],_,_,Br)|T]):-
        tab(Br),
        print_hor_bars0(T).

print_hor_bars0([tree(_,H,_,_,_Br)|T]):-
        print_hor_bars1(H),
        print_hor_bars0(T).

print_hor_bars1([]).
print_hor_bars1([tree(_,[],nobar,_,Br)|T]):-
	!,
        tab(Br),
        print_hor_bars1(T).

print_hor_bars1([tree(_,_I,_,_,Br)|T]):-
        divi(Br,Br2,Br3),
        tab(Br2),
        print_hor_bars2(T,Br3).

print_hor_bars2([],Br):-
	tab(Br).

print_hor_bars2([tree(empty,[],nobar,_,Br)|T],BrIn):-
        !,
        tab(Br),
        print_hor_bars2(T,BrIn).

print_hor_bars2([tree(label(_),_I,_,_,Br)],BrIn):-
        !,
        wrnum('_',BrIn),
        divi(Br,Br2,Br3),
        wrnum('_',Br2),
        tab(Br3).

print_hor_bars2([tree(_,_I,_,_,Br)|T],BrIn):-
        wrnum('_',BrIn),
        print_hor_bars2(T,Br).

empty([]).
empty([tree(empty,_,_,_,_)|Tail]):-
        empty(Tail).
        
wrnum(_,Num):-
	Num < 1,
        !.

wrnum(A,Num):-
        write(A),
        Num2 is Num - 1,
        wrnum(A,Num2).

tab2(A,B):-
        A,
        !,
	tab(B).
tab2(_,_).

even(X) :- 
        0 is X mod 2.

oneven(X) 
        :- 1 is X mod 2.

divi(Tot,A,A):-
        even(Tot),
        !,
        A is Tot // 2.

divi(Tot,A,B):-
        A is Tot // 2,
        B is A + 1.

graphic_daughters(Name,Pos,FS,[]):-
        nothigherone(Name,Pos,FS),
        !.

graphic_daughters(Name,Pos,FS,[Daughter|Daughters]):-
        graphic_daughter(Name,Pos,FS,Daughter),
        !,
        Pos2 is Pos + 1,
        graphic_daughters(Name,Pos2,FS,Daughters).

graphic_daughters(Name,Pos,FS,Ds):-
        Pos2 is Pos + 1,
        graphic_daughters(Name,Pos2,FS,Ds).

nothigherone(_Name,_Pos,Var):-
	var(Var),!.
nothigherone(Name,Pos,FS):-
        graphic_daughter(Name,Pos2,FS,_),
        Pos2 >= Pos,
        !,
        fail.
nothigherone(_,_,_).

atom_length(Atom,Length):-
        nonvar(Atom),
        name(Atom,List),
        length(List,Length).

write_list([]).
write_list([H|T]) :-
	write_term(H),
	write_list(T).

write_term(tab(I)) :-
	!,
	tab(I).
write_term(Term) :-
	write(Term).
