%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% (C) 1993  all rights reserved                           %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% prolog2latex2xdvi
% fs2latex2xdvi
%
% this file defines relations to pretty-print (using latex and Xdvi) arbitrary
% prolog terms, clauses and sets of clauses.
%
% furthermore, if the terms are used to represent feature-structures along the
% lines of the library(feature) package, then it is also possible to generate
% matrix notation for such terms / clauses / sets of clauses.

% uses: latex
%       xdvi
%       treemaker (dctree)
%       xterm

:- module(latex, [ window/0,
                   latex_fs/1,
		   latex_fs_clause/1,
		   latex_fs_listing/1,
                   latex_term/1,
		   latex_term_clause/1,
		   latex_term_listing/1,	
		   latex_tree/2,
		   latex_tree/1
                 ]).

:- use_module([library(feature),
	       library(gen_sym),
	       library(prettyvars)    % to know what variables are single
	      ]).

:- use_module( library(concat), [ concat/3,
	                          concat_all/2,
	                          concat_all/3,
				  term_atom/2]).

:- use_module( library(flags) ).
:- use_module( library(lists),  [ member/2,
	                          append/3 ]).
:- use_module( library(decons), [ prolog_conjunction/2 ] ).

:- use_module( library(env),    [ get_env_var/2 ]).

%%%%%%%%%%%%%%%%%%%
% main predicates %
%%%%%%%%%%%%%%%%%%%

% files used as input/output for latex and xdvi

dir(TmpDir) :-
	get_env_var('TMPDIR',TmpDir),!.
dir('/tmp').

dir0(Tmp) :-
	dir(Tmp0),
	concat(Tmp0,'/',Tmp).

files(A,B,C,D,E) :-
	files0(A,B,C,D,E),!.

files(Tex,Dvi,Tree,Tex0,Lab) :-
	dir0(Tmp),
	concat(Tmp,'_fs.tex',Tex),
	concat(Tmp,'_fs.dvi',Dvi),
	concat(Tmp,'_fs.tree',Tree),
	concat(Tmp,'_fs0.tex',Tex0),
	concat(Tmp,'_fslab.tex',Lab),
	asserta(files0(Tex,Dvi,Tree,Tex0,Lab)).

% the structures are embedded in a document, hence we set up the
% document:
start_docu(Tree) :-
        write('\documentstyle[matrix,12pt]{article}'),nl,
	(Tree == tree -> write('\input{tmaker}'),nl ; true ),
	write('\setlength{\parindent}{-100pt}'),nl,
        write('\addtolength{\textheight}{1000ex}'),nl,
	write('\begin{document}'),nl,
	write('\thispagestyle{empty}'),nl,
	write('\topmargin -100pt'),nl,
%%	write('\oddsidemargin -50pt'),nl,
	nl.

% and finish the document:
end_docu :-
        nl,write('\end{document}'),nl.

% starts xdvi 
window :-
	flag(window,_,off),
	start_window_up.

shell_call0(Cmds) :-
	append([xterm,'-iconic','-e','sh -c "'],Cmds,Cmds1),
	append(Cmds1,['" &'],Cmds2),
	concat_all(Cmds2,Cmd,' '),
	unix(shell(Cmd)).

% starts xdvi if not started yet
start_window_up :-
	flag(window,on),!.
start_window_up :-
	files(_,Dvi,_,_,_),
	concat_all([xdvi,'-geometry 800x480+100+800','-paper a1',Dvi,' &'],
	           Cmd,' '),
	unix(shell(Cmd)),
	flag(window,_,on).

% starts latex and xdvi in one command to be sure that the dvi file exists!
latex_and_xdvi :-
	flag(window,on),
	!,
	latex_cmd.

latex_and_xdvi :-
	dir(Dir),
	files(Tex,Dvi,_,_,_),
	shell_call0(['( cd ',Dir,' ; ',latex,Tex,' );',
                    xdvi,'-geometry 800x480+100+800','-paper a1',Dvi]),
        flag(window,_,on).

latex_cmd :-
	dir(Dir),
	files(Tex,_,_,_,_),
	shell_call0(['( cd ',Dir,' ; ',latex,Tex,' )']).

latex_and_dctree_and_latex_and_xdvi :-
	flag(window,on),!,
	dir(Dir),
	files(Tex,_,_,Tex0,_),
	shell_call0(['( cd ',Dir,' ; ',
                    latex,Tex0,';',
                    dctree,Tex0,';',
                    latex,Tex,' )']).

latex_and_dctree_and_latex_and_xdvi :-
	files(Tex,Dvi,_,Tex0,_),
	dir(Dir),
	shell_call0(['( cd ',Dir,' ; ',
                    latex,Tex0,';',
                    dctree,Tex0,';',
                    latex,Tex,';',
		    xdvi,'-geometry 800x480+100+800','-paper a1',Dvi, ')']),
        flag(window,_,on).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% prolog terms as feature structures %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% First: prolog terms / clauses / listing where the terms that
%% occur might represent feature-structures.
%%
%% cf. library(feature) & library p_feature
%%
%% note that the file library(p_feature) defines similar predicates for tty output

latex_fs(fs(FS)) :-
	!,
	print_latex_fs(fs(FS)).

latex_fs(clause(FS)) :-
	!,
	print_latex_fs(clause(FS)).

latex_fs(clause(FS,Body)) :-
	!,
	print_latex_fs(clause(FS,Body)).

latex_fs([H|T]) :-
	!,
	print_latex_fs([H|T]).

latex_fs(listing(X)) :-
	!,
	print_latex_fs(listing(X)).

latex_fs(FS) :-
	print_latex_fs(fs(FS)).

latex_fs_clause((H:-B0)) :-
	!,
%	ensure_convert_to_list(B0,B),          % lazyness
	ensure_prolog_conjunction(B0,B),
	print_latex_fs(clause(H,B)).

latex_fs_clause(Unit) :-
	print_latex_fs(clause(Unit)).

latex_fs_listing([H|T]) :-
	print_latex_fs(listing([H|T])).

print_latex_fs(Thing) :-
	flag(vspace,_,on),
	files(Tex,_,_,_,_),
	telling(Old), tell(Tex),
        start_docu(_),
        print_it_fs(Thing),
        end_docu,
	told, telling(Old),
	latex_and_xdvi.

print_it_fs(listing(L)) :-
	!,
	( member(Thing,L),
	  (   Thing = (H:-B0) 
%          ->  ensure_convert_to_list(B0,B),
	  ->  ensure_prolog_conjunction(B0,B),
	      print_it_fs(clause(H,B))
          ;   print_it_fs(clause(Thing))
          ),
	  fail
        ; true ).

print_it_fs([]).
print_it_fs([H|T]) :-
	print_it_fs(H),
	print_it_fs(T).

print_it_fs(Thing0) :-
	write('\begin{flushleft}'),nl,
%%	shorten_it(Thing0,Thing1),
	call_residue(copy_term(Thing0,Thing1),Cons0),
	user:rewrite_body(Cons0,[],Cons,[]),            % note: from link_clause.pl!!
	add_cons(Thing1,Cons,Thing2),
	shorten_it(Thing2,Thing3),
	prettyvars(Thing3,0,No),
	change_it_fs(Thing3,Thing,No),
	write_it_fs(Thing),
	write('\end{flushleft}').

shorten_it(clause(H0,B0),clause(H,B)) :-
	shorten_goal(H0,H),
	shorten_goals(B0,B).
shorten_it(clause(H0),clause(H)) :-
	shorten_goal(H0,H).
shorten_it(fs(X0),fs(X)) :-
	shorten_label(X0,X).
shorten_it(fs(X0,Y0),fs(X,Y)) :-
	shorten_label(X0,X),
	shorten_goals(Y0,Y).

add_cons(clause(H,B0),C,clause(H,B)) :-
	append(B0,C,B).
add_cons(clause(H),[],clause(H)).
add_cons(clause(H),[C0|C],clause(H,[C0|C])).
add_cons(fs(X),[C0|C],fs(X,[C0|C])).
add_cons(fs(X),[],fs(X)).
add_cons(fs(X,C0),C1,fs(X,C)) :-
	append(C0,C1,C).

change_it_fs(clause(H0),clause(H),No) :-
	feature:change_pretty_constraints([H0],[H],[],No).

change_it_fs(clause(H0,B0),clause(H,B),No) :-
	feature:change_pretty_constraints([H0|B0],[H|B],[],No).

change_it_fs(fs(FS0,C0),fs(FS,C),No) :-
	feature:change(FS0,FS,[],Out,No,OutNo),
	feature:change_pretty_constraints(C0,C,Out,OutNo).
	
change_it_fs(fs(FS0),fs(FS),No) :-
	feature:change(FS0,FS,[],_Out,No,_OutNo).
	
write_it_fs(fs(FS,C)) :-
	tex_begin_line,
	pp(FS),tex_if,
	tex_end_line,
	write_pretty_constraints(C).

write_it_fs(fs(FS)) :-
	tex_begin_line,
	pp(FS),
	tex_end_line.

write_it_fs(clause(H)) :-
	tex_begin_line,
	write_pretty_constraint(H),
	tex_end_line.

write_it_fs(clause(H,B)) :-
	tex_begin_line, 
	write_pretty_constraint(H), 
	tex_if,
	tex_end_line, 
	write_pretty_constraints(B).

write_pretty_constraints([H|T]) :-
	tex_begin_line, 
	write('~~~~'),
	write_pretty_constraint(H),
	write_pretty_constraints0(T).

write_pretty_constraints0([]) :-
	put(0'.), tab(1), tex_end_line.
write_pretty_constraints0([H|T]) :-
	write(','),tab(1),
	tex_end_line,
	tex_begin_line, 
	write('~~~~'),
	write_pretty_constraint(H),nl,
	write_pretty_constraints0(T).

write_pretty_constraint(H) :-
	H =.. [F|Args],
	write_relation(F),
	write_begin_functor(Args),
	write_pretty_arguments0(Args),
	write_end_functor(Args).

write_begin_functor([]).
write_begin_functor([_|_]):-
	write('(').

write_end_functor([]).
write_end_functor([_|_]) :-
	write(')').

write_pretty_arguments0([H|T]):-
	pp(H),
	write_pretty_arguments(T).

write_pretty_arguments([]).
write_pretty_arguments([H|T]):-
	write(','),
	pp(H),
	write_pretty_arguments(T).

fs :-
    write('\avm{'),nl.
fs(Var):-
    write('\avm[{'),
    tex_var(Var),
    write('}]{').
fsfs :-
    write('}').

pp(_Var/n=FS):-
	!,
        ppl_latex(FS,no).

pp(Var/_='R'):-
        !,
	tex_var(Var).

pp(Var/y=[]):-
        !,
	tex_var(Var).

pp(Var/y='$VAR'(_)):-
	!,
	tex_var(Var).

pp(Var/y=FS):-
%	tex_var(Var),
        ppl_latex(FS,yes(Var)).

pp(lex(W)) :-
	write(W).

all_empty([]).
all_empty([a(_,H)|T]):-
	do_not_print_latex(H),
	all_empty(T).

do_not_print_latex(_Var/n='$VAR'(_)).

ppl_latex([a(_Att,Thing)|Rest],Tab):-
	do_not_print_latex(Thing),!,
	ppl_latex(Rest,Tab).

ppl_latex([a(type,['.']),a(_,Head),a(_,Tail)],no) :-
	!,
	ppl_list(Head,Tail).

ppl_latex([a(type,['.']),a(_,Head),a(_,Tail)],yes(Var)) :-
	!,
	tex_var(Var),write('~'),
	ppl_list(Head,Tail).

ppl_latex([a(type,[[]])],_) :-
	!,
	write('\langle\rangle').

ppl_latex([a(type,Types)|T],no):-
	all_empty(T),
	!,
%	write(' \{ '),
	feature:write_as_conj(Types,PTypes),
	write_type(PTypes).
%	write(' \} ').

ppl_latex([a(type,Types)|T],yes(Var)):-
	all_empty(T),
	!,
	tex_var(Var),write('~'),
%	write(' \{ '),
	feature:write_as_conj(Types,PTypes),
	write_type(PTypes).
%	write(' \} ').

ppl_latex([a(type,Types)|T],no):-
	!,
	fs,
%	write(' \{ '),
	feature:write_as_conj(Types,PTypes),
	write_type(PTypes),
%	write(' \} '),
	ppl2_latex(T,_Tab),
	fsfs.

ppl_latex([a(type,Types)|T],yes(Var)):-
	!,
	fs(Var),
%	write(' \{ '),
	feature:write_as_conj(Types,PTypes),
	write_type(PTypes),
%	write(' \} '),
	ppl2_latex(T,_Tab),
	fsfs.

ppl_latex([a('BOOLEAN',_Type,Val)|T],no):-
	all_empty(T),
	!,
	give_boolean_type(Val,Exp),
%	write(' \{ '),
	write_type(Exp).
%	write(' \} ').

ppl_latex([a('BOOLEAN',_Type,Val)|T],yes(Var)):-
	all_empty(T),
	!,
	tex_var(Var),write('~'),
	give_boolean_type(Val,Exp),
%	write(' \{ '),
	write_type(Exp).
%	write(' \} ').


ppl_latex([a('BOOLEAN',_Type,Val)|T],no):-
	give_boolean_type(Val,Exp),
	fs,
	write_type(Exp),
	ppl2_latex(T,_Tab),
	fsfs.

ppl_latex([a('BOOLEAN',_Type,Val)|T],yes(Var)):-
	give_boolean_type(Val,Exp),
	fs(Var),
	write_type(Exp),
	ppl2_latex(T,_Tab),
	fsfs.

ppl_latex([a('UNTYPED',_Att,Val)|T],no):-
	tex_term(Val),
	ppl2_latex(T,_Tab).

ppl_latex([a('UNTYPED',_Att,Val)|T],yes(Var)):-
	tex_var(Var),write('~'),
        tex_term(Val),
	ppl2_latex(T,_Tab).

ppl_latex([a('TREE',Mark0,Cat,Ds)|T],_) :-
	mrk_t(Mark0,Mark),
	write(Mark),tab(1),
	pp_ds([Cat|Ds],0),
	ppl2_latex(T,_).

ppl_latex([a(Att,FS)|T],no):-
        !,
	fs,
	write_attribute(Att),
	write(':'),
        pp(FS),
        ppl2_latex(T,_Tab),
	fsfs.

ppl_latex([a(Att,FS)|T],yes(Var)):-
        !,
	fs(Var),
	write_attribute(Att),
	write(':'),
        pp(FS),
        ppl2_latex(T,_Tab),
	fsfs.

ppl_latex([],_).

ppl_latex('$VAR'(_),_) :-
	write('\_').
%%	write('$VAR'(X)).

pp_ds([H],_) :-
	!,
	pp(H).

pp_ds([H|T],I0) :-
	fs,
	pp_ds1([H|T],I0),
	fsfs.

pp_ds1([],_).
pp_ds1([H|T],I0) :-
	write(I0),
	write(':'),
	pp(H),
	I is I0 + 1,
	pp_ds2(T,I).

pp_ds2([],_) :- 
	!.
pp_ds2([H|T],I0) :-
	!,
	write('\\'),
	write(I0),
	write(':'),
	pp(H),
	I is I0 + 1,
	pp_ds2(T,I).
pp_ds2('$VAR'(_),_) :-
	write('\_').

mrk_t(foot,'=') :- 
	!.
mrk_t(subs,'*') :- 
	!.
mrk_t(_,'').

pplist_latex(atom(X),I,_Tab):-
	write(' \\ '),
	nl,
%	write_list(Tab),
	write(I),
	write(X).
pplist_latex([],_,_).
pplist_latex([H|T],I,Tab):-
%	append(Tab,[I],Tab2),
	write(' \\ '),
	nl,
%	write_list(Tab2),
	fs,
	pp(H),
	fsfs,
	I2 is I + 1,
	pplist_latex(T,I2,Tab).
pplist_latex('$VAR'(_),_,_).

ppl2_latex([a(_Att,Thing)|Rest],Tab):-
	do_not_print_latex(Thing),!,
	ppl2_latex(Rest,Tab).

ppl2_latex([a(Att,FS)|T],Tab):-
        !,
	write(' \\ '),
	nl,
        write_attribute(Att),
	write(':'),
        pp(FS),
        ppl2_latex(T,Tab).

ppl2_latex([],_).
ppl2_latex('$VAR'(_),_).

ppl_list(Head,Tail) :-
	write('\langle '),
	ppx(Head),
	ppl_list(Tail),
	write('\rangle ').

ppl_list(V/y='R') :-
	!,
	write(' | '),
	ppx(V/y='R').

ppl_list(V/YN='$VAR'(_)) :-
	!,
	write(' | '),
	ppx(V/YN='$VAR'(_)).
	
ppl_list(_Var/_YN=[a(type,[[]])]) :-
	!.

ppl_list(_Var/_YN=[a(type,['.']),a(_,Head),a(_,Tail)]) :-
	write(' , '),
	ppx(Head),
	ppl_list(Tail).

ppx(ListEl) :-
	pp(ListEl),!.

ppx(_) :-
	write(' \_ ').

write_attribute(A) :-
	write(' \mbox{\it '),
	tx_atom(A),
	write('} ').

write_relation(A) :-
%%	write(' \mbox{\it '),
	tex_atom(A).
%%	write('} ').

write_type(A) :-
	write(' \mbox{\sc '),
	tx_atom(A),
	write('} ').

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% latex output for ordinary prolog terms %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% latex_term/1,
% latex_term_clause/1,
% latex_term_listing/1

latex_term([H|T]) :-
	!,
	print_latex_term([H|T]).

latex_term(clause(H)) :-
	!,
	print_latex_term(clause(H)).

latex_term(clause(H,B)) :-
	!,
	print_latex_term(clause(H,B)).

latex_term(term(H)) :-
	!,
	print_latex_term(term(H)).

latex_term(listing(H)) :-
	!,
	print_latex_term(listing(H)).

latex_term(T) :-
	print_latex_term(term(T)).

latex_term_clause((H:-B)) :-
	!,
	print_latex_term(clause(H,B)).
latex_term_clause(H) :-
	print_latex_term(clause(H)).

latex_term_listing([L0|L]) :-
	print_latex_term(listing([L0|L])).

print_latex_term(Thing) :-
	flag(vspace,_,on),
	files(Tex,_,_,_,_),
	telling(Old), tell(Tex),
        start_docu(_),
        ( print_it(Thing), fail ; true ),   % no side effects.
        end_docu,
        told, tell(Old),
	latex_and_xdvi.

print_it([]).
print_it([H|T]) :-
	print_it(H),
	print_it(T).

print_it(term(T0)) :-
	call_residue(copy_term(T0,T),_),
	prettyvars(T),
	tex_begin_line,
	write_goal(T, 1199, 0, Co),
	write_fullstop(Co),
	tex_end_line.

print_it(fs(T0)) :-
	call_residue(copy_term(T0,T),_),
	prettyvars(T),
	tex_begin_line,
	write_goal(T, 1199, 0, Co),
	write_fullstop(Co),
	tex_end_line.

print_it(clause(H0)) :-
	call_residue(copy_term(H0,H),_),
	prettyvars(H),
	tex_begin_line,
	portray_clause1(H, Co),
	write_fullstop(Co),
	tex_end_line.

print_it(clause(H0,B0)) :-
	call_residue(copy_term(H0/B0,H/B1),_),
	ensure_prolog_conjunction(B,B1),
	prettyvars((H:-B)),
	tex_begin_line,
	portray_clause1((H:-B), Co),
	write_fullstop(Co),
	tex_end_line.

print_it(listing([H|T])) :-
	( H = (B:-C) -> print_it(clause(B,C)) ; print_it(clause(H))),
	nl,   %extra newline to make sure that a paragraph is started
	print_it(listing(T)).

% This must be careful not to bind any variables in Clause.
portray_clause_l([]).
portray_clause_l([Clause|_]) :-
	prettyvars(Clause),
	tex_begin_line,
	portray_clause1(Clause, Co),
	write_fullstop(Co),
	tex_end_line,
	fail.
portray_clause_l([_|T]) :-
	portray_clause_l(T).

%%% changed the following so as to generate latex code...
%%% buggy.
%%% 

%% ADAPTED FROM:
%   File   : WRITE.PL
%   Author : Richard A. O'Keefe
%   Updated: 22 October 1984
%   Purpose: Portable definition of write/1 and friends.

% Priority 999 is o.k. if printed e.g. as elements of a list. /MC

%   maybe_paren(P, Prio, Char, Ci, Co)
%   writes a parenthesis if the context demands it.
%   Context = 2'000 for alpha
%   Context = 2'001 for quote
%   Context = 2'010 for other
%   Context = 2'100 for punct

maybe_paren(P, Prio, Lpar, '(', _, 2'100) :-
	P > Prio, !,
	prolog:'$display'(Lpar).
maybe_paren(_, _, Lpar, Lpar, C, C).

maybe_paren(P, Prio, _, 2'100) :-
	P > Prio, !,
	prolog:'$display'(')').
maybe_paren(_, _, C, C).

%   maybe_space(LeftContext, TypeOfToken)
%   generates spaces as needed to ensure that two successive
%   tokens won't run into each other.

maybe_space(Ci, Co) :-
	(   Ci\/Co<2'100, Ci#Co<2'010 -> tex_tab(1)  %%put(0' )
	;   true
	).

/*
sticky_contexts(alpha, alpha).
sticky_contexts(quote, quote).
sticky_contexts(other, other).
sticky_contexts(alpha, quote).
sticky_contexts(quote, alpha).
*/

%   write_out(Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co)
%   writes out a Term in given SynStyle, LexStyle
%   at nesting depth Depth
%   in a context of priority Priority (that is, expressions with
%   greater priority must be parenthesized), 
%   and prefix operators =< PrePrio must be parenthesized,
%   where the last token to be
%   written was of type Ci, and reports that the last token it wrote
%   was of type Co.

write_out(Term, _, _, _, _, _, _, Ci, 2'000) :-
	var(Term), !,
	maybe_space(Ci, 2'000),
	tex_variable(Term).
write_out('$VAR'(N), SynStyle, LexStyle, _, _, Depth, _, Ci, Co) :- !,
	Depth1 is Depth+1,
	write_VAR(N, SynStyle, LexStyle, Depth1, Ci, Co).
write_out(_, print(Limit), _, _, _, Depth, _, Ci, 2'010) :-
	Depth >= Limit, !,
	maybe_space(Ci, 2'010),
	tex_dots.
/*
write_out(Term, print(_), _, _, _, _, _, _, 2'000) :-
	(   \+call_user_def(portray(Term), user) ->
	    fail		 % portray might bind variables
	;   true
	), !.
*/
write_out(Atom, _, LexStyle, _, PrePrio, _, Lpar, _, 2'100) :-
	atom(Atom),
	prolog:current_prefixop(Atom, P, _),
	P =< PrePrio, !,
	prolog:'$display'(Lpar),
	write_atom(LexStyle, Atom, 2'100, _),
	put(0')).
write_out(Atom, _, LexStyle, _, _, _, _, Ci, Co) :-
	atom(Atom), !,
	write_atom(LexStyle, Atom, Ci, Co).
write_out(N, _, _, _, _, _, _, Ci, 2'000) :-
	number(N), !,
	(   N < 0 -> maybe_space(Ci, 2'010)
	;   maybe_space(Ci, 2'000)
	),
	write(N).
write_out(Term, noop, LexStyle, _, _, Depth, _, Ci, 2'100) :-
	functor(Term, Atom, Arity), !,
	write_atom(LexStyle, Atom, Ci, _),
	Depth1 is Depth+1,
	write_args(0, Arity, Term, noop, LexStyle, Depth1).
write_out({Term}, SynStyle, LexStyle, _, _, Depth, _, _, 2'100) :- !,
%%	put(0'{),
        write('\{ '),
	Depth1 is Depth+1,
	write_out(Term, SynStyle, LexStyle, 1200, 0, Depth1, '(', 2'100, _),
	write('\} ').
%%	put(0'}).
write_out([Head|Tail], SynStyle, LexStyle, _, _, Depth, _, _, 2'100) :- !,
	put(0'[),
	Depth1 is Depth+1,
	write_out(Head, SynStyle, LexStyle, 999, 0, Depth1, '(', 2'100, _),
	write_tail(Tail, SynStyle, LexStyle, Depth1).
write_out((A,B), SynStyle, LexStyle, Prio, _, Depth, Lpar, Ci, Co) :- !,
	%  This clause stops writeq quoting commas.
	Depth1 is Depth+1,
	maybe_paren(1000, Prio, Lpar, Lpar1, Ci, C1),
	write_out(A, SynStyle, LexStyle, 999, 0, Depth1, Lpar1, C1, _),
	put(0',),
	write_out(B, SynStyle, LexStyle, 1000, 1000, Depth1, '(', 2'100, C2),
	maybe_paren(1000, Prio, C2, Co).
write_out(Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co) :-
	functor(Term, F, N),
	Depth1 is Depth+1,
	write_out(N, F, Term, SynStyle, LexStyle, Prio, PrePrio, Depth1, Lpar, Ci, Co).

write_out(1, F, Term, SynStyle, LexStyle, Prio, _, Depth, Lpar, Ci, Co) :-
	prolog:current_postfixop(F, P, O), !,
	(prolog:current_infixop(F, _, _, _) -> O1=1200; O1=O),
	maybe_paren(O1, Prio, Lpar, Lpar1, Ci, C1),
	arg(1, Term, A),
	write_out(A, SynStyle, LexStyle, P, 1200, Depth, Lpar1, C1, C2),
	write_atom(LexStyle, F, C2, C3),
	maybe_paren(O1, Prio, C3, Co).
write_out(1, F, Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co) :-
	F \== -,
        prolog:current_prefixop(F, O, P), !,
	(PrePrio=1200 -> O1 is P+1; O1=O),	% for "fy X yf" etc. cases
	maybe_paren(O1, Prio, Lpar, _, Ci, C1),
	write_atom(LexStyle, F, C1, C2),
	arg(1, Term, A),
	write_out(A, SynStyle, LexStyle, P, P, Depth, ' (', C2, C3),
	maybe_paren(O1, Prio, C3, Co).
write_out(2, F, Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co) :-
        prolog:current_infixop(F, P, O, Q), !,
	(PrePrio=1200 -> O1 is Q+1; O1=O),	% for "U xfy X yf" etc. cases
	maybe_paren(O1, Prio, Lpar, Lpar1, Ci, C1),
	arg(1, Term, A),
	write_out(A, SynStyle, LexStyle, P, 1200, Depth, Lpar1, C1, C2),
	write_atom(LexStyle, F, C2, C3),
	arg(2, Term, B),
	write_out(B, SynStyle, LexStyle, Q, Q, Depth, '(', C3, C4),
	maybe_paren(O1, Prio, C4, Co).
write_out(N, F, Term, SynStyle, LexStyle, _, _, Depth, _, Ci, 2'100) :-
	write_atom(LexStyle, F, Ci, _),
	write_args(0, N, Term, SynStyle, LexStyle, Depth).

write_VAR(N, SynStyle, _, _, Ci, 2'000) :-
	integer(N), N >= 0,
	SynStyle \== noop, !,
	maybe_space(Ci, 2'000),
	tex_var(N).

write_VAR(String, SynStyle, _, _, Ci, Co) :-
	nonvar(String),
	(   prolog:'$atom_chars'(Atom, String) -> true
	;   Atom = String
	),
	atom(Atom),
	SynStyle \== noop, !,
	write_atom(noquote,Atom,Ci,Co).
write_VAR(X, SynStyle, LexStyle, Depth, Ci, 2'100) :-
	write_atom(LexStyle, '$VAR', Ci, _),
	write_args(0, 1, '$VAR'(X), SynStyle, LexStyle, Depth).

write_atom(noquote, Atom, Ci, Co) :-
	prolog:'$atom_mode'(Atom, Co),
	maybe_space(Ci, Co),
	tex_atom(Atom).
write_atom(quote, Atom, Ci, Co) :-
	prolog:'$atom_mode'(Atom, Co),
	maybe_space(Ci, Co),
	tex_atom(Atom).

%   write_args(DoneSoFar, Arity, Term, SynStyle, LexStyle, Depth)
%   writes the remaining arguments of a Term with Arity arguments
%   all told in SynStyle, LexStyle, given that DoneSoFar have already been written.

write_args(N, N, _, _, _, _) :- !,
	put(0')).
write_args(I, _, _, print(Limit), _, Depth) :-
	Depth >= Limit, !,
	write_args(I, Depth),
	tex_dots,
	put(0')).
write_args(I, N, Term, SynStyle, LexStyle, Depth) :-
	write_args(I, Depth),
	J is I+1,
	arg(J, Term, A),
	write_out(A, SynStyle, LexStyle, 999, 0, Depth, '(', 2'100, _),
	write_args(J, N, Term, SynStyle, LexStyle, Depth).

write_args(0, _) :- !, put(0'().
write_args(_, 0) :- !, prolog:'$display'(', ').
write_args(_, _) :- put(0',).



%   write_tail(Tail, SynStyle, LexStyle, Depth)
%   writes the tail of a list of a given SynStyle, LexStyle, Depth.

write_tail(Var, _, _, _) :-			%  |var]
	var(Var), !,
	put(0'|),
	prolog:'$write'(Var),
	put(0']).
write_tail([], _, _, _) :- !,			%  ]
	put(0']).
write_tail(_, print(Limit), _, Depth) :-
	Depth >= Limit, !,
	put(0'|),
	tex_dots,
	put(0']).
write_tail([Head|Tail], SynStyle, LexStyle, Depth) :- !, %  ,Head tail
	put(0',),
	write_out(Head, SynStyle, LexStyle, 999, 0, Depth, '(', 2'100, _),
	Depth1 is Depth+1,
	write_tail(Tail, SynStyle, LexStyle, Depth1).
write_tail(Other, SynStyle, LexStyle, Depth) :-	%  |junk]
	put(0'|),
	write_out(Other, SynStyle, LexStyle, 999, 0, Depth, '(', 2'100, _),
	put(0']).

portray_clause1(:-(Command), Co) :-
	functor(Command, Key, 1),
	current_op(_, fx, Key), !,
	arg(1, Command, Body),
	'list clauses'(Body, :-(Key), 8, Co).
portray_clause1((Pred:-true), Co) :- !,	
	 write_goal(Pred, 1199, 0, Co).
portray_clause1((Pred:-Body), Co) :- !,	
	write_goal(Pred, 1199, 1200, _),
	'list clauses'(Body, 0, 8, Co).
portray_clause1((Pred-->Body), Co) :- !,
	write_goal(Pred, 1199, 1200, _),
	'list clauses'(Body, 2, 8, Co).
portray_clause1(Pred, Co) :-
	write_goal(Pred, 1199, 0, Co).


write_goal(M:Goal, Prio, PrePrio, C) :- !,
	write_out(M:Goal, op, quote, Prio, PrePrio, -2, '(', 2'100, C).
write_goal(Goal, Prio, PrePrio, C) :-
	write_out(Goal, op, quote, Prio, PrePrio, -1, '(', 2'100, C).

write_fullstop(Ci) :-
	maybe_space(Ci, 2'010),
	put(0'.), tex_nl.


'list clauses'((A,B), L, D, Co) :- !,
	'list clauses'(A, L, D, _),
	'list clauses'(B, 1, D, Co).
'list clauses'((A;B), L, D, 2'100) :- !,
	'list magic'(L, D),
	'list disj'(A, 3, D),
	'list disj'(B, D).
'list clauses'((A->B), L, D, 2'100) :- !,
	'list magic'(L, D),
	E is D+4,
	'list clauses'(A, 3, E, _),
	'list clauses'(B, 5, E, _),
	tex_nl, tex_tab(D),
	put(0')).
'list clauses'(!, 0, _, 2'100) :- !,
	prolog:'$display'(' :- !').
'list clauses'(!, 1, _, 2'100) :- !,
	prolog:'$display'(', !').
'list clauses'(!, 2, _, 2'100) :- !,
	prolog:'$display'(' --> !').
'list clauses'(Goal, L, D, Co) :- !,
	'list magic'(L, D),
	write_goal(Goal, 999, 0, Co).


'list magic'(0, D) :-
	tex_if,
	tex_nl, tex_tab(D).
'list magic'(1, D) :-
	put(0',),
	tex_nl, tex_tab(D).
'list magic'(2, D) :-
	prolog:'$display'(' -->'),
	tex_nl, tex_tab(D).
'list magic'(3, _) :-
	prolog:'$display'('(   ').
'list magic'(4, _) :-
	prolog:'$display'(';   ').
'list magic'(5, D) :-
	prolog:'$display'(' ->'),
	tex_nl, tex_tab(D).
'list magic'(:-(Key), D) :-
	tex_if,
	prolog:'$write'(Key),
	tex_nl, tex_tab(D).

'list disj'((A;B), D) :- !,
	'list disj'(A, 4, D),
	'list disj'(B, D).
'list disj'(Conj, D) :-
	'list disj'(Conj, 4, D),
	put(0')).

'list disj'((A->B), L, D) :- !,
	E is D+4,
	'list clauses'(A, L, E, _),
	'list clauses'(B, 5, E, _),
	tex_nl, tex_tab(D).
'list disj'(A, L, D) :-
	E is D+4,
	'list clauses'(A, L, E, _),
	tex_nl, tex_tab(D).


%%% these are all supposed to be in mathematical mode !
tex_tab(0) :-
	!.
tex_tab(N0) :-
	write('~'),
	N is N0-1,
	tex_tab(N).

tex_nl :-
	tex_end_line,
	tex_begin_line.

tex_end_line :-
	(  flag(vspace,on) 
        -> write('$\\')
        ;  write('$')
        ),nl.

tex_begin_line :-
	write('$'),nl.

tex_variable(Term) :-
	write(Term), 
	nl.  % added for long lines

tex_dots :-
	write('\dots ').

tex_atom([]) :-
	!,
	write('[~]').
tex_atom(A) :-
	write(' \mbox{'),  % removed \it
	tx_atom(A),
	write('} '),
	nl.           % added to make lines not too long..

tx_atom(A) :-
	escape_chars(A,A2),
	write(A2).

escape_chars(A,A2):-
	term_atom(A,A1),
	name(A1,Chars),
	escape_l(Chars,Chars2),
	name(A2,Chars2).

escape_l([],[]).
escape_l([H|T],Out):-
	escape_c(H,Rest,Out),
	escape_l(T,Rest).

% 38: &
% 95: _
% 94: ^ replaced by $\uparrow$
% 92: \ replaced by $\backslash$

% 92: \
escape_c(38,R,[92,38|R]) :- !.
escape_c(92,R,[36,92,98,97,99,107,115,108,97,115,104,36|R]) :-!.
escape_c(94,R,[36,92,117,112,97,114,114,111,119,36|R]) :- !.
escape_c(95,R,[92,95|R]) :- !.

% catch all:
escape_c(C,R,[C|R]).

tex_var(N) :-
	Letter is N mod 26 + 0'A,
	write('\mbox{'),put(Letter),write('}'),
	(   N>=26 ->
	    Rest is N//26, write('_{'),write(Rest),write('}')
	;   true
	).

tex_term(Val) :-
	write_goal(Val,1199,0,_Co).    % cf below!

tex_if :-
	write('{\mbox{\tt :-}}').

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%% T R E E S  with   T R E E M A K E R %%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% for the new Treemaker package we need to do more complex things:
%  - write mother file that calls .tdf file and defines new commands
%  - latex mother file
%  - dctree
%  - write mother file that calls .tex file
%  - latex mother file again
%  - xdvi it


% tree(Label,_,ListOfDs)
%
%
% latex_tree(+Tree)
% latex_tree(+Mode,+Tree)

:- flag(nodeskip,_,45).

latex_tree(Kind,Tree0) :-
	user:graphic_path(Kind,Tree0,Tree1),
	p_tree:tree_def_to_tree(Kind,Tree1,Tree),
	latex_tree(Tree).

latex_tree(Tree0) :-
	flag(fs_tree,off),!,
	flag(vspace,_,off),
	files(TexFile,_,TreeFile,TexFile0,LabelFile),
	telling(Old), tell(LabelFile),
	define_labels(Tree0,Tree),
	told, tell(TexFile),
	input_cmd(LabelFile),
        start_docu(tree), 
        print_tree_second(TreeFile), 
        end_docu,
	told, tell(TexFile0),
	input_cmd(LabelFile),
        start_docu(tree),
        print_tree_first(Tree,TreeFile),
        end_docu,
	told, tell(Old),
        latex_and_dctree_and_latex_and_xdvi.

latex_tree(Tree0) :-
	shorten_tree(Tree0,Tree1),
	prettyvars(Tree1,0,No),
	feature:change(Tree1,Tree2,[],_,No,_),
	flag(vspace,_,off),
	files(TexFile,_,TreeFile,TexFile0,LabelFile),
	telling(Old), tell(LabelFile),
	define_labels_fs(Tree2,Tree),
	told, tell(TexFile),
	input_cmd(LabelFile),
        start_docu(tree), 
        print_tree_second(TreeFile), 
        end_docu,
	told, tell(TexFile0),
	input_cmd(LabelFile),
        start_docu(tree),
        print_tree_first(Tree,TreeFile),
        end_docu,
	told, tell(Old),
        latex_and_dctree_and_latex_and_xdvi.

input_cmd(File) :-
	write('\input{'),write(File),write('}'),nl.

gen_cmd_name(Name) :-
	gen_atom(Name0),
	concat('\',Name0,Name).

% each label is defined as a seperate command - in order to get no errors
define_labels(tree(L0,_,Ds0),tree(L,_,Ds)) :-
	define_label(L0,L),
	define_label_ds(Ds0,Ds).

define_label_ds([],[]).
define_label_ds([H0|T0],[H|T]) :-
	define_labels(H0,H),
	define_label_ds(T0,T).

define_label(L,Cmd) :-
	gen_cmd_name(Cmd),
	write('\long\def'),
	write(Cmd),
	write('{'),
	tree_label(L),      % eg turn into feature structure...
	write('}'),nl.

define_labels_fs(_/_=[a('TREE',_,L0,Ds0)],tree(L,_,Ds)) :-
	define_label_fs(L0,L),
	define_label_ds_fs(Ds0,Ds).

define_label_ds_fs([],[]).
define_label_ds_fs([H0|T0],[H|T]) :-
	define_labels_fs(H0,H),
	define_label_ds_fs(T0,T).

define_label_fs(L,Cmd) :-
	gen_cmd_name(Cmd),
	write('\long\def'),
	write(Cmd),
	write('{'),
	tree_label_fs(L),      % eg turn into feature structure...
	write('}'),nl.

tree_label(A) :-
	user:tree_label(A),
	!.
tree_label(A) :-
	write('$ '), tex_term(A), write(' $'), nl.

tree_label_fs(A) :-
	write_it_fs(fs(A)).

print_tree_first(Tree,File0) :-
	dir0(Path),
	name(Path,PathStr),
	name(File0,File0Str),
	append(PathStr,FileStr,File0Str),
	name(File,FileStr),
	flag(nodeskip,NodeSkip),
	write('\nodeskip{'),
	write(NodeSkip),
	write('pt}'),nl,
	write('\outputquality{low}'),nl,
        write('\opentree{'),write(File),write('}'),nl,
	print_tree(Tree).

print_tree(tree(L,_,[])) :-
	write('\leaf{'),
	write('\noexpand'),write(L),
	write('}'),nl.

print_tree(tree(L,_,[H|T])) :-
	write('\tree{'),
	write('\noexpand'),write(L),
	write('}'),nl,
	print_tree_ds([H|T]),
	write('\endtree'),
	nl.

print_tree_ds([]).
print_tree_ds([H|T]) :-
	print_tree(H),
	print_tree_ds(T).

print_tree_second(File) :-
	write('\input{'),
	write(File),
	write('}'),
	nl.

tt :-
	latex_tree(tree(s,_,[ tree(np,_,[]), tree(vp,_,[tree(v,_,[])])  ])).




shorten_tree(tree(L0,M,Ds0),tree(L,M,Ds)) :-
	shorten_label(L0,L),
	shorten_ds(Ds0,Ds).

shorten_ds([],[]).
shorten_ds([H0|T0],[H|T]) :-
	shorten_tree(H0,H),
	shorten_ds(T0,T).

shorten_goal(M:G0,M:G) :-
	!,
	shorten_goal(G0,G).

shorten_goal(when(C,G0),when(C,G)) :-
	!,
	shorten_goal(G0,G).

shorten_goal(G0,G) :-
	G0 =.. [F|T0],
	 shorten_labels(T0,T),
	 G =.. [F|T].

shorten_goals([],[]).
shorten_goals([H0|T0],[H|T]) :-
	shorten_goal(H0,H),
	shorten_goals(T0,T).

shorten_labels([],[]).
shorten_labels([H0|T0],[H|T]) :-
	shorten_label(H0,H),
	shorten_labels(T0,T).


shorten_label(V0,V) :-
	var(V0),
	!,
	V0=V.
shorten_label(V0,V) :-
	user:shorten_label(V0,V),
	!.
shorten_label(L,L).

% convert only if not already converted..
ensure_prolog_conjunction(A,B) :-
	prolog_conjunction(A,B),
	!.
ensure_prolog_conjunction(A,A).
