%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%% MELLISH' DISJUNCTION AND NEGATION OF ATOMIC TYPES %%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
% package to compile expressions with negation and disjunction into
% 'Mellish-types' (see Computational Linguistics 1988), and vice
% versa.
%
% note that this is NOT a seperate module
% declarations of the form boolean_type(Name,SetofSets)
% eg. boolean_type(agr,[[1,2,3],[sg,pl],[mas,fem,neut]])
%            
%
% after compilation of such boolean types 'compile_boolean_types'
% the following two predicates are useful:
%
% eval_b_type(+Exp,-MellishTerm)
%
% give_boolean_type(+MellishTerm,-Exp)
%
% 
% 
% syntax of Exp:  atom
%                 (Exp & Exp)
%                 ~Exp
%                 (Exp ; Exp)
%
% where all atoms occuring in an expression are taken from the same 
% boolean_type definition

:- use_module(library(lists), [ substitute/4,
	                        select/3,
	                        member/2,
	                        append/3 ] ).

compile_boolean_types:-
	abolish(btype,3),
	( user:boolean_type(Type,ListOfSets),
          ff1(ListOfSets,Ar),
          Ar20 is Ar + 1,
	  ( user:intensional(Type) -> Ar2 is Ar20 + 1 ; Ar2 is Ar20 ),
          functor(Term,Type,Ar2),
          arg(1,Term,0),
          arg(Ar20,Term,1),
          setof(X,poss(X,ListOfSets),List),
          assertz(btype(Type,List,Term)),
	  fail
        ; true).

find_super_type(Atom,Type):-
	atomic(Atom),!,
	user:boolean_type(Type,LList),
	member(List,LList),
	member(Atom,List).
find_super_type(~X,Type):-
	find_super_type(X,Type).
find_super_type(A&_B,Type):-
	find_super_type(A,Type).
find_super_type((A;_B),Type):-
	find_super_type(A,Type).

eval_b_type(Var,Var2):-
	var(Var),!,
	Var = Var2.

eval_b_type(Exp,Sign):-
	find_super_type(Exp,Type),!, %unique
	evl(Exp,Type,Sign),
	\+ Sign = fail.

evl(Exp,Type,Sign):-
	btype(Type,List,Sign),
	eval(Exp,Type,List,Sign),
	!.
evl(_,_,fail).

eval((A&B),Type,_List,Sign):-
	evl(A,Type,Sign),
	evl(B,Type,Sign).

eval((A;B),Type,_List,Sign):-
	evl(~A,Type,Sign1),
	evl(~B,Type,Sign1),
	negate(Type,Sign1,Sign).

eval(~A,Type,_List,Sign):-
	evl(A,Type,Sign1),
	negate(Type,Sign1,Sign).

eval(Atom,_,List,Sign):-
	atomic(Atom),
	eval_atom(List,1,2,Atom,Sign).

eval_atom([],_,_,_,_).
eval_atom([H|T],I,J,Atom,Sign):-
	(  member(Atom,H)
        -> true
        ;  arg(I,Sign,X),
           arg(J,Sign,X)
        ),
	K is J + 1,
	eval_atom(T,J,K,Atom,Sign).

negate(Type,fail,Term):-
	!,
	btype(Type,_,Term).
negate(_,Term,Term2):-
	functor(Term,F,A0),
	functor(Term2,F,A0),
	arg(A0,Term,PosR),
	( var(PosR) -> A is A0 - 1 ; A0 = A ),
	negate1(A,Term,Term2),
	!.
negate(_,_,fail).

negate1(1,_,_):-
	!.
negate1(I,A,B):-
	J is I-1,
	arg(J,A,A1),
	arg(I,A,A2),
	arg(J,B,B1),
	arg(I,B,B2),
	negate2(A1,A2,B1,B2),
	negate1(J,A,B).

negate2(A1,A2,B1,B2) :-
	(  A1 == A2
        -> true
        ; B1 = B2
        ).

poss([],[]).
poss([H|T],[L|L2]):-
	member(H,L),
	poss(T,L2).

ff1([],0).
ff1([H|T],Out):-
	ff([H|T],1,Out).

ff([],In,In).
ff([H|T],In,Out):-
	length(H,I),
	In2 is In * I,
	ff(T,In2,Out).

give_boolean_type(Obj,Exp2):-
	btype(Type,List,Obj),
	compall(List,Obj,fail,Exp,1,2),
	rewrite_disj(Type,Exp,Exp2).

compall([],_Obj,E,E,_,_).
compall([_H|T],Obj,In,Out,I1,I2):-
	arg(I1,Obj,X),
	arg(I2,Obj,Y),
	\+ X == '$VAR'('_'), % ..
        \+ Y == '$VAR'('_'), % to recognize prettyvarred objects
	X==Y,!,
	I3 is I2 + 1,
	compall(T,Obj,In,Out,I2,I3).
compall([H|T],Obj,fail,Out,_I1,I2):-!,
	I3 is I2 + 1,
	compall(T,Obj,[H],Out,I2,I3).
compall([H|T],Obj,In,Out,_I1,I2):-
	I3 is I2 + 1,
	compall(T,Obj,[H|In],Out,I2,I3).

/* examples
t(A):-
	eval_b_type(A,X),
	give_boolean_type(X,Y),
	write(X),nl,
	write(Y),nl,fail.
	
t1(X,Y) :-
	eval_b_type((~2;masc),X),
	give_boolean_type(X,Y).


t2(X,Y) :-
	eval_b_type((2;(masc&sg)),X),
	give_boolean_type(X,Y).


t3(X,Y) :-
	eval_b_type(sg& ~1,X),
	give_boolean_type(X,Y).


t4(X,Y) :-
	eval_b_type(~ ((sg;masc)&(pl;fem)),X),
	give_boolean_type(X,Y).

tf(X,Y) :-
	eval_b_type(~ ((sg;masc)&(pl;fem)),X),
	give_boolean_type(X,Y).


t5(X,Y) :-
	eval_b_type(masc,X),
	give_boolean_type(X,Y).

t6 :- write((3;sg;(pl&1))),nl,t((3;sg;(pl&1))).

boolean_type(agr,[[1,2,3],[sg,pl],[mas,fem,neut]]).
*/

rewrite_disj(Type,P0,P):-
	user:boolean_type(Type,Set),
	remove_unspecified_values(Set,P0,P1),
	write_as_disj(P1,P).

remove_doubles(X,Y):-
	select(V,X,X2),
	member(V,X2),!,
	remove_doubles(X2,Y).
remove_doubles(X,X).

%% 
%% if for some value each possibility is multiplied because
%% it is simply not known
%%
%% [[a,b,c],[x,y]]
%% instead a&x ;b&x ;c&x yields x
remove_unspecified_values(Tuples,I,Out):-
	select(F,I,In),
	find_candidate(Tuples,F,In,In2),!,
	remove_unspecified_values(Tuples,In2,Out).
remove_unspecified_values(_Tuples,I,I).

find_candidate(Tuples,F,In,[F2|Out]):-
	select(L,F,F2),
	find_tuple(L,Tuples,Others),
	remove_others(F,L,Others,In,Out).

find_tuple(L,Tuples,Others):-
	member(Tuple,Tuples),
	select(L,Tuple,Others).

remove_others(_F,_L,[],In,In).
remove_others(F,L,[H|T],In,Out):-
	%% replace(L,H,F,L2),!,
	substitute(L,F,H,L2),!,
	select(L2,In,In2),
	remove_others(F,L,T,In2,Out).

write_as_disj([H],H2):-
	write_as_conj(H,H2).
write_as_disj([H|T],(H2;T2)):-
	write_as_conj(H,H2),
	write_as_disj(T,T2).

write_as_conj([H],H).
write_as_conj([H|T],H&T2):-
	write_as_conj(T,T2).



