:- compile(ops).


:- use_module( library(lists), [ select/3,
	                         member/2,
	                         append/3 ]).
:- use_module( library(flags) ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% A %%%%%%%%%%%%% DEFINITION OF TYPE SYSTEM %%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Each type is specified for: 
% - a conjunction of (exclusive) disjunctions of types (called subtypes)
% - appropriate features (i.e. attributes)
% - constraints (implementation not complete (nor sound, probably) OK, eg try 
%   define_type(x,[],[y],X,X:y => x). infinite loop)
%   not very carefully thought out yet.
%
% a type type(Type,[[A1..An],[B1..Bn],...,[Z1..Zn]],[Att1..Attn]) 
%
% defines a term Type(Ai',Bi',..,Zi',Att1',..,Attn',_)
% 
% for example: type(sign,[[basic,complex],[nominal,verbal]],[mor,sem])
% 
% implies that everything of type sign is represented with a term
% sign(BorC,NorV,Mor,Sem)


% Assumptions:
%
% top has no appropriate features, will always be denoted with Variable
% bottom has no appropriate features, will not be denoted -> failure
% hence top is only specified along one `dimension'!
%
% other type can be further specified along several dimensions, hence
% it can have more than one subtype. 
%
% OR-Subtypes of a type are mutually exclusive!! 
%
% all types describe intensional objects (as in PATR II)
% that is, in the compilation an extra argument position is added
% to which you cannot refer!
%%
% define_type(Name,SubTypes,App,X,Constraint(X)).
%
% appropriate features are only labels, not necc. specified for type
% they can be specified for type by using the extra constraints.
%
% the same reasoning applies for multiple inheritence: you are always a subtype
% of one immediate supertype, other inheritence defined in extra constraints.
%
% unique name assumption: types with same name are the same!
%
% appropriate features: there is a minimal type for which a 
% feature is appropriate
% i.e. if you know that some structure has a feature then you know 
% already something about its type!!!
%
%
% after defining your type system, compile it with the predicate
% type_compiler/0.
%
%
% To pretty print the type system, use the package
% p_type.pl 
%
% One you have compiled the type system, you can use all kinds
% of constraints, cf. below (B).

% some abbreviations...
define_type(top,[Subs],[],_,true) :-
	user:top(Subs).

define_type(Type,Subs,Atts,_,true) :-
	user:type(Type,Subs,Atts).

define_type(Type,[],[],_,true) :-
	user:at(Type).

define_type(Type,Subs,Atts,Var,ConstrOnVar) :-
	user:define_type(Type,Subs,Atts,Var,ConstrOnVar).

x_define_type(A,B,C,D,E) :-
	define_type(A,B,C,D,E).
x_define_type('.',[],[H,T],_,true) :-
	user:list_type(H,T).
x_define_type([],[],[],_,true) :-
	user:list_type(_,_).

reset_type_compiler :-
	abolish(has_type,3),
	abolish(e,3),
	abolish(xtype,4).

type_compiler :-
	reset_type_compiler,
	check_supertype,
	first_phase,   % assert xtype/4, e/3
	add_list_type,
	second_phase,  % assert has_type/3
	third_phase,   % retracts e/3 and asserts e/3 (with constraints instantiated)
	check_unique_types,
	check_unique_atts.

type_compiler.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%% first phase %%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the first phase takes the type definitions and compiles
% an xtype definition for each type. This xtype def contains
% a term that will be used as the internal rep of the type.
% However, not all associated constraints of the type are
% evaluated yet - these remain in the third argument pos. of
% xtype. In the second phase the constraints will be evaluated.
%
% Furthermore definition for e/3 is built which picks out the 
% value of an attribute for a given internal rep. Similarly, 
% associated constraints are not yet evaluated. In the third
% phase they are... 

% Probably something is wrong here, since
% the second phase uses e/3 definitions without constraints..
% Also note that modules & constraints is not taken care of.
% As long as constraints are simple type assignments this all
% seems to work.

% Note that if there weren't any constraints, the first phase
% would be all there is to it.

first_phase :-
	define_type(top,[List],[],_,true),
	assertz( xtype(top,Var,true,Var) ),
	first_phase(List,Term,Term,true,first).

first_phase([],_,_,_,_).
first_phase([H0|T],Term,Top,Cin,ForNf):-
	(  (  define_type(H0,HList,AList,Top,Cons)
	   -> length(HList,HL),
	      length(SubTypes,HL),
	      length(AList,AL),
	      length(Attributes,AL),
	      H0 =.. [H|Type],
	      append(Type,SubTypes,For),
	      append(For,Attributes,NewList0),
	      f_or_nf(ForNf,H,NewList0,NewList),
	      Term =.. [H|NewList],
	      assertz( xtype(H0,Top,(Cin,Cons),SubTypes) ),
	      P is HL + 1,
	      assert_all_e(AList,P,Term,Top),
	      subtypes(HList,1,Term,Top,(Cin,Cons))
	   ;  %%msg(['Type not defined: ',H0,nl])
	       format(user_output,
"Type not defined: ~w~n",[H0])
           ),
	   fail
        ;  first_phase(T,Term,Top,Cin,ForNf)
        ).

% NOTE: extra argument position to represent
% `reentrancy'. This position will never get
% instantiated, nor can constraints refer to it.
% the pretty printer detects such reentrancies.
f_or_nf(nfirst,_Type,L,L).
f_or_nf(first,H,L0,L) :-
	f_or_nf2(H,L0,L).

f_or_nf2(Type,L,L) :-
	user:extensional(Type),!.
f_or_nf2(_Type,L0,L) :-
	append(L0,[_ReentVar],L).       

assert_all_e([],_,_,_).
assert_all_e([H|T],P,Term,Top):-
	( arg(P,Term,El),
	  assertz( e(H,Top,El) ),
	  fail
        ; P2 is P + 1,
	  assert_all_e(T,P2,Term,Top)
        ).

subtypes([],_,_,_,_).
subtypes([H|T],P,Term,Top,C):-
	( arg(P,Term,HArg),
	  first_phase(H,HArg,Top,C,nfirst),
	  fail
        ; P2 is P + 1,
	  subtypes(T,P2,Term,Top,C)
        ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%% second phase %%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the second phase takes all xtype definitions and
% tries to evaluate the constraints of each one. The
% evaluated xtypes are asserted as has_type/3.
second_phase :-
	findall(xtype(Type,Term,Cons,Sub),xtype(Type,Term,Cons,Sub),Xtypes),
	eval_constraints(Xtypes).

% Non-deterministically picks an xtype-def whose constraints
% can be evaluated already. This gives rise to has_type def.
% that may be useful for other constraints! 
% If not all xtypes can have their constraints evaluated, this
% results in an error. This can happen e.g. if type definitions
% are associated with constraints that use that very same type:
% type(x,[],[],X,X:v => x). 
eval_constraints([]).
eval_constraints(List):-
	(  ( select(xtype(A,B,Cons,Sub),List,List2),
	     call(Cons))
	-> assertz(has_type(A,B,Sub)),
	   eval_constraints(List2)
        ;  %%msg(['Could not compile the following types:'|List]),
	   %%nl,
	   format(user_output,
"Could not compile the following types: ~w~n",[List]),
	  fail
        ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%% third phase %%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% after the second_phase the `has_type' is correct (?)
%% however, the `e' is not, because eval_path may
%% introduce types, hence it should do inheritance of
%% the extra constraints...

third_phase :-
	findall(e(A,B,C),retract(e(A,B,C)),List),
	assert_es(List).

assert_es([]).
assert_es([e(A,B,C)|T]):-
	functor(B,Type,_),
	has_type(Type,B,_),
	assertz(e(A,B,C)),
	assert_es(T).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%% add_list_type %%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% for ordinary prolog lists!


add_list_type :-
	user:list_type(Head,Tail),   % defines the attributes to point to head and tail
	!,
	assertz(xtype([],[],true,[])),
	assertz(xtype('.',[_|_],true,[])),
	assertz(e(Head,[H|_],H)),
	assertz(e(Tail,[_|T],T)).
add_list_type.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%% error detection %%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

check_supertype :-
	define_type(T,_,_,_,_),
	(  super_type(T)
	-> fail
	;  %% msg(['error: type ',T,' has no super type',nl]),
           format(user_output,"error: type ~w has no super type~n",[T]),
	   fail
	).
check_supertype.

super_type(top).
super_type(T):-
	define_type(_,L,_,_,_),
	member(S,L),
	member(T,S).

% to detect some errors:
check_unique_types :-
	define_type(T,_,_,_,_),
	(  unique_type(T)
	-> fail
	;  %%msg(['error: type ',T,' is not unique',nl]),
	   format(user_output,"error: type ~w is not unique~n",[T]),
	   fail
	).
check_unique_types.

unique_type(T):-
	bagof(X,htype(T,X),[_]).

htype(T,X):-
	has_type(T,X,_).
htype(T,_X):-
	user:boolean_type(T,_).
htype(T,_X):-
	user:boolean_type(_F,M),
	member(S,M),
	member(T,S).

% to detect some errors:
check_unique_atts :-
	e(Att,_,_),
	(  unique_att(Att)
	-> fail
	;  %%msg(['error: attribute ',Att,' is not unique',nl]),
	   format(user_output,"error: attribute ~w is not unique~n",[Att]),
	   fail
	).
check_unique_atts.

unique_att(T):-
	bagof(N,A^e(T,N,A),[_X]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%% B %%%%%%%%%%%% CONSTRAINTS %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% defines some PATR like equations and paths
%%% note that =>, <=>, ==>, and : are newly defined operators
%%%
%%% Path is
%%%    Term:Att0: .. : Attn (evaluates to the value of Att1..Attn in Term)
%%%    @Macro (i.e. Prolog relation, see below)
%%%    `Type  (evaluates to the internal rep. of Type)
%%%              note that type may also be from libraray(mellish)
%%%    =UntypedPrologTerm (evaluates to UntypedPrologTerm)
%%%    PrologTerm that is not one of the previous (evaluates to itself)
%%% 
%%% Patha <=> Pathb (evaluates Patha and Pathb and unifies the results)
%%%
%%% Path => Type    =def   Path <=> `Type
%%% Path ==> Val    =def   Path <=> =Term
%%% Path =*> Macro  =def   Path <=> @Macro
%%%
%
% a `macro' is a Prolog call minus its first argument
% the value of this first argument is what the macro
% evaluates to!
% for example @np evaluates to X, where user:np(X).
%
% find_type(?InternalRep,-Types)
% gives back the most informative types of InternalRep
% THIS IS CLEARLY META-LOGICAL
%
% unify_except(?Rep1,?Rep2,Path)
% unifies the internal reps Rep1 and Rep2, except for the value of
% the path Path.
% IS THIS LOGICAL?
%
% overwrite(?Rep1,?Rep2,Path,Val)
% unifies the internal reps Rep1 and Rep2, except for the value of
% the path path. Moreover, in Rep2 the value of Path is assigned Val.
% IS THIS LOGICAL?
%
% subtype(?Rep,+Type,?SubType(s))
% is true if Rep is of type Type and all info embedded under
% this type is ?SubType(s).
%
%
% To pretty print internal representations, use the package
% p_feature.pl tex_feature.pl

Path => Type :-
	eval_path(Path,Val),
	eval_type(Type,Val).

Path ==> Val :-
	eval_path(Path,Val).

Path1 <=> Path2 :-
	eval_path(Path1,Val),
	eval_path(Path2,Val).

Path1 =*> Call0 :-
	eval_path(Path1,Val),
	eval_macro(Call0,Val).

% eval_macro(+Macro,?Val)
eval_macro(M0,Val) :-
	M0 =.. [F|Tail],
	M =.. [F,Val|Tail],
	user:M.

% eval_type(+Type,?Val)
eval_type(Type,Val):-
	has_type(Type,Val0,_),
	!,
	Val = Val0.

eval_type(Type,Val):-
	btype(Type,_,Val0),
	!,
	Val0=Val.

eval_type(Exp,Val):-
	eval_b_type(Exp,Val0),
	!,
	Val = Val0.

/*
eval_type([],_) :- !.
eval_type([H|T],Val):-
	!,
	eval_type(H,Val),
	eval_type(T,Val).
*/
eval_type(NoType,_) :-
%	msg(['error: ',NoType,' is not a proper type',nl]),
	format(user_output,"error: ~w is not a proper type~n",[NoType]),
	fail.

% eval_path(?Path,?Val)

eval_path(Var,X):-  % path consisists of variable only
	var(Var),!,
	Var = X.

eval_path(@Macro,Val) :-
	!,
	eval_macro(Macro,Val).

eval_path(`Type,Val) :-
	!,
	eval_type(Type,Val).

eval_path(=Term,Term0) :-
	!,
	Term=Term0.

eval_path(Obj:Path,Val):-  % path starts with (instantiated) variable
	!,
	eval_path(Path,Obj,Val).

eval_path(X <=> Y, Val):-  % to allow A <=> B <=> C.
	!,
	eval_path(X,Val),
	eval_path(Y,Val).

eval_path(X,Y):-
	user:user_defined_eval(X),!,
	user:user_eval(X,Y).

eval_path(X & Y,Val) :-
	!,
	eval_path(X,Val),
	eval_path(Y,Val).

% catch_all
eval_path(X,X).   % path consists of instantiated variable only

% eval_path/3
eval_path(Var,_,_) :-
	var(Var),!,
	write_list(['error: variable path',nl]),
	fail.
eval_path(PH:PT,Obj,Val):-
	e(PH,Obj,V),
	eval_path(PT,V,Val).

eval_path(PH,Obj,Val):-
	atomic(PH),
	e(PH,Obj,Val).

%unify_except_l(?Node,?Node,ListofPaths).
unify_except_l(Node,Node,[]).
unify_except_l(N1,N2,[H|T]):-
	unify_except(N1,X,H),
	unify_except_l(X,N2,T).

unify_except(N1,N2,Path) :-
	N1:Path <=> _,      % make Path defined
        N2:Path <=> _,      % make Path defined
	Var:Path <=> Val,   % Path should make sense,
        Val = '***',        % *** remembers which part should not be unified
        unify_except2(N1,N2,Var).

unify_except2(N1,N2,N3):-
	var(N3),!,
	N1 = N2.
unify_except2(_,_,'***') :-
	!.
unify_except2(N1,N2,_):-
	var(N1),
	var(N2),!,
	N1 = N2.
unify_except2(N1,N2,N3):-
	var(N1),!,
	unify_except2(N2,N1,N3).
unify_except2(N1,N2,N3):-
	functor(N1,F,Ar),
	functor(N2,F,Ar),
	functor(N3,F,Ar),
	unify_except3(N1,N2,N3,Ar,0).

unify_except3(_,_,_,I,I):-
	!.
unify_except3(N1,N2,N3,I,J):-
	arg(I,N1,A1),
	arg(I,N2,A2),
	arg(I,N3,A3),
	unify_except2(A1,A2,A3),
	I2 is I-1,
	unify_except3(N1,N2,N3,I2,J).

% overwrite(?InFS,?OutFS,+Path,+Type)
overwrite(FS,FS2,Path,Type) :-
	unify_except(FS2,FS,Path),
	FS2:Path => Type.

% find_type(Term,MinimalTypes)
% find_type(Term,MinimalTypes,Attributes)

find_type(Var,Top):-
	var(Var),!,
	Top = [top].
find_type(Term,T):-
	find_type(Term,T,_Atts).

% catches some cases of untyped that look typed:

find_type([],[[]],[]):-
	user:list_type(_,_).
find_type([_|_],['.'],[H,T]):-
	user:list_type(H,T).
find_type(Term,_Ts,_):-
	functor(Term,F,_),
	x_define_type(F,_,_,_,_),
	x_define_type(top,S,_,_,_),
        \+ mem_mem(F,S),!,fail.
% otherwise:
find_type(Term,Ts,Atts):-
	find_type(Term,[top],Ts,[],Atts).  

mem_mem(F,S):-
	member(L,S),
	member(F,L),!.

find_type(Var,T,T,A,A):-
	var(Var),!.
find_type('$VAR'(_),T,T,A,A):-!.
find_type(Term,T1,T,A1,A):-
	Term =.. [Fun|Ar],
	x_define_type(F,Subs,Atts,_,_),
	F =.. [Fun|Begin],
	append(Begin,_Rest,Ar),
	replace_type(F,T1,T2),
	append(A1,Atts,A2),
	find_types(Subs,Term,T2,T,A2,A,1).

find_types([],_,T,T,A,A,_):-
	!.
find_types([_H|Ta],Term,T1,T,A1,A,P):-
	arg(P,Term,V),
	find_type(V,T1,T2,A1,A2),
	P2 is P + 1,
	find_types(Ta,Term,T2,T,A2,A,P2).

replace_type(New,Types,[New|Res]):-
	select(T,Types,Res),
	x_define_type(T,ConjSubs,_,_,_),
	member(Subs,ConjSubs),
	member(New,Subs),!.
replace_type(New,T,[New|T]).   % in case top type was already removed..

/* not used
%% map_extract(Path,Set,Dset).
map_extract(_Path,[],[]).
map_extract(Path,[H|T],[H2|T2]):-
	H:Path <=> H2,
	map_extract(Path,T,T2).
*/

% subtype(?Rep,+Type,?SubType(s))
% is true if Rep is of type Type and all info embedded under
% this type is ?SubType(s).

subtype(Rep,Type,Sub) :-
	has_type(Type,Rep,Sub).

% unify_except_type(?Rep0,+Type,Rep).
% is true if Rep0 is of type Type and all info embedded under
% this type is forgotten in Rep.
unify_except_type(Rep0,Rep,Type) :-
	Rep0 => Type,     % 
	Rep => Type,      % 
	has_type(Type,_,Lt),   % in order to know how many types there are
	Rep0 =.. [_|L0],
	Rep  =.. [_|L],
	forget(Lt,L0,L).

forget([],Rep,Rep).
forget([_|T],[_|R0],[_|R]):-
	forget(T,R0,R).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% C %%%%%%%%%%% partial evaluation facility %%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% this allows `functional' use of feature expressions in head and 
% body of clauses. To put this to work, define
% term_expansion(Clause0,Clause) :- 
%          eval_funct(Clause0,Clause).
%
%
% For example, given an appropriately compiled type system consisting of
%
% define_type(top,[[list,..]],[],_,true).
% define_type(list,[[elist,nelist]],[],_,true).
% define_type(elist,[],[],_,true).
% define_type(nelist,[],[f,r],_,true).
%
% the following definition 
%
% append(Elist,List,List) :-
%    Elist <=> `elist.
% append(HT,X,HT2) :-
%    HT:f <=> HT2:f,
%    HT:r <=> T,
%    HT2:r <=> T2,
%    append(T,X,T2)
%
% can now be abbreviated as:
%
% append(`elist,List,List).
% append(HT,X,HT2) :-
%    HT:f <=> HT2:f,
%    append(HT:r,X,HT2:r).
%
%
% so anywhere in a definition where a Path occurs, this Path
% is evaluated.
% X:Path
% `Type
% @Macro
% {X} evaluates to X (escape)
%  =Term
% Path & Path
% 
%
%
% ONLY MAKES SENSE FOR PURE PROLOG DEFINITIONS
% and not for definitions using disjunction, etc.
%
%

:- user:del_expansion(feature:eval_head_body).
	
eval_head_body(Head0,Body0,Head,Body) :-
	flag(eval_feature,on),
	evalt(Head0,Head),
	eval_body(Body0,Body).

evall([],[]).
evall([H0|T0],[H|T]):-
	evalt(H0,H),
	evall(T0,T).

evalt(Var0,Var) :-
	var(Var0),!,
	Var=Var0.
evalt(Term0,Term) :-
	functor(Term0,Fun,Ar),
	evalt1(Fun,Ar,Term0,Term).

evalt1(Fun,Ar,Term0,Term) :-
	a_eval_op(Fun/Ar,Type),!,
	eval_op(Type,Term0,Term).
evalt1(Fun,Ar,Term0,Term) :-
	Term0 =.. [Fun|L0],
	length(L,Ar),
	Term  =.. [Fun|L],
	evall(L0,L).

eval_body(Body0,Body) :-
	eval_body_l(Body0,Body).
eval_body({X},X).

eval_body_l(V0,V) :-
	var(V0),!,
	V0=V.
eval_body_l([],[]).
eval_body_l([H|T],Out) :-
	eval_goal(H,T2,Out),
	eval_body_l(T,T2).

eval_goal({H},T,[H|T]) :-
	!.
eval_goal(Call0,T,T) :-
	a_call(Call0,Call),!,
	call(Call).
eval_goal(Term,T,[NewTerm|T]) :-
	evalt(Term,NewTerm).

a_call(C,C) :-
	eval_a_call(C,no).
a_call(C0,C) :-
	eval_a_call(C0,yes),
	evalt(C0,C).

a_call(C,user:C) :-
	user:eval_a_call(C,no).
a_call(C0,user:C) :-
	user:eval_a_call(C0,yes),
	evalt(C0,C).

eval_a_call( _ ==> _ , no).
eval_a_call( _ <=> _ , no).
eval_a_call( _  => _ , no).
eval_a_call(unify_except(_,_,_), yes).       % for third argument should be no
eval_a_call(unify_except_l(_,_,_), yes).     %               ,,
eval_a_call(overwrite(_,_,_,_), yes).

%%eval_a_call(Term,Yn) :-
%%	user:eval_a_call(Term,Yn).

a_eval_op('{}'/1,escape). % escape
a_eval_op(':'/2,eval).  % `real' path
a_eval_op('`'/1,eval).  % type
a_eval_op('@'/1,eval).  % macro
a_eval_op('&'/2,eval).  % conjunction of paths
a_eval_op('='/1,eval).  % untyped prolog term
a_eval_op('<=>'/2,eq). % equation
a_eval_op( '=>'/2,eq). % ,,
a_eval_op('==>'/2,eq). % ,,
a_eval_op(F/Ar,eval):-
	functor(Term,F,Ar),
	user:user_defined_eval(Term).

eval_op(escape,{T0},T) :-      % escape
	T0=T.
eval_op(eval,Term0,Term) :-       % so Term0 is a `path'
	Term0 <=> Term.        % hence evaluates to Term
eval_op(eq,A <=> B,Term) :-
	A <=> B <=> Term.
eval_op(eq,A => B, Term) :-
	A => B,
	lvar(A,X),
	X <=> Term.
eval_op(eq,A ==> B, Term) :-
	A ==> B,
	lvar(A,X),
	X <=> Term.

lvar(Var0,Var) :-
	var(Var0),
	!,
	Var0=Var.
lvar(Var0:_,Var) :-
	!,
	Var0=Var.
lvar(Term,Term).



:- user:add_expansion(feature:eval_head_body).

