:- use_module( library( flags )).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%% pretty printing of trees %%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% datastructure for derived trees is as follows:
%% tree(Mother,Marker,ListOfDaughters)
%% where in the case of a terminal symbol Term
%% Mother = lex(Term) and
%% ListOfDaughters = []
%%
%% Marker is sometimes used to indicate whether
%% the node is a substitution node (`subs') or
%% a foot node (`foot'). If not it is variable.

%% The Marker position is also used to represent
%% the derivation tree of a parse. Cf. below.

:- use_module(library(lists), [ nth/3,
	                        reverse/2,
	                        member/2,
	                        append/3 ] ).

:- del_graphic(t_path,t_label,t_daughter).

t_path(V,X,X) :-
	user_show(V,_,_).

t_label(V,tree(Cat,Kind,_),C) :-
	user_show(V,Cat,C0),
	add_kind(Kind,C0,C).
t_label(V,tree(lex(Cat0),_,_),Cat) :-
	del_index(Cat0,Cat),
	user_show(V,_,_).

t_daughter(V,P,tree(_,_,Ds),El):-
	user_show(V,_,_),
	nth(P,Ds,El).

del_index(W0/_I,W):-
	!,
	W0 = W.
del_index(W,W).

add_kind(Var,C0,C) :-
	var(Var),!,C0=C.
add_kind(subs,C,*C).
add_kind(foot,C,=C).
add_kind(_,C,C).

:- add_graphic(t_path,t_label,t_daughter).

%% datastructure for derived trees is
%% dtree(Name,Address,ListOfDaughters

:- del_graphic(d_t,d_l,d_d).

d_t(dt,tree(_,D,_),D).

d_l(dt,dtree(Name,Address,_),Label) :-
	(  var(Address)   % root 
        -> Name = Label
        ;  Label = Address:Name
        ).

d_d(dt,P,dtree(_,_,Ds),El) :-
	nth(P,Ds,El).

:- add_graphic(d_t,d_l,d_d).

:- usr_cmd([alias,d,s,dt]).
:- usr_cmd([alias,dg,s,dt,gm]).

%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% for pp parsers: %%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%

:- del_portray(pr).

pr(Node) :-
	user_syn(Node,_),
	gm_show(n(Node),N),
	write(N).

pr(Exp0) :-
	gm_show(Exp0,Exp),
	write(Exp).

:- add_portray(pr).

pred_to_edge(_:parse(Cat,P0,P,_,_,_),Cat,[],P0,P).

% is this still used??
gm_show(subs(Node),*L):-
	user:user_show(syn,Node,L).
gm_show(lex(Word/_),Word).
gm_show(lex_head(Word/_,ToParse),L):-
	gm_show(ToParse,Word,L).
gm_show(subs_head(Cat,ToParse),L):-
	gm_show(subs(Cat),C),
	gm_show(ToParse,C,L).

gm_show(e_head(Cat,ToParse),L):-
	gm_show(subs(Cat),C),
	gm_show(ToParse,C,L).

gm_show(current(Cat,ToParse),L):-
	gm_show(Cat,C),
	gm_show(ToParse,+C,L).

gm_show([],H,H).
gm_show([t(A,L,R)|T],Head,Res):-
	gm_show(n(A),NewHead),
	gm_show_t(L,L1),
	reverse(L1,Lefties),
	gm_show_t(R,Righties),
	append(Lefties,[Head|Righties],Daughters),
	NH =.. [NewHead|Daughters],
	gm_show(T,NH,Res).

gm_show_t([],[]).
gm_show_t([H|T],[NH|NT]):-
	gm_show(H,NH),
	gm_show_t(T,NT).

gm_show(lex(W),W).

gm_show(n(Node),L):-
	user:user_show(syn,Node,L).

gm_show(Node,L):-
	user:user_show(syn,Node,L).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%% show elementary trees %%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- del_command(tag_cmd).

tag_cmd(show:sh(Type,Output,Things)) -->
	[w],
	show:type(Type),
	show:output(Output),
	sequence_of_terms(Words,+),
	{ sel_words(Words,Things) }.

:- add_command(tag_cmd).

sel_words(Ws,Ts) :-
	findall(T,(member(W,Ws),sel_word(W,T)),Ts).

sel_word(H,object(Name,o(Obj,_,_))) :-
	ele(H,Name,Obj).

% key is either a terminal symbol or the rule name
ele(Word,initial(Name),Tree) :-
	ini(Word,Tree,Name).

ele(Word,auxiliary(Name),Tree) :-
	au(Word,Tree,Name).

ele(Name,initial(Name),Tree) :-
	ini(Name,Tree).

ele(Name,auxiliary(Name),Tree) :-
	au(Name,Tree).

ini(Word,Tree,Name) :-
	init_rule(_,Word,List,Name),
	to_tree(List,tree(lex(Word),none,[]),Tree).

ini(Name,Tree) :-
	init_rule(_,Word,List,Name),
	to_tree(List,tree(lex(Word),none,[]),Tree).

au(Word,Tree,Name) :-
	aux_rule(Foot,_,List,Words,Name),
	member(Word/_,Words),
	to_tree(List,tree(Foot,foot,[]),Tree).

au(Name,Tree) :-
	aux_rule(Foot,_,List,_,Name),
	to_tree(List,tree(Foot,foot,[]),Tree).

to_tree([],T,T).
to_tree([t(Node,L,R)|Rest],T0,T) :-
	to_tree_ls(L,[T0|Rds],Ds),
	to_tree_rs(R,Rds),
	to_tree(Rest,tree(Node,none,Ds),T).

to_tree_rs([],[]).
to_tree_rs([H|T],[H0|T0]):-
	to_tree(H,H0),
	to_tree_rs(T,T0).

to_tree_ls([],I,I).
to_tree_ls([H|T],I0,I):-
	to_tree(H,H0),
	to_tree_ls(T,[H0|I0],I).

to_tree(lex_head(Word,Chain),Tree) :-
	to_tree(Chain,tree(lex(Word),none,[]),Tree).

to_tree(subs_head(Subs,Chain),Tree) :-
	to_tree(Chain,tree(Subs,subs,[]),Tree).

to_tree(e_head(Node,Chain),Tree) :-
	to_tree(Chain,tree(Node,none,[]),Tree).


tree_label(FS) :-
	latex:print_it_fs(fs(FS)).

:- initialize_flag(nodeskip,150).

:- flag(show_object,_,No^show_object_canvas_tree(semx,No)).
