:- module(right_chart_array,[]).

:- use_module(library(flags)).
:- use_module(library(between), [ between/3 ]).

:- add_flag(parser_mode,right_chart_array).

% inactive chart parser. 
%
% Does not check for more/less general items on the
% chart.
%
% Does not uses Prolog database for edges, but percolates an extra argument:
% the chart that is gradually instantiated.
%
% The chart is a two-dimensional n x n array. The elements of the array
% are variable-tail-lists of categories. The chart thus gradually
% becomes further instantiated by adding edges.
%
% for sentence of three words [john,kisses,mary], we have:
%
% chart(B,C,D) where B, C and D represent the edges _ending_ in resp. position
% 1, 2 and 3 (no edges end in 0)
% each of these have the form
% state(W,X,Y) where W, X, Y represent the edges _beginning_ in resp. position
% 0, 1, 2 (no edges start in 3)
%
% A complete chart for this example may consist of (note that lexical categories
% are not explicitly encoded on the chart in the parser, in this example they are
% for illustration)
%
% chart(
%       state([np|_],_,_),                      % john 0-1
%       state(_,[v|_],_),                       % kisses 1-2
%       state([s|_],[vp|_],[np|_])).            % mary 2-3, kisses mary 1-3, john kisses mary 0-3
%
% Objective: do not use assert/retract. Especially with the use of constraints this
% saves a lot of administration. We should get more benefit from Prolog's compilation.
% Further possibilities: add more indexing in the chart, e.g. indexing by basic syntactic
% category. This techniques allows to add more indexing..
%
%
% Evaluation: seems to be slightly slower after all. Might be due to explicit searching
% (essentially `member') in a list of categories in the chart, 
% rather than use Prolog's built-in indexing and search

clean.
count.
count(size).

parse(o(Node,String,_)) :-
	length(String,Max),
	new_chart(Max,Chart),
	bu_lr(0,Max,Chart),
	size(Chart),
	member_chart(Node,0,Max,Chart).

bu_lr(I,I,_C) :- !.
bu_lr(I0,I,C):-
	findall(Edge,lookup(I0,Edge),Agenda),
	process(Agenda,C),
	I1 is I0 + 1,
	bu_lr(I1,I,C).

lookup(I0,inactive(M,I0,I)):-
	user:lex(I0,I,M,_Name).

% process(+ListOfEdges)
% process a list of edges
process([],_C). 
process([Edge|OldAgenda],C):-
	findall(NewEdge,completer(Edge,NewEdge,C),NewEdges),
	add_list_chart(NewEdges,C),
	process(NewEdges,C),
	process(OldAgenda,C).

% completer(+Cat,+P1,+P,-Edge)
% selects rule whose rightmost symbol matches Cat
completer(inactive(Cat,P1,P),inactive(M,P0,P),Chart):-
	user:r_rule(Cat,M,Lefties,_MR),
	select_lefties(Lefties,P0,P1,Chart).

select_lefties([],P,P,_).
select_lefties([H|T],P0,P,Chart):-
	an_inactive(P,P1,H,Chart),
	select_lefties(T,P0,P1,Chart).

an_inactive(P,P0,H,Chart) :-
	member_chart(H,P0,P,Chart).
an_inactive(P,P0,H,_) :-
	user:lex(P0,P,H,_).

add_list_chart([],_C).
add_list_chart([inactive(Cat,P0,P)|T],C) :-
	add_chart(Cat,P0,P,C),
	add_list_chart(T,C).

add_chart(Cat,P0,P,Chart) :-
	arg(P,Chart,StateSet),
	P1 is P0 + 1,
	arg(P1,StateSet,Set),
	add(Cat,Set).

member_chart(Cat,P0,P,Chart) :-
	arg(P,Chart,StateSet),
	var_arg(P0,StateSet,Set),
	member_set(Cat,Set).

var_arg(P0,Term,Arg) :-
	nonvar(P0),
	!,
	P is P0 + 1,
	arg(P,Term,Arg).
var_arg(P0,Term,Arg) :-
	functor(Term,_,Ar),
	between(1,Ar,P),
	arg(P,Term,Arg),
	P0 is P-1.

member_set(_,Var) :-
	var(Var),!,
	fail.
member_set(Cat,[Cat|_]).
member_set(Cat,[_|T]):-
	member_set(Cat,T).

add(Cat,Var) :-
	var(Var),!,
	Var = [Cat|_].
add(Cat,[_|T]) :-
	add(Cat,T).

new_chart(M,Term) :-
	functor(Term,chart,M),
	fill_new_chart(Term,M,M).

fill_new_chart(_Term,0,_):-
	!.  
fill_new_chart(Term,A0,Max):-
	arg(A0,Term,StateSet),
	functor(StateSet,state,Max),
	A is A0-1,
	fill_new_chart(Term,A,Max).

size(Chart) :-
	functor(Chart,_,Ar),
	size(Chart,1,Ar).

size(_Chart,P,Q) :-
	P > Q,!.
size(Chart,P0,P) :-
	arg(P0,Chart,Set),
	size_set(Set,1,P),
        nl,
        P1 is P0 + 1,
	size(Chart,P1,P).

size_set(_Set,Q,P) :-
	Q > P,!.
size_set(Set,P0,P) :-
	arg(P0,Set,List),
	vlength(List,0,L),!,
	( L =:= 0 -> tab(4) ; write(L), tab(3) ),
	P1 is P0 + 1,
	size_set(Set,P1,P).

vlength(Var,X,Y) :-
	var(Var),!,
	X=Y.
vlength([_H|T],X,Y):-
	X1 is X + 1,
	vlength(T,X1,Y).


