:- module(sikkel,[]).

:- use_module([library(database),
	       library(flags)]).

:- use_module(library(lists), [ append/3 ]).

:- use_module( library(between), [ between/3 ]).

:- use_module( library(count_edges), 
	[ count_edges/2,
	  report_count_edges/1 ]).


% an Sikkel parser for CFG
% * no look-ahead
% * bottom-up, head-driven
% * depth-first
% * parser

:- add_flag(parser_mode,sikkel).


clean :-
	clean_up_database(sikkel:predict_item(_l,_r,_A)),
	clean_up_database(sikkel:dd_item(_Lhs,_Lrhs,_Rrhs,_i,_j,_Rule,_Hist)),
	clean_up_database(sikkel:cyk_item(_a,_j_1,_j)),
	clean_up_database(sikkel:tt(_,_,_,_)).

parse(o(Node,String,_)) :-
	flag(ticks,_,0),
	length(String,Max),
	store(predict_item(is/0,is/Max,Node),Edge),
	process([Edge]),
	left_right(String,0,Max),
	packing:apply_semantics(Node,0,Max,sikkel).

left_right([],Max,Max).   % finished
left_right([_|T],I1,Max) :-
	lexical_lookup(I1,Agenda),   % scan next one
	process(Agenda),         % process resulting edges
	I2 is I1 + 1,            
	left_right(T,I2,Max).   % move to the right in sentence

lexical_lookup(I0,Agenda) :-
	findall(Edge,lookup(I0,Edge),Agenda).

lookup(I0,Edge):-
	user:ign_lex(I0,I,M,Name),
	store(dd_item(M,[],[],I0,I,[],Name),Edge).

% process(+ListOfEdges)
% process a list of edges
process([]).
process([Edge|OldAgenda]):-
	process_one(Edge,SubAgenda),         %% process each one in turn
	append(SubAgenda,OldAgenda,Agenda),  %% depth first!
	process(Agenda).                     %% process the rest

% process_one(+Edge,-Agenda)
% depending on the form of Edge, builds all new edges from Edge 
% by prediction or completion

process_one(predict_item(A,B,C),Agenda) :-
	findall(Edge1,head_corner(predict_item(A,B,C),Edge1), Agenda1),
	findall(Edge2,predict(    predict_item(A,B,C),Edge2), Agenda2),
	findall(Edge3,complete(   predict_item(A,B,C),Edge3), Agenda3),
	append(Agenda1,Agenda2,Agenda12),
	append(Agenda12,Agenda3,Agenda).

process_one(dd_item(A,B,C,D,E,F,G),Agenda) :-
	findall(Edge1,predict(    dd_item(A,B,C,D,E,F,G), Edge1), Agenda1),
	findall(Edge2,complete(   dd_item(A,B,C,D,E,F,G), Edge2), Agenda2),
	append(Agenda1,Agenda2,Agenda).

process_one(cyk_item(A,B,C), Agenda) :-
	findall(Edge1,head_corner(cyk_item(A,B,C),Edge1), Agenda1),
	findall(Edge2,complete(   cyk_item(A,B,C),Edge2), Agenda2),
	append(Agenda1,Agenda2,Agenda).

% case a: we found some X from i to j
%         furthermore X is the head of a rule with mother B
%         furthermore B can be linked to a goal A we already had
head_corner(cyk_item(X,I,J), Edge) :-
	user:ign_h_rule(X,B,Alfa,Beta,Name),
	predict_item(Lop/L,Rop/R,A),
	L =< I,I =< J,J =< R,
	head_relation(B,A),
	hlink(x(X,I,J),x(A,Lp,Rp),Lp,Lop,L,Rp,Rop,R),
	store(dd_item(B,Alfa,Beta,I,J,[t(I,J,X)],Name),Edge).

% case a: we found some X from i to j
%         furthermore X is the head of a rule with mother B
%         furthermore B can be linked to a goal A we already had

head_corner(predict_item(Lop/L,Rop/R,A), Edge) :-
	cyk_item(X,I,J),
	L =< I,I =< J,J =< R,
	user:ign_h_rule(X,B,Alfa,Beta,Name),
	head_relation(B,A),
	hlink(x(X,I,J),x(A,Lp,Rp),Lp,Lop,L,Rp,Rop,R),
	store(dd_item(B,Alfa,Beta,I,J,[t(I,J,X)],Name),Edge).

% case b: we have a goal A between l and r
%         furthermore, B is a gap
%         furthermore, B can be linked to A
%         hence B occurs anywhere between l and r
head_corner(predict_item(_/L,_/R,A), Edge) :-
	user:ign_h_gap(B,Name),
	head_relation(B,A),
	between(L,R,J),
	store(dd_item(B,[],[],J,J,[],Name),Edge).

% case c: we have a goal A between l and r
%         and we have a dd-item whose mother node B can be linked to A
%         then we predict the daughter left of the left-dot to occur

predict(predict_item(_/L,_/R,A), Edge) :-
	dd_item(B,AlfaC,_Gamma,I,J,_His,_Rule),
	head_relation(B,A),
	L =< I,I =< J,J =< R,
	append(_Alfa,[C],AlfaC),
	store(predict_item(lr/L,is/I,C),Edge).

% case d: we have a goal A between l and r
%         and we have a dd-item whose mother node B can be linked to A
%         then we predict the daughter right of the right-dot to occur

predict(predict_item(_/L,_/R,A), Edge) :-
	dd_item(B,_Alfa,[C|_Gamma],I,J,_His,_Rule),
	head_relation(B,A),
	L =< I,I =< J,J =< R,
	store(predict_item(is/J,sm/R,C),Edge).

% case c: we have a goal A between l and r
%         and we have a dd-item whose mother node B can be linked to A
%         then we predict the daughter left of the left-dot to occur


predict(dd_item(B,AlfaC,_Gamma,I,J,_His,_Rule),Edge):-
	predict_item(_/L,_/R,A),
	head_relation(B,A),
	L =< I,I =< J,J =< R,
	append(_Alfa,[C],AlfaC),
	store(predict_item(lr/L,is/I,C),Edge).

% case d: we have a goal A between l and r
%         and we have a dd-item whose mother node B can be linked to A
%         then we predict the daughter right of the right-dot to occur

predict(dd_item(B,_Alfa,[C|_Gamma],I,J,_His,_Rule),Edge):-
	predict_item(_/L,_/R,A),
	head_relation(B,A),
	L =< I,I =< J,J =< R,
	store(predict_item(is/J,sm/R,C),Edge).

% case e. we have a goal A between l and r
%         and we found some X between i and j
%         and we had a dd-item whose daughter left of the left dot is X
%         and there is a link

complete(predict_item(_Lop/L,_Rop/R,A),Edge) :-
	cyk_item(X,I,J),
	dd_item(B,AlfaX,Gamma,J,K,His,Rule),
	append(Alfa,[X],AlfaX),
	L =< I,I =< J,J =< K,K =< R,
	head_relation(B,A),
	store(dd_item(B,Alfa,Gamma,I,K,[t(I,J,X)|His],Rule),Edge).

% case e. we have a goal A between l and r
%         and we found some X between i and j
%         and we had a dd-item whose daughter left of the left dot is X
%         and there is a link
complete(cyk_item(X,I,J),Edge) :-
	dd_item(B,AlfaX,Gamma,J,K,His,Rule),
	append(Alfa,[X],AlfaX),
	head_relation(B,A),
	predict_item(_Lop/L,_Rop/R,A),
	L =< I,I =< J,J =< K,K =< R,
	store(dd_item(B,Alfa,Gamma,I,K,[t(I,J,X)|His],Rule),Edge).

% case e. we have a goal A between l and r
%         and we found some X between i and j
%         and we had a dd-item whose daughter left of the left dot is X
%         and there is a link
complete(dd_item(B,AlfaX,Gamma,J,K,His,Rule),Edge) :-
	cyk_item(X,I,J),
	append(Alfa,[X],AlfaX),
	head_relation(B,A),
	predict_item(_Lop/L,_Rop/R,A),
	L =< I,I =< J,J =< K,K=<R,
	store(dd_item(B,Alfa,Gamma,I,K,[t(I,J,X)|His],Rule),Edge).

% case f. we have a goal A between l and r
%         and we found some X between i and j
%         and we had a dd-item whose daughter right of the right dot is X
%         and there is a link

complete(predict_item(_Lop/L,_Rop/R,A),Edge) :-
	dd_item(B,[],[X|Gamma],I,J,His,Rule),
	cyk_item(X,J,K),
	head_relation(B,A),
	L =< I,I =< J,J =< K,K=<R,
	append(His,[t(J,K,X)],His2),
	store(dd_item(B,[],Gamma,I,K,His2,Rule), Edge).

% case f. we have a goal A between l and r
%         and we found some X between i and j
%         and we had a dd-item whose daughter right of the right dot is X
%         and there is a link
complete(dd_item(B,[],[X|Gamma],I,J,His,Rule),Edge) :-
	predict_item(_Lop/L,_Rop/R,A),
	cyk_item(X,J,K),
	head_relation(B,A),
	L =< I,I =< J,J =< K,K=<R,
	append(His,[t(J,K,X)],His2),
	store(dd_item(B,[],Gamma,I,K,His2,Rule), Edge).

% case f. we have a goal A between l and r
%         and we found some X between i and j
%         and we had a dd-item whose daughter right of the right dot is X
%         and there is a link
complete(cyk_item(X,J,K),Edge) :-
	dd_item(B,[],[X|Gamma],I,J,His,Rule),
	predict_item(_Lop/L,_Rop/R,A),
	head_relation(B,A),
	L =< I,I =< J,J =< K,K=<R,
	append(His,[t(J,K,X)],His2),
	store(dd_item(B,[],Gamma,I,K,His2,Rule), Edge).

% store(+Edge,-Edge)
% stores an edge, and depending on form gives different representation
% of the same edge back. If the edge already exists, then failure.
store(_,_):-
	flag(ticks,V),
	V2 is V+1,
	flag(ticks,_,V2),
	fail.
store(predict_item(A,B,C), predict_item(A,B,C)) :-
	assertz_most_general(sikkel:predict_item(A,B,C),no).

store(dd_item(A,[],[],D,E,His,Rule), cyk_item(A,D,E)) :-
	!,
	assertz_most_general(sikkel:tt(D,E,Rule,His)),
	assertz_most_general(sikkel:cyk_item(A,D,E),no).

store(cyk_item(A,D,E), cyk_item(A,D,E)) :-
	assertz_most_general(sikkel:cyk_item(A,D,E),no).

store(dd_item(A,B,C,D,E,H,R), dd_item(A,B,C,D,E,H,R)) :-
	assertz_most_general(sikkel:dd_item(A,B,C,D,E,H,R),no).

head_relation(Cat0,Cat):-
	user:hfc(Cat0,Cat,_,_,_,_),
	user:ign_h_link(x(Cat0,_,_),x(Cat,_,_)).

hlink(X,Y,Lp,Lop,L,Rp,Rop,R) :-
	user:ign_h_link(X,Y),
	match_i(Lop,Lp,L),
	match_i(Rop,Rp,R).

match_i(is,I,I).
match_i(lr,I,J):-
	larger(I,J).
match_i(sm,I,J):-
	larger(J,I).

larger(Var,_):-
	var(Var),!.
larger(_,Var):-
	var(Var),!.
larger(I,J):-
	I >= J.

list :- 
	listing(sikkel:dd_item),
	listing(sikkel:cyk_item),
	listing(sikkel:predict_item),
	listing(sikkel:tt).

count :-
	count_edges(sikkel:dd_item(_,_,_,_,_,_,_),I),
	count_edges(sikkel:cyk_item(_,_,_),J),
	count_edges(sikkel:predict_item(_,_,_),K),
	count_edges(sikkel:tt(_,_,_,_),M),
	flag(ticks,L),
	write('dd_items:      '),write(I),nl,
	write('cyk_items:     '),write(J),nl,
	write('predict_items: '),write(K),nl,
	write('ticks:         '),write(L),nl,
	write('tt:            '),write(M),nl.


count(Total) :-
	count_edges(sikkel:dd_item(_,_,_,_,_,_,_),I),
	count_edges(sikkel:cyk_item(_,_,_),J),
	count_edges(sikkel:predict_item(_,_,_),K),
	Total is I + J + K.


pack_rule(Name,Mother,Ds) :-
	user:rule(Mother,Ds,Name).










