;;;  Pict-views are views which cache their image as a PICT,
;;;  and display them by replaying this PICT
(in-package :ccl)

(export '(
          *pict-view* 
          start-picture get-picture draw-picture kill-picture
          save-picture install-picture 
          view-draw-contents erase-view
          pict-size set-pict-size
          move-to move line-to line fill-rect erase-rect paint-rect frame-rect
          pen-position stream-tyo
          
          pen-size set-pen-size pen-mode
          set-pen-mode pen-pattern set-pen-pattern pen-state
          set-pen-state pen-normal

          field-size set-field-size

          *scrolling-window* *scroller-view* scroll-view
          ))

(eval-when (eval compile)
  (require 'records)
  (require 'traps))

(require 'scrollers "ccl;examples:scrollers")
(require 'pict-scrap "ccl;examples:pict-scrap")

(defobject *pict-view* *view*)

(defmacro with-rectangle-arg ((var left top right bottom) &body body)
  "takes a rectangle, two points, or four coordinates and makes a rectangle.
body is evaluated with VAR bound to that rectangle."
  `(rlet ((,var :rect))
     (cond (,bottom
            (rset ,var rect.topleft (make-point ,left ,top))
            (rset ,var rect.bottomright (make-point ,right ,bottom)))
           (,right
            (error "Illegal rectangle arguments: ~s ~s ~s ~s"
                   ,left ,top ,right ,bottom))
           (,top
            (rset ,var rect.topleft (make-point ,left nil))
            (rset ,var rect.bottomright (make-point ,top nil)))
           (t (setq ,var (pointer-arg ,left))))
     ,@body))

(defun pointer-arg (thing)
  (if (pointerp thing)
      thing
      (error "Argument: ~a is not a Macintosh pointer" thing)))

(defun picture-arg (thing)
  (if (handlep thing)
      thing
      (error "Argument: ~a is not a Macintosh handle" thing)))

(defun mode-arg (thing)
  (or
   (and (fixnump thing) (<= 0 thing 15) thing)
   (position thing *pen-modes*)
   (error "Unknown pen mode: ~a" thing)))

(proclaim '(object-variable picture wptr pict-size))

(defconstant *pict-length* 10000)

(ask *pict-view* (have 'pict-size (make-point *pict-length* *pict-length*)))

;;;Pictures

(defobfun (view-close *pict-view*) ()
  (when (and (boundp 'picture) picture)
    (kill-picture picture))
  (usual-view-close))

(defobfun (start-picture *pict-view*) ()
  (if (rref wptr window.picsave)
   (error "A picture may not be started for window: ~a.
           since one is already started" (self)))
 (with-rectangle-arg (r 0 0 *pict-length* *pict-length*)
  (with-focused-view (self)
    (_cliprect :ptr r) 
    (have 'my-hPic (_OpenPicture :ptr r :ptr))))
 nil)

(defobfun (get-picture *pict-view*) ()
  (declare (object-variable my-hPic))
 (if (and (boundp 'my-hPic) my-hPic (rref wptr window.picSave))
  (prog1
    my-hPic
    (with-port wptr (_ClosePicture))
    (setq my-hPic nil))
  (error "Picture for window: ~a is not started" (self))))

(defobfun (draw-picture *pict-view*) (picture)
 (picture-arg picture)
 (with-rectangle-arg (r 0 0 *pict-length* *pict-length*)
   (with-focused-view (self)
     (_DrawPicture :ptr picture :ptr r)))
 picture)

(defun kill-picture (picture)
  (_KillPicture :ptr (picture-arg picture)))



(defobfun (save-picture *pict-view*) ()
  (when (and (boundp 'picture) picture)
    (kill-picture picture))
  (have 'picture (get-picture))
  (erase-view)
  (draw-picture picture))

(defobfun (install-picture *pict-view*) (draw-fn)
  (without-interrupts 
   (unwind-protect (progn
                     (start-picture)
                     (funcall draw-fn))
     (save-picture))))

(defobfun (view-draw-contents *pict-view*) ()
  (when (and (boundp 'picture) picture)
    (draw-picture picture))
  (usual-view-draw-contents))

(defobfun (erase-view *pict-view*) ()
  (with-rectangle-arg (r (- *pict-length*) (- *pict-length*) *pict-length* *pict-length*)
    (with-focused-view (self)
      (erase-rect r))))

(defobfun (pict-size *pict-view*) ()
  (objvar pict-size))
          
(defobfun (set-pict-size *pict-view*) (h v)
  (have 'pict-size (make-point h v)))

;From: Espen Jarle Vestre <espen%math.uio.no@RELAY.CS.NET>
;
;Hi!
;I have detected a minor error in the AV Parser which I corrected.  The problem
;is that trees and avms which are copied to the clipboard, are cut to the size o
;the window.  Although I haven't completely understood how _CLIPRECT interacts
;with the WITH-FOCUSED-VIEW macro, I understood that the clipping rectangle set
;a call to _CLIPRECT in the *PICT View* function COPY seems to be overriden, and
;the cause seems to be that the call to DRAW-PICTURE is outside the scope of WIT-
;FOCUSED-VIEW.  So the fix is just to move the call to DRAW-PICTURE inside the
;scope of WITH-FOCUSED-VIEW, which makes the function look like this:

(defobfun (copy *pict-view*) ()
  (if (rref wptr window.picsave)
    (error "A picture may not be started for window: ~a.
           since one is already started" (self)))
  (with-rectangle-arg (r 0 pict-size)
    (with-focused-view (self)
      (_cliprect :ptr r)
      (have 'my-hPic (_OpenPicture :ptr r :ptr))
      (draw-picture picture)))
  (put-scrap :pict (get-picture)))

#|
(defobfun (copy *pict-view*) ()
  "The old copy method for *pict-views*"
  (if (rref wptr window.picsave)
    (error "A picture may not be started for window: ~a.
           since one is already started" (self)))
  (with-rectangle-arg (r 0 pict-size)
    (with-focused-view (self)
      (_cliprect :ptr r) 
      (have 'my-hPic (_OpenPicture :ptr r :ptr))))
  (draw-picture picture)
  (put-scrap :pict (get-picture)))
|#

(defobfun (pen-size *view*) ()
  (rref wptr window.pnsize))

(defobfun (set-pen-size *view*) (h &optional v &aux (pt (make-point h v)))
  (with-port wptr (_PenSize :long pt))
  pt)

(defobfun (pen-mode *view*) ()
  (elt *pen-modes* (rref wptr window.pnmode)))

(defobfun (set-pen-mode *view*) (new-mode)
  (with-port wptr (_PenMode :word (mode-arg new-mode))))

(defobfun (pen-pattern *view*) (&optional (save-pat (make-record :pattern)))
  (copy-record
   (rref wptr window.pnPat) :pattern (pointer-arg save-pat))
  save-pat)

(defobfun (set-pen-pattern *view*) (new-pattern)
  (with-port wptr
  (_PenPat :ptr (pointer-arg new-pattern)))
  new-pattern)

(defobfun (pen-state *view*) (&optional (save-state (make-record :penstate)))
 (with-focused-view (self)
   (_GetPenState :ptr (pointer-arg save-state)))
 save-state)

(defobfun (set-pen-state *view*) (new-state)
  (with-focused-view (self)
    (_SetPenState :ptr (pointer-arg new-state)))
  new-state)

(defobfun (pen-normal *view*) ()
  (with-focused-view (self) (_PenNormal)))

;;;  Here is the quickdraw interface

(defobfun (move-to *pict-view*) (h &optional v)
  (with-port wptr (_MoveTo :long (setq h (make-point h v))))
  h)

(defobfun (move *pict-view*) (h &optional v)
  (with-port wptr (_Move :long (setq h (make-point h v))))
  h)

(defobfun (line-to *pict-view*) (h &optional v)
  (with-port wptr (_LineTo :long (setq h (make-point h v))))
  h)

(defobfun (line *pict-view*) (h &optional v)
  (with-port wptr (_Line :long (setq h (make-point h v))))
  h)

(defobfun (fill-rect *pict-view*) (pattern left &optional top right bot)
  (with-port wptr
    (with-rectangle-arg (r left top right bot)
       (_FillRect :ptr r :ptr (pointer-arg pattern)))))

(defobfun (erase-rect *pict-view*) (left &optional top right bot)
  (with-port wptr
    (with-rectangle-arg (r left top right bot) (_EraseRect :ptr r))))

(defobfun (paint-rect *pict-view*) (left &optional top right bot)
  (with-port wptr
    (with-rectangle-arg (r left top right bot) (_PaintRect :ptr r))))

(defobfun (frame-rect *view*) (left &optional top right bot)
  (with-port wptr
   (with-rectangle-arg (r left top right bot) (_FrameRect :ptr r))))

(defobfun (pen-position *view*) ()
  (rref wptr window.pnloc))

(defobfun (stream-tyo *view*) (char)
  (ask (view-window (self))
    (stream-tyo char)))

(defobfun (window-font *pict-view*) ()
  (ask (view-window (self)) (window-font)))

(defobfun (set-window-font *pict-view*) (font-spec)
  (ask (view-window (self)) (set-window-font font-spec)))

(defobfun (pen-normal *view*) ()
  (with-port wptr (_PenNormal)))


;;;   First some mods to *scroller*

(defobfun (field-size *scroller*) ()
  (objvar field-size))

(defobfun (set-field-size *scroller*) (hmax &optional vmax)
  (let* ((p (make-point hmax vmax))
         (s (view-size))
         (field-size (make-point (max (point-h p) (point-h s)) 
                                 (max (point-v p) (point-v s)))))
    (have 'field-size field-size)
    (set-view-scroll-position #@(0 0))
    (ask (objvar h-scroller) 
      (set-scroll-bar-value 0)
      (set-scroll-bar-max (point-h (objvar field-size))))
    (ask (objvar v-scroller) 
      (set-scroll-bar-value 0)
      (set-scroll-bar-max (point-v (objvar field-size))))))

;;;;  Here's where *scrolling-windows* are defined

(defobject *scrolling-window* *dialog*)
(defobject *scroller-view*  *scroller* *pict-view*)

(defobfun (set-field-size *scroller-view*) (h &optional v)
  (set-pict-size h v)
  (usual-set-field-size h v))

(defobfun (exist *scrolling-window*) (init-list)
  (usual-exist (init-list-default init-list 
                                  :window-type :document-with-zoom
                                  :close-box-p t))
  (have 'scroll-view (oneof (getf init-list :view-class *scroller-view*)
                            :view-container (self)
                            :view-size (subtract-points (window-size) #@(15 15))
                            :view-position #@(0 0)
                            :draw-outline nil
                            :field-size (getf init-list :field-size
                                              (make-point 2000 2000)))))

(defobfun (set-window-size *scrolling-window*) (h &optional v)
  (without-interrupts
   (usual-set-window-size (setq h (make-point h v)))
   (let* ((new-size (subtract-points (window-size) #@(15 15))))
     (ask (objvar scroll-view)
       (set-view-size new-size)))))

(defobfun (window-zoom-event-handler *scrolling-window*) (message)
  (without-interrupts
   (usual-window-zoom-event-handler message)
   (let* ((new-size (subtract-points (window-size) #@(15 15))))
     (ask (objvar scroll-view)
       (set-view-size new-size)))))

(defobfun (copy *scrolling-window*) ()
  (ask (objvar scroll-view) (copy)))

(defobfun (window-close *scrolling-window*) ()
  (ask (objvar scroll-view) (view-close))
  (usual-window-close))

