
;;===========================================================================
;; Subset algorithm for NEITHER
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof.
;; -------------------------------------------------------------------------
;;
;; These routines are used by generalization and sepcialization as part of
;; the bookkeeping necessary to determine which examples are covered by a
;; fix. Specifically, when a fix is made to the theory, both generalization
;; and specialization need to know what examples are related to each element
;; of the fix. This is because a fix is not applied "en-masse", but rather
;; one component at a time. Thus a fix with four components would be applied
;; to the theory in four steps. For each component, we need to know which
;; examples (of those that were covered by the fix) *required* that component
;; of the fix. The "covered-by-subset?" routine gets passed two fixes, and
;; determines which elements of the second fix are necessary to account for
;; the first fix.
;;
;; CHANGE HISTORY
;;
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;===========================================================================
;; FIX SUBSET ROUTINES
;;
;; These next three routines are used to determine if the elements of one
;; fix can be accounted for by a SUBSET of the elements in another fix.
;;===========================================================================


(defun covered-by-subset? (fix-1 fix-2 mark-list)
  "Returns nil if the elements of fix-1 are NOT covered by a subset of the
elements in fix-2. If they are covered by a subset, then mark-list is
updated with 1s in the positions of the required elements and 0s elsewhere
and the routine returns t."
  ;;-------------------------------------------------------------------------
  ;; `fix-2' is checked to see if a subset of its elements will account for
  ;; the elements in `fix-1'. The `mark-list' is assumed to come in to this
  ;; routine from the caller, having been created there for repeated calls to
  ;; this routine with the same value for fix-2. This routine starts, then,
  ;; by resetting all the elements of mark-list to 0. Then each element in
  ;; fix-1 is compared against fix-2. If no covering element can be found
  ;; in fix-2, this routine returns nil. Otherwise, t is returned with the
  ;; mark-list having been updated.
  ;;
  ;; NOTE: the `mark-list' argument is DESTRUCTIVELY MODIFIED.
  ;;-------------------------------------------------------------------------
  (do ((m mark-list (cdr m)))
      ((null m) nil)
    (rplaca m 0))
  (loop for elem in fix-1
	if (not (element-in-fix? elem fix-2 mark-list))
	do (return nil)
	finally (return t)))


(defun element-in-fix? (element fix mark-list)
  "Returns nil if the element is not covered by any of the elements in
the fix. If it is covered, then the mark-list is destructively modified by
changing the element of mark-list in the same position as the element in
fix from 0 to 1."
  ;;-------------------------------------------------------------------------
  ;; `mark-list' is a list of 0s and 1s the same length as the fix, initially
  ;; set to all 0s. If an element is found in `fix' that covers the
  ;; incoming `element' argument, it's 0 is set to 1. Routine returns T if
  ;; element is found, else nil.
  ;;
  ;; NOTE: the `mark-list' argument is DESTRUCTIVELY MODIFIED.
  ;;-------------------------------------------------------------------------
  (loop for elem in fix
	with cur-mark = mark-list
	if (is-covered-by? element elem)
	do (rplaca cur-mark 1) (return t)
	finally (return nil)
	do
	(setf cur-mark (cdr cur-mark))))

  
(defun is-covered-by? (element-1 element-2)
  "Returns T if element-1 is covered by (meaning a subset of) element-2."
  ;;-------------------------------------------------------------------------
  ;; A simple comparison routine, that checks if element-1 is equal to
  ;; element-2. NOTE: This routine is REDEFINED by code which uses the
  ;; covered-by-subset? routine above to customize the comparisons.
  ;;-------------------------------------------------------------------------
  (equal element-1 element-2))
