
;;===========================================================================
;; Rule Retraction Module for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof.
;; -------------------------------------------------------------------------
;;
;; These routines are the analog of "abduce" (see abduce.lisp) but for rule
;; deletion rather than antecedent abduction. The idea is to find a set of
;; rules that, if deleted, will prevent a negative example from being
;; provable. The main routine is "retract-rules".
;;
;; DELETION FORMAT
;; ----------------
;; The return value of "retract-rules" is a FIX (see documentation in file
;; structrues.lisp) which is a LIST of DELETIONS. Note that multiple rule
;; deletions from the same parent vertex would be listed separately in the
;; fix. Said another way, each deletion has one parent and ONE RULE.
;;
;; CHANGE HISTORY
;;
;; 22-JUN-92: (ptb) Fixed a bug in the "retract-child" routine.
;; 22-SEP-92: (ptb) Updated to use structures in "structures.lisp".
;; 23-DEC-92: (ptb) Updated to work for m-of-n rules. Note however that this
;;            file now makes the implicit assumption that all rules will have
;;            a threshold value. This is ok for non m-of-n rules since they
;;            can simply use a 0 (zero) threshold. If the user specifies
;;            no threshold, the default is automatically set to zero so this 
;;            code will work by default (see io.lisp).
;; 03-JAN-93: (ptb) Updated file to work with the candidate structure (see
;;            structures.lisp).
;; 16-MAR-93: (ptb) Fixed another bug in the rule retraction (see comments
;;            in provable-without-all-rules routine).
;;===========================================================================


(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;===========================================================================
;; GLOBAL VARIABLE REFERENCE
;;
;; Four global variables are referenced here.
;; *trace-neg-fix* 
;;    used to print out messages as to how a retraction is found for a given 
;;    example.
;; *neither-examples* 
;;    used as a default for printout out tracing info.
;; *neither-theory*
;;    the default theory used to compute abductions.
;; *fix-marker*
;;     Used to make retraction more efficient. The principle is the same as 
;;     in deduce.lisp: avoid revisiting parts of the theory which have 
;;     already been traversed. This global is defined in abduce.lisp
;;===========================================================================
#+:cltl2(declaim (special *fix-marker*)
                 (special *trace-neg-fix*)
                 (special *neither-theory*)
                 (special *neither-examples*))
#-:cltl2(progn
          (proclaim '(special *fix-marker*))
          (proclaim '(special *trace-neg-fix*))
          (proclaim '(special *neither-theory*))
          (proclaim '(special *neither-examples*)))


(defun retract-rules (example category good-cat-flag
			      &optional (theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; Given an example (a false negative), a category which the example proves
  ;; but should not, and a flag indicating if the example also proved its
  ;; desired category; return a list of rules that, if retracted, will 
  ;; prohibit the proof of the example in the category WITHOUT inhibiting the
  ;; proof of the correct category. Calls "find-category" to get the root of
  ;; the tree (in "theory") that contains the rules to prove the category.
  ;; Then calls "retract-tree" to extract a minimal set of rules. 
  ;;
  ;; RETURNS a LIST of the deletions which, if made, will prevent the example
  ;; from being proved for category "category".
  ;;
  ;; NOTE: I set up 3 global variables dynamically scoped for the recursion.
  ;; Since these values remain constant throughout the recursion, there is no
  ;; need to continuously pass them as parameters.
  ;;
  ;; (ptb 12-22-92) Adding a comment to describe "pop up" phenomenon. The 
  ;; essential need for this extra code is to cover the event that no 
  ;; retractions are returned from the call to retract-tree. In this case, 
  ;; the theory is made maximally specific and returns a retraction which 
  ;; makes the category trivially unprovable.
  ;;
  ;; 16-MAR-93 (ptb) added a reference to *fix-marker* which is a variable 
  ;; used here and in abduce to mark the theory as it is traversed to avoid
  ;; recalculating revisions. This is completely analygous to the marking 
  ;; done during deduction (see deduce.lisp) and saves a good deal of time
  ;; when the theory has shared concepts. This is especially true because 
  ;; each revision must call "prove-categories" to verify the revision, which
  ;; is the most time-expensive routine I have. NOTE, however, that because
  ;; prove-categories is called during retraction, we cannot use the same
  ;; flag to mark the elements of the theory. Thus, I had to have two flags
  ;; *fix-marker* and *deduction-marker* and two sets of fields in the vertex
  ;; structure for storing information.
  ;;-------------------------------------------------------------------------
  (incf *fix-marker*)
  (let ((*example* example)
	(*exvals*  (example-values example))
	(*good-cat-flag* good-cat-flag)
	(parent-vert (find-category category theory))
	rval)
    (declare (special *example* *exvals* *good-cat-flag*))

    ;; do a printout for tracing purposes
    (when (and *trace-neg-fix*
               (member (position example *neither-examples*) *trace-neg-fix*))
      (push 'yes *trace-neg-fix*)
      (format t "~%~%* Retractions to DISprove the following as ~A"
	      category)
      (format t "~%  example number: ~D" (position example *neither-examples*))
      (ppexample *example*)
      (format t "~%  incoming args: category ~A  flag ~A"
	      category good-cat-flag))

    ;; perform the actual retraction. Save results
    (setf rval (candidate-deletions (retract-tree parent-vert t)))

    ;; do another printout for tracing purposes before returning
    (when (and *trace-neg-fix* (eq (car *trace-neg-fix*) 'yes))
      (format t "~%~%Total retractions for example ~D"
	      (position example *neither-examples*))
      (loop for r in rval do (pprule r))
      (pop *trace-neg-fix*))
    rval))


(defun retract-tree (parent-vert &optional (top-level nil))
  ;;-------------------------------------------------------------------------
  ;; Given a vertex which is the top of a sub tree in the theory, return a
  ;; CANDIDATE structure consisting of TWO values (1) t or nil indicating
  ;; whether or not the vertex is still provable using the *exvals* (see the
  ;; retract-rules routine above) and (2) the list of deletion structures 
  ;; (see structures.lisp) which will make the vertex not provable (note 
  ;; this could be passed back from a recursive call).
  ;;
  ;; The basic algorithm is to check each child of the vertex. There are 
  ;; three cases. (1) If the child is not provable (provable=nil) don't worry
  ;; about it. (2) If the child is provable, but has deletions which make it
  ;; non-provable (provable=t and deletions=non-nil) then append the list of
  ;; deletions to a saved variable. (3) If the child is provable and has no
  ;; deletions which make it non-provable (provable=t and deletions=nil) then
  ;; we must make a check. (3a) If removal of the child rule will NOT cause 
  ;; the CORRECT category to be unprovable, then create a new deletion of the
  ;; parent and this child and add to the list of deletions. (3b) Otherwise, 
  ;; there is no way to keep the parent-vert from being unprovable and we 
  ;; punt, returning '(t nil). 
  ;;
  ;; If all the children are checked without punting on case 3, this routine
  ;; creates a candidate structure and sets "provable" to t or nil, depending
  ;; upon whether ANY of the children required deletions (t if yes) and sets
  ;; "deletions" to the list of deletions saved from cases 1-3 above.
  ;;
  ;; 07-JAN-93 (ptb): Discovered a bug. When this routine is called
  ;; initially by retract-rules, that routine checks for pop up (ie, when the
  ;; last disjunct is being deleted). In that event, the caller will delete
  ;; all the rules anyway, which is exactly what caused the pop up in the 
  ;; first place. Unfortunately, the caller has less information, notably, it
  ;; lacks the threshold information returned by the individual calls to 
  ;; retract-child here. So, I added an optional variable that will be set to
  ;; t only for the first call. If we get to the deletion of all disjuncts
  ;; at the top level call, then go ahead and delete anyway since that is 
  ;; what the top level call would have to do.
  ;;
  ;; 16-MAR-93 (ptb) this routine was modified to look at the parent vertex
  ;; to see if it had already been traversed for the current example, and 
  ;; return the old value if so. This saves repetitious work. Also, this code
  ;; had to have "case 2" changed to eliminate duplicate deletions when 
  ;; combining results of two child retractions in the event that multiple
  ;; children proposed the same retraction. Note that since this routine 
  ;; stores the candidate value in "vertex-Fvisited", we can use an "eq" test
  ;; for duplication (since any duplicate would revisit the same parent and
  ;; thus return the same pointer).
  ;;-------------------------------------------------------------------------
  (when (= *fix-marker* (vertex-Fvisited parent-vert))
    (return-from retract-tree (vertex-Fvalue parent-vert)))
  (setf (vertex-Fvisited parent-vert) *fix-marker*)
  (loop for child in (vertex-children parent-vert)
	for cand = (retract-child child)
	for provable = (candidate-provable cand)    ;; indicates if provable
	for deletions = (candidate-deletions cand)  ;; rules to delete
        with saved = nil
        with any-provable = nil
	
        if provable                                 ;; eliminates case 1
        do (setf any-provable t)
           (if deletions
             (dolist (del deletions)                        ;; case 2
               (unless (member del saved :test #'eq) 
                 (setf saved (cons del saved))))
             (if (or (and (not (last-disjunct parent-vert))
                          (provable-without-rule child))
                     top-level)
               (setf saved                                  ;; case 3a
                 (cons (make-deletion :parent parent-vert :child child
                         :threshold-delta (candidate-threshold-delta cand))
                       saved))
               (progn (when (and *trace-neg-fix*            ;; case 3b
                                 (eq (car *trace-neg-fix*) 'yes))
                        (format t "~%  for consequent ~A" 
                                (vertex-prop parent-vert))
	                (format t "~%  returning: (t nil)"))
                      (setf (vertex-Fvalue parent-vert)
                            (make-candidate :provable t))
	              (return (vertex-Fvalue parent-vert)))))
	finally 
	(progn
	  (when (and *trace-neg-fix* (eq (car *trace-neg-fix*) 'yes))
	    (format t "~%  for consequent ~A" (vertex-prop parent-vert))
	    (format t "~%  returning: ~A" any-provable)
	    (when saved
	      (format t " and")
	      (loop for r in saved do (pprule r))))
          (setf (vertex-Fvalue parent-vert)
                (make-candidate :provable any-provable :deletions saved))
          (return (vertex-Fvalue parent-vert)))))


(defun provable-without-rule (rule)
  ;;-------------------------------------------------------------------------
  ;; Given a rule that is proposed for a negative fix, this routine
  ;; temporarily deletes that rule and sees if the example stored in the
  ;; global *example* (see retract-rules routine above) can still prove its
  ;; desired category.
  ;;
  ;; There are two cases here. First, it may be that *example* could not
  ;; prove its desired category. In this case, we don't have to check to see
  ;; if *example* can now prove the desired category. This first case is
  ;; tested by looking at the value of the global *good-cat-flag*. Second, if
  ;; the desired category was provable, then the rule is temporarily removed
  ;; and "prove-categories" is called to see if the desired category is still
  ;; provable. If not, we return nil.
  ;;-------------------------------------------------------------------------
  (declare (special *example* *good-cat-flag*))
  (if *good-cat-flag*
      (let (cats)
        (set-rule-used rule nil)
	(setf cats (prove-categories *example*))
        (set-rule-used rule t)
	(member (example-name *example*) cats))
      t))


(defun last-disjunct (parent)
  ;;-------------------------------------------------------------------------
  ;; Returns t if the rule passed in is the last undeleted rule of the parent
  ;; vertex. If not, returns nil
  ;;-------------------------------------------------------------------------
  (= (loop for child in (vertex-children parent)
	   sum (if (rule-used child) 1 0))
     1))

  
(defun retract-child (child)
  ;;-------------------------------------------------------------------------
  ;; The incoming child is a rule from a vertex, and thus has an antecedent
  ;; list. It also has a flag, indicating if the rule has been retracted from
  ;; the theory (nil=retracted). If retracted, then return (nil nil), ie, not
  ;; provable and no retractions are required to make the rule non-provable.
  ;;
  ;; If still used, this antecedent list is checked one antecedent at a time.
  ;; Each is passed to the "retract-ante" routine to determine if provable
  ;; and the needed retractions to make the antecedent unprovable.
  ;;
  ;; (ptb 12-22-92): updated this routine to handle threshold rules. This was
  ;; substantially different from the modifications made to the abduction 
  ;; code. Here, I needed to take the threshold of the rule into account 
  ;; since the choice of which retraction to use must be made locally. The 
  ;; new algorithm works as follows. (1) loop for all antecedents in the rule
  ;; regardless of whether or not they are provable (2) compute the number of
  ;; forced failures (antecedents which can be made false by retractions) and
  ;; unforced failures (antecedents which are false with no retractions).
  ;; Then (3) compute the return value based on the following cases:
  ;;    case 1: (# unforced failues > rule threshold) => return (nil nil)
  ;;            In this case, there are enough antecedents which naturally 
  ;;            return false that the entire rule can be considered false 
  ;;            without needing any retractions.
  ;;    case 2: (# forced + # unforced <= rule threshold) => return (t nil)
  ;;            In this case, no matter how many forced failures are created
  ;;            they are still covered by the threshold, thus the rule is
  ;;            true and cannot be made false.
  ;;    case 3: (# forced + # unforced > threshold AND
  ;;             # unforced failures <= threshold ) => return (t ...)
  ;;            Here the "..." is the N best forced deletions, where N equals
  ;;            (threshold - # unforced failures + 1). 
  ;;            In this case, there are not enough unforced failures to make
  ;;            the rule naturally false, but there are more than enough to 
  ;;            FORCE the rule to be false. So, we subtract out the number of
  ;;            unforced from the threshold to find the number of required 
  ;;            forced failures. The best from among the forced failures are 
  ;;            selected, and enough are selected to ensure the rule's 
  ;;            failure (thus, the +1).
  ;;
  ;; Returns a candidate structure indicating whether the child rule is 
  ;; provable; if not, then how many deletions were required; and finally, 
  ;; how to change the threshold to make the rule unprovable without doing
  ;; deletions. (Note that the change in threshold = -1 * N).
  ;;
  ;; 16-MAR-93 (ptb) this routine had to be changed to return deletions=nil
  ;; when all the deletions proposed for the child TOGETHER could not be
  ;; deleted. Recall that each proposed rule deletion is tested for validity
  ;; INDIVIDUALLY in the retract-tree routine. However, these deletions must
  ;; also be tested collectively, which is done here (see provable-without-
  ;; all-rules routine). If this fails, then the result is treated the same
  ;; as case 2.
  ;;-------------------------------------------------------------------------
  (if (rule-used child)
      (loop for ante in (rule-antecedents child)
            with cand = nil
            with provable = nil
            with dels = nil
            with threshold = (rule-threshold child)
            with forced = 0
            with unforced = 0
            with saved = nil

            if (not (antecedent-abduced? ante))
            do (setf cand     (retract-ante ante)
                     provable (candidate-provable cand)
                     dels     (candidate-deletions cand))
            
               ;; update forced, unforced and saved variables
               (if (not provable)
                   (incf unforced)
                   (when dels
                     (incf forced)
                     (setf saved (cons dels saved))))
            
            ;; quit early if unforced exceeds threshold, case 1
            if (> unforced threshold) return (make-candidate :provable nil)
            
            finally
            (let ((N (1+ (- threshold unforced))))
              ;; *****
              ;; set *foo* = t in retract-rules when example position
              ;; equals the desired example number
              ;;(when *foo*
              ;;  (format t "~%~D:~A <-" (rule-threshold child) 
              ;;          (rule-consequent child))
              ;;  (dolist (ante (rule-antecedents child))
              ;;    (format t " ~A" (antecedent-prop ante)))
              ;;  (format t "~%->ALL rules together = ~A"
              ;;          (trash-provable-without-all-rules saved)))
              ;; *****
              (return
               (if (<= (+ forced unforced) threshold)
                 (make-candidate :provable t :threshold-delta (* -1 N))
                 (let ((best (pick-n-best saved N)))
                   (if (provable-without-all-rules best)
                       (make-candidate :provable t :deletions best)
                       (make-candidate :provable t 
                                       :threshold-delta (* -1 N))))))))
      (make-candidate :provable nil)))


(defun pick-n-best (deletions n)
  ;;-------------------------------------------------------------------------
  ;; Selects the n best deletions from the LIST OF DELETION LISTS passed in 
  ;; via "deletions". That is, "deletions" is a list of deletion lists, 
  ;; representing the retractions necessary to make various antecedents of a
  ;; rule unprovable. The criterion for "best" here is to compare two
  ;; deletion lists using the "smaller-retraction-p" predicate below.
  ;;
  ;; Once the n best deletions are chosen, note that they must be appended 
  ;; together. I don't think append here is going to cause too much garbage
  ;; formation, but it can be changed later to nconc if it turns out to be
  ;; too wasteful.
  ;;-------------------------------------------------------------------------
  (loop for pick in (subseq (sort deletions #'smaller-retraction-p) 0 n)
        append pick))


(defun smaller-retraction-p (dlist1 dlist2)
  ;;-------------------------------------------------------------------------
  ;; Called by "retract-child" to compare two deletion lists and return t if
  ;; and only if the first is smaller than the second. The length of each 
  ;; deletion list is the sum of the lengths of the rules it deletes. The 
  ;; length of each rule is the number of literals in the rule.
  ;;-------------------------------------------------------------------------
  (< (loop for del in dlist1
	   sum (1+ (count-antes (deletion-child del))))
     (loop for del in dlist2
	   sum (1+ (count-antes (deletion-child del))))))


(defun provable-without-all-rules (del-list)
  "Temporarily makes all deletions in del-list and checks to see if the good
category can still be proved. Must be called from within rule-retract to have
access to dynamically scoped special variables."
  ;;-------------------------------------------------------------------------
  ;; 16-MAR-93 (ptb): I added this routine to fix a bug. With threshold rules
  ;; one must be careful when checking if the proposed deletions are valid 
  ;; for an example, even if NOT doing threshold revision! The problem is 
  ;; that any group of deletions may be valid when considered singly (as is
  ;; done by the provable-without-rule routine) and yet be invalid when taken
  ;; as a group. Specifically, all the deletions may be below a threhsold by
  ;; themselves and exceed that threshold when taken as a group. This routine
  ;; simply does the same task as provable-without-rule, but after making all
  ;; the deletions in del-list, rather than just one.
  ;;-------------------------------------------------------------------------
  (declare (special *example* *good-cat-flag*))
  (if *good-cat-flag*
      (let (cats)
        (dolist (del del-list) (set-rule-used (deletion-child del) nil))
	(setf cats (prove-categories *example*))
        (dolist (del del-list) (set-rule-used (deletion-child del) t))
        (member (example-name *example*) cats))
      t))


(defun retract-ante (ante)
  ;;-------------------------------------------------------------------------
  ;; Given an antecedent (from an antecedent list of a rule) this routine
  ;; checks to see if the antecedent is provable using the values in
  ;; *exvals* (see retract-rules above). There are four cases: (1) The
  ;; antecedent is already abduced, thus it should be ignored. (2) The
  ;; antecedent points to a vertex which is an undefined intermediate
  ;; concept. In this case, the antecedent is not provable. (3) The
  ;; antecedent refers to an operational vertex. Here we check to see if the
  ;; vertex is provable using the exvals and the condition stored in the prop
  ;; field of the ANTECEDENT structure. (4) The vertex involved is the top of
  ;; a tree. Here we recurse to "retract-tree".
  ;;
  ;; Note that in the first three cases no rules are available for
  ;; retraction. Only the last case (which recurses) can return a list of
  ;; rule retractions required for making the antecedent non provable. Return
  ;; value is a CANDIDATE structure (see structures.lisp).
  ;;-------------------------------------------------------------------------
  (declare (special *exvals*))
  (let ((vert (antecedent-vertex ante)))
    (cond ((antecedent-abduced? ante) (make-candidate :provable t))
	  ((vertex-no-rules? vert) (make-candidate :provable nil))
	  ((leaf-vertex vert)
	   (if (leaf-test (antecedent-prop ante)
			  (aref *exvals* (vertex-example-index vert)))
	       (make-candidate :provable t)
	       (make-candidate :provable nil)))
	  (t (retract-tree vert)))))
