
;;===========================================================================
;; IO - Compile and disassemble examples and trees for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Christopher M. Whatley, Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; There are several functions of interest in this file. They are:
;;
;; (1) neither-load-data:
;;     loads file with theory and examples. Should be called BEFORE any other
;;     calls to NEITHER functions.
;; 
;; (2) prepare-theory-and-examples:
;;     Must be called on an example theory list and example list (see
;;     description of external globals and discussion below). Converts the
;;     theory and examples to the internal NEITHER format.
;;
;; (3) add-examples, set-examples:
;;     Used to add or examples to NEITHER's internal list or to reset that
;;     internal list.
;;
;; (4) decompile-theory:
;;     A useful routine for converting the internal NEITHER format for a
;;     theory back to a readable form.
;;
;; FORMATS:
;;
;; The routines here depend heavily upon the format of the theory and
;; examples defined in the files loaded by "neither-load-data". The expected
;; formats are as follows:
;;
;; THEORY: is a list of definite clauses (ie Horn clauses with exactly one
;; consequent). The preferred format for each clause is
;;
;;    (consequent <- antecedent-1 antecedent-2 ...)
;;
;; where the "consequent" is assumed to be a symbol representing either a
;; category of classification or an intermediate concept. Antecedents are
;; also expected as symbols, unless they are predicates which accept
;; arguments. Thus, an input clause might look like:
;;
;;    (a <- b (c v1) d)
;;
;; with "a" as the consequent, "b" and "d" as singleton (true/false)
;; predicates and "c" as a proposition with the value "v1". Due to older
;; versions of code, NEITHER is also compatible with the following forms for
;; the same clause:
;;
;;    (<- a b (c v1) d)
;;    ((a) <- (b) (c v1) (d))
;;    (<- (a) (b) (c v1) (d))
;;
;; Some assumptions about the theory:
;; (1) all consequents are singletons, ie "graspable" or "(graspable)". Thus
;;     a rule like "((man socrates) <- (human socrates) (male socrates))" is
;;     forbidden.
;; (2) no rules of the form "(graspable <-)", ie with no antecedents.
;;
;; EXAMPLES: are lists of category-observables lists, where the "category" is
;; one from among the list of values in *categories* and the "observables"
;; are legal values for the features in *feature-names*. The category and
;; features listed together. As with theories, singletons in the examples are
;; preferred as symbols, but will be accepted as singleton lists. For
;; features whose values in *domains* are (true false), using the feature
;; name as a singleton is equivalent to listing it with the value "true". As
;; an illustration, given the following *feature-names*, *categories*, and
;; *domains*: 
;;
;;    (defvar *categories* '(cat1 cat2 ct3))
;;    (defvar *feature-names* '(f1 f2 f3 f4))
;;    (defvar *domains* '((true false) (true false) (v1 v2 v3) (v4 v5)))
;;
;; the following are legal examples:
;;
;;    (cat1 f1 f2 (f3 v1) (f4 v4))
;;    ((cat1) (f1) (f2) (f3 v1) (f4 v4))
;;    (cat2 (f1 true) f2 (f3 v2) (f4 v4))
;;    (cat3 f1 (f2 false) (f3 v2) (f4 v4)) ;; f2 explicitly set false
;;    (cat3 f1 (not f2) (f3 v1) (f4 v5))   ;; f2 explicitly set false
;;    (cat3 (f1 true) (f3 v3) (f4 v5))     ;; f2 absent so assumed false
;;
;; Some assumptions about examples:
;; (1) right now, all examples are assumed to be provable in only ONE
;; category. 
;;
;; CHANGE HISTORY
;;
;; 02-JUN-92: (ptb) Wrote all comments. Deleted several global variables
;;            and functions not used in any code. Many routines also needed
;;            small fixes here. Most notably, I changed the "prepare-theory-
;;            and-examples" routine to redefine the "clause-consequent" 
;;            routine to handle different rule formats.
;; 16-JUN-92: (ptb) Reworked most of the routines to handle the new
;;            "antecedent" structure (see deduce.lisp).
;; 22-JUN-92: (ptb) Changed the decompiling routines to print out the rules 
;;            in a nicer format and to skip any rules or antecedents that are
;;            ignored in the theory (ie, their "abduced?" or "rule-used" flag
;;            is set).
;; 01-SEP-92: (ptb) Fixed bug in "convert-theory" routine so it would work 
;;            with theories that are already converted. Also, changed the 
;;            theory-to-decision-tree routine to return only one value (no 
;;            callers needed two values).
;; 22-OCT-92: Major rewrite to use structures in structures.lisp file.
;; 12-MAR-93: (ptb) modified to work with updated rule structure.
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;===========================================================================
;; GLOBAL VARIABLES
;;===========================================================================

(defvar *deduce-hash* nil
  "Used in translation of theory to internal format")

(defvar *neither-theory* nil
  "NEITHER's current theory")

(defvar *neither-examples* nil
  "NEITHER's current working set of example vectors.")

;;---------------------------------------------------------------------------
;; EXTERNAL GLOBAL VARIALBES
;;
;; These are the globals which should be defined in a NEITHER data file. Each
;; is described in turn. These variables are also described in the file
;; "data-utilities.lisp"
;;---------------------------------------------------------------------------

#-:cltl2
(progn
  (proclaim '(special *domains*))
  ;;-------------------------------------------------------------------------
  ;; A list of domain value lists, where each domain value list contains the
  ;; legal values for a feature of *feature-names*. The feature and the
  ;; domain value list should be at corresponding positions in their
  ;; respective lists.
  ;;-------------------------------------------------------------------------

  (proclaim '(special *feature-names*))
  ;;-------------------------------------------------------------------------
  ;; A list of the observable features of the theory. Each should be a
  ;; symbol, and the position of each feature in *feature-names* should
  ;; correspond to a set of its legal values at the same position in
  ;; *domains*
  ;;-------------------------------------------------------------------------

  (proclaim '(special *categories*))
  ;;-------------------------------------------------------------------------
  ;; A list of the possible categories which the theory may prove. Each
  ;; should be a symbol
  ;;-------------------------------------------------------------------------

  (proclaim '(special *correct-theory*))
  ;;-------------------------------------------------------------------------
  ;; A list of rules representing a theory which will correctly classify all
  ;; examples as exactly one of the categories of *categories*
  ;;-------------------------------------------------------------------------

  (proclaim '(special *theory*))
  ;;-------------------------------------------------------------------------
  ;; A list of rules which represent an incorrect theory
  ;;-------------------------------------------------------------------------

  (proclaim '(special *raw-examples*))
  ;;-------------------------------------------------------------------------
  ;; A list of example lists, where each example list is a list of the
  ;; category in which the example should be proved followed by a list of
  ;; observed values.
  ;;-------------------------------------------------------------------------

  (proclaim '(special *deduction-marker*))
  (proclaim '(special *fix-marker*))
  ;;-------------------------------------------------------------------------
  ;; Used in deduction (*deduction-marker*) and abduction and retraction
  ;; (*fix-marker*) to avoid revisiting parts of the theory.
  ;;-------------------------------------------------------------------------

  (proclaim '(special *student-model*))
  ;;-------------------------------------------------------------------------
  ;; Used in model.lisp to store the revisions made to the theory.
  ;;-------------------------------------------------------------------------
)
#+:cltl2
(declaim (special *domains*)
         (special *feature-names*)
         (special *categories*)
         (special *correct-theory*)
         (special *theory*)
         (special *raw-examples*)
         (special *deduction-marker*)
         (special *fix-marker*)
         (special *student-model*))


;;===========================================================================
;; SUPPORT ROUTINES
;;===========================================================================

(defun example-preparation-hook (x) x) ;;dummy routine. see neither-load-data

(defun load-neither-data (filename)
  ;;-------------------------------------------------------------------------
  ;; Loads a data file which is assumed to set the external global variables
  ;; above. Since the file may or may not define a routine called "example-
  ;; preparation-hook", that function is unbound BEFORE loading the file.
  ;; This is essential for the "prepare-theory-and-examples" routine to work
  ;; properly. 
  ;;-------------------------------------------------------------------------
  (fmakunbound 'example-preparation-hook)
  (fmakunbound 'back-xlate-example)
  (makunbound '*negative-category*)
  (setf (symbol-function 'clause-consequent) #'(lambda (c) (second c)))
  (load filename)
  (unless (boundp '*negative-category*)
    (set '*negative-category* 'negative)))


  ;;-------------------------------------------------------------------------
  ;; for referencing parts of clauses. Note that "clause-consequent" may be
  ;; redefined by "prepare-theory-and-examples", depending upon the format of
  ;; the theory.
  ;;-------------------------------------------------------------------------
(defun clause-antecedents (c) (cdddr c))
(defun clause-threshold (c) (car c))
(defun clause-consequent (c)  (second c))


(defun find-feature-index (prop)
  ;;-------------------------------------------------------------------------
  ;; Returns what will be the eventual position of the values for "prop" in
  ;; the example feature vector. Uses position in the *feature-names* list.
  ;;-------------------------------------------------------------------------
  (position prop *feature-names*))


(defun hash-vertices (vertices)
  ;;-------------------------------------------------------------------------
  ;; Hashes the elements of vertices into a global table. Use the proposition
  ;; stored in the "prop" field of the vertex as the key. Note that Chris
  ;; made the implicit assumption here that all "prop" fields would be
  ;; symbols rather than lists or negated symbols. Thus, calls below to
  ;; "find-hashed-vertex", which call "prop-name" to make sure the
  ;; proposition is a symbol, will access the correct vertex of the hash
  ;; table. 
  ;;-------------------------------------------------------------------------
  (setf *deduce-hash* (make-hash-table))
  (loop for v in vertices
	do (setf (gethash (vertex-prop v) *deduce-hash*) v)))


(defun find-hashed-vertex (prop) 
  ;;-------------------------------------------------------------------------
  ;; retrieve vertex from hash table. Note that the incoming proposition is
  ;; not assumed to be a symbol here, and therefore MUST be converted to a
  ;; symbol (because the hash table used symbols as its keys). 
  ;;-------------------------------------------------------------------------
  (gethash (prop-name prop) *deduce-hash*))


(defun prop-name (p)
  ;;-------------------------------------------------------------------------
  ;; Recursively strips away any surrounding negations until it gets to the
  ;; use of the proposition. Then, if the proposition is a singleton, it
  ;; returns that, otherwise it returns the car of the proposition. Thus
  ;; something like "(not graspable)" returns "graspable" and something like
  ;; "(not (birth live))" returns "birth"
  ;;-------------------------------------------------------------------------
  (if (listp p)
      (if (eq (first p) 'not)
	  (prop-name (second p))
	  (first p))
      p))


;;===========================================================================
;; THEORY TRANSLATION ROUTINES
;;===========================================================================

(defun prepare-theory-and-examples (raw-theory raw-examples
					       &optional (silent nil))
  "Compile and convert theory and examples to internal format. Returns
nothing and sets the globals *neither-theory* and *neither-examples*"
  ;;-------------------------------------------------------------------------
  ;; Converts the input "raw-theory" and "raw-examples" into an internal
  ;; format stored in *neither-theory* and *neither-examples*.
  ;;
  ;; An interesting feature of this function is its ability to allow any
  ;; format for examples via the "example-preparation-hook" user-defined
  ;; function. The idea is to check to see if the routine
  ;; "example-preparation-hook" is defined and, if so, to call that function
  ;; on the examples to convert them to the standard NEITHER format. The
  ;; result is a preliminary conversion of the examples to the normal "raw"
  ;; format. This raw format is then passed to the "build-example-vectors"
  ;; routine to construct the example vectors for *neither-examples*. Note
  ;; that this routine is unbound when a new file of examples is loaded.
  ;;-------------------------------------------------------------------------
  ;; translate the theory
  (setf *neither-theory*
	(theory-to-decision-tree (convert-theory raw-theory)))

  ;; translate the examples
  (setf *neither-examples* (translate-examples raw-examples silent))

  ;; reset the student model
  (setf *student-model* nil)

  ;; printout and return nil
  (when (not silent)
    (format t "~%Examples are now in *neither-examples* ")
    (format t "and theories in *neither-theory*"))
  (setf *deduction-marker* 0)
  (setf *fix-marker* 0))


(defun add-examples (exs)
  ;;-------------------------------------------------------------------------
  ;; Use this to add new examples to existing compiled set
  ;;-------------------------------------------------------------------------
  (nconc *neither-examples* (translate-examples exs)))


(defun set-examples (exs &optional (silent nil))
  ;;-------------------------------------------------------------------------
  ;; Use this to replace *neither-examples* with a new set.
  ;;
  ;; 23-MAR-93 (ptb) added a reset for *student-model* so that any old
  ;; revisions are deleted.
  ;;-------------------------------------------------------------------------
  (setf *neither-examples* (translate-examples exs silent))
  (setf *student-model* nil)
  (unless silent
    (format t "~%Examples in *neither-examples* have been updated.")))


(defun translate-examples (ex-list &optional (silent nil))
  ;;-------------------------------------------------------------------------
  ;; Translates the incoming list of examples and returns them listed in the 
  ;; new format.
  ;;-------------------------------------------------------------------------
  (let ((raw (cond ((fboundp 'example-preparation-hook)
                    (unless silent
                      (format t "~%Preparing examples with ~A"
                              (documentation 'example-preparation-hook
                                             'function)))
                    (example-preparation-hook ex-list))
                   (ex-list))))
    (build-example-vectors (convert-examples raw))))


(defun set-theory (raw-theory &optional (silent nil))
  ;;-------------------------------------------------------------------------
  ;; Use this to replace *neither-theory* with a new theory.
  ;;
  ;; 23-MAR-93 (ptb) added a reset for *student-model* so that any old
  ;; revisions are deleted.
  ;;-------------------------------------------------------------------------
  (setf *neither-theory*
	(theory-to-decision-tree (convert-theory raw-theory)))
  (setf *deduction-marker* 0)
  (setf *fix-marker* 0)
  (setf *student-model* nil)
  (unless silent
    (format t "~%Theory in *neither-theory* has been updated.")))


;;===========================================================================
;; THEORY CONVERTION ROUTINES
;;===========================================================================

(defun convert-theory (th)
  "Converts old-style theory to new-style"
  ;;-------------------------------------------------------------------------
  ;; NEITHER theories in the past represented every proposition as a list.
  ;; Thus even something like "graspable" from the cup theory looked like
  ;; "(graspable)". This routine converts old theories to a new format, which
  ;; leaves singletons as atoms rather than lists. Note, however, that
  ;; theories already in the new format are uneffected.
  ;;-------------------------------------------------------------------------
  (loop for rule in th
	for first = (car rule)
	collect
	(if (numberp first)
	    (convert-rule rule)
	    (cons 0 (convert-rule rule)))))
   


(defun convert-rule (rule)
  ;;-------------------------------------------------------------------------
  ;; Called by "convert-theory" to convert a given rule into the new format
  ;; understood by NEITHER. The idea here is to loop through the rule,
  ;; collecting new versions of every element of the rule. The rules for
  ;; conversion are simple: (1) if the element is an atom, leave it alone (2)
  ;; if its a 1-element list, strip it of its "()" (3) if it's a negation,
  ;; then pull out the "not" and list it with the stripped version of the
  ;; proposition (assumes all negations are on single propositions). The
  ;; default case is to leave the element alone.
  ;;-------------------------------------------------------------------------
  (loop for i in rule
	collect 
	(cond ((atom i) i)
	      ((= (length i) 1) (first i))
	      ((and (eq (first i) 'not) (listp (second i)))
	       `(not ,(first (second i))))
	      (t i))))


(defun theory-to-decision-tree (theory)
  "Converts human-readable theory into a decision tree of vertex structures"
  ;;-------------------------------------------------------------------------
  ;; Builds a list of "category trees" (see structures.lisp) which it returns 
  ;; as the final converted form of "theory". Assumes that "theory" has 
  ;; already been run through the "convert-theory" routine.
  ;;
  ;; Chris set this up to work in three stages. First, the theory is
  ;; "dissected" into four distince parts: observables (ie operational
  ;; features), undefined propositions that are not observables, defined
  ;; propositions (ie, consequents of rules) that are neither observables nor
  ;; categories, and categories. Second, a list of vertices is created, where
  ;; each type of vertex is tailored to fit the type of vertex (e.g.
  ;; observable vertices have the "example-index" field set whereas undefined
  ;; propositions have the "no-rules?" field set). This list of vertices is
  ;; then hashed into a table. Third (and last), a list of "category trees"
  ;; is constructed using the hash table.
  ;;
  ;; (ptb-10/20/92) changed this routine to handle rules with thresholds. The
  ;; assumption is that the input theory will have been through the
  ;; "conversion" process (see convert theory) which will place the threshold
  ;; for the rule at the beginning of the list representing the rule.
  ;;
  ;; 12-MAR-93 (ptb): changed the call to make-rule to work with the new 
  ;; field added to the rule structure.
  ;;-------------------------------------------------------------------------

  ;; before translating the theory, check the format of the rules. If they do
  ;; not begin with "<-" redefine the "clause-consequent" access routine
  (if (eq (second (first theory)) '<-)
      (setf (symbol-function 'clause-consequent) #'(lambda (c) (third c)))
      (setf (symbol-function 'clause-consequent) #'(lambda (c) (second c))))

  (multiple-value-bind (observables undefined-inter categories)
      (dissect-theory theory)

    ;; Make observable verticies
    (let (trees 
	  ;; convert operationals into vertices. note there will be the same
	  ;; number as feature-names (thus one vertex for each feature).
	  ;; Also, when these get hashed below, they are guaranteed to have
	  ;; symbol names for the "prop" field (see "hash-vertices" routine)
	  (observable-verts
	   (loop for ob in observables
		 collect (make-vertex 
			  :prop ob
			  :example-index (find-feature-index ob)))))

      ;; The next step is to collect a list of top-level vertices and store
      ;; them in the "trees" local variable. Begin by creating a vertex for
      ;; each consequent of the theory (set its children to its antecedents)
      ;; Note that all consequents are guaranteed to be symbols, required for
      ;; the call to "hash-vertices" below. Also, for those consequents that
      ;; are defined by multiple rules, note that by using "member" to check
      ;; for the consequent on "trees", we end up grouping all antecedents
      ;; under the same vertex (children = list of antecedent lists).
      (loop for clause in theory
	    for conseq = (clause-consequent clause)
	    for ants = (clause-antecedents clause)
	    for threshold = (clause-threshold clause)
	    for vertex = (first 
			  (or (member conseq trees :test #'vertex-member-test)
			      (push (make-vertex :prop conseq) trees)))
	    do
	    (push (make-rule :used nil :threshold threshold
			     :antecedents ants :consequent conseq)
		  (vertex-children vertex)))

      ;; Then, add a vertex for every undefined proposition; be sure to set
      ;; the "no-rules?" flag (see deduce.lisp)
      (loop for i in undefined-inter
	    do (push (make-vertex :prop i :no-rules? t) trees))

      ;; Finally build a hash table out of the vertices in "trees" and the
      ;; vertices for the observable features. Store table globally.
      (hash-vertices (nconc trees observable-verts))

      ;; Construct the final converted theory by building a tree for each
      ;; category to be proven, using the elements of the hash table.
      (loop for cat in categories
	    for top = (find-hashed-vertex cat)
	    if top
	    collect (build-tree top)))))


(defun dissect-theory (theory)
  "Takes a theory as a list of clauses, and extracts the observable features,
intermediate concepts (both undefined and defined) and categories which are
returned as four values" 
  ;;-------------------------------------------------------------------------
  ;; Called after theory and access functions are converted to the new
  ;; format. This routine pulls out four lists of information from the
  ;; theory. Two of these are defined by the user in the theory-and-example
  ;; file: *feature-names* and *categories*. The other two are computed by
  ;; looping through the theory (which, by the way, is assumed to have
  ;; already passed through the "convert-theory" routine). The defined
  ;; intermediates are those consequents which do not appear as categories,
  ;; but have rules for their definitions. The undefined intermediates are
  ;; any other symbols which are referenced in antecedent expressions, but
  ;; are not defined with rules.
  ;;
  ;; NOTE: this routine assumes there are no "empty" rules of the form
  ;; "graspable <-". Also assumes that all consequents are SINGLETONS.
  ;;-------------------------------------------------------------------------
  (let* ((features *feature-names*)
	 (categories *categories*)

	 ;; Every rule consequent is a proposition defined by the theory.
	 ;; Don't include those that are categories.
	 (defined-intermediates
	   (loop for k in theory
		 for cc = (clause-consequent k)
		 unless (or (member cc foo) (member cc *categories*))
		 collect cc into foo
		 finally (return foo)))

	 
	 ;; make a list of all propositions referenced in the theory that are
	 ;; defined.
	 (referenced 
	  (append *categories* *feature-names* defined-intermediates))

	 ;; make a single list of all propositions in the theory (stripping
	 ;; out the "<-" symbol).
	 (everything 
	  (loop for k in theory
		for cl = (cons (clause-consequent k) (clause-antecedents k))
		append cl))

	 ;; pull out intermediates (ie, non listed propositions) which are
	 ;; not features, categories or defined intermediates (ie, are not in
	 ;; the "referenced" list).
	 (undefined-intermediates
	  (loop for i in everything
		for j = (prop-name i)
		if (not (or (member j foo) (member j referenced)))
		collect j into foo
		finally (return foo))))

    (values features undefined-intermediates categories)))
   

(defun build-tree (vert)
  ;;-------------------------------------------------------------------------
  ;; Recursive routine to construct the category tree for vertex "vert".
  ;; Basically, the idea is to loop through the children of the incoming
  ;; vertex (ie, "vert") creating antecedent lists for each child. Each
  ;; antecedent list may call build-tree again to construct a tree for the
  ;; antecedent vertex.
  ;;
  ;; Recursion only occurs when the incoming vertex has not had its children
  ;; expanded. The test for this is to see if the "used" flag of the rule has
  ;; been set to "t" yet (initially, this flag is set to nil). If not, each 
  ;; of the antecedents of the rules in children must be converted to
  ;; antecedent structures.
  ;;
  ;; To convert, each child is traversed. For each child, "t" (for the used
  ;; flag) is set and the antecedents field of the child rule is set to a
  ;; list of antecedent structures created from the antecedents of the rule
  ;; (which are the original contents of the "antecedents" field). As each
  ;; antecedent structure is created, its "prop" field is set to the
  ;; condition in the original rule. Thus if the original theory looked like:
  ;;
  ;;   ((a <- (b v1)) (a <- (not c) d))
  ;;
  ;; then the vertex "a" would come into this routine with children set to:
  ;;
  ;;   ((b v1) ((not c) d))
  ;;
  ;; and this routine would set the antecedents field to:
  ;;
  ;;   (ante[prop=(b v1)] ante[prop=(not c)] ante[prop=d])
  ;;
  ;; where the antecedent structures are shown with their contents enclosed
  ;; in square brackets"[]". The updated "vert" is returned.
  ;;
  ;; NOTE: before this routine is called, each of the propositions in the
  ;; theory has already had a vertex constructed for it and stored in the
  ;; hash table. Thus, the only change made here is to "flesh out" those
  ;; vertices which represent propositions that are consequents of rules.
  ;;
  ;; (ptb-10/20/92): Changed this routine to deal with rule thresholds.
  ;;-------------------------------------------------------------------------
  (if (and (vertex-children vert)
	   (not (rule-used (first (vertex-children vert)))))
      (loop for child in (vertex-children vert)
	    for antes = (rule-antecedents child)
	    do
	    (setf (rule-used child) t)
	    (setf (rule-antecedents child)
		  (loop for i in antes
			collect
			(make-antecedent
			 :prop i :abduced? nil
			 :vertex (build-tree (find-hashed-vertex i)))))))
  vert)
						  

;;===========================================================================
;; EXAMPLE CONVERSION ROUTINES
;;===========================================================================

(defun convert-examples (examples)
  ;;-------------------------------------------------------------------------
  ;; Performs a function parallel to the "convert-theory" routine. For each
  ;; example, it finds any singleton list and strips off the parentheses. If
  ;; the examples are already in this format, nothing changes. Returns a list
  ;; of the examples in the same form (list of lists).
  ;;-------------------------------------------------------------------------
  (loop for ex in examples
	collect
	(loop for it in ex
	      collect (cond ((symbolp it) it)
			    ((and (listp it) (= 1 (length it))) (first it))
			    (t it)))))


(defun build-example-vectors (examples)
  ;;-------------------------------------------------------------------------
  ;; Once the examples have been converted to lists of (category features ..)
  ;; this routine is called to produce the final format for the global
  ;; *neither-examples*. 
  ;;
  ;; Chris wrote this routine to use two internal functions. The first takes
  ;; an instance of an observation in an example, and returns the "value"
  ;; associated with that observation. Thus, if the example were
  ;;
  ;;   (cup graspable (width small) liftable)
  ;;
  ;; where "cup" is the category for the example and the other elements are
  ;; observed features, then the value of "graspable" and "liftable" would be
  ;; "true" whereas the value of "width" would be "small". The second routine
  ;; creates an initialized vector for an example, setting any domains which
  ;; have "true/false" values to "false", and other domains to nil. 
  ;;
  ;; The overall routine then proceeds by looping through each example,
  ;; creating an initialized vector, and then looping through all the
  ;; observations for the example to update the vector. Values from the
  ;; observables are placed into the vector at the corresponding location of
  ;; the observable in the *feature-names* list (see "find-feature-index").
  ;; Lastly, once the vector is set up, a new example structure is created
  ;; and added to the list to be returned.
  ;;-------------------------------------------------------------------------
  (let ((nobs (length *feature-names*)))
    (flet ((obs-value (obs)
	     (cond ((listp obs) (second obs))
		   (t 'true)))
	   (make-default-feature-array ()
	     (make-array nobs :initial-contents
			 (loop for i from 1 to (length *feature-names*)
			       for dom in *domains*
			       collect (cond ((member 'true dom) 'false)
					     (t nil))))))
      (loop for ex in examples
	    for name = (first ex)
	    for newvec = (make-default-feature-array)
	    do (loop for obs in (rest ex)
		     for prop = (prop-name obs)
		     for idx = (find-feature-index prop)
		     if idx
		     do (setf (aref newvec idx) (obs-value obs)))
	    collect (make-example :name name :values newvec)))))
