
;;===========================================================================
;; Deduce for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Christopher M. Whatley, Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; Deduce uses decision trees to classify the object given as input. The
;; input should be a vector of the form: 
;;
;;     #(v1 v2 v3 v4 v5 ...)  ;; note: "v1" etc are values.
;;
;; to allow for fast, circuit-like truth determination. The vertices of the
;; decision tree graph are described below.
;;
;; CHANGE HISTORY
;;
;; 26-MAY-92: (ptb) fixed bug in "test-deduce" to work for NEGATIVE examples.
;;            Added comments to code.
;; 16-JUN-92: (ptb) modified to use "antecedent" structure. Makes sharing
;;            vertices possible and more flexible and fixes incorrect use of
;;            undefined intermediate concepts.
;; 19-NOV-93: (ptb) modified to work with theshold (m-of-n) rules and to 
;;            avoid any reproving by using a marking method.
;; 02-MAR-93: (ptb) added code to print out the rules which are satisfied by
;;            a given example. Note that this will work if the theory has not
;;            been numbered, but all the default numbers on the rules are set
;;            to -1 so you won't get much information. 
;; 15-MAR-93 
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;---------------------------------------------------------------------------
;; GLOBAL VARIABLE REFERENCES
;;
;; The code here references three global variables which it assumes are
;; defined elsewhere in the code.
;;---------------------------------------------------------------------------
#-:cltl2(progn
          (proclaim '(special *neither-theory*))    ;; in io.lisp
          (proclaim '(special *neither-examples*))  ;; in io.lisp
          (proclaim '(special *negative-category*)) ;; in user data file
          (proclaim '(special *trace-deduce*)))     ;; in debug.lisp
#+:cltl2(declaim (special *neither-theory*)
                 (special *neither-examples*)
                 (special *negative-category*)
                 (special *trace-deduce*))


;;===========================================================================
;; SUPPORT MACRO
;;===========================================================================

(defmacro negation (p)
  ;;-------------------------------------------------------------------------
  ;; Checks to see if "p" is a negation. This is done by checking whether the
  ;; first element is "not". The "p" input is assumed to be a proposition
  ;; such as is stored in the "prop" fields of vertices or antecedents (see
  ;; structure definitions above).
  ;;-------------------------------------------------------------------------
  `(when (listp ,p)
     (eq (first ,p) 'not)))


;;===========================================================================
;; DEDUCE ROUTINES
;; 
;; The routines below are used for proving an example using the theory tree.
;;===========================================================================

(defvar *deduction-marker* 0)

(defun prove-categories (example &optional (theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; Returns a list of categories provable for the example. Example is
  ;; assumed to be of the form indicated by the example structure (see above)
  ;; and theory is assumed to be a list of category trees.
  ;;
  ;; Works as follows: call "satisfy-children" on each category tree in the
  ;; theory. Collect the proposition (prop) field of the top level vertex for
  ;; each category tree which is successfully proved. 
  ;;
  ;; 19-NOV-92 (ptb): added a global variable to indicate how many times the
  ;; prove-categories routine is called. It acts as a marker that changes 
  ;; with each invocation of the code. This allows marking which elements of
  ;; the proof tree have been visited, without having to clean up the marks
  ;; from the last call.
  ;;-------------------------------------------------------------------------
  (incf *deduction-marker*) 
  (loop for cat in theory
	do (trace-print *trace-deduce* "~%~%=>Trying to prove category ~A"
			(vertex-prop cat))
	when (satisfy-children cat (vertex-children cat)
			       (example-values example))
	collect (vertex-prop cat)))


(defun satisfy-children (parent children example-vals)
  ;;-------------------------------------------------------------------------
  ;; Inputs: "children" is a list of rule structures (see structures.lisp).
  ;; Each child has a list of antecedents representing a rule (ie a
  ;; conjunction of antecedents) which, if satisfied, will suffice to prove
  ;; the parent (the caller of this routine). "example-vals" comes in as a
  ;; vector of values (see the example structure) NOT as the whole example
  ;; structure.
  ;;
  ;; Chris set this routine up to return "nil" as the default. Thus, if all
  ;; children are checked with no success, nil is returned. Then, he has two
  ;; nested loops, each of which will quit as soon as possible. The outer
  ;; loop checks through all the children any ONE of which must be true for
  ;; the whole routine to return true. The inner loop checks all the
  ;; antecedents for a given child and returns false if the number of 
  ;; failures exceeds the threshold. Otherwise, it returns true causing the
  ;; outer loop to quit.
  ;;
  ;; 20-NOV-92 (ptb): changed this routine to use a threshold to do the
  ;; deduction. The idea now is to quit the inner loop when the number of
  ;; failed antecedents exceeds the max allowed by the threshold.
  ;; Furthermore, if the threshold is negative, the rule can NEVER fire and
  ;; I quit immediately (this only happens during specialization in the event
  ;; that a threshold is set negative. Eventually, the specialization code
  ;; will delete such a rule). Also modified the code to avoid reproving (by 
  ;; checking if the mark on the parent is equal to *deduction-marker* -- 
  ;; see structures.lisp).
  ;;-------------------------------------------------------------------------
  (if (= *deduction-marker* (vertex-Dvisited parent))
      (return-from satisfy-children (vertex-Dvalue parent)))
  (setf (vertex-Dvisited parent) *deduction-marker*)
  (setf (vertex-Dvalue parent) nil)
  (loop for r in children  ;; for each rule supporting the parent
	for antes = (rule-antecedents r)
	for num-ok-failures = (rule-threshold r)
	when *trace-deduce*
	do (format t "~%~%trying to prove rule:")
	   (loop for k in (rule-antecedents r)
	         do (format t " ~A" (antecedent-prop k)))
	when 
	(and 
	 (rule-used r)
	 (>= num-ok-failures 0)
	 (loop for k in antes   ;; loop to test if child satisfiable
	       for v = (antecedent-vertex k)
	       for idx = (vertex-example-index v)
	       unless (if idx   ;; test to see if antecedent is provable
			  (or (antecedent-abduced? k);; No test if abduced
			      (leaf-test (antecedent-prop k) 
					 (aref example-vals idx)))
			  (or (antecedent-abduced? k);; No idx then recurse
			      (if (negation (antecedent-prop k))
				  (not (satisfy-children v
					 (vertex-children v) example-vals))
				  (satisfy-children v
				    (vertex-children v) example-vals))))
	       do (trace-print *trace-deduce*
			       "~%failed to prove antecedent ~A"
			       (antecedent-prop k))
	          (if (= num-ok-failures 0)
		      (return nil)
		      (incf num-ok-failures -1))

	       ;; an ante will pass the above test and still not be provable
               ;; if it is an unprovable intermediate. Test for that here. 
	       if (and (vertex-no-rules? v) (not (antecedent-abduced? k)))
	       do (if (= num-ok-failures 0)
		      (return nil)
		      (incf num-ok-failures -1))
	       
	       finally   ;; else if enough succed, return t
	       (return t)))
	do     ;; when one of the children succeds, quit with t
	(progn
	  (setf (vertex-Dvalue parent) t)
	  (return t))))
	

(defun leaf-test (condition example-value)
  ;;-------------------------------------------------------------------------
  ;; Called on a vertex when that vertex has an example index to see if the
  ;; conditions aresatisfied for the vertex.  Returns "t" or "nil" depending
  ;; upon whether the example value satisfies the condition.
  ;;
  ;; INPUTS: "condition" is the value for the "prop" field of the vertex in
  ;; question, and "example-value" is the value from the vector stored for
  ;; the example being tested, extracted by using the index of the
  ;; "example-index" field of the vertex.
  ;;
  ;; The implementation "cases" off the results of a "cond" statement. It is
  ;; assumed that the cond will return true, t, false or nil, and the case
  ;; then serves only to convert the return value to "t" or "nil". The cond
  ;; statement does all the work. Based upon the type of the condition
  ;; (which, at present, can be told by its length) the cond statement tests
  ;; the example value. There are currently 4 cases.
  ;; (1) If the condition is unary, it is assumed to be a true/false
  ;; predicate. The example value is just returned (here is where  the result
  ;; of the cond might be "true" or "false" as well as "t" or  "nil",
  ;; depending upon the example domain given to NEITHER). 
  ;; (2) If the condition is negated, then it looks like a normal condition
  ;; with a "not" wrapped around it, ie (not condition). Here leaf-test is
  ;; simply called recursively after stripping off the not, and the negation
  ;; of the results is returned.
  ;; (3) Conditions of length 2 are propositions like "(foot-type hoof)"
  ;; where the second element of the proposition is considered a value from
  ;; among a set of values allowable for the proposition. This second member
  ;; should match the example value.
  ;; (4) Finally, ithe condition is of length 3 then it is a a relational
  ;; proposition from among (>,<,>=,<=). These propositions look like "(size
  ;; <= 3)" and are evaluated by calling the second member as a function with
  ;; the example value as its first argument and the third member as its
  ;; second. >>> NOTE THIS SHOULD BE UPDATED TO HANDLE PROPOSITIONS LIKE:
  ;;  (size >= 3 < 5) , IE RANGE VALUES IN **ONE** PROPOSITION NOT TWO <<<
  ;;-------------------------------------------------------------------------
  (case  (cond ((atom condition)	; t or nil in vector
		example-value)
	       ((negation condition) 
		(not (leaf-test (second condition)  example-value)))
	       ((= (length condition) 2) 
		(equal (second condition) example-value))
	       ((= (length condition) 3)
		(funcall (second condition) example-value (third condition)))
	       (t (error "other things not done yet")))
    ((true t) t)
    ((false nil) nil)))


;;===========================================================================
;; TESTING CODE
;;
;; "test-deduce" is a diagnostic routine used to get information about the
;; current accuracy of a theory on a given set of examples.
;;===========================================================================

(defun back-xlate-example (neither-example)
  "Redefine this routine to print a number corresonding to the example."
  ;;-------------------------------------------------------------------------
  ;; A dummy routine which may be redefined in a data file if appropriate. It
  ;; takes an example in neither format and prints out some representation of
  ;; the example, typically a number for the example, using format.
  ;;
  ;; The idea behind this routine is that some data files may have a unique
  ;; numbering scheme for the examples which is not loaded into the example
  ;; in NEITHER format. Specifically, the routine "example-preparation-hook"
  ;; would translate from one example format to another that is compatible
  ;; with the "set-theory" routine (see io.lisp). 
  ;;
  ;; The "test-deduce" routine below provides a hook to print out examples in
  ;; the original numbering scheme by redefinition of this routine. In the 
  ;; past, I have redefined this routine in conjunction with a global 
  ;; variable defined and set in the data file which holds the currently 
  ;; loaded examples in the original untranslated format. Then, I can use 
  ;; "position" and "nth" to relate the position of the example in 
  ;; *neither-examples* to the position in the original list of examples. For
  ;; an example of how this routine would be written, see "nurse2.lisp".
  ;;-------------------------------------------------------------------------
  neither-example)


(defun test-deduce (&optional (silent nil) (use-float nil)
			      (examples *neither-examples*)
			      (theory *neither-theory*))
  "Returns three values: accuracy, number of failing positives, number of 
failing negatives. When silent=nil, prints out proof results for all examples
with asterisks marking failures."
  ;;-------------------------------------------------------------------------
  ;; A simple test of a theory. Loops through all examples, proving each in
  ;; turn using the theory. When an example proves more than its desired
  ;; category, it is collected for the final return. The list of examples
  ;; which were not accurately proved is returned.
  ;;-------------------------------------------------------------------------
  (if  examples
    (loop for ex in examples
          for cats = (prove-categories ex theory)
          with total = 0
          with num-correct = 0
          with num-pos = 0
          with num-neg = 0
          finally (progn
                    (when (not silent)
                      (format t "~%~%Percent correct: ~D% of total=~D"
                              (round (* 100 (/ num-correct total))) total)
                      (format t "~%Number failing positives: ~D" num-pos)
                      (format t "~%Number failing negatives: ~D" num-neg))
                    (return (values
			     (if use-float (/ num-correct total)
				 (round (* 100 (/ num-correct total))))
			     num-pos num-neg)))
          if (or (and (member (example-name ex) cats :test #'eq)
                      (= (length cats) 1))
                 (and (not cats)
                      (eq (example-name ex) *negative-category*)))
          do (incf num-correct)
             (if (not silent) (format t "~%  "))
          else do (if (not silent) (format t "~%**"))
                  (if (eq (example-name ex) *negative-category*)
                    (incf num-neg (length cats))
		    (if (member (example-name ex) cats :test #'eq)
			(incf num-neg (1- (length cats)))
			(progn (incf num-pos)
			       (incf num-neg (length cats)))))
          do (when (not silent)
               (format t "Example ~D(" total)
               (when (fboundp 'back-xlate-example)
                 (back-xlate-example ex))
               (format t "): ~A proven to be ~A"(example-name ex) cats))
             (incf total))
      (values 100 0 0)))


;;===========================================================================
;; SATISFACTION CODE
;;===========================================================================

(defun rules-satisfied (example &optional (theory *neither-theory*))
  "Given an example in neither format, prints out the numbers of the rules
satisfied by the example. Call number-theory to number a neither theory. See
also decompile-theory routine."
  ;;-------------------------------------------------------------------------
  ;; Prints a list of the numbers of the rules satisfied by the example.
  ;; Works in the same general fashion as the prove-categories code above,
  ;; except that it tries all rules to see if they are satisfied. The example
  ;; is assumed to have been already translated into NEITHER format.
  ;;
  ;; NOTE: this routine should be run only after the theory in question has
  ;; been numbered (see number-theory routine in debug.lisp).
  ;;-------------------------------------------------------------------------
  (incf *deduction-marker*)
  (format t "~%")
  (loop for cat in theory
	do (children-satisfied cat (vertex-children cat)
				 (example-values example))))


(defun children-satisfied (parent children example-vals)
  ;;-------------------------------------------------------------------------
  ;; Works similarly to satisfy-children routine except that it checks all
  ;; children.
  ;;-------------------------------------------------------------------------
  (if (= *deduction-marker* (vertex-Dvisited parent))
      (return-from children-satisfied (vertex-Dvalue parent)))
  (setf (vertex-Dvisited parent) *deduction-marker*)
  (setf (vertex-Dvalue parent) nil)
  (loop for r in children  ;; for each rule supporting the parent
	for antes = (rule-antecedents r)
	for num-ok-failures = (rule-threshold r)
	do
	;;(format t "~%~%trying to prove rule ~D" (rule-number r))
	(and 
	 (rule-used r)
	 (>= num-ok-failures 0)
	 (loop for k in antes   ;; loop to test if child satisfiable
	       for v = (antecedent-vertex k)
	       for idx = (vertex-example-index v)
	       with rule-satisfied = t
	       unless (if idx   ;; test to see if antecedent is provable
			  (or (antecedent-abduced? k);; No test if abduced
			      (leaf-test (antecedent-prop k) 
					 (aref example-vals idx)))
			  (or (antecedent-abduced? k);; No idx then recurse
			      (if (negation (antecedent-prop k))
				  (not (children-satisfied v
					  (vertex-children v) example-vals))
				  (children-satisfied v
				     (vertex-children v) example-vals))))
	       do (if (= num-ok-failures 0)
		      (setf rule-satisfied nil)
		      (incf num-ok-failures -1))
	       
	       ;; an ante will pass the above test and still not be provable
               ;; if it is an unprovable intermediate. Test for that here. 
	       if (and (vertex-no-rules? v) (not (antecedent-abduced? k)))
	       do (if (= num-ok-failures 0)
		      (setf rule-satisfied nil)
		      (incf num-ok-failures -1))
	       
	       finally   ;; else if enough succed, print rule number
	       (progn
		 (setf (vertex-Dvalue parent)
		       (or rule-satisfied (vertex-Dvalue parent)))
		 (when rule-satisfied
		   (format t " ~D" (rule-number r))
		   (if (> (rule-threshold r) 0)
		       (format t "(~D)" (+ (- (length antes) (rule-threshold r))
					   num-ok-failures)))))))
	 finally
	 (return (vertex-Dvalue parent))))
	
