
;;===========================================================================
;; Debugging code for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof.
;; -------------------------------------------------------------------------
;;
;; This routine contains all the formatting routines for printing out various
;; data structures.
;;
;; CHANGE HISTORY
;;
;; 01-OCT-92: (ptb) altered the decompile theory routine to print out the number 
;;            of times each element of each rule was visited.
;; 17-DEC-92: (ptb) altered the decompile theory routine again to take an 
;;            optional argument dictating whether or not to print out the 
;;            visit count information.
;; 02-MAR-93: (ptb) added code to assign numbers to a theory and to print the
;;            theory out using numbers (see decompile-theory routine).
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

#+:cltl2(declaim (special *neither-theory*)
                 (special *neither-examples*))
#-:cltl2(progn
          (proclaim '(special *neither-theory*)) 
          (proclaim '(special *neither-examples*))) ;; see io.lisp


(defun pprint-vertex (v &optional (recurse t) (level 0))
  ;;-------------------------------------------------------------------------
  ;; Print routine for a vertex. Prints the vertex, then recursively prints
  ;; all its children. Each recursive call is indented by 2 spaces.
  ;; Definition for vertex is in ``structures.lisp''.
  ;;-------------------------------------------------------------------------
  (format t "~%~V@T-----" level)
  (format t "~%~V@TPROP:~A" level (vertex-prop v))
  (format t "~%~V@TEXAMPLE-INDEX:~A" level (vertex-example-index v))
  (format t "~%~V@TNO-RULES?:~A" level (vertex-no-rules? v))
  (format t "~%~V@TCHILDREN:" level)
  (incf level 2)
  (if (vertex-children v)
      (dolist (ante-lists (vertex-children v))
	(format t "~%~V@T(" level)
	(dolist (a (rule-antecedents ante-lists))
	  (format t "~%~V@TABDUCED?:~A" level (antecedent-abduced? a))
	  (format t "~%~V@TPROP:~A" level (antecedent-prop a))
	  (if recurse
	      (pprint-vertex (antecedent-vertex a) recurse level)))
	(format t "~%~V@T)" level))
      (format t "()")))


;;===========================================================================
;; DECOMPILING ROUTINES	(for data structure created in ``io.lisp'')
;;===========================================================================

(defun decompile-theory (&optional (show-calls nil) (show-nums t)
                                   (compiled-theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; Takes a theory of the form stored in *neither-theory* and returns a
  ;; human-readable format of the theory (ie, something much like what it
  ;; read in). Can work with a theory that is a list of category trees (as is
  ;; typical of *neither-theory*) or will work with a single category tree.
  ;; Useful for printout statements during the specialization and
  ;; generalization processes.
  ;; (ptb 12-17-92) modified the routine to take an optional argument telling
  ;; whether or not to print out the number of calls made to each vertex of 
  ;; the theory.
  ;; (ptb 3-2-93) modified to print out the number of the rule as well, if
  ;; the appropriate optional argument is set to t (note, default = t).
  ;;-------------------------------------------------------------------------
  (let (*has-printed*)
    (declare (special *has-printed*))
    (cond ((listp compiled-theory)
	   (dolist (r (loop for vert in compiled-theory
			    nconcing
			    (tree-to-theory vert show-calls show-nums)))
	     (format t "~%~A" r)))
	  ((vertex-p compiled-theory)
	   (dolist (r (tree-to-theory compiled-theory show-calls show-nums)) 
             (format t "~%~A" r)))
	  (t (error "~A not a theory~%" compiled-theory)))))


(defun tree-to-theory (theory-tree &optional (show-calls nil) (show-nums t))
  ;;-------------------------------------------------------------------------
  ;; A recursive routine to "print" the theory stemming from the vertex
  ;; "theory-tree". Works in two steps. First, all children of the
  ;; "theory-tree" vertex are turned into "rules" with the "prop" field as
  ;; the head. Recall that "vertex-children" returns a list of antecedent
  ;; lists; thus, the prop fields of the vertices in each antecedent list are
  ;; collected together to form a rule. Each antecedent list forms a separate
  ;; rule. Next, a recursive call is made to expand each of the children
  ;; (which will only be expanded if they have further children). Recursion
  ;; ends for vertices with no children. All results are nconc'ed together.
  ;; (ptb 3-2-93) modified to print out the numbers of rules. The two
  ;; routines after this (number-theory and number-rules) will number each
  ;; rule of the theory in order of depth first search. This routine simply
  ;; checks to see if "show-nums" is t and, if so, prints out the number
  ;; assigned by number-theory.
  ;;-------------------------------------------------------------------------
  (declare (special *has-printed*))
  (let ((kids (vertex-children theory-tree))
	(myprop (vertex-prop theory-tree)))
    (unless (member theory-tree *has-printed*)
      (setf *has-printed* (cons theory-tree *has-printed*))
      (nconc (loop for k in kids
		   for thresh = (rule-threshold k)
		   if (rule-used k)
		   collect 
		   (loop for i in (rule-antecedents k)
			 if (not (antecedent-abduced? i))
			 collect (if show-calls
                                   (cons 
                                    (vertex-Dvisited (antecedent-vertex i))
                                    (antecedent-prop i))
                                   (antecedent-prop i)) into clause
			 finally
			 (return (if (and show-nums (>= (rule-number k) 0))
				     (format nil "~D: ~A" (rule-number k)
					     `(,thresh ,myprop <- ,@clause))
				     `(,thresh ,myprop <- ,@clause)))))
	     (loop for k in kids
		   if (rule-used k)
		   nconc 
		   (loop for i in (rule-antecedents k)
			 for vert = (antecedent-vertex i)
			 nconc
			 (tree-to-theory vert show-calls show-nums)))))))


(defun number-theory (&optional (compiled-theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; Takes a theory of the form stored in *neither-theory* and assigns
  ;; numbers to the rules in that theory.
  ;;-------------------------------------------------------------------------
  (let (*has-numbered* (*number-val* 0))
    (declare (special *has-numbered*) (special *number-val*))
    (cond ((listp compiled-theory)
	   (loop for vert in compiled-theory
                 do (number-rules vert)))
	  ((vertex-p compiled-theory)
	   (number-rules compiled-theory))
	  (t (error "~A not a theory~%" compiled-theory)))))


(defun number-rules (theory-tree)
  ;;-------------------------------------------------------------------------
  ;; A recursive routine to assign numbers to the rules of a vertex. Works in
  ;; two steps. First, all children of the "theory-tree" vertex assigned a
  ;; number, then all the children are visited again with a recursive call to
  ;; this routine.
  ;;-------------------------------------------------------------------------
  (declare (special *has-numbered*) (special *number-val*))
  (let ((kids (vertex-children theory-tree)))
    (unless (member theory-tree *has-numbered*)
      (setf *has-numbered* (cons theory-tree *has-numbered*))
      (nconc (loop for k in kids
		   if (rule-used k)
		   do (setf (rule-number k) *number-val*)
		      (incf *number-val*))
	     (loop for k in kids
		   if (rule-used k)
		   do (loop for i in (rule-antecedents k)
			    for vert = (antecedent-vertex i)
			    do (number-rules vert)))))))



;;===========================================================================
;; COVER PRINTING ROUTINES
;;
;; The following are used to print out elements of the rule cover and
;; antecedent cover "fixes" (see definitions in generalize.lisp and
;; specialize.lisp).
;;===========================================================================

(defun pprule-fix (fix)
  ;;-------------------------------------------------------------------------
  ;; Given a rule cover where "cover" is of the form (del del ...) and each
  ;; del is a deletion which uses the fields (parent child), this routine
  ;; prints out all the dels using the "pprule" routine below. 
  ;;-------------------------------------------------------------------------
  (format t "~%* ")
  (pprule (first fix) nil)
  (loop for del in (cdr fix) do (format t "~%  ") (pprule del nil)))


(defun pprule (deletion &optional (nl t) (print-anyway nil))
  ;;-------------------------------------------------------------------------
  ;; takes a deletion (see structures.lisp) which has a "parent" vertex and a
  ;; "child" rule which is deleted from the list of rules for that parent.
  ;;
  ;; This routine prints out the rule using a <- notation. No newlines or
  ;; spaced are put before or after the rule.
  ;;-------------------------------------------------------------------------
  (if nl (format t "~%"))
  (format t "~D:~A <-" (rule-threshold (deletion-child deletion))
	  (vertex-prop (deletion-parent deletion)))
  (dolist (ante (rule-antecedents (deletion-child deletion)))
    (if (or (not (antecedent-abduced? ante))
            (member ante print-anyway))
      (format t " ~A" (antecedent-prop ante)))))


(defun ppabduction (abduction)
  ;;-------------------------------------------------------------------------
  ;; Given an abduction which is a deletion structure that uses all three
  ;; fields of the deletion (see structure.lisp) prints out the rule followed
  ;; by the antecedents to be deleted from the rule. 
  ;;-------------------------------------------------------------------------
  (format t "~%  in rule:~%    ")
  (pprule abduction nil (deletion-antes abduction))
  (format t "~%  remove the following:")
  (dolist (ante (deletion-antes abduction))
    (format t "~%    ~A" (antecedent-prop ante))))


(defun ppfix-positive (positive-fix)
  ;;-------------------------------------------------------------------------
  ;; Prints a positive fix, which consists of one or more abductions listed
  ;; together.
  ;;-------------------------------------------------------------------------
  (dolist (ab positive-fix) (ppabduction ab)))


(defun ppfix-list (fix-list genl-or-spec)
  ;;-------------------------------------------------------------------------
  ;; Given fix-list, a list of solution structures (see structure.lisp), this
  ;; routine prints out each fix and the example associated with it. Based on
  ;; the value of "genl-or-spec" this routine will call different routines to
  ;; treat the various structure fields differently.
  ;;-------------------------------------------------------------------------
  (loop for soln in fix-list
	for fix = (solution-fix soln)
	for ex  = (solution-example soln)
	do
	(format t "~%~%* For the following example:")
        (if (eq genl-or-spec 'genl)
	  (ppexample ex)
          (ppexample (provable-neg-example ex)))
	(format t "~%~%  Fix as follows:")
        (if (eq genl-or-spec 'genl)
	  (ppfix-positive fix)
          (dolist (del fix) (pprule del t)))))


(defun ppexample-num (example-pos)
  ;;-------------------------------------------------------------------------
  ;; prints out an example. Must look through the value stored in the example
  ;; array and the *feature-names* list to reconstruct the example
  ;;-------------------------------------------------------------------------
  (let ((example (nth example-pos *neither-examples*)))
    (format t "~%  category: ~A" (example-name example))
    (format t "~%  features:")
    (loop for f in *feature-names*
	  with inx = 0
	  with values = (example-values example)
	  do
	  (if (evenp inx) (format t "~%  "))
	  (format t " (~A ~A)" f (aref values inx))
	  (incf inx))))


(defun ppexample (example)
  ;;-------------------------------------------------------------------------
  ;; prints out an example. Must look through the value stored in the example
  ;; array and the *feature-names* list to reconstruct the example
  ;;-------------------------------------------------------------------------
  (format t "~%  category: ~A" (example-name example))
  (format t "~%  features:")
  (loop for f in *feature-names*
	with inx = 0
	with values = (example-values example)
	do
	(if (evenp inx) (format t "~%  "))
	(format t " (~A ~A)" f (aref values inx))
	(incf inx)))


(defun common-feature-values (examples &optional feature value)
  ;;-------------------------------------------------------------------------
  ;; Given a list of examples in NEITHER's internal format, this routine will
  ;; first search the examples for those which have the given feature-value 
  ;; pair. With this second subset of examples, the routine searchs again, 
  ;; returning a list of all the feature-value pairs that the second list of 
  ;; examples share in common.
  ;;-------------------------------------------------------------------------
  (let (inx subset fv-pairs)
    (setf subset examples)
    (when (and feature value)
      (setf inx (position feature *feature-names*))
      (setf subset (loop for e in examples
			 if (equal value (aref (example-values e) inx))
			 collect e into rval
			 finally (return rval))))
    (setf fv-pairs (loop for f in *feature-names*
			 for inx from 0 to (1- (length *feature-names*))
			 with exvals = (example-values (car subset))
			 collect (cons f (aref exvals inx))))
    (loop for ex in (cdr subset)
	  for exvals = (example-values ex)
	  do
	  (setf fv-pairs
		(loop for fv in fv-pairs
		      for inx = (position (car fv) *feature-names*)
		      if (equal (cdr fv) (aref exvals inx))
		      collect fv)))
    (format t "~%Total examples with ~a = ~a: ~D" feature value (length subset))
    (format t "~%Feature-Value pairs in common:")
    (loop for fv in fv-pairs do (format t "~%  ~A = ~A" (car fv) (cdr fv)))))


(defun translate-example (example)
  ;;-------------------------------------------------------------------------
  ;; translates the incoming example back into a list format
  ;;-------------------------------------------------------------------------
  (cons (example-name example)
	(loop for f in *feature-names*
	      with inx = 0
	      with values = (example-values example)
	      collect
	        (list f (aref values inx))
	      do (incf inx))))

	
;;===========================================================================
;; TRACE ROUTINES
;;===========================================================================

(defvar *trace-spec* nil)
(defvar *trace-genl* nil)
(defvar *trace-deduce* nil)
(defvar *trace-pos-fix* nil)
(defvar *trace-neg-fix* nil)
(defvar *trace-common-model* nil)


(defun trace-neither-status ()
  "Prints the values of the NEITHER trace flags."
  ;;-------------------------------------------------------------------------
  ;; Prints the values of the NEITHER trace flags.
  ;;-------------------------------------------------------------------------
  (format t "~%NEITHER trace flag values:")
  (format t "~%   *trace-deduce*  = ~D" *trace-deduce*)
  (format t "~%   *trace-genl*    = ~D" *trace-genl*)
  (format t "~%   *trace-pos-fix* = ~D" *trace-pos-fix*)
  (format t "~%   *trace-spec*    = ~D" *trace-spec*)
  (format t "~%   *trace-neg-fix* = ~D" *trace-neg-fix*))


(defun trace-neither (&key (spec nil) (genl nil) (deduce nil) (pos-fix nil)
			  (neg-fix nil) (all-on nil) (all-off nil))
  ;;-------------------------------------------------------------------------
  ;; Used to set the *trace-spec* and *trace-genl* flags, which print out
  ;; messages during specialization and generalization respectively.
  ;;-------------------------------------------------------------------------
  (cond (all-off (setf *trace-spec* nil
		       *trace-genl* nil
		       *trace-pos-fix* nil
		       *trace-neg-fix* nil
		       *trace-deduce* nil))
	(all-on (setf *trace-spec* t
		      *trace-pos-fix* t
		      *trace-neg-fix* t
		      *trace-deduce* t
		      *trace-genl* t))
	(t
	 (setf *trace-genl* genl
	       *trace-spec* spec
	       *trace-pos-fix* pos-fix
	       *trace-neg-fix* neg-fix
	       *trace-deduce* deduce))))
