#include  "defns.i"
#include  "extern.i"

/*****************************************************************************/
/*                                                                           */
/*  Find potential orderings between pairs of arguments of each relation     */
/*  - for type number TN                                                     */
/*                                                                           */
/*****************************************************************************/


    FindArgumentOrders(TN)
/*  ------------------  */

int TN;

{
    int i;
    Relation R;

    VERBOSE(2) 
        printf("\nFinding argument orders for type %s\n", Type[TN]->Name);
    ForEach(i, 1, MaxRel)
    {
        R = Reln[i];
	if( R==SAMEVAR || R==CONTGT) 
	{
	    R->PosOrder = CreateOrderArray(2);
	    R->NegOrder = CreateOrderArray(2);
        }
	else if( R->Arity == 1 )
	{
	    R->PosOrder = CreateOrderArray(1);
	    R->NegOrder = CreateOrderArray(1);
	}
        else
	{
	    FindArgumentOrder(R,TN);
	}
    }
}

/*****************************************************************************/
/*                                                                           */
/*  Find potential orderings between pairs of arguments of R                 */
/*  where relevant to type number TN                                         */
/*                                                                           */
/*****************************************************************************/


    FindArgumentOrder(R, TN)
/*  ----------------    */

Relation R;
int TN;

{
    int FirstVar, SecondVar, N;
    Boolean AllOrderedPos;

    int i, NumberNeg, Size;
    Tuple Case;

    TypeInfo T=Type[TN];

    VERBOSE(2) printf("\tChecking argument order of %s\n", R->Name);

    N = R->Arity;

    R->PosOrder = CreateOrderArray(N);
    R->NegOrder = CreateOrderArray(N);

    AllOrderedPos = true;
    ForEach(FirstVar, 1, N-1)
    {
        ForEach(SecondVar, FirstVar+1, N)
	{
	    VERBOSE(2) 
	        printf("\t\tChecking arguments %d %d: ", FirstVar, SecondVar);
	    if(Compatible[R->Type[FirstVar]][TN]
	       && Compatible[R->Type[SecondVar]][TN])
	    { 
	        if(PossibleOrder(T, R->Pos, FirstVar, SecondVar))
		{
		    if(CheckForOrder(T, R->Pos, FirstVar, SecondVar, false))
		    {
		        VERBOSE(2) printf("possible order is consistent");
			R->PosOrder[FirstVar][SecondVar]
			  = R->PosOrder[SecondVar][FirstVar] = true;
		    }
		    else
		    {
		        VERBOSE(2) printf("possible order is inconsistent");
			AllOrderedPos = false;
		    }
		}
		else
		{
		    VERBOSE(2) printf("no constraint on ordering this type");
		    AllOrderedPos = false;
		}
	    }
	    else
	    {
	        VERBOSE(2) printf("irrelevant for ordering this type");
	    }
	    VERBOSE(2) printf("\n");
	}
    }
    if(AllOrderedPos || !NEGLITERALS) return;

    VERBOSE(2) printf("\tConsider argument order of ~%s\n", R->Name);

    if(!R->Neg) /* Need to create negative tuples */
    {
        NumberNeg = 1;
        ForEach(i, 1, R->Arity)
	{
            NumberNeg*= R->TypeRef[i]->NValues;
	}
	NumberNeg = NumberNeg - Number(R->Pos);

	if(NumberNeg>=Number(R->Pos))
	{
	    VERBOSE(2)
	    {
	        printf("\tbut number of negative tuples >= positive tuples\n");
	    }
	    return;
	}

	R->Neg = (Tuple*) pmalloc((NumberNeg+1) * sizeof(Tuple));
	NumberNeg = 0;

	MaxVar = R->Arity;
	Size = (R->Arity+1) * sizeof(Const);
        Case = Nil;
        while ( (Case = NextConstTuple(R, Case)) )
	{
            if ( ! Join(R->Pos, R->PosIndex, DefVars, Case, MaxVar, true) )
	    {
                R->Neg[NumberNeg] = (Tuple) pmalloc(Size);
                memcpy(R->Neg[NumberNeg], Case, Size);
		R->Neg[NumberNeg][0] = 0;
                NumberNeg++;
	    }
	}
	R->Neg[NumberNeg] = Nil;
    }

    ForEach(FirstVar, 1, N-1)
    {
        ForEach(SecondVar, FirstVar+1, N)
	{
	    VERBOSE(2)
	        printf("\t\tChecking arguments %d %d: ", FirstVar, SecondVar);
	    if(Compatible[R->Type[FirstVar]][TN]
	       && Compatible[R->Type[SecondVar]][TN]
	       && PossibleOrder(T, R->Neg, FirstVar, SecondVar))
	    {
	        if(CheckForOrder(T, R->Neg, FirstVar, SecondVar, false))
		{
		    VERBOSE(2) printf("possible order is consistent");
		    R->NegOrder[FirstVar][SecondVar]
		      = R->NegOrder[SecondVar][FirstVar] = true;
		}
		else
		{
		    VERBOSE(2) printf("possible order is inconsistent");
		}
	    }
	    else
	    {
	        VERBOSE(2) printf("irrelevant for ordering this type");
	    }
	    VERBOSE(2) printf("\n");
	}
    }

    return;
}

/*****************************************************************************/
/*                                                                           */
/*  Create a zeroed boolean (N(+1))x(N+1) table                              */
/*                                                                           */
/*****************************************************************************/

Boolean **CreateOrderArray(N)
/*        ----------------   */

int N;

{
    int i;
    Boolean **Result;

    Result = (Boolean**) pmalloc( (N+1)*sizeof(Boolean*));
    Result[0] = Nil;
    ForEach(i, 1, N)
    {
        Result[i] = pcalloc((N+1), sizeof(Boolean));
    }

    return Result;
}

/*****************************************************************************/
/*                                                                           */
/* OrderTheConstants - order all the constants                               */
/*                                                                           */
/*****************************************************************************/

      OrderTheConstants()
/*    -----------------  */

{
    int i;

    ForEach(i,1,MaxType)
    {
        if(Type[i]->FixedPolarity) continue;
        VERBOSE(2) printf("\nOrdering constants of type %s\n",Type[i]->Name);
        FindArgumentOrders(i);
	OrderCsForType(i);
    }
    return;
}


/*****************************************************************************/
/*                                                                           */
/* OrderCsForType - order the constants for a given type                     */
/*                                                                           */
/*****************************************************************************/

      OrderCsForType(TypeN)
/*    --------------  */

int TypeN;

{
    int **POList, *partPOList;
    int i, j, k, l, n_pos, N;
    Relation R;
    int *BestSet, *CurrentSet;
    Boolean *POClosure, *BestPOClosure;
    int *ConstantOrder;
    TypeInfo T;

    T = Type[TypeN];
    N = T->NValues;

    n_pos = 0;
    ForEach(i, 1, MaxRel)
    {
        R = Reln[i];
	ForEach(j, 1, R->Arity-1)
	{
	    ForEach(k, j+1, R->Arity)
	    {
	        if(R->PosOrder[j][k]) n_pos++;
		if(R->NegOrder[j][k]) n_pos++;
	    }
	}
    }
    n_pos*=2; /* Considering both polarities */

    if(n_pos)
    {
        T->Ordered = true; /* An order exists for the type */
        VERBOSE(2)
	  printf("\n\t%d possible ordered argument pairs\n", n_pos);
    }
    else
    {
        T->Ordered = false; /* No order exists for the type */
	VERBOSE(1)
	  printf("\n\tConstants of type %s are unordered\n", T->Name);
	return;
    }

    POList = (int**) pmalloc((n_pos+1)*sizeof(int*));

    n_pos = 1;
    ForEach(i, 1, MaxRel)
    {
        R = Reln[i];
	ForEach(j, 1, R->Arity-1)
	{
	    ForEach(k, j+1, R->Arity)
	    {
	        if(R->PosOrder[j][k])
		{
		    partPOList = (int*) pmalloc(4*sizeof(int));
		    partPOList[0] = i;
		    partPOList[1] = j;
		    partPOList[2] = k;
		    partPOList[3] = 1;
		    POList[n_pos++] = partPOList;
		    partPOList = (int*) pmalloc(4*sizeof(int));
		    partPOList[0] = i;
		    partPOList[1] = k;
		    partPOList[2] = j;
		    partPOList[3] = 1;
		    POList[n_pos++] = partPOList;
		}
	        if(R->NegOrder[j][k])
		{
		    partPOList = (int*) pmalloc(4*sizeof(int));
		    partPOList[0] = i;
		    partPOList[1] = j;
		    partPOList[2] = k;
		    partPOList[3] = 0;
		    POList[n_pos++] = partPOList;
		    partPOList = (int*) pmalloc(4*sizeof(int));
		    partPOList[0] = i;
		    partPOList[1] = k;
		    partPOList[2] = j;
		    partPOList[3] = 0;
		    POList[n_pos++] = partPOList;
		}
	    }
	}
    }

    if(n_pos-1) /* There is at least one partial ordering constraint */
    {
        VERBOSE(2)
	{
	    for(i = 1 ; i < n_pos ; i++)
	    {
	        printf("\t\t%d: %s %s %s>%s\n", i, 
		   POList[i][3] ? " ":"~", Reln[POList[i][0]]->Name, 
	           VariableName[POList[i][1]], 
		   VariableName[POList[i][2]]);
	     }
	}

        /* Set up for call to find most compatible set of POs, as measured by
           the number of pairs of relation arguments that are ordered */

	CurrentSet = Nil;
	POClosure = Nil;
	BestPOClosure = pmalloc((T->NValues)*(T->NValues));
	memset(BestPOClosure, false, (T->NValues)*(T->NValues));
	BestSet = (int*) pcalloc(n_pos, sizeof(int));

	MostCompatiblePOs(T, n_pos-1, POList, BestSet, CurrentSet, 
			  BestPOClosure, POClosure);

	ConstantOrder = (int*) pmalloc(N*sizeof(int));

	ChooseAnOrder(T, ConstantOrder, BestPOClosure);

	free(BestPOClosure);
	free(BestSet);
    }
    else /* No Partial Order Constraint */
    {
        ConstantOrder = (int*) pmalloc(N*sizeof(int));
	for(i=0;i<N;i++) ConstantOrder[i]=i;
    }

    /* Implement the chosen constant order */

    VERBOSE(1)
    {
        printf("\n\tConstants of type %s are in the order:\n\t\t", T->Name);
    }

    j = 0;
    while(j<N)
    {
        k = T->Value[ConstantOrder[j]];
        VERBOSE(1)
	{
	    printf("%s", ConstName[k]);
	    if(j<N-1)printf(",");
	}
        T->CollSeq[k] = ++j;
    }
    VERBOSE(1) printf("\n");

    for(i=1;i<n_pos;i++)
    {
        free(POList[i]);
    }
    free(POList);
    free(ConstantOrder);

    return;
}

/*****************************************************************************/
/*                                                                           */
/* MostCompatiblePOs - find a compatible set of POs so that the number of    */
/*                     interargument orders is maximised                     */
/*                                                                           */
/*****************************************************************************/

MostCompatiblePOs(T,N,POList, BestSet, CurrentSetIn,BestPOClosure, POClosureIn)
/*---------------                                                            */

int N, **POList, *BestSet, *CurrentSetIn;
Boolean *POClosureIn, *BestPOClosure;
TypeInfo T;

{
    int i, *CurrentSet;
    Boolean *POClosure, CarryOn;
    int j;

    if(CurrentSetIn)
    {
        CurrentSet = (int*) pmalloc((N+1)*sizeof(int));
	memcpy(CurrentSet, CurrentSetIn, (N+1)*sizeof(int));
        POClosure = pmalloc((T->NValues)*(T->NValues));
	memcpy(POClosure, POClosureIn, (T->NValues)*(T->NValues));
	VERBOSE(2)
	{
	    printf("\t ");
	    for( i = 1 ; i <= CurrentSet[0] ; i++ ) 
	        printf(" %d", CurrentSet[i]);
	}
	CarryOn = StillConsistent(T, POClosure, CurrentSet, POList);
	VERBOSE(2) 
	{
	    if(CarryOn&&CurrentSet[0]>1) printf(" is consistent");
	    else 
	        if(!CarryOn) printf(" is inconsistent");
	}

	if(CarryOn && CurrentSet[0]>BestSet[0])
	{
	    for( i = 0 ; i <= CurrentSet[0] ; i++ ) BestSet[i] = CurrentSet[i];
	    memcpy(BestPOClosure, POClosure, (T->NValues)*(T->NValues));
	    VERBOSE(2)
	    {
	        printf(" - best set so far\n");
	    }
	}
	else
	{	
	    VERBOSE(2) printf("\n");
	}
    }
    else
    {
        CurrentSet = (int*) pcalloc(N+1, sizeof(int));
	POClosure = pmalloc((T->NValues)*(T->NValues));
	memset(POClosure, false, (T->NValues)*(T->NValues));
	CarryOn = true;
    }

    if(CarryOn && CanExtend(N, CurrentSet))
    {
        while(CheckNext(N, CurrentSet, BestSet))
	{
	    MostCompatiblePOs(T, N, POList, BestSet, CurrentSet, 
			      BestPOClosure, POClosure);
	}
    }

    free(CurrentSet);
    free(POClosure);

    return;

}

/*****************************************************************************/
/*                                                                           */
/* CanExtend - trivial function to check if this set can be extended to form */
/*             a larger one as yet unsearched                                */
/*                                                                           */
/*****************************************************************************/

Boolean CanExtend(N, Set)
/*      ---------        */

int N, *Set;

{
    if(Set[Set[0]]<N)
    {
	if(Set[0]) /* Can skip inverse of preceding order */
	{
	    Set[Set[0]+1] = Set[Set[0]] + (Set[Set[0]]%2);
	}
	else
	{
	    Set[1] = -1; /* Initialisation for CheckNext */
	}
	(Set[0])++;
	return true;
    }
    else return false;
}

/*****************************************************************************/
/*                                                                           */
/* CheckNext  - determines if there is a further set to be tried at the      */
/*              current level                                                */
/*                                                                           */
/*****************************************************************************/

Boolean CheckNext(N, Set, Best)
/*      ---------             */

int N, *Set, *Best;

{
    if(Set[Set[0]]<N)
    {
        (Set[Set[0]])++;
	if(Set[0]==1) Set[1]++; /* Skip inverse of preceding order at top */
	if( (Set[0]+N/2-(Set[Set[0]]+1)/2) > Best[0] ) return true;
    }
    return false;
}

/*****************************************************************************/
/*                                                                           */
/* ChooseAnOrder - given the closure of the best set of partial orders for   */
/*                 type T, select an order for the constants of type T       */
/*                 Method uses a quicksort with ordering criterion being the */
/*                 number of constants less than the constant in the closure */
/*                 - tie-break on order in type as entered                   */
/*                                                                           */
/*****************************************************************************/

    ChooseAnOrder(T, ConstantOrder, POClosure)
/*  -------------                            */

int *ConstantOrder;
Boolean *POClosure;
TypeInfo T;

{
    int i, j, k, swap, N;
    Boolean *Used;
    int *SubOrder;
    int *NLT;

    N = T->NValues;

    /* Find counts of Number of Constants Less Than */

    NLT = (int*) pmalloc(N*sizeof(int));
    ForEach(i,1,N)
    {
        NLT[i-1] = 0;
	ForEach(j,1,N) if(OrderCheck(POClosure,N,i,j)) NLT[i-1]++;
    }

    /* Partial Order Based Quick Sort */

    for(i=0;i<N;i++) ConstantOrder[i] = i;

    POQS(ConstantOrder,0,N-1,NLT);

    return;
}

/*****************************************************************************/
/*                                                                           */
/* AddOrderingsDueTo(L) Update the order information to take account of new  */
/*                      recursive literal                                    */
/*                                                                           */
/*****************************************************************************/

AddOrderingsDueTo(L)

Literal L;

{
    int i, j, N, t, *Seq;
    Relation R;

    R = L->Rel;
    N = R->Arity;

    if(NRecLitCl && ((NRecLitCl%100)==0) )
        RecLitCl = (Ordering**)
            prealloc(RecLitCl, (101+100*(NRecLitCl/100))*sizeof(Ordering*));

    RecLitCl[++NRecLitCl] = (Ordering*) pmalloc((N+1)*sizeof(Ordering));

    ForEach(i,1,N) RecLitCl[NRecLitCl][i] = PartialOrder[L->Args[i]][i];

    L->RefOrderedArg = (Ordering*) pmalloc(N+1);
    ForEach(i,1,N) L->RefOrderedArg[i] = PartialOrder[L->Args[i]][i];

    return;
}

/*****************************************************************************/
/*                                                                           */
/* RecursiveCallOK - Determines whether the argument orderings on a          */
/*                   recursive call are OK                                   */
/*                                                                           */
/*****************************************************************************/

Boolean RecursiveCallOK(A)

Vars A;

{
    int i, j, k, N, NColumnLeft, NRowLeft;
    int ordercount, bestcount, bestcolumn;
    Ordering *ThisCall, ThisOrder;
    Boolean *ColumnLeft, *RowLeft;

    N = Target->Arity;

    /* First need to establish ordering constraints for these arguments */

    ThisCall = RecLitCl[0];
    ForEach(i,1,N)
    {
        if(A[i]>MaxVar)
	{
	    ThisCall[i] = '#';
	}
	else
	{
	    ThisCall[i] = PartialOrder[A[i]][i];
	}
    }

    /* Now use Quinlan's procedure to check ordering constraints
       first cut implementation - crude handling of column and row deletion,
       but faster on examples tried than constructing linked list */

    NColumnLeft = N;
    ColumnLeft = (Boolean*) pmalloc(N+1);
    memset(ColumnLeft, true, N+1);
    NRowLeft = NRecLitCl+1;
    RowLeft = (Boolean*) pmalloc(NRecLitCl+1);
    memset(RowLeft, true, NRecLitCl+1);

    while(NRowLeft&&NColumnLeft)
    {
        bestcolumn = 0;
	bestcount = 0;
	ForEach(k,1,N)
	{
	    if(!ColumnLeft[k]) continue;
	    ordercount = 0;
	    ThisOrder = '=';
	    ForEach(j,0,NRecLitCl)
	    {
	        if(!RowLeft[j]) continue;
	        if(ThisOrder == '<')
		{
		    if(RecLitCl[j][k] == '<')
		        ordercount++;
		    else if(RecLitCl[j][k] != '=')
		        ThisOrder = '#';
		}
		else if(ThisOrder == '>')
		{
		    if(RecLitCl[j][k] == '>')
		        ordercount++;
		    else if(RecLitCl[j][k] != '=')
		        ThisOrder = '#';
		}
		else if(ThisOrder == '=')
		{
		    ThisOrder = RecLitCl[j][k];
		    if( (ThisOrder=='<') || (ThisOrder=='>') )
		        ordercount++;
		}
		if(ThisOrder == '#') /* Column Unordered */
		{
		    ordercount = 0;
		    break;
		}
	    }
	    if(ordercount>bestcount)
	    {
	        bestcount = ordercount;
		bestcolumn = k;
	    }
	}

	if(!bestcount) /* Recursive Call Not OK */
	{
	    free(ColumnLeft);
	    free(RowLeft);
	    return false;
	}

	/* Process best column */

	ForEach(j,0,NRecLitCl)
	{
	    if( RowLeft[j] && (RecLitCl[j][bestcolumn] != '=') )
	    {
	        RowLeft[j] = false; /* delete row */
		NRowLeft--;
	    }
        }

	ColumnLeft[bestcolumn] = false; /* delete column */
	NColumnLeft--;
    }

    free(ColumnLeft);
    free(RowLeft);

    if(!NRowLeft) return true;
    else return false;

}

/*****************************************************************************/
/*                                                                           */
/* PossibleOrder - check if there's a possible ordering constraint for type T*/
/*                 given by ordering the arguments First and Second in the   */
/*                 TupleSet                                                  */
/*                                                                           */
/*****************************************************************************/

Boolean PossibleOrder(T, TupleSet, First, Second)
/*      -------------                         */

int First, Second;
Tuples TupleSet;
TypeInfo T;

{
    int i, j, FirstConst, SecondConst, swap;
    Boolean Possible;

    Possible = false;

    while( *TupleSet && !Possible )
    {
        FirstConst = (*TupleSet)[First];
	SecondConst = (*TupleSet)[Second];

	if(T->CollSeq[FirstConst] && T->CollSeq[SecondConst])
	    Possible = true;

	TupleSet++;
    }

    return Possible;
}
        
/*****************************************************************************/
/*                                                                           */
/* POQS - Sorts v[left]...v[right] according to NLT criterion                */
/*        Tie break on numbering of items                                    */
/*                                                                           */
/*****************************************************************************/

    POQS(v, left, right, NLT)
/*  -----------                            */

int *v, left, right;
int *NLT;

{
    int i, last, swap;

    if(left>=right) return;

    swap = v[left];
    v[left] = v[(left+right)/2];
    v[(left+right)/2] = swap;

    last = left;

    for(i = left+1; i <= right; i++)
        if( (NLT[v[i]]<NLT[v[left]]) ||
	    (NLT[v[i]]==NLT[v[left]] && v[i]<v[left]) )
	{
	    swap = v[++last];
	    v[last] = v[i];
	    v[i] = swap;
	}

    swap = v[left];
    v[left] = v[last];
    v[last] = swap;

    POQS(v, left, last-1, NLT);
    POQS(v, last+1, right, NLT);

    return;
}

/*****************************************************************************/
/*                                                                           */
/*  Determine whether the constants can be ordered so as to generate an order*/
/*  for type T, between the First and Second arguments in the TupleSet       */
/*                                                                           */
/*****************************************************************************/

Boolean CheckForOrder(T, TupleSet, First, Second)
/*      -------------                           */

int First, Second;
Tuples TupleSet;
TypeInfo T;

{
    int i, j, N, *CS, FirstConst, SecondConst;
    Boolean *OrderTable, Consistent;

    N = T->NValues;
    CS = T->CollSeq;

    OrderTable = pmalloc(N*N);
    memset(OrderTable, false, N*N);

    Consistent = true;
    while( *TupleSet && Consistent )
    {
        FirstConst = (*TupleSet)[First];
	SecondConst = (*TupleSet)[Second];
	if( !(MissingValueEncountered
	      &&((FirstConst==MISSING_DISC)||(SecondConst==MISSING_DISC)))
	    &&((i=CS[FirstConst])&&(j=CS[SecondConst])) )
	{
	   Consistent = ElementOrderCheck(OrderTable, N, i, j);
	}
	TupleSet++;
    }

    free(OrderTable);
    return Consistent;
}
        
/*****************************************************************************/
/*                                                                           */
/* ElementOrderCheck - Process ordering between two elements                 */
/*                                                                           */
/*****************************************************************************/

Boolean ElementOrderCheck(OrderTable, N, FirstConst, SecondConst)
/*      -----------------                                     */

Boolean *OrderTable;
int N, FirstConst, SecondConst;

{
    int i, j;

    /* Missing Values are assumed to be appropriate to permit order */

    if(OrderCheck(OrderTable,N,FirstConst,SecondConst)) return true;
    if(OrderCheck(OrderTable,N,SecondConst,FirstConst)) return false;

    ForEach(i, 1, N)
    {
        if( i==FirstConst || 
	    OrderCheck(OrderTable,N,i,FirstConst) )
	{
	    ForEach(j, 1, N)
	    {
	        if( j==SecondConst || 
		    OrderCheck(OrderTable,N,SecondConst,j))
		{
		    OrderCheck(OrderTable,N,i,j) = true;
		    if( OrderCheck(OrderTable,N,j,i) )
		    {
		        return false;
		    }
		}
	    }
	}
    }
    return true;
}

/*****************************************************************************/
/*                                                                           */
/* StillConsistent - determines if the addition of the last partial order    */
/*                   in the set leaves the partial orders in the set still   */
/*                   consistent                                              */
/*                                                                           */
/*****************************************************************************/

Boolean StillConsistent(T, OrderTable, Set, POList)
/*      ---------------                           */

int *Set, **POList;
Boolean *OrderTable;
TypeInfo T;

{
    int First, Second;
    Tuples TupleSet;
    int i, j, FirstConst, SecondConst, NewOrder;
    Boolean Consistent;
    int N, *CS;

    N = T->NValues;
    CS = T->CollSeq;

    NewOrder = Set[Set[0]];
    if(POList[NewOrder][3])
        TupleSet = Reln[POList[NewOrder][0]]->Pos;
    else
        TupleSet = Reln[POList[NewOrder][0]]->Neg;

    First = POList[NewOrder][1];
    Second= POList[NewOrder][2];

    Consistent = true;
    while( *TupleSet && Consistent )
    {
        FirstConst = (*TupleSet)[First];
	SecondConst = (*TupleSet)[Second];

	if( !(MissingValueEncountered
	      &&((FirstConst==MISSING_DISC)||(SecondConst==MISSING_DISC)))
	    &&((i=CS[FirstConst])&&(j=CS[SecondConst])) )
	{
	   Consistent = ElementOrderCheck(OrderTable, N, i, j);
	}
	TupleSet++;
    }

    return Consistent;
}
        

