;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-

;;;	File:		TL-Trace.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	TimeLogic tracing, debugging
;;;	Last Edit:	3/03/89 05:26:12
;;;
;;;	Copyright (c) 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;
;;;
;;;	This file provides tracing and display facilities for tracking
;;;	interval relationships as they are evolving.

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))

;;; When tracing in visual mode, the head and tail point slots are used to
;;; contain endpoints, which consists of an ordinal (either NIL, or a number, or
;;; an endpoint equal to it), a flag indicating whether this endpoint  is
;;; related to any other interval's endpoint, and a list of endpoints that
;;; preceed it.


;;; User Stuff


(defun display-intervals (&key ints context (clear t))
  "
    (DISPLAY-INTERVALS &key INTS CONTEXT (CLEAR T))
	Attempts to graphically display the relations between INTS
	relative to CONTEXT.  If INTS = NIL, all defined intervals
	will be displayed.  If INTS = T, only traced intervals are
	displayed.  If CLEAR, terminal is first wiped.
  "
  (let ((*tl-current-context* (tl-find-context context)))
    (unless *tl-display-initialized-p* (tl-init-display))
    (if clear (tl-clear-display))
    (cond ((null ints)
	   (tl-display-ints *tl-defined-intervals-list*))
	  ((consp ints)
	   (tl-display-ints
	     (mapcan #'(lambda (txname)
			 (let ((tx (tl-interval-defined-p txname)))
			   (if tx (list tx))))
		     ints)))
	  ((eq ints t)
	   (when (consp *tl-traced-intervals*)
	     (tl-display-ints
	       (mapcan #'(lambda (namewithpair)
			   (let ((tx (tl-interval-defined-p (car namewithpair))))
			     (if tx (list tx))))
		       *tl-traced-intervals*))))
	  (t (error "Expecting NIL, T, or a list of intervals;  not ~S" ints))))
)


(defun graph-intervals
#-xerox (&optional intervals)
#-xerox 
  "
    (GRAPH-INTERVALS &optional INTERVALS)
	Puts up a graph of the reference hierarchy for the
	indicated INTERVALS (or all defined intervals if NIL).
  "
#+xerox (&optional intervals title format)
#+xerox
  "
    (GRAPH-INTERVALS &optional INTERVALS TITLE FORMAT)
	Puts up a window containing a graph of the reference hierarchy
	for the indicated INTERVALS (or all defined intervals if NIL).
	If TITLE is given, it becomes the graph window's title.
	FORMAT is passed onto the function IL:LAYOUTGRAPH.
  "
#-(or xerox symbolics)
  "
    (GRAPH-INTERVALS &optional INTERVALS)
	Does nothing unless you're running Xerox or Symbolics Lisp!
  "
#-(or xerox symbolics)
   (format t "~%Graphing only available in Xerox or Symbolics Lisp!~%")
#+xerox
   (prog (gnodes roots graph region window)
         (dolist (int (tl-get-ints intervals))
                (push (il:nodecreate (interval-name int)
                             (interval-name int)
                             nil
                             (mapcar #'interval-name
                                    (tl-sort-ints (tl-get-referrals int) t)))
                      gnodes)
                (unless (tl-get-referents int)
                       (push int roots)))
         (setq roots (mapcar #'interval-name (tl-sort-ints roots t)))
	 (prog ((realroot :root))
               (when (and *tl-auto-reference-p*
                          (not (member realroot roots)))
                     (push (il:nodecreate realroot realroot nil roots)
                           gnodes)
                     (setq roots (list realroot))))
         (setq graph (il:layoutgraph gnodes roots format))
         (setq region (il:createregion 0 0 500 650))
         (setq window (il:createw region title))
         (il:movew window)
         (il:showgraph graph window))
#+symbolics
   (let ((stream *standard-output*))
     (fresh-line stream)
     (fresh-line stream)
     (ZL:::SCL:formatting-graph (stream)
       (let ((ints (tl-get-ints intervals))
	     (realroot :root)
	     roots)
	 ;; First create all nodes
	 (dolist (int ints)
	   (ZL:::SCL:formatting-graph-node (stream :id int)
	     (prin1 (interval-name int) stream))
	   (unless (tl-get-referents int)
	     (push int roots)
	     (if (and realroot (eql (interval-name int) realroot))
		 (setq realroot nil))))
	 (when realroot
	   (ZL:::SCL:formatting-graph-node
	     (stream :id realroot
		     :connections
		     (mapcan #'(lambda (root)
				 (list :before (ZL:::dw:find-graph-node stream root)))
			     (tl-sort-ints roots t)))
	     (prin1 realroot stream)))
	 ;; Now link them all
	 (dolist (int ints)
	   (let ((node (ZL:::dw:find-graph-node stream int)))
	     (dolist (subint (tl-sort-ints (tl-get-referrals int) t))
	       (ZL:::dw:connect-graph-nodes
		 stream node
		 (list :before (ZL:::dw:find-graph-node stream subint)))))))))
   intervals
)

(defun trace-interval (int &optional withints)
  "
    (TRACE-INTERVAL INT &optional WITHINTS)
	Provided the :TRACE property is not :OFF, causes a message to be
	printed whenever changes are made to the relational or durational
	constraint between INT and any other interval on the list WITHINTS
	(or all intervals if WITHINTS is NIL).
  "
  (prog (current)
	(cond ((or (null *tl-traced-intervals*)
		   (null (setq current (assoc int *tl-traced-intervals*))))
	       (setq current (cons int (cond ((consp withints)
					      withints)
					     (withints (list withints)))))
	       (push current *tl-traced-intervals*))
	      (withints (setq withints (union (cdr current)
					      (cond ((consp withints)
						     withints)
						    (withints (list withints)))))
			(rplacd current (intersection withints withints)))
	      (t (rplacd current nil))))
  int
)


(defun untrace-interval (&optional int)
  "
    (UNTRACE-INTERVAL &optional INT)
	Undoes the effect of TRACE-INTERVAL.
	If INT is NIL, all traced intervals are untraced.
  "
  (prog (entry)
	(cond ((null int)
	       (setq *tl-traced-intervals* nil))
	      ((consp int)
	       (dolist (i int)
		 (untrace-interval i)))
	      ((setq entry (assoc int *tl-traced-intervals*))
	       (setq *tl-traced-intervals* (remove entry *tl-traced-intervals*))))
	(return int))
)


;;; Functions that do I/O


(defun report-rlinks (tx)
  (cond (*tl-report-rlinks-p*
	 (prog ((rlive 0)
		(rtotal 0))
	       (dolist (ilink (tlr-get-ilinks tx t))
		 (incf rtotal)
		 (unless (ilink-broken-p ilink)
		   (incf rlive)))
	       (format t "~16A : ~D/~D~%" (interval-name tx) rlive rtotal))))
)


(defun show-interval-constraints (int &key with-ints context type)
  "
    (SHOW-INTERVAL-CONSTRAINTS INT &key WITH-INTS CONTEXT TYPE)
	Prints significant constraints of TYPE (both :REL and :DUR if TYPE is
	NIL) relative to CONTEXT in effect between interval INT and all
	intervals on the list WITH-INTS (or all defined intervals if NIL).
  "
  (unless with-ints
    (setq with-ints (related-intervals int :context context :type type)))
  (case (and type (tl-constraint-type type))
    (:rel (tl-show-rels int with-ints))
    (:dur (tl-show-durs int with-ints))
    (t  (tl-show-both int with-ints)))
)


(defun tl-show-constraint (txname tyname constraint-type constraint-onlyp)
  (let ((constraint (get-interval-constraint txname tyname :type constraint-type)))
    (cond ((null constraint))
	  ((not constraint-onlyp)
	   (format t "~&~16A ~16A ~S~%" txname tyname constraint))
	  (t (format t "~&~34T~S~%" constraint))))
)


(defun tl-show-durs (txname tynames)
  (if *tl-durations-enabled-p*
      (dolist (tyname tynames)
	(tl-show-constraint txname tyname :dur nil)))
)


(defun tl-show-rels (txname tynames)
  (if *tl-relations-enabled-p*
      (dolist (tyname tynames)
	(tl-show-constraint txname tyname :rel nil)))
)


(defun tl-show-both (txname tynames)
  (dolist (tyname tynames)
    (if *tl-relations-enabled-p*
	(tl-show-constraint txname tyname :rel nil))
    (if *tl-durations-enabled-p*
	(tl-show-constraint txname tyname :dur t)))
)


(defun timelogic-stats nil
  "
    (TIMELOGIC-STATS)
	Prints a number of statistics gathered since
	the property :STATS was last given the value :RESET.
  "
  (prog ((ints 0)
	 (rmin 999999)
	 (rmax 0)
	 (rsum 0)
	 (rdel 0)
	 (rsin 0)
	 (rupcnt 0)
	 (rsidecnt 0)
	 (dmin 999999)
	 (dmax 0)
	 (dsum 0)
	 (ddel 0)
	 (dsin 0)
	 rcnt ract dcnt dact)
	(mapc #'(lambda (tx)
		  (incf ints)
		  (setq rcnt 0)
		  (setq ract 0)
		  (setq dcnt 0)
		  (setq dact 0)
		  (dolist (ilink (tlr-get-ilinks tx t))
		    (incf rcnt)
		    (if (ilink-broken-p ilink)
			(incf rdel)
			(incf ract))
		    (if (ilink-unique-p ilink)
			(incf rsin))
		    (case (ilink-class ilink)
		      (:up (incf rupcnt))
		      (:side (incf rsidecnt))))
		  (dolist (ilink (tld-get-ilinks tx t))
		    (incf dcnt)
		    (if (ilink-broken-p ilink)
			(incf ddel)
			(incf dact))
		    (if (ilink-unique-p ilink)
			(incf dsin)))
		  (cond ((< ract rmin)
			 (setq rmin ract)))
		  (cond ((> ract rmax)
			 (setq rmax ract)))
		  (setq rsum (+ rcnt rsum))
		  (cond ((< dact dmin)
			 (setq dmin dact)))
		  (cond ((> dact dmax)
			 (setq dmax dact)))
		  (setq dsum (+ dcnt dsum)))
	      *tl-defined-intervals-list*)

	(let ((fmt0 "~2&~A:")
	      (fmt1 "~26T~12A:~8D~%")
	      (fmt2 "~26T~12A:~10,1F~%")
	      (fmt3 "~26T~12A:~5D:~4,1,,,'0F"))
	 
	  (format t fmt0 "Defined")
	  (format t fmt1 "intervals" ints)

	  (when (and *tl-posting-runtime* *tl-fetching-runtime*)
	    (let ((psecs (/ *tl-posting-runtime*
			    (float internal-time-units-per-second)))
		  (fsecs (/ *tl-fetching-runtime*
			    (float internal-time-units-per-second))))
	      (format t fmt0 "Runtime")
	      (multiple-value-bind (mins secs) (floor psecs 60)
		(format t fmt3 "posting" mins secs))
	      (terpri t)
	      (multiple-value-bind (mins secs) (floor fsecs 60)
		(format t fmt3 "fetching" mins secs))))

	  (format t fmt0 "Relational links")
	  (format t fmt1 "total" (floor rsum 2))
	  (unless (zerop rsum)
	    (format t fmt1 "unique" (floor rsin 2))
	    (format t fmt1 "broken" (floor rdel 2))
	    (format t fmt1 "uplinks" rupcnt)
	    (format t fmt1 "sidelinks" (floor rsidecnt 2))
	    (format t fmt1 "min fan-out" rmin)
	    (format t fmt1 "max fan-out" rmax)
	    (format t fmt2 "avg fan-out" (/ (- rsum rdel) (float ints)))
	    (when *tlr-assert-count*
	      (format t fmt0 "Relational constraints")
	      (format t fmt1 "asserted" *tlr-assert-count*)
	      (format t fmt1 "attempted" *tlr-try-add-count*)
	      (format t fmt1 "multiplied" *tlr-multiply-count*)))
	  (format t fmt0 "Durational links")
	  (format t fmt1 "total" (floor dsum 2))
	  (unless (zerop dsum)
	    (format t fmt1 "unique" (floor dsin 2))
	    (format t fmt1 "broken" (floor ddel 2))
	    (format t fmt1 "min fan-out" dmin)
	    (format t fmt1 "max fan-out" dmax)
	    (format t fmt2 "avg fan-out" (/ (- dsum ddel) (float ints)))
	    (when *tld-assert-count*
	      (format t fmt0 "Durational constraints")
	      (format t fmt1 "asserted" *tld-assert-count*)
	      (format t fmt1 "attempted" *tld-try-add-count*)
	      (format t fmt1 "multiplied" *tld-multiply-count*)))
	  (format t fmt0 "")))
)


;;; Trace Support

(defvar *tl-last-ord* 0 "for interval sorting")


(defun tl-assign-rank-values (ints)
  (dolist (tx ints)
    (tl-ordinal-value (interval-headpoint tx))
    (tl-ordinal-value (interval-tailpoint tx)))
  (dolist (tx ints)
    (let ((pt (interval-tailpoint tx)))
      (unless (endpoint-related pt)
	(setf (endpoint-ordinal pt) *tl-last-ord*))))
)


(defun tl-clear-display nil
  (eval *tl-clear-display-form*)
)


(defun tl-display-ints (ints)
  (when ints (tl-do-display (tl-sort-ints ints)))
)


(defun tl-do-display (ints)
  (let ((maxnamewidth 5))
    (dolist (tx ints)
      (setq maxnamewidth (max maxnamewidth (length (symbol-name (interval-name tx))))))
    (setq maxnamewidth (+ maxnamewidth 2))
    (setq *tl-segment-width*
	  (max 1 (floor (/ (- *tl-display-width* maxnamewidth (1+ *tl-last-ord*))
			 *tl-last-ord*))))
    (dolist (tx ints)
      (tl-do-display-int tx maxnamewidth)))
)


(defun tl-do-display-int (tx maxnamewidth)
  (format *tl-display-stream* "~%~v@<~A~>" maxnamewidth (interval-name tx))
  (tl-draw-line (interval-headpoint tx) (interval-tailpoint tx))
)


(defun tl-draw-line (headpt tailpt)
  (prog* ((headpos (tl-ordinal-value headpt))
	  (tailpos (tl-ordinal-value tailpt))
	  (h (1+ (* headpos *tl-segment-width*)))
	  (n (* *tl-segment-width* (- tailpos headpos))))
	 #+xerox
	 (cond ((il:windowp *tl-display-stream*)
		(prog (x1 y1 x2 y2 x3 y3 x4 y4
		       (w (il:constant (il:charwidth (il:charcode "-"))))
		       (dsp (il:windowprop *tl-display-stream* 'il:dsp)))
		      (setq x1 (+ (il:dspxposition nil dsp)
				  (* *tl-segment-width* headpos w)
				  (/ w 2)))
		      (setq y1 (il:dspyposition nil dsp))
		      (setq x2 (+ x1 (+ (* (1- n) w) (/ w 2))))
		      (setq y2 y1)
		      (setq x3 (- x1 2))
		      (setq y3 (+ y1 4))
		      (setq x4 (+ x2 2))
		      (setq y4 (- y1 4))
		      (il:drawline x3 y3 x3 y4 2 nil dsp)
		      (il:drawline x4 y3 x4 y4 2 nil dsp)
		      (il:drawline x1 y1 x2 y2 2 nil dsp))
		(return nil)))
	 ;; Tab to character position corresponding to HEADPOS,
	 ;; then output two vertical bars seperated by N dashes
	 (format *tl-display-stream* "~v<~C~>~v,1,0,'-<~C~>"
		 h (if (endpoint-related headpt) #\| #\?)
		 n (if (endpoint-related tailpt) #\| #\?)))
)


(defun tl-init-end-points (ints)
  (let (hp tp)
    (dolist (tx ints)
      (setf (interval-headpoint tx)
	    (setq hp (make-endpoint)))
      (setf (interval-tailpoint tx)
	    (setq tp (make-endpoint :leaders (list hp))))))
)


(defun tl-init-display nil
#+xerox
  (progn (setq *tl-display-initialized-p* t)
	 (setq *tl-display-stream*
	       (il:createw '(0 0 700 400) "TimeLogic display" 4 t))
	 (il:dspfont (il:fontcreate '(il:gacha 10 il:bold))
		     (il:windowprop *tl-display-stream* 'il:dsp))
	 (il:windowaddprop
	   *tl-display-stream*
	   'il:reshapefn
	   #'(lambda nil
	       (setq *tl-display-width*
		     (/ (il:windowprop *tl-display-stream* 'il:width)
			(il:charwidth (il:charcode "-")
				      (il:dspfont nil *tl-display-stream*))))))
	 (setq *tl-display-width*
	       (/ (il:windowprop *tl-display-stream* 'il:width)
		  (il:charwidth (il:charcode "-")
				(il:dspfont nil *tl-display-stream*))))
	 (setq *tl-clear-display-form* '(il:clearw *tl-display-stream*)))
#+franz
  (progn (setq *tl-display-initialized-p* t)
	 (case (getenv "TERM")
	   ((sun sun24 sun34 sun44 sun54)
	    (setq *tl-display-width* 71)
	    (setq *tl-clear-display-form* '(princ "[h[j")))
	   ((t1061 teleray)
	    (setq *tl-display-width* 79)
	    (setq *tl-clear-display-form* '(princ "")))
	   ((guru-36x80 guru-40x80 guru-42x80 guru-46x80 guru-50x80 guru-54x80
			guru-60x80)
	    (setq *tl-display-width* 79)
	    (setq *tl-clear-display-form* '(princ "[H[J")))
	   (t (setq *tl-display-width* 60)
	      (setq *tl-clear-display-form* '(progn (exec "clear") (terpri))))))
#+Symbolics
  (progn (setq *tl-display-initialized-p* t)
	 (setq *tl-display-stream* t)
	 (setq *tl-display-width* 128)
	 (setq *tl-clear-display-form* nil))
  (unless *tl-display-initialized-p*
    (setq *tl-display-initialized-p* t)
    (setq *tl-display-stream* t)
    (setq *tl-display-width* 78)
    (setq *tl-clear-display-form* nil))
)


(defun tl-int-sort (tx ty)
  (prog (xpt ypt xord yord)
	(setq xpt (interval-headpoint tx))
	(setq ypt (interval-headpoint ty))
	(setq xord (tl-ordinal-value xpt))
	(setq yord (tl-ordinal-value ypt))
	(cond ((< xord yord) (return t))
	      ((> xord yord) (return nil))
	      ((not (endpoint-related xpt)) (return t))
	      ((not (endpoint-related ypt)) (return nil)))
	;; headpoints are equal
	(setq xpt (interval-tailpoint tx))
	(setq ypt (interval-tailpoint ty))
	(setq xord (tl-ordinal-value xpt))
	(setq yord (tl-ordinal-value ypt))
	(cond ((< xord yord) (return nil))
	      ((> xord yord) (return t))
	      ((not (endpoint-related xpt)) (return t))
	      ((not (endpoint-related ypt)) (return nil)))
	;; tailpoints are equal
	(return t))
)


(defun tl-max-ord-value (endpoints)
  (let ((maxval 0))
    (dolist (endpt endpoints maxval)
      (setq maxval (max maxval (tl-ordinal-value endpt)))))
)


(defun tl-next-ord-value (endpoints)
  (let ((ordval (1+ (tl-max-ord-value endpoints))))
    (if (> ordval *tl-last-ord*)
	(setq *tl-last-ord* ordval))
    ordval)
)


(defun tl-ordinal-value (endpt)
  (if (numberp (endpoint-ordinal endpt))
      (endpoint-ordinal endpt)
      (setf (endpoint-ordinal endpt)
	    (cond ((endpoint-ordinal endpt)
		   (tl-ordinal-value (endpoint-ordinal endpt)))
		  ((endpoint-leaders endpt)
		   (tl-next-ord-value (endpoint-leaders endpt)))
		  (t 0))))
)


(defun tl-plink-from-rlink (hx? hy? rlink)
  (if (tlr-all-rlink-p rlink)
      (tlr-const :a :b :e)
      (let (aset bset eset)
	(cond ((and hx? hy?)
	       (setq aset (tlr-const :a :d :f :mi :oi))
	       (setq bset (tlr-const :b :c :fi :m :o))
	       (setq eset (tlr-const :e :s :si)))
	      (hx? (setq aset (tlr-const :a))
		   (setq bset (tlr-const :b :c :d :e :f :fi :m :o :oi :s :si))
		   (setq eset (tlr-const :mi)))
	      (hy? (setq aset (tlr-const :a :c :d :e :f :fi :mi :o :oi :s :si))
		   (setq bset (tlr-const :b))
		   (setq eset (tlr-const :m)))
	      (t (setq aset (tlr-const :a :c :mi :oi :si))
		 (setq bset (tlr-const :b :d :m :o :s))
		 (setq eset (tlr-const :e :f :fi))))
	(cond ((tlr-subset-p rlink aset)
	       (tlr-const :a))
	      ((tlr-subset-p rlink bset)
	       (tlr-const :b))
	      ((tlr-subset-p rlink eset)
	       (tlr-const :e))
	      ((tlr-subset-p rlink (tlr-unite-rlinks aset bset))
	       (tlr-const :a :b))
	      ((tlr-subset-p rlink (tlr-unite-rlinks aset eset))
	       (tlr-const :a :e))
	      ((tlr-subset-p rlink (tlr-unite-rlinks bset eset))
	       (tlr-const :b :e))
	      (t (tlr-const :a :b :e)))))
)


(defun tl-rank-after (xpt ypt)
  (cond ((endpoint-ordinal xpt)
	 (tl-rank-after (endpoint-ordinal xpt) ypt))
	((endpoint-ordinal ypt)
	 (tl-rank-after xpt (endpoint-ordinal ypt)))
	((member ypt (endpoint-leaders xpt)))
	(t (setf (endpoint-related xpt) t)
	   (setf (endpoint-related ypt) t)
	   (push ypt (endpoint-leaders xpt))))
)


(defun tl-rank-end-points (ints)
  (setq *tl-last-ord* -1)
  (tl-rank-intervals ints)
  (tl-assign-rank-values ints)
)


(defun tl-rank-end-points-of-intervals (tx ty)
  (prog ((rlink (tlr-find-rlink tx ty)))
	(tl-rank-ordinals (interval-headpoint tx)
			  (interval-headpoint ty)
			  (tl-plink-from-rlink t t rlink))
	(tl-rank-ordinals (interval-tailpoint tx)
			  (interval-headpoint ty)
			  (tl-plink-from-rlink nil t rlink))
	(tl-rank-ordinals (interval-headpoint tx)
			  (interval-tailpoint ty)
			  (tl-plink-from-rlink t nil rlink))
	(tl-rank-ordinals (interval-tailpoint tx)
			  (interval-tailpoint ty)
			  (tl-plink-from-rlink nil nil rlink)))
)


(defun tl-rank-equal (xpt ypt)
  (cond ((eq xpt ypt) nil)
	((endpoint-ordinal xpt)
	 (tl-rank-equal (endpoint-ordinal xpt) ypt))
	((endpoint-ordinal ypt)
	 (tl-rank-equal xpt (endpoint-ordinal ypt)))
	(t (setf (endpoint-ordinal xpt) ypt)
	   (setf (endpoint-related xpt) t)
	   (setf (endpoint-related ypt) t)
	   (when (endpoint-leaders xpt)
	     (setf (endpoint-leaders ypt)
		   (union (endpoint-leaders xpt)
			  (endpoint-leaders ypt)))
	     (setf (endpoint-leaders xpt) nil))))
)


(defun tl-rank-intervals (ints)
  (do* ((txp ints (cdr txp))
	(tx (car txp) (car txp)))
       ((null (cdr txp)))
    (dolist (ty (cdr txp))
      (tl-rank-end-points-of-intervals tx ty)))
)


(defun tl-rank-ordinals (xpt ypt plink)
  (cond ((tlr-same-rlink-p plink (tlr-const :a))
	 (tl-rank-after xpt ypt))
	((tlr-same-rlink-p plink (tlr-const :b))
	 (tl-rank-after ypt xpt))
	((tlr-same-rlink-p plink (tlr-const :e))
	 (tl-rank-equal ypt xpt)))
)


(defun tl-sort-ints (ints &optional sort-always-p)
  (when ints
    (tl-init-end-points ints)
    (tl-rank-end-points ints)
    (if (or sort-always-p *tl-sort-ints-before-display-p*)
	(sort (copy-list ints) #'tl-int-sort)
	ints))
)


(defun tl-subtrace-1 (lbl txname)
  (format *tl-display-stream* "~&TL~20A ~16A~%" lbl txname))

(defun tl-subtrace-2 (lbl txname tyname)
  (format *tl-display-stream* "~&TL~20A ~16A ~16A~%" lbl txname tyname))

(defun tl-subtrace-3 (lbl txname tyname xyrel)
  (format *tl-display-stream* "~&TL~A [~D]:~22T ~16A ~16A ~S (context: ~S)~%"
	  lbl *tl-propagation-level* txname tyname xyrel (CURRENT-CONTEXT)))

(defun tl-subtrace (kind txname tyname)
  (cond (*tl-display-enabled-p*
	 (unless *tl-display-initialized-p* (tl-init-display))
	 (tl-clear-display)))
  (case kind
    (:newcon (tl-subtrace-1 "+ context:" txname))
    (:delcon (tl-subtrace-1 "- context:" txname))
    (:switch (tl-subtrace-1 "> context:" txname))
    (:newint (tl-subtrace-1 "+ interval:" txname))
    (:delint (tl-subtrace-1 "- interval:" txname))
    (:newref (tl-subtrace-2 "+ reference:" txname tyname))
    (:delref (tl-subtrace-2 "- reference:" txname tyname))
    ((:newrel :push-rlink :pop-rlink)
     (tl-subtrace-3 (if (eq kind :pop-rlink)
			"^ relation"
			"+ relation")
		    txname tyname 
		    (get-interval-constraint txname tyname))
     (cond ((and *tl-display-enabled-p*
		 (or (eq kind :newrel)
		     (tl-traced-int-p txname tyname)
		     (tl-traced-int-p tyname txname)))		     
	    (display-intervals :ints (eq *tl-trace-mode* :on) :clear nil))))
    ((:newdur :push-dlink :pop-dlink)
     (tl-subtrace-3 (if (eq kind :pop-dlink )
			"^ duration"
			"+ duration")
		    txname tyname
		    (get-interval-constraint txname tyname :type :dur))
     (cond ((and *tl-display-enabled-p*
		 (or (eq kind :newdur)
		     (tl-traced-int-p txname tyname)
		     (tl-traced-int-p tyname txname)))
	    (display-intervals :ints (eq *tl-trace-mode* :on) :clear nil))))
    (:break-rlink   (tl-subtrace-2 "~ relation" txname tyname))
    (:break-dlink   (tl-subtrace-2 "~ duration" txname tyname))
    (:unbreak-rlink (tl-subtrace-2 "& relation" txname tyname))
    (:unbreak-dlink (tl-subtrace-2 "& duration" txname tyname))
    (t (error "Unsupported TRACE flag: " kind)))
)


(defun tl-traced-int-p (txname tyname)
  (let ((withints (assoc txname *tl-traced-intervals*)))
    (if withints
	(or (null (cdr withints))
	    (and (member tyname (cdr withints)) t))))
)

(defun tl-trace (kind tx ty)
  (prog ((txname (cond ((context-p tx) (context-name tx))
		       ((interval-p tx) (interval-name tx))
		       (t tx)))
	 (tyname (cond ((context-p ty) (context-name ty))
		       ((interval-p ty) (interval-name ty))
		       (t ty))))
	(cond ((eq *tl-trace-mode* :verbose)
	       (when (or (eq kind :newrel)
			 (eq kind :newdur))
		 (return nil)))
	      ((eq *tl-trace-mode* :all)
	       (cond ((or (eq kind :newrel)
			  (eq kind :newdur)
			  (and (interval-p tx)
			       (interval-generated-p tx))
			  (and (interval-p ty)
			       (interval-generated-p ty)))
		      (return nil))))
	      ((and (interval-p tx)
		    (tl-traced-int-p txname tyname)))
	      ((and (interval-p ty)
		    (tl-traced-int-p tyname txname)))
	      (t (return nil)))
	(tl-subtrace kind txname tyname)
	(if *tl-trace-wait* (tl-trace-wait)))
)


#-xerox
(defun dismiss (msecs) (sleep (ceiling msecs 1000)))

(defun tl-trace-wait nil
  (cond ((numberp *tl-trace-wait*)
	 (dismiss (cond ((< *tl-trace-wait* 1000)
			 (* *tl-trace-wait* 1000))
			(t *tl-trace-wait*))))
	(t (terpri)
	   (cond ((not (y-or-n-p "Continue? "))
		  (error "TimeLogicWait interrupted")))))
)



;; End of file TL-TRACE
