;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-RLinks.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	Manipulation of relational constraints
;;;	Last Edit:	3/02/89 17:32:56
;;;
;;;	Copyright (c) 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))


(defun tlr-add-constraint (tx ty rlink)
  ;; If no error occurred sofar and the interval logic is
  ;; enabled, then if an rlink already exists adds the given
  ;; restriction, otherwise adds restriction to whatever can
  ;; be found through the hierarchy.
  (cond
    ((eq tx ty)
     (if (tlr-no-rlink-p (tlr-intersect-rlinks rlink (tlr-const :e)))
	 (tlr-incompat-error tx ty rlink (tlr-const :e))))
    ((and *tl-relations-enabled-p* (not (tlr-all-rlink-p rlink)))
     (tlr-add-restriction tx ty rlink)
     (unless *tl-error-occurred-p*
       (tl-record-event :newrel tx ty))))
)


(defun tlr-add-restriction (tx ty rlink)
  (unless (tlr-all-rlink-p rlink)
    (if *tlr-try-add-count* (incf *tlr-try-add-count*))
    (let* ((oldrlink (tlr-get-rlink tx ty))
	   (currlink (or oldrlink (tlr-find-rlink tx ty)))
	   (newrlink (tlr-intersect-rlinks rlink currlink)))
      (cond ((tlr-no-rlink-p newrlink)
	     (tlr-incompat-error tx ty rlink currlink))
	    ((or (null oldrlink)
		 (not (tlr-same-rlink-p newrlink oldrlink)))
	     (tlr-assert-rlink
	       tx ty newrlink (tlr-invert-rlink newrlink))
	     (when *tl-durations-enabled-p*
	       (tld-add-restriction
		 tx ty (tl-convert-rlink-to-dlink newrlink))))))))


(defun tlr-assert-rlink (tx ty rlink invrlink)
  (tlr-push-rlink tx ty rlink invrlink)
  (if *tlr-assert-count* (incf *tlr-assert-count*))
  (if *tl-auto-reference-p* (tlr-queue-autoref-p tx ty))
  (if *tlr-assert-hook* (tlr-hook-enqueue tx ty rlink invrlink))
  (if *tl-propagate-depth-first-p*
      (tlr-propagate-constraint tx ty rlink invrlink
				(1+ *tl-propagation-level*))
      (tlr-enqueue tx ty rlink invrlink))
)


(defun tlr-assert-sidelink (xy-ilink)
  ;; If a path exists thru ref hierarchy from TX to TY, break current link
  (let ((tx (ilink-source xy-ilink))
	(ty (ilink-target xy-ilink)))
    (when (ilink-unique-p xy-ilink)
      (dolist (xz-ilink (tlr-get-ilinks tx))
	(let ((tz (ilink-target xz-ilink)))
	  (when (tlr-break-ilink-p tx tz ty xz-ilink xy-ilink)
	    (tlr-break-ilink tx ty)
	    (return))))))
)


(defun tlr-assert-uplink (xy-ilink)
  ;; tx is contained in ty so make ty a reference interval
  ;; of tx, unless there is an indirect path between tx and
  ;; ty already.
  (let ((tx (ilink-source xy-ilink))
	(ty (ilink-target xy-ilink)))

    (cond ((tl-indirect-ref-p tx ty)
	   (tl-retract-referent tx ty))
	  ((tl-assert-referent tx ty)
	   (tl-retract-indirect-refs tx ty)
	   (tl-unbreak-direct-refd-ilinks tx ty)))

    ;; Now decide if any links can be broken
    ;; First of all, if this uplink is unique, it might be ok to break it

    (when (ilink-unique-p xy-ilink)
      (dolist (xz-ilink (tlr-get-ilinks tx))
	(let ((tz (ilink-target xz-ilink)))
	  (when (tlr-break-ilink-p tx tz ty xz-ilink xy-ilink)
	    (tlr-break-ilink tx ty)
	    (return)))))

    ;; Next see if there is a tx-ty-tz such that tx-tz can be broken

    (dolist (xz-ilink (tlr-get-ilinks tx))
      (let ((tz (ilink-target xz-ilink)))
	(if (tlr-break-ilink-p tx ty tz xy-ilink xz-ilink)
	    (tlr-break-ilink tx tz))))

    ;; Finally see if there is a tz-tx-ty such that tz-ty can be broken

    (dolist (yz-ilink (tlr-get-ilinks ty))
      (let ((tz (ilink-target yz-ilink)))
	(if (tlr-break-ilink-p tz tx ty nil nil)
	    (tlr-break-ilink tz ty)))))
)


(defun tlr-before-p (tx ty)
  ;; See if TX before TY
  (not (tlr-intersect-p (tlr-find-rlink tx ty)
			(tlr-const :a :d :f :mi :oi :si)))
)


(defun tlr-break-ilink-p (tx ty tz xy-ilink xz-ilink)
  ;; Should the link between tx and tz (if any) be retracted?
  ;; Yes if distinct(tx,ty,tz) and uplink(tx,ty) and unique(tx,tz) and
  ;; and xz can be obtained indirectly from xy and yz 
  (let (yz-rlink)
    (and (not (eq tx ty))
	 (not (eq tx tz))
	 (not (eq ty tz))
	 (not (tl-common-reference-p tx tz))
	 (or xz-ilink (setq xz-ilink (tlr-get-readable-ilink tx tz)))
	 (not (ilink-broken-p xz-ilink))
	 (or xy-ilink (setq xy-ilink (tlr-get-readable-ilink tx ty)))
	 (eq (ilink-class xy-ilink) :up)
	 (ilink-unique-p xz-ilink)
	 (setq yz-rlink (tlr-find-rlink ty tz))
	 (not (tlr-all-rlink-p yz-rlink))
	 (tlr-same-rlink-p (ilink-current xz-ilink)
			   (tlr-multiply-rlinks (ilink-current xy-ilink)
						yz-rlink))))
)


(defun tlr-break-ilink (tx ty)
  ;; Break the relational links between TX and TY and vice versa
  (let ((ilink (tlr-get-readable-ilink tx ty)))
    (unless (and ilink (ilink-broken-p ilink))
      (let ((clink (tl-get-writable-clink tx)))
	(setf ilink (tlr-get-writable-ilink tx ty))
	(setf (ilink-broken-p ilink) t)
	(setf (clink-rel-ilinks-list clink)
	      (remove ilink (clink-rel-ilinks-list clink)
		      :count 1 :test #'eq)))
      (let ((clink (tl-get-writable-clink ty)))
	(setf ilink (tlr-get-writable-ilink ty tx))
	(setf (ilink-broken-p ilink) t)
	(setf (clink-rel-ilinks-list clink)
	      (remove ilink (clink-rel-ilinks-list clink)
		      :count 1 :test #'eq)))
      (tl-record-event :break-rlink tx ty)))
)



;;; The following is used to compute the transitivity function
;;; TLR-MULTIPLY-RLINKS.  Note that the functions TLR-ENCODE-CONSTRAINTS-TABLE
;;; and TLR-COMPILE-CONSTRAINTS-TABLE must be called (in that
;;; order) whenever the value of *TLR-DECODED-CONSTRAINTS-TABLE*
;;; is changed.  Note that TLR-SLOW-MULTIPLY-RLINKS is a doubly
;;; looping variant of TLR-MULTIPLY-RLINKS where the latter
;;; is automatically generated with the loops unrolled.


(defun tlr-compile-constraints-table nil
  ;; This function compiles the transitivity table into code that
  ;; will compute the transitive relation as the function TLR-MULTIPLY-RLINKS
  (eval `(defun tlr-multiply-rlinks (rlink1 rlink2)

	   ;; DO NOT EDIT THIS FUNCTION!!!

	   ;; It is created by the function TLR-COMPILE-CONSTRAINTS-TABLE
	   ;; on the basis of the variable *TLR-ENCODED-CONSTRAINTS-TABLE*

	   (cond (*tlr-multiply-count* (incf *tlr-multiply-count*)))
	   (prog ((newrlink (cond ((tlr-intersect-p rlink1 (tlr-const :e))
				   (cond ((tlr-intersect-p rlink2 (tlr-const :e))
					  (tlr-unite-rlinks rlink1 rlink2))
					 (t rlink2)))
				  ((tlr-intersect-p rlink2 (tlr-const :e))
				   rlink1)
				  (t (tlr-const nil)))))
		 (cond ((tlr-all-rlink-p newrlink)
			(return newrlink)))
		 ,@(mapcar #'tlr-compile-constraints-table-entry
			   *tlr-encoded-constraints-table*)
		 (return newrlink))))
  'tlr-multiply-rlinks
)


(defun tlr-compile-constraints-table-entry (entry)
  `(cond ((tlr-intersect-p rlink1 ,(car entry))
	  ,@(mapcar #'tlr-compile-constraints-table-subentry (cdr entry))))
)


(defun tlr-compile-constraints-table-subentry (subentry)
  (cond ((tlr-all-rlink-p (cdr subentry))
	 `(and (tlr-intersect-p rlink2 ,(car subentry))
	       (return ,(cdr subentry))))
	(t `(and (tlr-intersect-p rlink2 ,(car subentry))
		 (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink ,(cdr subentry))))
		 (return newrlink))))
)


(defun tlr-compress-table-pairs (pairs)
  (let (newpairs)
       (dolist (pair pairs)
	       (or (dolist (newpair newpairs)
			   (cond ((tlr-same-rlink-p (cdr pair)
						    (cdr newpair))
				  (setf (car newpair)
					(tlr-unite-rlinks (car pair)
							  (car newpair)))
				  (return t))))
		   (push pair newpairs)))
       newpairs)
)


(defun tlr-compute-table-pairs (origpairs)
  ;; Build from the original symbolic list of pairs a list
  ;; of encoded pairs for each bit in the second rlink, where
  ;; the CAR of the pair is the rlink corresponding to that
  ;; bit, and the CDR is the resulting rlink multiplication.
  (mapcan #'(lambda (pair)
	      (unless (eq (car pair) :e)
		(list (cons (tlr-encode-rlink (car pair))
			    (tlr-encode-rlink (cdr pair))))))
	  origpairs)
)


;;; The following defn is presumably faster than the one after it (and
;;; potentially more dangerous) as it will not detect inconsistency, at least
;;; with these intervals.  However, it turns out that the cost of calling
;;; tlr-get-readable-ilink (gethash) exceeds the cost of doing the
;;; tlr-multiply-rlinks and tlr-add-restriction!
;(defun tlr-constrain (tx xyrlink ty yzrlink tz)
;  (declare (ignore ty))
;  (when (let ((ilink (tlr-get-readable-ilink tx tz)))
;	  (if *tl-auto-reference-p*
;	      (or (null ilink)
;		  (not (ilink-broken-p ilink)))
;	      (or ilink (tl-shared-refs-p tx tz))))
;    (tlr-add-restriction tx tz (tlr-multiply-rlinks xyrlink yzrlink)))
;)
(defun tlr-constrain (tx xyrlink ty yzrlink tz)
  (declare (ignore ty))
  (when (or *tl-auto-reference-p*
	    (tlr-get-readable-ilink tx tz)
	    (tl-shared-refs-p tx tz))
    (tlr-add-restriction tx tz (tlr-multiply-rlinks xyrlink yzrlink)))
)


(defun tlr-unique-downlink-p (rlink tx ty)
  (or (tlr-same-rlink-p rlink (tlr-const :c))
      (tlr-same-rlink-p rlink (tlr-const :fi))
      (tlr-same-rlink-p rlink (tlr-const :si))
      (and (tl-older-interval-p ty tx)
	   (tlr-same-rlink-p rlink (tlr-const :e))))
)


(defun tlr-downlink-p (rlink tx ty)
  (if *tlr-autoref-unique-uplinks-p*
      (tlr-unique-downlink-p rlink tx ty)
      (tlr-subset-p rlink (if (tl-older-interval-p ty tx)
			      (tlr-const :c :e :fi :si)
			      (tlr-const :c :fi :si))))
)


(defun tlr-encode-constraints-table nil
  ;; This encodes, compresses and sorts the constraint propagation table
  ;; for relations (rlinks, internally) from the decoded constraints
  ;; table.  Compression is possible because several combinations of
  ;; rlinks may have the same transitivity result.  Sorting is based on
  ;; cardinality of the resulting rlinks, to maximize the possibility
  ;; that the transitivity computation can be cut because the result
  ;; will be (TLR-CONST :ALL)
  (setq *tlr-encoded-constraints-table*
	(mapcan #'(lambda (entry)
		    (unless (eq (car entry) :e)
		      (list (cons (tlr-encode-rlink (car entry))
				  (tlr-sort-table-pairs
				    (tlr-compress-table-pairs
				      (tlr-compute-table-pairs (cdr entry))))))))
		*tlr-decoded-constraints-table*))
)


(defun tlr-hook-enqueue (tx ty rlink invrlink)
  ;; Keep until we're done propagating.
  ;; Note this does not avoid duplicates!
  (push (list tx rlink ty) *tlr-hook-constraints*))


(defun tlr-enqueue (tx ty rlink invrlink)
  ;; Queue is mainained as a list of constraint entries, where
  ;; each entry is a list of the intervals tx and ty and the
  ;; new link values.  If tx ty already there, replaces the
  ;; old entry with the new one.  Otherwise the entry is added
  ;; at the back
  (prog (prev entry (queue *tlr-constraint-queue*)
	 (newentry (list tx ty rlink invrlink (1+ *tl-propagation-level*))))
	(cond ((null queue)
	       (setq *tlr-constraint-queue* (list newentry))
	       (return nil)))
	(loop (setq entry (car queue))
	      (cond ((or (and (eq tx (car entry))
			      (eq ty (cadr entry)))
			 (and (eq ty (car entry))
			      (eq tx (cadr entry))))
		     (rplaca queue newentry)
		     (return nil))
		    ((null (setq queue (cdr (setq prev queue))))
		     (rplacd prev (list newentry))
		     (return nil)))))
)


(defun tlr-find-rlink (tx ty)
  (cond ((or (null tx)
	     (null ty))
	 (tlr-const :all))
	((eq tx ty)
	 (tlr-const :e))
	((tlr-find-rlink-aux-x tx ty))
	(t (tlr-const :all)))
)

(defun tlr-find-rlink-aux-x (tx ty)
  (or (tlr-get-rlink tx ty)
      (tlr-find-rlink-thru-yrefs tx ty)
      (tlr-find-rlink-thru-xrefs tx ty))
)

(defun tlr-find-rlink-aux-y (tx ty)
  (or (tlr-get-rlink tx ty)
      (tlr-find-rlink-thru-yrefs tx ty))
)

(defun tlr-find-rlink-thru-xrefs (tx ty)
  ;; Goes up the hierarchy from TX
  (let (tx-rx rx-ty tx-ty tx*ty)
    (dolist (rx (tl-get-referents tx) tx*ty)
      (cond ((null (setq tx-rx (tlr-get-rlink tx rx))))
	    ((tlr-all-rlink-p tx-rx))
	    ((null (setq rx-ty (tlr-find-rlink-aux-x rx ty))))
	    ((null (setq tx-ty (tlr-multiply-rlinks tx-rx rx-ty))))
	    (tx*ty (setq tx*ty (tlr-intersect-rlinks tx*ty tx-ty)))
	    (*tl-search-all-paths-p* (setq tx*ty tx-ty))
	    (t (return tx-ty)))))
)

(defun tlr-find-rlink-thru-yrefs (tx ty)
  ;; goes up the hierarchy from TY
  (let (tx-ry ry-ty tx-ty tx*ty)
    (dolist (ry (tl-get-referents ty) tx*ty)
      (cond ((null (setq ry-ty (tlr-get-rlink ry ty))))
	    ((tlr-all-rlink-p ry-ty))
	    ((null (setq tx-ry (tlr-find-rlink-aux-y tx ry))))
	    ((null (setq tx-ty (tlr-multiply-rlinks tx-ry ry-ty))))
	    (tx*ty (setq tx*ty (tlr-intersect-rlinks tx*ty tx-ty)))
	    (*tl-search-all-paths-p* (setq tx*ty tx-ty))
	    (t (return tx-ty)))))
)


(defun tlr-get-rlink (tx ty)
  (let ((ilink (tlr-get-readable-ilink tx ty)))
    (and ilink (ilink-current ilink)))
)


(defun tlr-incompat-error (tx ty rlink old-rlink)
  (cond (*tl-auto-backtrack-p*
	 (setq *tl-error-occurred-p* t)
	 (tl-backtrack nil t))
	(t (tl-error "Incompatible relational constraint between ~S and ~S:~%~10Told: ~S~%~10Tnew: ~S"
		     (interval-name tx)
		     (interval-name ty)
		     (tlr-decode-rlink old-rlink)
		     (tlr-decode-rlink rlink))))
)


(defun tlr-invert-rlink (rlink)
  (cond ((tlr-all-rlink-p rlink) rlink)
	(t (logior (logand rlink (tlr-const :e))
		   (logand (ash rlink -1)
			   (tlr-const :b :d :f :m :o :s))
		   (logand (ash rlink 1)
			   (tlr-const :a :c :fi :mi :oi :si)))))
)


(defun tlr-make-autorefs ()
  (dolist (xy-ilink *tlr-uplink-queue*)
    (tlr-assert-uplink xy-ilink))
  (dolist (xy-ilink *tlr-sidelink-queue*)
    (tlr-assert-sidelink xy-ilink))
  (setf *tlr-uplink-queue* nil)
  (setf *tlr-sidelink-queue* nil)
)

(defun tlr-queue-autoref-p (tx ty)
  ;; If a new uplink or sidelink has been asserted, put it on the appropriate
  ;; queue.  Can't quite assert it yet because the structure being added to the
  ;; network may not be consistent.  Better wait until all propagation of
  ;; asserted rlinks and dlinks is finished.
  (let ((xy-ilink (tlr-get-readable-ilink tx ty)))
    (case (ilink-class xy-ilink)
      (:up (pushnew xy-ilink *tlr-uplink-queue*))
      (:down (pushnew (tlr-get-readable-ilink ty tx) *tlr-uplink-queue*))
      (:side (pushnew xy-ilink *tlr-sidelink-queue*))))
)


(defun tlr-multiply-rlinks (rlink1 rlink2)

  ;; DO NOT EDIT THIS FUNCTION!!!

  ;; It is created by the function TLR-COMPILE-CONSTRAINTS-TABLE
  ;; on the basis of the variable *TLR-ENCODED-CONSTRAINTS-TABLE*

  (declare (optimize speed))

  (cond (*tlr-multiply-count* (incf *tlr-multiply-count*)))
  (prog ((newrlink (cond ((tlr-intersect-p rlink1 (tlr-const :e))
			  (cond ((tlr-intersect-p rlink2 (tlr-const :e))
				 (tlr-unite-rlinks rlink1 rlink2))
				(t rlink2)))
			 ((tlr-intersect-p rlink2 (tlr-const :e))
			  rlink1)
			 (t (tlr-const nil)))))
	(cond ((tlr-all-rlink-p newrlink)
	       (return newrlink)))
	(cond ((tlr-intersect-p rlink1 2)
	       (and (tlr-intersect-p rlink2 1)
		    (return 8191))
	       (and (tlr-intersect-p rlink2 1348)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 662)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2746)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 1)
	       (and (tlr-intersect-p rlink2 2)
		    (return 8191))
	       (and (tlr-intersect-p rlink2 660)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1349)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 3433)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 8)
	       (and (tlr-intersect-p rlink2 4)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 7996)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 361)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2698)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1344)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 296)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 656)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2568)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2088)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 8)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 4)
	       (and (tlr-intersect-p rlink2 8)
		    (return 8191))
	       (and (tlr-intersect-p rlink2 2560)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 662)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 288)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1349)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1044)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 4)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 65)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 130)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 16)
	       (and (tlr-intersect-p rlink2 8)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2698)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2560)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 642)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 256)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1284)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 32)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 4144)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1028)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 4)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 130)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 64)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 64)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 16)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 16)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 32)
	       (and (tlr-intersect-p rlink2 2)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2698)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 640)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2568)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 16)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 4144)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 4)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1284)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1280)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 256)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2056)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 8)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 64)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 64)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 32)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 32)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 64)
	       (and (tlr-intersect-p rlink2 2)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2698)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 532)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1284)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 128)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 4144)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 361)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 3072)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 64)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 128)
	       (and (tlr-intersect-p rlink2 1)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 361)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1284)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 532)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 64)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 7168)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2698)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 48)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 128)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 256)
	       (and (tlr-intersect-p rlink2 512)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 7996)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 8)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 361)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2698)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 288)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 321)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 20)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1284)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2048)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 296)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 128)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2568)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 65)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1024)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 256)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 512)
	       (and (tlr-intersect-p rlink2 256)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 7996)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 8)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2698)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 361)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2560)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 642)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1028)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 532)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 64)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 296)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 32)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2568)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 130)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 16)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 512)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 1024)
	       (and (tlr-intersect-p rlink2 8)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 361)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 288)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 321)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2048)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 7168)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 512)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 532)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 20)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 4)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 65)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1024)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 1024)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 128)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 128)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))))
	(cond ((tlr-intersect-p rlink1 2048)
	       (and (tlr-intersect-p rlink2 1)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 361)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 320)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 296)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 1024)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 7168)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 4)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 532)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 528)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 512)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 40)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 8)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2048)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2048)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 128)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 128)))
		    (return newrlink))
	       (and (tlr-intersect-p rlink2 2)
		    (tlr-all-rlink-p (setq newrlink (tlr-unite-rlinks newrlink 2)))
		    (return newrlink))))
	(return newrlink))
  )

(defun tlr-multiply-rlinks-slowly (rlink1 rlink2)
  ;; Calculate transitivity relation A x B and B y C --> A (x*y) C by table lookup.
  (prog (table entry pair (newrlink (tlr-const nil)))
	(cond (*tlr-multiply-count* (incf *tlr-multiply-count*)))
	(cond ((tlr-intersect-p rlink1 (tlr-const :e))
	       (setq newrlink (tlr-unite-rlinks newrlink rlink2))))
	(cond ((tlr-intersect-p rlink2 (tlr-const :e))
	       (setq newrlink (tlr-unite-rlinks newrlink rlink1))))
	(cond ((tlr-all-rlink-p newrlink)
	       (return newrlink)))
	(setq table *tlr-encoded-constraints-table*)
    outerloop
	(setq entry (car table))
	(unless (tlr-intersect-p rlink1 (car entry))
	  (go outernext))
	(setq entry (cdr entry))
    innerloop
	(setq pair (car entry))
	(unless (tlr-intersect-p rlink2 (car pair))
	  (go innernext))
	(setq newrlink (tlr-unite-rlinks newrlink (cdr pair)))
	(cond ((tlr-all-rlink-p newrlink)
	       (return newrlink)))
    innernext
	(cond ((setq entry (cdr entry))
	       (go innerloop)))
    outernext
	(cond ((setq table (cdr table))
	       (go outerloop)))
	(return newrlink))
)


(defun tlr-pop-rlink (tx ty)
  (let (class)
    (let ((ilink (tlr-get-writable-ilink tx ty)))
      (setf (ilink-current ilink)
	    (pop (ilink-previous ilink)))
      (setf class (tlr-rlink-class tx ty (ilink-current ilink)))
      (setf (ilink-class ilink) class)
      (setf (ilink-unique-p ilink) nil))
    (let ((ilink (tlr-get-writable-ilink ty tx)))
      (setf (ilink-current ilink)
	    (pop (ilink-previous ilink)))
      (setf (ilink-class ilink)
	    (case class
	      (:up :down)
	      (:down :up)
	      (t class)))
      (setf (ilink-unique-p ilink) nil))
    (tl-record-event :pop-rlink tx ty))
)


(defun tlr-propagate-constraint (tx ty rlink invrlink *tl-propagation-level*)
  (let (tz zrlink)
    (dolist (ilink (tlr-get-ilinks ty))
      (when (and (setq zrlink (ilink-current ilink))
		 (not (eq tx (setq tz (ilink-target ilink)))))
	(tlr-constrain tx rlink ty zrlink tz)))

    (dolist (ilink (tlr-get-ilinks tx))
      (when (and (setq zrlink (ilink-current ilink))
		 (not (eq ty (setq tz (ilink-target ilink)))))
	(tlr-constrain ty invrlink tx zrlink tz))))
)


(defun tlr-push-rlink (tx ty newrlink invrlink)
  (let ((class (tlr-rlink-class tx ty newrlink))
	(unique-p (tlr-single-rlink-p newrlink)))
    (let ((ilink (tlr-get-writable-ilink tx ty)))
      (push (ilink-current ilink)
	    (ilink-previous ilink))
      (setf (ilink-current ilink) newrlink)
      (setf (ilink-class ilink) class)
      (setf (ilink-unique-p ilink) unique-p))
    (let ((ilink (tlr-get-writable-ilink ty tx)))
      (push (ilink-current ilink)
	    (ilink-previous ilink))
      (setf (ilink-current ilink) invrlink)
      (setf (ilink-class ilink)
	    (case class
	      (:up :down)
	      (:down :up)
	      (t class)))
      (setf (ilink-unique-p ilink) unique-p)))
  (tl-record-event :push-rlink tx ty)
)


(defun tlr-rlink-class (tx ty rlink)
  (cond ((tlr-sidelink-p rlink) :side)
	((tlr-uplink-p rlink tx ty) :up)
	((tlr-downlink-p rlink tx ty) :down)
	(t nil))
)


(defun tlr-sidelink-p (rlink)
  (if *tlr-autoref-unique-sidelinks-p*
      (tlr-unique-sidelink-p rlink)
      (tlr-subset-p rlink (tlr-const :a :b :m :mi)))
)


(defun tlr-single-rlink-p (rlink)
  (or (tlr-same-rlink-p rlink (tlr-const :a))
      (tlr-same-rlink-p rlink (tlr-const :b))
      (tlr-same-rlink-p rlink (tlr-const :c))
      (tlr-same-rlink-p rlink (tlr-const :d))
      (tlr-same-rlink-p rlink (tlr-const :e))
      (tlr-same-rlink-p rlink (tlr-const :f))
      (tlr-same-rlink-p rlink (tlr-const :fi))
      (tlr-same-rlink-p rlink (tlr-const :m))
      (tlr-same-rlink-p rlink (tlr-const :mi))
      (tlr-same-rlink-p rlink (tlr-const :o))
      (tlr-same-rlink-p rlink (tlr-const :oi))
      (tlr-same-rlink-p rlink (tlr-const :s))
      (tlr-same-rlink-p rlink (tlr-const :si)))
)


(defun tlr-sort-table-pairs (pairs)
  (sort pairs #'(lambda (p1 p2)
		  (let ((c1 (tlr-card (cdr p1)))
			(c2 (tlr-card (cdr p2))))
		    (or (> c1 c2)
			(and (= c1 c2)
			     (>= (tlr-card (car p1))
				 (tlr-card (car p2))))))))
)


(defun tlr-test-rlink (tx ty test-rlink test)
  (let ((rlink (tlr-find-rlink tx ty)))
    (case test
      (:equal (tlr-same-rlink-p rlink test-rlink))
      (:subset (tlr-subset-p rlink test-rlink))
      (:intersect (tlr-intersect-p rlink test-rlink))))
)


(defun tlr-unbreak-ilink (tx ty)
  (let ((ilink (tlr-get-readable-ilink tx ty)))
    (when (and ilink (ilink-broken-p ilink))
      (let ((clink (tl-get-writable-clink tx)))
	(setf ilink (tlr-get-writable-ilink tx ty))
	(setf (ilink-broken-p ilink) nil)
	(push ilink (clink-rel-ilinks-list clink)))
      (let ((clink (tl-get-writable-clink ty)))
	(setf ilink (tlr-get-writable-ilink ty tx))
	(setf (ilink-broken-p ilink) nil)
	(push ilink (clink-rel-ilinks-list clink)))
      (tl-record-event :unbreak-rlink tx ty)))
)


(defun tlr-unique-sidelink-p (rlink)
  (or (tlr-same-rlink-p rlink (tlr-const :a))
      (tlr-same-rlink-p rlink (tlr-const :b))
      (tlr-same-rlink-p rlink (tlr-const :m))
      (tlr-same-rlink-p rlink (tlr-const :mi)))
)


(defun tlr-unique-uplink-p (rlink tx ty)
  (or (tlr-same-rlink-p rlink (tlr-const :d))
      (tlr-same-rlink-p rlink (tlr-const :f))
      (tlr-same-rlink-p rlink (tlr-const :s))
      (and (tl-older-interval-p tx ty)
	   (tlr-same-rlink-p rlink (tlr-const :e))))
)


(defun tlr-uplink-p (rlink tx ty)
  (if *tlr-autoref-unique-uplinks-p*
      (tlr-unique-uplink-p rlink tx ty)
      (tlr-subset-p rlink (if (tl-older-interval-p tx ty)
			      (tlr-const :d :e :f :s)
			      (tlr-const :d :f :s))))
)


(defun tlr-related-ints (tx)
  (let (relints)
    (dolist (ilink (tlr-get-ilinks tx) relints)
      (unless (tlr-all-rlink-p (ilink-current ilink))
	(push (ilink-target ilink) relints))))
)



;;; End of file TL-RLINKS
