;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-RLink-Codes.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	Run- and compile-time defs for relational links
;;;	Last Edit:	12/13/88 10:43:31
;;;
;;;	Copyright (c) 1988  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;
;;;
;;;	This file contains definitions for all TimeLogic functions dealing with
;;;	encoding and decoding of relational constraints.  This is in a seperate
;;;	file since it must be loaded at compile-time -- it is necessary for
;;;	expanding the macro TLR-CONST in the rest of the system.

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))

(defun tlr-encode-rlink-p (rel)
  (cond ((null rel) *tlr-nil*)
	((consp rel)
	 (let (carlink cdrlink)
	      (and (setq carlink (tlr-encode-rlink-p (car rel)))
		   (setq cdrlink (tlr-encode-rlink-p (cdr rel)))
		   (tlr-unite-rlinks carlink cdrlink))))
	(t (case rel
	     (:a   *tlr-a*)
	     (:b   *tlr-b*)
	     (:c   *tlr-c*)
	     (:d   *tlr-d*)
	     (:e   *tlr-e*)
	     (:f   *tlr-f*)
	     (:fi  *tlr-fi*)
	     (:m   *tlr-m*)
	     (:mi  *tlr-mi*)
	     (:o   *tlr-o*)
	     (:oi  *tlr-oi*)
	     (:s   *tlr-s*)
	     (:si  *tlr-si*)
	     (:all *tlr-all*)
	     (:dur *tlr-dur*)
	     (:con *tlr-con*)
	     (:dis *tlr-dis*)
	     (:int *tlr-int*)
	     (:neq *tlr-neq*))))
)


(defun tlr-encode-rlink (rel)
  (or (tlr-encode-rlink-p rel)
      (tl-error "Unsupported relational constraint specification: ~S" rel))
)


(defun tlr-decode-rlink (rlink)
  (cond ((tlr-all-rlink-p rlink) :all)
	(t (let (rels)
	     (if (tlr-intersect-p rlink *tlr-si*)
		 (push :si rels))
	     (if (tlr-intersect-p rlink *tlr-s*)
		 (push :s rels))
	     (if (tlr-intersect-p rlink *tlr-oi*)
		 (push :oi rels))
	     (if (tlr-intersect-p rlink *tlr-o*)
		 (push :o rels))
	     (if (tlr-intersect-p rlink *tlr-mi*)
		 (push :mi rels))
	     (if (tlr-intersect-p rlink *tlr-m*)
		 (push :m rels))
	     (if (tlr-intersect-p rlink *tlr-fi*)
		 (push :fi rels))
	     (if (tlr-intersect-p rlink *tlr-f*)
		 (push :f rels))
	     (if (tlr-intersect-p rlink *tlr-e*)
		 (push :e rels))
	     (if (tlr-intersect-p rlink *tlr-d*)
		 (push :d rels))
	     (if (tlr-intersect-p rlink *tlr-c*)
		 (push :c rels))
	     (if (tlr-intersect-p rlink *tlr-b*)
		 (push :b rels))
	     (if (tlr-intersect-p rlink *tlr-a*)
		 (push :a rels))
	     rels)))
)



;;; The relational constraint transitivity table

(defvar *tlr-decoded-constraints-table*
  '((:a (:a :a)
	(:b :all)
	(:c :a)
	(:d :a :d :f :mi :oi)
	(:e :a)
	(:f :a)
	(:fi :a)
	(:m :a :d :f :mi :oi)
	(:mi :a)
	(:o :a :d :f :mi :oi)
	(:oi :a)
	(:s :a :d :f :mi :oi)
	(:si :a))
    (:b (:a :all)
	(:b :b)
	(:c :b)
	(:d :b :d :m :o :s)
	(:e :b)
	(:f :b :d :m :o :s)
	(:fi :b)
	(:m :b)
	(:mi :b :d :m :o :s)
	(:o :b)
	(:oi :b :d :m :o :s)
	(:s :b)
	(:si :b))
    (:c (:a :a :c :mi :oi :si)
	(:b :b :c :fi :m :o)
	(:c :c)
	(:d :c :d :e :f :fi :o :oi :s :si)
	(:e :c)
	(:f :c :oi :si)
	(:fi :c)
	(:m :c :fi :o)
	(:mi :c :oi :si)
	(:o :c :fi :o)
	(:oi :c :oi :si)
	(:s :c :fi :o)
	(:si :c))
    (:d (:a :a)
	(:b :b)
	(:c :all)
	(:d :d)
	(:e :d)
	(:f :d)
	(:fi :b :d :m :o :s)
	(:m :b)
	(:mi :a)
	(:o :b :d :m :o :s)
	(:oi :a :d :f :mi :oi)
	(:s :d)
	(:si :a :d :f :mi :oi))
    (:e (:a :a)
	(:b :b)
	(:c :c)
	(:d :d)
	(:e :e)
	(:f :f)
	(:fi :fi)
	(:m :m)
	(:mi :mi)
	(:o :o)
	(:oi :oi)
	(:s :s)
	(:si :si))
    (:f (:a :a)
	(:b :b)
	(:c :a :c :mi :oi :si)
	(:d :d)
	(:e :f)
	(:f :f)
	(:fi :e :f :fi)
	(:m :m)
	(:mi :a)
	(:o :d :o :s)
	(:oi :a :mi :oi)
	(:s :d)
	(:si :a :mi :oi))
    (:fi (:a :a :c :mi :oi :si)
	 (:b :b)
	 (:c :c)
	 (:d :d :o :s)
	 (:e :fi)
	 (:f :e :f :fi)
	 (:fi :fi)
	 (:m :m)
	 (:mi :c :oi :si)
	 (:o :o)
	 (:oi :c :oi :si)
	 (:s :o)
	 (:si :c))
    (:m (:a :a :c :mi :oi :si)
	(:b :b)
	(:c :b)
	(:d :d :o :s)
	(:e :m)
	(:f :d :o :s)
	(:fi :b)
	(:m :b)
	(:mi :e :f :fi)
	(:o :b)
	(:oi :d :o :s)
	(:s :m)
	(:si :m))
    (:mi (:a :a)
	 (:b :b :c :fi :m :o)
	 (:c :a)
	 (:d :d :f :oi)
	 (:e :mi)
	 (:f :mi)
	 (:fi :mi)
	 (:m :e :s :si)
	 (:mi :a)
	 (:o :d :f :oi)
	 (:oi :a)
	 (:s :d :f :oi)
	 (:si :a))
    (:o (:a :a :c :mi :oi :si)
	(:b :b)
	(:c :b :c :fi :m :o)
	(:d :d :o :s)
	(:e :o)
	(:f :d :o :s)
	(:fi :b :m :o)
	(:m :b)
	(:mi :c :oi :si)
	(:o :b :m :o)
	(:oi :c :d :e :f :fi :o :oi :s :si)
	(:s :o)
	(:si :c :fi :o))
    (:oi (:a :a)
	 (:b :b :c :fi :m :o)
	 (:c :a :c :mi :oi :si)
	 (:d :d :f :oi)
	 (:e :oi)
	 (:f :oi)
	 (:fi :c :oi :si)
	 (:m :c :fi :o)
	 (:mi :a)
	 (:o :c :d :e :f :fi :o :oi :s :si)
	 (:oi :a :mi :oi)
	 (:s :d :f :oi)
	 (:si :a :mi :oi))
    (:s (:a :a)
	(:b :b)
	(:c :b :c :fi :m :o)
	(:d :d)
	(:e :s)
	(:f :d)
	(:fi :b :m :o)
	(:m :b)
	(:mi :mi)
	(:o :b :m :o)
	(:oi :d :f :oi)
	(:s :s)
	(:si :e :s :si))
    (:si (:a :a)
	 (:b :b :c :fi :m :o)
	 (:c :c)
	 (:d :d :f :oi)
	 (:e :si)
	 (:f :oi)
	 (:fi :c)
	 (:m :c :fi :o)
	 (:mi :mi)
	 (:o :c :fi :o)
	 (:oi :oi)
	 (:s :e :s :si)
	 (:si :si))))

(defvar *tlr-encoded-constraints-table* nil)


;;; do this when *tlr-decoded-constraints-table* changes!!!
;;; (progn (tlr-encode-constraints-table)
;;;        (tlr-compile-constraints-table))



;;; End of file TL-RLINK-CODES
