;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-Main.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	Main TimeLogic interface, assertions, retractions
;;;	Last Edit:	3/03/89 02:02:02 (but see below)
;;;
;;;	Copyright (c) 1993, 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;

;;; 13May93 by miller - change printing of backtrack points to include some data, and follow cl conventions on unreadability.

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))

;;; User functions


(defun define-interval (&optional txname refname &key context)
  "
    (DEFINE-INTERVAL &optional TXNAME REFNAME &key CONTEXT)
	Adds interval TXNAME to the TimeLogic database if not already present.
	If TXNAME = NIL, generates a new name.
	If given, makes interval REFNAME a reference for TXNAME in CONTEXT.
  "
  (let ((*tl-current-context* (tl-find-context context)))
    (timelogic-checkpoint)			; mark current state
    (prog1 (interval-name (tl-defint txname refname))
	   (timelogic-checkpoint)))		; mark new state
)


(defun interval-defined-p (name)
  "
    (INTERVAL-DEFINED-P NAME)
	Returns T if NAME is a defined interval.
  "
  (not (null (tl-interval-defined-p name)))
)


(defun defined-intervals ()
  "
    (DEFINED-INTERVALS)
	Returns a list of all intervals defined since the last TIMELOGIC-INIT.
  "
  (mapcar #'interval-name *tl-defined-intervals-list*)
)


(defun reference-intervals (txname &optional inverse-p &key context)
  "
    (REFERENCE-INTERVALS TXNAME &optional INVERSE-P &key CONTEXT)
	Returns a list of reference intervals of TXNAME	in CONTEXT.
	If INVERSE-P, return intervals for which TXNAME is a reference
	interval instead.
  "
  (let* ((tx (tl-get-interval txname))
	 (*tl-current-context* (tl-find-context context)))
    (mapcar #'interval-name
	    (if inverse-p
		(clink-referrals (tl-get-readable-clink tx))
		(clink-referents (tl-get-readable-clink tx)))))

)


(defun add-interval-constraint (txname constraint tyname &key context (type :rel))
  "
    (ADD-INTERVAL-CONSTRAINT TXNAME CONSTRAINT TYNAME &key CONTEXT (TYPE :REL))
	Adds relational CONSTRAINT (or durational if TYPE = :DUR) to intervals
	TXNAME and TYNAME, relative to CONTEXT (current context if NIL).
	Errors cause changed constraints to be reset if :auto-backtrack is :on.
	Returns NIL if a failure occurred, otherwise a backtracking point which
	can be given to TIMELOGIC-BACKTRACK to return the TimeLogic DB back to
	the state it was in *prior* to this constraint addition.
  "
  (tl-time-it *tl-posting-runtime*
    (let ((*tl-error-occurred-p* nil)
	  (*tlr-constraint-queue* nil)
	  (*tld-constraint-queue* nil)
	  (*tlr-hook-constraints* nil)
	  (*tld-hook-constraints* nil)
	  (*tlr-uplink-queue* nil)
	  (*tlr-sidelink-queue* nil))
      ;; These are specials rather than globals, to make this reentrant.  This
      ;; is necessary because we may be called recursively through the hooks!
      (let ((tx (tl-get-interval txname))
	    (ty (tl-get-interval tyname))
	    (*tl-current-context* (tl-find-context context))
	    (btpoint (timelogic-checkpoint)))	; mark current state
	(case (tl-constraint-type type)
	  (:rel (tlr-add-constraint tx ty (tlr-encode-rlink constraint)))
	  (:dur (tld-add-constraint tx ty (tld-encode-dlink constraint))))
	(tl-propagate)
	(unless *tl-error-occurred-p*
	  (tl-run-assert-hooks))
	(unless *tl-error-occurred-p*
	  (setf *tl-clean-p* nil)
	  (if *tl-auto-reference-p* (tlr-make-autorefs))
	  (timelogic-checkpoint)		; mark new state
	  btpoint)))))


(defun get-interval-constraint (txname tyname &key context (type :rel))
  "
    (GET-INTERVAL-CONSTRAINT TXNAME TYNAME &key CONTEXT (TYPE :REL))
	Returns the current relation (or duration if TYPE = :DUR)
	between intervals TXNAME and TYNAME relative to CONTEXT.
  "
  (tl-time-it *tl-fetching-runtime*
    (let ((*tl-current-context* (tl-find-context context))
	  (tx (tl-interval-defined-p txname))
	  (ty (tl-interval-defined-p tyname)))
      (and tx ty (case (tl-constraint-type type)
		   (:rel (tlr-decode-rlink (tlr-find-rlink tx ty)))
		   (:dur (tld-decode-dlink (tld-find-dlink tx ty)))))))
)


(defun related-intervals (txname &key context type)
  "
    (RELATED-INTERVALS TXNAME &key CONTEXT TYPE)
	Returns a list of all intervals that are *directly* related
	to TXNAME in the given CONTEXT (or current context if NIL).
	If TYPE = NIL, uses both :REL and :DUR.
  "
  (let ((*tl-current-context* (tl-find-context context))
	(tx (tl-interval-defined-p txname)))
    (mapcar #'interval-name
	    (cond ((null tx) nil)
		  ((null type) (union (tlr-related-ints tx)
				      (tld-related-ints tx)))
		  (t (case (tl-constraint-type type)
		       (:rel (tlr-related-ints tx))
		       (:dur (tld-related-ints tx)))))))
)


(defun sort-intervals (&optional intervals &key context)
  "
    (SORT-INTERVALS &optional INTERVALS &key CONTEXT)
	Returns a copy of INTERVALS sorted by (partial!) relational
	order relative to CONTEXT (or current context if NIL).
  "
  (let ((*tl-current-context* (tl-find-context context)))
    (mapcar #'interval-name
	    (tl-sort-ints (tl-get-ints intervals) t)))
)


(defun test-interval-constraint (txname constraint tyname
				 &key context (type :rel) (test :intersect))
  "
    (TEST-INTERVAL-CONSTRAINT TXNAME CONSTRAINT TYNAME
			      &key CONTEXT (TYPE :REL) (TEST :INTERSECT))
	Returns T iff CONSTRAINT is compatible with the current constraint
	between TXNAME and TYNAME.  Compatibility is determined by TEST, which
	is either :EQ (equality), :SUBSET (subset) or :INTERSECT (intersection)
  "
  (tl-time-it *tl-fetching-runtime*
    (let ((*tl-current-context* (tl-find-context context)))
      (case (tl-constraint-type type)
	(:rel (tlr-test-rlink
		(tl-get-interval txname)
		(tl-get-interval tyname)
		(tlr-encode-rlink constraint)
		(tl-constraint-test test)))
	(:dur (tld-test-dlink
		(tl-get-interval txname)
		(tl-get-interval tyname)
		(tld-encode-dlink constraint)
		(tl-constraint-test test))))))
)


(defun timelogic-backtrack (&optional btpoint)
  "
    (TIMELOGIC-BACKTRACK &optional BTPOINT)
	Return the TimeLogic DB back to the state in which BTPOINT was issued
	(or the last backtracking point issued if BTPOINT = NIL).
  "
  (cond ((null btpoint)
	 (tl-backtrack))
	((not (btpoint-p btpoint))
	 (tl-error "Invalid backtracking point: ~S" btpoint))
	((btpoint-used-p btpoint)
	 (warn "Backtracking point already used: ~S" btpoint))
	(t (tl-backtrack btpoint)))
)


(defun timelogic-checkpoint ()
  "
    (TIMELOGIC-CHECKPOINT)
	Returns an object that, when given to TimeLogic-Backtrack,
	will return TimeLogic back to the current state"
  (if (and (not (null *tl-backtrack-point*))
	   (eq (btpoint-kind *tl-backtrack-point*) :checkpoint))
      *tl-backtrack-point*			; don't make 2 in a row
      (tl-record-event :checkpoint))
)


(defun timelogic-checkpoint-p (&optional btpoint)
  "
    (TIMELOGIC-CHECKPOINT-P &optional BTPOINT)
	Returns T if BTPOINT (or the current state) is a valid checkpoint.
  "
  (let ((btp (or btpoint *tl-backtrack-point*)))
    (and (not (null btp))
	 (not (btpoint-used-p btp))
	 (eq :checkpoint (btpoint-kind btp))))
)


(defun timelogic-init nil
  "
    (TIMELOGIC-INIT)
	Resets the TimeLogic DB, wiping out all current information.
	Current TimeLogic property settings are not affected.
  "
  (let (*tl-trace-enabled-p*)
    (when (and *tl-display-enabled-p*
	       *tl-display-initialized-p*)
      (tl-clear-display))
    (tl-reset)
    (tl-init-contexts)
    (setf *tl-clean-p* t)
    (if *tl-props-reset-p*
	(tl-collect-props)
	(timelogic-reset-props)))
)


(defun timelogic-reset-props (&key (auto-define :on)
				   (all-paths :on)
				   (auto-backtrack :on)
				   (auto-reference :on)
				   (depth-first :off)
				   (display :off)
				   (durations :off)
				   (floats :off)
				   (leaves-only :warn)
				   (relations :on)
				   (sort :off)
				   (stats :off)
				   (tolerance 0.001)
				   (trace :off)
				   (wait :off))
  "
    (TIMELOGIC-RESET-PROPS &rest PROPERTY-KEYWORDS-&-VALUES)
	Resets the TimeLogic properties to their default values, which may
	be overriden by the supplying the appropriate props as keyword args.
  "
  (tl-putprop :auto-define auto-define)
  (tl-putprop :all-paths all-paths)
  (tl-putprop :auto-backtrack auto-backtrack)
  (tl-putprop :auto-reference auto-reference)
  (tl-putprop :depth-first depth-first)
  (tl-putprop :display display)
  (tl-putprop :durations durations)
  (tl-putprop :floats floats)
  (tl-putprop :leaves-only leaves-only)
  (tl-putprop :relations relations)
  (tl-putprop :sort sort)
  (tl-putprop :stats stats)
  (tl-putprop :tolerance tolerance)
  (tl-putprop :trace trace)
  (tl-putprop :wait wait)
  (setf *tl-props-reset-p* t)
  (tl-collect-props)
)

(defun timelogic-prop (prop &optional (newval nil newval-given-p))
  "
    (TIMELOGIC-PROP PROP &optional NEWVAL)
	Returns the *current* value of the TimeLogic property PROP.
	If NEWVAL is given, makes it the new value of PROP.
  "
  (unless *tl-props-reset-p* (timelogic-reset-props))
  (prog1 (cdr (assoc prop *tl-props*))
	 (if newval-given-p (tl-putprop prop newval))))

(defun tl-putprop (prop newval)
  (labels ((init-ok nil
	     (if *tl-clean-p*
		 t
		 (tl-error "TimeLogic property ~S may only be changed~%~A"
			   prop
			   (FORMAT NIL "   immediately following ~S" '(Timelogic-Init)))))
	   (badputprop nil
	     (tl-error "Unsupported value (~S) for property ~S" newval prop))
	   (y-or-n nil (case newval
			 (:on t)
			 (:off nil)
			 (t (badputprop)))))
    (setf *tl-error-occurred-p* nil)
    (case prop
      (:all-paths
       (unless (eq *tl-search-all-paths-p* (y-or-n))
	 (if (init-ok)
	     (setq *tl-search-all-paths-p* (not *tl-search-all-paths-p*)))))
      (:auto-backtrack
       (unless (eq *tl-auto-backtrack-p* (y-or-n))
	 (if (init-ok)
	     (setq *tl-auto-backtrack-p* (not *tl-auto-backtrack-p*)))))
      (:auto-define (setq *tl-auto-define-p* (y-or-n)))
      (:auto-reference
       (unless (eq *tl-auto-reference-p* (y-or-n))
	 (if (init-ok)
	     (setq *tl-auto-reference-p* (not *tl-auto-reference-p*)))))
      (:depth-first (setq *tl-propagate-depth-first-p* (y-or-n)))
      (:display (setq *tl-display-enabled-p* (y-or-n)))
      (:durations
       (unless (eq *tl-durations-enabled-p* (y-or-n))
	 (if (init-ok)
	     (setq *tl-durations-enabled-p* (not *tl-durations-enabled-p*)))))
      (:floats
       (unless (eq *tld-floats-enabled-p* (y-or-n))
	 (if (init-ok)
	     (setq *tld-floats-enabled-p* (not *tld-floats-enabled-p*)))))
      (:leaves-only (case newval
		      (:on (setq *tl-context-leaves-only* t))
		      (:off (setq *tl-context-leaves-only* nil))
		      (:warn (setq *tl-context-leaves-only* :warn))
		      (t (badputprop))))
      (:relations
       (unless (eq *tl-relations-enabled-p* (y-or-n))
	 (if (init-ok)
	     (setq *tl-relations-enabled-p* (not *tl-relations-enabled-p*)))))
      (:sort (setq *tl-sort-ints-before-display-p* (y-or-n)))
      (:stats (case newval
		(:on (or *tl-stats-p* (tl-reset-stats t)))
		(:off (tl-reset-stats nil))
		(:reset (tl-reset-stats t))
		(t (badputprop))))
      (:tolerance
	(let ((tol (cond ((null newval) nil)
			 ((or (not (floatp newval))
			      (< newval 0))
			  (badputprop))
			 ;; Make sure tolerance is between 1 and 0.5 (eg, 0.999)
			 ((< newval 0.5) (- 1 newval))
			 ((< newval 1) newval)
			 (t (- newval 1)))))
	  (unless (equal *tld-tolerance* tol)
	    (if (init-ok)
		(setq *tld-tolerance* tol)))))
      (:trace (setq *tl-trace-enabled-p*
		    (case (setq *tl-trace-mode* newval)
		      (:off nil)
		      ((:on :all :verbose) t)
		      (t (badputprop)))))
      (:wait (setq *tl-trace-wait*
		   (if (numberp newval) newval (y-or-n))))
      (t (tl-error "Unsupported property: ~S" prop)))
    (unless *tl-error-occurred-p*
      (let ((entry (assoc prop *tl-props*)))
	(if (null entry)
	    (setq *tl-props* (acons prop newval *tl-props*))
	    (setf (cdr entry) newval)))))
)

(defun tl-collect-props ()
  (mapcan #'identity
	  (sort (mapcar #'(lambda (prop) (list (car prop) (cdr prop))) *tl-props*)
		#'string-lessp :key #'car))
)


(defun timelogic-dump (pathname)
  ;; NEEDS MORE WORK!!!
  (let ((*package* (find-package "TIMELOGIC"))
	(*print-pretty* t))
    (with-open-file (dump (merge-pathnames pathname)
			  :direction :output)
      (format dump ";;; -*- Package: TimeLogic; Mode: Lisp;")
      (format dump " Syntax: Common-Lisp; Base: 10. -*-~%;;;~%")
      (multiple-value-bind (secs mins hrs day month year) (get-decoded-time)
	(format dump ";;; TimeLogic dump of ~2,'0d/~2,'0d/~2,'0d ~2,'0d:~2,'0d:~2,'0d~2%"
		month day (rem year 100) hrs secs mins))
      (print `(tl-restore :version ',*tl-dump-version*
			  :props ',*tl-props*
			  :contexts ',(context-tree t)
			  :intervals ',(tl-dump-get-intervals)
			  :rlinks ',(tl-dump-get-rlinks)
			  :dlinks ',(tl-dump-get-dlinks))
	     dump)))
)

;;; Implementation


;;; Intervals are connected to each other using relational
;;; links (RLINKs) and/or durational links (DLINKs).  RLINKs
;;; are internally represented by 13 element bitstrings (i.e.,
;;; integers), each bit indicating the presence of the corresponding
;;; relation.  See TLR-ENCODE-RLINK, TLR-DECODE-RLINK, and TLR-INVERT-RLINK


;;; DLINKs are internally represented by a CONS cell whose
;;; CAR is the lower bound and whose CDR is the upper bound.
;;; The upper bound may be Infinity, which is represented
;;; by NIL.  Either bound may be open, which is represented
;;; by a negative value bound.  0 and Infinity are by definition
;;; open bounds.


;;; General internal names are prefixed with TL, names related
;;; to manipulation of RLINKs are prefixed with TLR, and names
;;; related to manipulation of DLINKs are prefixed with TLD.


(defun tl-constraint-type (constraint-type)
  (case constraint-type
    (:rel	 :rel)
    (:relation	 :rel)
    (:relational :rel)
    (:dur	 :dur)
    (:duration	 :dur)
    (:durational :dur)
    (t (tl-error "Unrecognized constraint type: ~S" constraint-type)))
)


(defun tl-constraint-test (constraint-test)
  (case constraint-test
    (:eq	   :equal)
    (:eql	   :equal)
    (:equal	   :equal)
    (:subset	   :subset)
    (:intersect	   :intersect)
    (:intersection :intersect)
    (t (tl-error "Unrecognized constraint test: ~S" constraint-test)))
)


(defun tl-backtrack (&optional frombtp only-if-needed confirm-msg)
  "Removes (undoes) all changes to links inserted during last propagation.
   If FROMBTP is non-NIL, undo all that happened after FROMBTP, assuming 
   it is a backtracking point handed to the user (cf. TimeLogic-CheckPoint)."
  (let ((*tl-backtrack-in-progress* t) btp tx ty kind quitok)
    (when (and frombtp (not (eq (btpoint-kind frombtp) :checkpoint)))
      (error "Invalid backtrack checkpoint: ~S" frombtp))
    (setf quitok only-if-needed)
    (loop
      (unless (setf btp *tl-backtrack-point*)
	(return))
      (unless (btpoint-used-p btp)
	(setf kind (btpoint-kind btp))
	(case kind
	  (:checkpoint (if (or (eq btp frombtp)
			       (and quitok (null frombtp)))
			   (return)))
	  (:newrel nil)
	  (:newdur nil)
	  (t (when confirm-msg
	       (if (yes-or-no-p confirm-msg)
		   (setf confirm-msg nil)
		   (return)))
	     (setf tx (btpoint-tx btp))
	     (setf ty (btpoint-ty btp))
	     (case kind
	       (:newint (tl-backtrack-newint tx))
	       (:newref (tl-backtrack-newref tx ty))
	       (:delref (tl-backtrack-delref tx ty))
	       (:push-rlink (tl-backtrack-push-rlink tx ty))
	       (:push-dlink (tl-backtrack-push-dlink tx ty))
	       (:break-rlink (tl-backtrack-break-rlink tx ty))
	       (:unbreak-rlink (tl-backtrack-unbreak-rlink tx ty))
	       (t (error "Unrecognized kind of backtrack point: ~S"
			 kind)))))
	(setf (btpoint-used-p btp)
	      (setf quitok t)))
      (setq *tl-backtrack-point* (btpoint-prev btp))))
)


(defun tl-backtrack-break-rlink (tx ty)
  (tlr-unbreak-ilink tx ty)
)


(defun tl-backtrack-unbreak-rlink (tx ty)
  (tlr-break-ilink tx ty)
)


(defun tl-backtrack-delref (tx rx)
  (tl-assert-referent tx rx)
)


(defun tl-backtrack-push-dlink (tx ty)
  (tld-pop-dlink tx ty)
)


(defun tl-backtrack-newint (tx)
  ;; Since it doesn't matter having an interval defined or not, in terms of
  ;; constraints, ignore interval defs wrt backtracking
  (declare (ignore tx))
  nil
)


(defun tl-backtrack-push-rlink (tx ty)
  (tlr-pop-rlink tx ty)
)


(defun tl-backtrack-newref (tx rx)
  (tl-retract-referent tx rx)
)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Breaking ILinks because of Auto-Referencing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun tl-retract-indirect-refs (tx ty)
  ;; TY is a newly added reference interval of TX.  For all
  ;; TZ s.t.  TZ is a reference interval of TY, break the
  ;; reference link between TX and TZ.  For all TZ s.t.  TX
  ;; is a reference interval of TZ, break the reference link
  ;; between TY and TZ.
  (tl-retract-indirect-uprefs tx ty)
  (tl-retract-indirect-downrefs tx ty)
)

(defun tl-retract-indirect-uprefs (tx ty)
  (dolist (tz (tl-get-referents ty))
    (or (tl-retract-referent tx tz)
	(tl-retract-indirect-uprefs tx tz)))
)

(defun tl-retract-indirect-downrefs (tx ty)
  (dolist (tv (tl-get-referrals tx))
    (or (tl-retract-referent tv ty)
	(tl-retract-indirect-downrefs tv ty)))
)


(defun tl-unbreak-direct-refd-ilinks (tx ty)
  ;; ty is a new reference interval for tx.  Make sure tx
  ;; does not have a broken link with any tz under ty.
  (dolist (tz (tl-get-referrals ty))
    (tlr-unbreak-ilink tx tz))
)


(defun tl-circular-ref-p (tx rx)
  (or (eq tx rx)
      (dolist (rxref (tl-get-referents rx))
	(if (tl-circular-ref-p tx rxref)
	    (return t))))
)


(defun tl-common-reference-p (tx ty)
  (let ((txrefs (tl-get-referents tx)))
    (when txrefs
      (let ((tyrefs (tl-get-referents ty)))
	(when tyrefs
	  (dolist (tz txrefs)
	    (if (member tz tyrefs)
		(return t)))))))
)


(defun tl-convert-dlink-to-rlink (dlink)
  (cond ((tld-same-dlink-p dlink (tld-const :=))
	 ;; Everything but :CON or :DUR
	 (tlr-const :a :b :e :m :mi :o :oi))
	((tld-subset-p dlink (tld-const :<))
	 ;; Everything but equality or :CON
	 (tlr-const :a :b :d :f :m :mi :o :oi :s))
	((tld-subset-p dlink (tld-const :>))
	 ;; Everything but equality or :DUR
	 (tlr-const :a :b :c :fi :m :mi :o :oi :si))
	((tld-subset-p dlink (tld-const :<=))
	 ;; Everything but :CON
	 (tlr-const :a :b :d :e :f :m :mi :o :oi :s))
	((tld-subset-p dlink (tld-const :>=))
	 ;; Everything but :DUR
	 (tlr-const :a :b :c :e :fi :m :mi :o :oi :si))
	((tld-subset-p dlink (tld-const :<>))
	 ;; Everything but equality or :CON or :DUR
	 (tlr-const :a :b :m :mi :o :oi))
	(t (tlr-const :all)))
)


(defun tl-convert-rlink-to-dlink (rlink)
  (cond ((tlr-same-rlink-p rlink (tlr-const :e))
	 (tld-const :=))
	((tlr-subset-p rlink (tlr-const :d :f :s))
	 (tld-const :<))
	((tlr-subset-p rlink (tlr-const :c :fi :si))
	 (tld-const :>))
	((tlr-subset-p rlink (tlr-const :d :e :f :s))
	 (tld-const :<=))
	((tlr-subset-p rlink (tlr-const :c :e :fi :si))
	 (tld-const :>=))
	((tlr-subset-p rlink (tlr-const :c :d :f :fi :s :si))
	 (tld-const :<>))
	(t (tld-const (0 :INF))))
)


(defun tl-defint (origtxname refname)
  ;; If origtxname = NIL, generates new interval.  Returns
  ;; the interval structure (newly created only if necessary).
  (prog (tx (txname origtxname))
	(cond ((null txname)
	       ;; Create a new interval name.  Make sure it has not been
	       ;; used before
	       (loop (setq txname (gensym "GenInt"))
		     (or (setq tx (tl-interval-defined-p txname))
			 (return nil))))
	      ((setq tx (tl-interval-defined-p txname))
	       (return tx)))
	(setq tx (tl-create-interval txname (null origtxname)))
	(tl-record-event :newint tx)
	(cond ((null refname) nil)
	      ((not *tl-auto-reference-p*)
	       (tl-assert-referent tx (tl-defint refname nil)))
	      (t (tl-defint refname nil)))
	(setf *tl-clean-p* nil)
	(return tx))
)

(defun tl-create-interval (txname generated-p)
  (let ((tx (make-interval :name txname
			   :index (incf *tl-interval-count*)
			   :generated-p generated-p)))
    (push tx *tl-defined-intervals-list*)
    (setf (gethash txname *tl-defined-intervals-hash*) tx))
)

(defun tl-assert-referent (tx rx)
  (cond ((and (not *tl-auto-reference-p*)
	      (tl-circular-ref-p tx rx))
	 (tl-error "Cyclic reference structure not supported: ~S  -->  ~S"
		   (interval-name tx)
		   (interval-name rx)))
	((tl-add-referent tx rx)
	 (tl-record-event :newref tx rx)))
)

(defun tl-retract-referent (tx rx)
  (if (tl-del-referent tx rx)
      (tl-record-event :delref tx rx))
)


(defun tl-warn (message &rest msgargs)
  (warn "(TimeLogic) ~?~%" message msgargs)
  nil
)

(defun tl-error (message &rest msgargs)
  (setq *tl-error-occurred-p* t)
  (format *error-output* "~2&*** TimeLogic ERROR: ~?~%" message msgargs)
  (cond ((y-or-n-p "Break? ")
	 (apply #'error (cons message msgargs))))
  (tl-backtrack nil t "Undo last constraint propagation? ")
  (format *error-output*
	  "~2&*** Proceeding from TimeLogic ERROR ~A~%"
	  "with value NIL (may not work...)")
  (unless (or (null *tl-backtrack-point*)
	      (eq (btpoint-kind *tl-backtrack-point*) :checkpoint))
    (warn "DATABASE IS PROBABLY INCONSISTENT!!!   All bets are off..."))
  nil
)


(defun tl-get-interval (txname)
  (or (tl-interval-defined-p txname)
      (and *tl-auto-define-p* (tl-defint txname :root))
      (tl-error "Undefined interval: ~S" txname))
)

(defun tl-get-ints (intnames)
  (if intnames
      (mapcar #'tl-get-interval intnames)
      (copy-list *tl-defined-intervals-list*))
)


(defun tl-indirect-ref-p (tx ty)
  (dolist (tz (tl-get-referents tx))
    (if (tl-ref-path-p tz ty)
	(return t)))
)


(defun tl-print-btpoint (btpoint stream level)
  ;; Needed because backtracking points are returned from user-level
  ;; functions.  Defaulting on the print function causes infinite
  ;; recursion because of the circularity of the interval structure.
  (declare (ignore level))
  (format stream "#<TimeLogic Backtrack Point ~d> " (btpoint-id btpoint))
)


(defun tl-print-ilink (ilink stream level)
  ;;  Primarily for debugging purposes.  Users are not intended to ever
  ;;  see these.
  (write `(ilink :type ,(ilink-type ilink)
		 :class ,(ilink-class ilink)
		 :source ,(ilink-source ilink)
		 :target ,(ilink-target ilink)
		 :current ,(ilink-current ilink)
		 :unique-p ,(ilink-unique-p ilink)
		 :broken-p ,(ilink-broken-p ilink))
	 :stream stream
	 :level level)
)


(defun tl-print-interval (interval stream level)
  ;;  Primarily for debugging purposes.  Users are not intended to ever
  ;;  see these.
  (declare (ignore level))
  (print `(interval :name ,(interval-name interval)
		    :index ,(interval-index interval)
		    :generated-p ,(interval-generated-p interval))
	stream)
)


(defun tl-propagate nil
  (let (queue)
    (loop (cond (*tl-error-occurred-p* (return nil))
		((setq queue *tlr-constraint-queue*)
		 (setq *tlr-constraint-queue* (cdr queue))
		 (apply #'tlr-propagate-constraint
			(car queue)))
		((setq queue *tld-constraint-queue*)
		 (setq *tld-constraint-queue* (cdr queue))
		 (apply #'tld-propagate-constraint
			(car queue)))
		(t (return nil)))))
)


(defun tl-run-assert-hooks ()
  ;; Apply hooks to queued constraints changes, if any.
  ;; Signal incompatibility error if any hook returns NIL
  (let (con)
    (loop 
      (cond (*tlr-hook-constraints*
	     (unless (apply *tlr-assert-hook* (setq con (pop *tlr-hook-constraints*)))
	       (tlr-incompat-error (car con) (caddr con) (cadr con) (cadr con))))
	    (*tld-hook-constraints*
	     (unless (apply *tld-assert-hook* (setq con (pop *tld-hook-constraints*)))
	       (tld-incompat-error (car con) (caddr con) (cadr con) (cadr con))))
	    (t (return))))))

(defun tl-record-event (kind &optional tx ty)
  (if *tl-trace-enabled-p*
      (unless (eq kind :checkpoint)
	(tl-trace kind tx ty)))
  (unless *tl-backtrack-in-progress*
    (setf *tl-backtrack-point*
	  (make-btpoint :kind kind
			:tx tx
			:ty ty
			:prev *tl-backtrack-point*)))
  *tl-backtrack-point*
)


(defun tl-ref-path-p (tx ty)
  (let ((refs (tl-get-referents tx)))
    (or (member ty refs)
	(dolist (tz refs)
	(if (tl-ref-path-p tz ty)
	    (return t)))))
)


(defun tl-reset nil
  ;; Remove the circular structures of intervals and contexts,
  ;; so a refcounting gc won't choke on them when they get dropped.
  (mapc #'tl-reset-context *tl-contexts*)
  (setf *tl-contexts* nil)
  (setq *btpoint-id* 0)         ; bwm - for printing of backtrack points uniquely.
  
  (mapc #'tl-reset-interval *tl-defined-intervals-list*)
  (clrhash *tl-defined-intervals-hash*)
  (setf *tl-defined-intervals-list* nil)
  (setf *tl-interval-count* 0)

  (tl-reset-stats *tl-stats-p*)
  (setq *tl-backtrack-point* nil)
  (setq *tlr-constraint-queue* nil)
  (setq *tld-constraint-queue* nil)
  (setq *tl-error-occurred-p* nil)
)


(defun tl-reset-interval (tx)
  (setf (interval-clink tx) nil)
  (setf (interval-context tx) nil)
  (setf (interval-headpoint tx) nil)
  (setf (interval-tailpoint tx) nil)
)


(defun tl-reset-stats (on-p)
  (prog ((val (if on-p 0)))

	(setf *tl-posting-runtime* val)
	(setf *tl-fetching-runtime* val)

	;; Relational counters
	(setq *tlr-assert-count* val)
	(setq *tlr-constraint-count* val)
	(setq *tlr-multiply-count* val)
	(setq *tlr-try-add-count* val)

	;; Durational counters
	(setq *tld-assert-count* val)
	(setq *tld-constraint-count* val)
	(setq *tld-multiply-count* val)
	(setq *tld-try-add-count* val)

	;; report the stuff??
	(setq *tl-report-rlinks-p* (if on-p t))

	(return (setq *tl-stats-p* (if on-p t))))
)


(defun tl-setup-time-units nil
  (dolist (tu '(second minute hour day week month quarter year decade century))
    (define-interval tu 'time-units))
  (addintconq minute 60 second :type :dur)
  (addintconq hour 60 minute :type :dur)
  (addintconq day 24 hour :type :dur)
  (addintconq week 7 day :type :dur)
  (addintconq quarter 3 month :type :dur)
  (addintconq year 4 quarter :type :dur)
  (addintconq decade 10 year :type :dur)
  (addintconq century 10 decade :type :dur)
  ;; Can't handle variable generic intervals:
  (addintconq month (or 28 29 30 31) day :type :dur)
  ;; infer:  year (or 336 348 360 372) day
  (addintconq year (or 365 366) day :type :dur)
  ;; after propagation this causes a conflict between previously inferred dur link
  nil
)


(defun tl-shared-refs-p (tx ty)
  (prog (origxrefs origyrefs xrefs yrefs)

;;; Check if tx is a member of origyrefs or ty is a member
;;; of origxrefs

	(cond ((and (setq origxrefs (tl-get-referents tx))
		    (member ty origxrefs))
	       (return t))
	      ((and (setq origyrefs (tl-get-referents ty))
		    (member tx origyrefs))
	       (return t))
	      ((or (null origxrefs)
		   (null origyrefs))
	       (return nil)))

;;; Check if origxrefs and origyrefs intersect

	(setq xrefs origxrefs)
     loop1
	(cond ((member (car xrefs) origyrefs)
	       (return t))
	      ((setq xrefs (cdr xrefs))
	       (go loop1)))

	(if *tl-auto-reference-p* (return nil))

;;; Check if some xref is not disjoint from some yref

; This is not documented nor according to specs!
;	(setq xrefs origxrefs)
;     loop2
;	(setq yrefs origyrefs)
;     loop3
;	(cond ((not (tlr-subset-p (tlr-find-rlink (car xrefs)
;						  (car yrefs))
;				  (tlr-const :a :b :m :mi)))
;	       (return t))
;	      ((setq yrefs (cdr yrefs))
;	       (go loop3))
;	      ((setq xrefs (cdr xrefs))
;	       (go loop2))
;	      (t (return nil)))
	)
)

;;; Dump:  untangling the spaghetti!

(defun tl-dump-get-intervals ()
)

(defun tl-dump-get-rlinks ()
)

(defun tl-dump-get-dlinks ()
)


;;; End of file TL-MAIN
