;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-DLinks.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	Manipulation of durational constraints
;;;	Last Edit:	3/02/89 17:32:47
;;;
;;;	Copyright (c) 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))

(defun tld-get-dlink (tx ty)
  (let ((ilink (tld-get-readable-ilink tx ty)))
    (and ilink (ilink-current ilink)))
)

(defun tld-add-bounds (b1 b2)
  ;; Resulting bound is closed if both b1 and b2 are closed,
  ;; otherwise open.  Any bound plus Infinity = Infinity.
  (cond ((tld-inf-bound-p b1) b1)
	((tld-inf-bound-p b2) b2)
	(t (tld-make-bound (+ (tld-bound-value b1)
			      (tld-bound-value b2))
			   (or (tld-open-bound-p b1)
			       (tld-open-bound-p b2)))))
)


(defun tld-add-constraint (tx ty dlink)
  ;; If no error occurred sofar and the duration logic is
  ;; enabled, then if a dlink already exists adds the given
  ;; restriction, otherwise adds restriction to whatever can
  ;; be found through the hierarchy.
  (cond
    ((eq tx ty)
     (if (tld-no-dlink-p (tld-intersect-dlinks dlink (tld-const :=)))
	 (tld-incompat-error tx ty dlink (tld-const :=))))
    ((and *tl-durations-enabled-p* (not (tld-all-dlink-p dlink)))
     (tld-add-restriction tx ty dlink)
     (unless *tl-error-occurred-p*
       (tl-record-event :newdur tx ty))))
)


(defun tld-add-restriction (tx ty dlink)
  (unless (tld-all-dlink-p dlink)
    (if *tld-try-add-count* (incf *tld-try-add-count*))
    (let* ((olddlink (tld-get-dlink tx ty))
	   (curdlink (or olddlink (tld-find-dlink tx ty)))
	   (newdlink (tld-intersect-dlinks dlink curdlink)))
      (cond ((tld-no-dlink-p newdlink)
	     (tld-incompat-error tx ty dlink curdlink))
	    ((or (null olddlink)
		 (not (tld-same-dlink-p newdlink olddlink)))
	     (tld-assert-dlink tx ty newdlink (tld-invert-dlink newdlink))
	     (when *tl-relations-enabled-p*
	       (tlr-add-restriction
		 tx ty (tl-convert-dlink-to-rlink newdlink))))))))


(defun tld-add-ranges (dlink1 dlink2)
  (tld-make-range (tld-add-bounds (tld-lo-bound dlink1)
				  (tld-lo-bound dlink2))
		  (tld-add-bounds (tld-hi-bound dlink1)
				  (tld-hi-bound dlink2)))
)


(defun tld-all-range-p (range)
  (and (tld-inf-bound-p (tld-hi-bound range))
       (tld-nul-bound-p (tld-lo-bound range)))
)


(defun tld-assert-dlink (tx ty dlink invdlink)
  (tld-push-dlink tx ty dlink invdlink)
  (if *tld-assert-count* (incf *tld-assert-count*))
  (if *tld-assert-hook* (tld-hook-enqueue tx ty dlink invdlink))
  (if *tl-propagate-depth-first-p*
      (tld-propagate-constraint tx ty dlink invdlink
				(1+ *tl-propagation-level*))
      (tld-enqueue tx ty dlink invdlink)))


(defun tld-constrain (tx xydlink ty yzdlink tz)
  (declare (ignore ty))
  (when (or *tl-auto-reference-p*
	    (tld-get-readable-ilink tx tz)
	    (tl-shared-refs-p tx tz))
    (tld-add-restriction tx tz (tld-multiply-dlinks xydlink yzdlink)))
)


(defun tld-hook-enqueue (tx ty dlink invdlink)
  ;; Keep until we're done propagating.
  ;; Note this does not avoid duplicates!
  (push (list tx dlink ty) *tld-hook-constraints*))


(defun tld-enqueue (tx ty dlink invdlink)
  ;; Queue is mainained as a list of constraint entries, where
  ;; each entry is a list of the intervals tx and ty and the
  ;; new link values.  If tx ty already there, replaces the
  ;; old entry with the new one.  Otherwise the entry is added
  ;; at the back
  (prog (prev entry (queue *tld-constraint-queue*)
	 (newentry (list tx ty dlink invdlink (1+ *tl-propagation-level*))))
	(cond ((null queue)
	       (setq *tld-constraint-queue* (list newentry))
	       (return nil)))
	(loop (setq entry (car queue))
	      (cond ((or (and (eq tx (car entry))
			      (eq ty (caddr entry)))
			 (and (eq ty (car entry))
			      (eq tx (caddr entry))))
		     (rplaca queue newentry)
		     (return nil))
		    ((null (setq queue (cdr (setq prev queue))))
		     (rplacd prev (list newentry))
		     (return nil)))))
)


(defun tld-find-dlink (tx ty)
  (cond ((or (null tx)
	     (null ty))
	 (tld-const (0 :INF)))
	((eq tx ty)
	 (tld-const :=))
	((tld-find-dlink-aux-x tx ty))
	(t (tld-const (0 :INF))))
)

(defun tld-find-dlink-aux-x (tx ty)
  (or (tld-get-dlink tx ty)
      (tld-find-dlink-thru-yrefs tx ty)
      (tld-find-dlink-thru-xrefs tx ty))
)

(defun tld-find-dlink-aux-y (tx ty)
  (or (tld-get-dlink tx ty)
      (tld-find-dlink-thru-yrefs tx ty))
)

(defun tld-find-dlink-thru-xrefs (tx ty)
  ;; Goes up the hierarchy from TX
  (let (tx-rx rx-ty tx-ty tx*ty)
    (dolist (rx (tl-get-referents tx) tx*ty)
      (cond ((null (setq tx-rx (tld-get-dlink tx rx))))
	    ((tld-all-dlink-p tx-rx))
	    ((null (setq rx-ty (tld-find-dlink-aux-x rx ty))))
	    ((null (setq tx-ty (tld-multiply-dlinks tx-rx rx-ty))))
	    (tx*ty (setq tx*ty (tld-intersect-dlinks tx*ty tx-ty)))
	    (*tl-search-all-paths-p* (setq tx*ty tx-ty))
	    (t (return tx-ty)))))
)

(defun tld-find-dlink-thru-yrefs (tx ty)
  ;; Goes up the hierarchy from TY
  (let (tx-ry ry-ty tx-ty tx*ty)
    (dolist (ry (tl-get-referents ty) tx*ty)
      (cond ((null (setq ry-ty (tld-get-dlink ry ty))))
	    ((tld-all-dlink-p ry-ty))
	    ((null (setq tx-ry (tld-find-dlink-aux-y tx ry))))
	    ((null (setq tx-ty (tld-multiply-dlinks tx-ry ry-ty))))
	    (tx*ty (setq tx*ty (tld-intersect-dlinks tx*ty tx-ty)))
	    (*tl-search-all-paths-p* (setq tx*ty tx-ty))
	    (t (return tx-ty)))))
)


(defun tld-hi-bound-lessp (b1 b2)
  ;; Note: (n) < n
  (cond ((tld-inf-bound-p b2) t)
	((tld-inf-bound-p b1) nil)
	((and (tld-open-bound-p b1)
	      (not (tld-open-bound-p b2)))
	 (<= (tld-bound-value b1)
	     (tld-bound-value b2)))
	(t (< (tld-bound-value b1)
	      (tld-bound-value b2))))
)


(defun tld-incompat-error (tx ty dlink olddlink)
  (cond (*tl-auto-backtrack-p*
	 (setq *tl-error-occurred-p* t)
	 (tl-backtrack nil t))
	(t (tl-error "Incompatible durational constraint between ~S and ~S:~%~10Told: ~S~%~10Tnew: ~S"
		     (interval-name tx)
		     (interval-name ty)
		     (tld-decode-dlink olddlink)
		     (tld-decode-dlink dlink))))
)


(defun tld-intersect-dlinks (dlink1 dlink2)
  (prog (newdlink)
	(dolist (r1 dlink1)
	  (dolist (r2 dlink2)
	    (cond ((tld-range-lessp r1 r2)
		   (return nil))
		  ((not (tld-range-lessp r2 r1))
		   (setq newdlink
			 (tld-insert-range (tld-intersect-ranges r1 r2)
					   newdlink))))))
	(return newdlink))
)


(defun tld-intersect-p (dlink1 dlink2)
  (dolist (range1 dlink1)
    (if (dolist (range2 dlink2)
	  (cond ((tld-range-lessp range1 range2)
		 (return nil))
		((not (tld-range-lessp range2 range1))
		 (return t))))
	(return t)))
)


(defun tld-intersect-ranges (range1 range2)
  (tld-make-range (tld-max-lo-bound (tld-lo-bound range1)
				    (tld-lo-bound range2))
		  (tld-min-hi-bound (tld-hi-bound range1)
				    (tld-hi-bound range2)))
)


(defun tld-intersecting-ranges-p (range1 range2)
  (and (not (tld-range-lessp range1 range2))
       (not (tld-range-lessp range2 range1)))
)


(defun tld-invert-bound (bound)
  (cond ((tld-inf-bound-p bound) *tld-nul-bound*)
	((tld-nul-bound-p bound) *tld-inf-bound*)
	(t (tld-make-bound (/ 1 (tld-bound-value bound))
			   (tld-open-bound-p bound))))
)


(defun tld-invert-dlink (dlink)
  (prog (invdlink)
	(dolist (range dlink)
	  (push (tld-invert-range range)
		invdlink))
	(return invdlink))
)


(defun tld-invert-range (range)
  (tld-make-range (tld-invert-bound (tld-hi-bound range))
		  (tld-invert-bound (tld-lo-bound range)))
)


(defun tld-max-lo-bound (b1 b2)
  (cond ((tld-lo-bound-lessp b1 b2) b2)
	(t b1))
)


(defun tld-min-hi-bound (b1 b2)
  (cond ((tld-hi-bound-lessp b1 b2) b1)
	(t b2))
)


(defun tld-multiply-bounds (b1 b2)
  ;; Resulting bound is closed if both B1 and B2 are closed,
  ;; otherwise open.  any bound times infinity = infinity.
  (cond ((tld-nul-bound-p b1) b1)
	((tld-nul-bound-p b2) b2)
	((tld-inf-bound-p b1) b1)
	((tld-inf-bound-p b2) b2)
	(t (tld-make-bound (* (tld-bound-value b1)
			      (tld-bound-value b2))
			   (or (tld-open-bound-p b1)
			       (tld-open-bound-p b2)))))
)


(defun tld-multiply-dlinks (dlink1 dlink2)
  (cond (*tld-multiply-count* (incf *tld-multiply-count*)))
  (prog (newdlink)
	(dolist (range1 dlink1)
	  (dolist (range2 dlink2)
	    (setq newdlink (tld-insert-range (tld-multiply-ranges range1 range2)
					     newdlink))))
	(return newdlink))
)


(defun tld-multiply-ranges (range1 range2)
  (tld-make-range (tld-multiply-bounds (tld-lo-bound range1)
				       (tld-lo-bound range2))
		  (tld-multiply-bounds (tld-hi-bound range1)
				       (tld-hi-bound range2)))
)


(defun tld-pop-dlink (tx ty)
  (let ((ilink (tld-get-writable-ilink tx ty)))
       (setf (ilink-current ilink)
	     (pop (ilink-previous ilink)))
       (setf (ilink-unique-p ilink) nil))
  (let ((ilink (tld-get-writable-ilink ty tx)))
       (setf (ilink-current ilink)
	     (pop (ilink-previous ilink)))
       (setf (ilink-unique-p ilink) nil))
  (tl-record-event :pop-dlink tx ty)
)


(defun tld-propagate-constraint (tx ty dlink invdlink *tl-propagation-level*)
  (let (tz zdlink)
    (dolist (ilink (tld-get-ilinks tx))
      (when (and (setq zdlink (ilink-current ilink))
		 (not (eq ty (setq tz (ilink-target ilink)))))
	(tld-constrain ty invdlink tx zdlink tz)))

    (dolist (ilink (tld-get-ilinks ty))
      (when (and (setq zdlink (ilink-current ilink))
		 (not (eq tx (setq tz (ilink-target ilink)))))
	(tld-constrain tx dlink ty zdlink tz))))
)


(defun tld-push-dlink (tx ty dlink invdlink)
  (let ((unique-p (tld-single-dlink-p dlink)))
    (let ((ilink (tld-get-writable-ilink tx ty)))
      (push (ilink-current ilink)
	    (ilink-previous ilink))
      (setf (ilink-current ilink) dlink)
      (setf (ilink-unique-p ilink) unique-p))
    (let ((ilink (tld-get-writable-ilink ty tx)))
      (push (ilink-current ilink)
	    (ilink-previous ilink))
      (setf (ilink-current ilink) invdlink)
      (setf (ilink-unique-p ilink) unique-p))
    (tl-record-event :push-dlink tx ty))
)


(defun tld-related-ints (tx)
  (let (relints)
    (dolist (ilink (tld-get-ilinks tx) relints)
      (unless (tld-all-dlink-p (ilink-current ilink))
	(push (ilink-target ilink) relints))))
)


#+UNUSED
(defun tld-round-off (n)
  ;; This function rounds a numeric bound to some reasonable
  ;; number.  Integers > (LSH 1 16) are made into floats,
  ;; so that subsequent multiplication doesn't result in integer
  ;; overflow.  Floating point numbers are scaled up or down
  ;; to (LSH 1 16), rounded off by fixing, then scaled down
  ;; or up again. This rounding prevents excessive propagation
  ;; due to changing least significant digits.
  (cond ((= n 0) 0)
	((< n 0)
	 (- (tld-round-off (minus n))))
	((> n *tld-maxint*)
	 (tld-round-off-big n))
	((fixp n) n)
	((= 0.0 (remainder n 1.0))
	 (fix n))
	(t (tld-round-off-small n)))
)


#+UNUSED
(defun tld-round-off-big (n)
  (prog ((m (setq n (float n)))
	 (factor 1.0))
     loop
        (cond ((> m *tld-maxint*)
	       (setq factor (* 10.0 factor))
	       (setq m (/ n factor))
	       (go loop)))
	(return (* factor (fix (+ m .5)))))
)


#+UNUSED
(defun tld-round-off-small (n)
  (prog ((m n)
	 (factor 1.0))
     loop
	(cond ((lessp (* m 10) *tld-maxint*)
	       (setq factor (* 10.0 factor))
	       (setq m (* n factor))
	       (go loop)))
	(setq m (/ (fix (+ m .5)) factor))
	(return (cond ((= 0.0 (remainder m 1.0))
		       (fix m))
		      (t m))))
)


(defun tld-same-bound-p (b1 b2)
  (cond ((tld-inf-bound-p b1)
	 (tld-inf-bound-p b2))
	((tld-inf-bound-p b2) nil)
	((not (eq (tld-open-bound-p b1)
		  (tld-open-bound-p b2)))
	 nil)
	(t (let ((v1 (tld-bound-value b1))
		 (v2 (tld-bound-value b2)))
	     (cond ((= v1 v2) t)
		   ((null *tld-tolerance*) nil)
		   ((< v1 v2)
		    (> v1 (* v2 *tld-tolerance*)))
		   (t (> v2 (* v1 *tld-tolerance*)))))))
)

(defun tld-single-dlink-p (dlink)
  (and (null (cdr dlink))
       (tld-single-range-p (car dlink)))
)


(defun tld-single-range-p (range)
  (tld-same-bound-p (tld-lo-bound range)
		    (tld-hi-bound range))
)


(defun tld-subrange-p (range1 range2)
  (and (not (tld-lo-bound-lessp (tld-lo-bound range1)
				(tld-lo-bound range2)))
       (not (tld-hi-bound-lessp (tld-hi-bound range2)
				(tld-hi-bound range1))))
)


(defun tld-subset-p (dlink1 dlink2)
  (or (null dlink1)
      (let ((range1 (car dlink1))
	    (range2 (car dlink2)))
	(cond ((tld-range-lessp range1 range2) nil)
	      ((tld-range-lessp range2 range1) (tld-subset-p dlink1 (cdr dlink2)))
	      (t (tld-subrange-p range1 range2)))))
)


(defun tld-test-dlink (tx ty test-dlink test)
  (let ((dlink (tld-find-dlink tx ty)))
    (case test
      (:equal (tld-same-dlink-p dlink test-dlink))
      (:subset (tld-subset-p dlink test-dlink))
      (:intersect (tld-intersect-p dlink test-dlink))))
)



;;; End of file TL-DLINKS
