;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-DLink-Codes.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	Run- and compile-time defs for durational links
;;;	Last Edit:	2/08/89 11:19:29
;;;
;;;	Copyright (c) 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;
;;;
;;;	This file contains definitions for all TimeLogic functions dealing with
;;;	encoding and decoding of durational constraints.  It needs to be loaded
;;;	at compile-time since it is needed for expansion of the macro TLD-CONST
;;;	in the rest of the system.

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))


;;;  ENCODING DLINKS


(defun tld-encoding-error (dur)
  (tl-error "Unsupported durational constraint specification: ~S" dur)
)


(defun tld-hi-lo-bound-lessp (hb lb)
  ;; Note: n < (n)  AND  (n) < (n)  AND  (n) < n
  (cond ((tld-inf-bound-p lb) t)
	((tld-inf-bound-p hb) nil)
	((or (tld-open-bound-p hb)
	     (tld-open-bound-p lb))
	 (<= (tld-bound-value hb)
	     (tld-bound-value lb)))
	(t (< (tld-bound-value hb)
	      (tld-bound-value lb))))
)


(defun tld-range-lessp (range1 range2)
  (tld-hi-lo-bound-lessp (tld-hi-bound range1)
			 (tld-lo-bound range2))
)


(defun tld-lo-bound-lessp (b1 b2)
  ;; Note: n < (n)
  (cond ((tld-inf-bound-p b2) t)
	((tld-inf-bound-p b1) nil)
	((and (tld-open-bound-p b2)
	      (not (tld-open-bound-p b1)))
	 (<= (tld-bound-value b1)
	     (tld-bound-value b2)))
	(t (< (tld-bound-value b1)
	      (tld-bound-value b2))))
)


(defun tld-min-lo-bound (b1 b2)
  (if (tld-lo-bound-lessp b1 b2) b1 b2)
)


(defun tld-max-hi-bound (b1 b2)
  (if (tld-hi-bound-lessp b1 b2) b2 b1)
)


(defun tld-insert-range (newrange dlink)
  ;; Insert NEWRANGE into DLINK (which is a list of ranges),
  ;; assuming DLINK is ordered.
  (cond ((null dlink)
	 (list newrange))
	(t (let ((range (car dlink)))
	     (cond ((tld-range-lessp newrange range)
		    (cons newrange dlink))
		   ((tld-range-lessp range newrange)
		    (cons range (tld-insert-range newrange (cdr dlink))))
		   (t ;; Overlapping ranges, so merge
		    (tld-insert-range (tld-unite-ranges range newrange)
				      (cdr dlink)))))))
)


(defun tld-unite-ranges (range1 range2)
  (tld-make-range (tld-min-lo-bound (tld-lo-bound range1)
				    (tld-lo-bound range2))
		  (tld-max-hi-bound (tld-hi-bound range1)
				    (tld-hi-bound range2)))
)


(defun tld-unite-dlinks (dlink1 dlink2)
  (let ((newdlink dlink1))
    (dolist (range2 dlink2 newdlink)
      (setq newdlink (tld-insert-range range2 newdlink))))
)


(defun tld-encode-numeric-range (n &optional no-zerop)
  (cond ((or (not (numberp n))
	     (< n 0)
	     (and no-zerop (zerop n)))
	 (tld-encoding-error n))
	((and *tld-floats-enabled-p* (floatp n)) n)
	(t (rationalize n)))
)


(defun tld-encode-range (range)
  ;; Receives a range, returns a dlink!!!
  (cond ((symbolp range)
	 (case range				; non-keywords kept for
	   ((:< <)  *tld-lt*)			; backwards compatibility
	   ((:> >)  *tld-gt*)
	   ((:<= <=) *tld-le*)
	   ((:>= >=) *tld-ge*)
	   ((:= =)  *tld-eq*)
	   ((:<> <>) *tld-ne*)
	   (t  (tld-encoding-error range))))
	((numberp range)
	 (prog ((bound (tld-make-bound (tld-encode-numeric-range range t))))
	       (return (list (tld-make-range bound bound)))))
	((and (consp range)
	      (consp (cdr range)))
	 (prog (lopen (lo (car range))
		hopen (hi (cadr range)))
	       (cond ((consp lo)
		      (setq lo (car lo))
		      (setq lopen t)))
	       (cond ((consp hi)
		      (setq hi (car hi))
		      (setq hopen t)))
	       (cond ((numberp lo)
		      (setq lo (tld-encode-numeric-range lo))
		      (cond ((zerop lo)
			     (setq lopen t))))
		     (t (tld-encoding-error range)))
	       (cond ((or (eq hi :INF)
			  (eq hi '*))		; for backwards compatibility
		      (setq hi *tld-inf-value*))
		     ((numberp hi)
		      (setq hi (tld-encode-numeric-range hi t)))
		     (t (tld-encoding-error range)))
	       (cond ((and hi (< hi lo))
		      (tld-encoding-error range)))
	       (return (list (tld-make-range (tld-make-bound lo lopen)
					     (tld-make-bound hi hopen))))))
	(t (tld-encoding-error range)))
)


(defun tld-encode-dlink (dur)
  ;; The functions TLD-ENCODE-DLINK and TLD-DECODE-DLINK are
  ;; the only ones dealing with duration specs from outside.
  ;; Internally all is manipulated using dlinks.  A dlink
  ;; consists of list of ranges, where each range is a pair
  ;; (CONS) of TLDBOUND's.
  (cond ((and (consp dur)
	      (eq (car dur) 'or))
	 (let (dlink)
	   (dolist (range (cdr dur) dlink)
	     (setq dlink (tld-unite-dlinks dlink (tld-encode-range range))))))
	(t (tld-encode-range dur)))
)



;;; DECODING DLINKS


(defun tld-decode-bound (bound)
  (cond ((tld-inf-bound-p bound) :INF)
	((tld-nul-bound-p bound) 0)
	((tld-open-bound-p bound)
	 (list (tld-bound-value bound)))
	(t (tld-bound-value bound)))
)


(defun tld-decode-dlink (dlink)
  (cond ((null (cdr dlink))
	 (tld-decode-range (car dlink)))
	(t (cons 'or (mapcar #'tld-decode-range dlink))))
)


(defun tld-decode-range (range)
  (prog ((lo (tld-lo-bound range))
	 (hi (tld-hi-bound range)))
	(cond ((tld-inf-bound-p hi)
	       (cond ((= 1 (tld-bound-value lo))
		      (return (cond ((tld-open-bound-p lo) :>)
				    (t :>=))))))
	      ((tld-nul-bound-p lo)
	       (cond ((= 1 (tld-bound-value hi))
		      (return (cond ((tld-open-bound-p hi) :<)
				    (t :<=))))))
	      ((and (= 1 (tld-bound-value lo))
		    (= 1 (tld-bound-value hi)))
	       (return :=)))
	(setq lo (tld-decode-bound lo))
	(setq hi (tld-decode-bound hi))
	(return (cond ((eql lo hi) lo)
		      (t (list lo hi)))))
)



;;; End of file TL-DLINK-CODES
