;;; -*- Package: Cl-user; Syntax: Ansi-common-lisp; Mode: Lisp; Base: 10. -*-
;;;
;;;	File:		TL-Demos.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	TimeLogic system demonstrations
;;;	Last Edit:	02/11/89 16:09:50
;;;
;;;	Copyright (c) 1991, 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;

(defmacro catch-demo-aborts (&rest forms)
#+xerox
  `(il:nlsetq ,@forms)
#+explorer
  `(condition-case () (progn ,@forms) (sys:abort))
;; Modified 4/4/91 by miller@cs.rochester.edu; use the condition system.
#-(OR EXPLORER XEROX)
  `(WITH-SIMPLE-RESTART (ABORT "Go on to next demo") ,@FORMS)
)

(defun tl-demos ()
   (unless tl::*tl-display-initialized-p* (tl::tl-init-display))
   (format t "~3&Simple demo of TimeLogic: cooking-dining-dishes~%")
   (if (y-or-n-p "   Now? ")
       (catch-demo-aborts (tl-demo-1)))
   (format t "~3&Cooking-dining-dishes + context-dependent inference~%")
   (if (y-or-n-p "   Now? ")
       (catch-demo-aborts (tl-demo-2)))
   (loop (format t "~3&Switching the top two blocks on a stack~%")
	 (unless (y-or-n-p "   Now? ") (return))
	 (catch-demo-aborts (tl-demo-3)))
   'Done
)


;;; Simple demo of TimeLogic: cooking-dining-dishes

(defun tl-demo-1 ()
   (tl:timelogic-init)
   (tl:timelogic-reset-props :display :on
			     :sort :on
			     :trace :verbose
			     :wait :on)
   (tl:defintq cooking cook-demo)
   (tl:defintq dining cook-demo)
   (tl:defintq dishes cook-demo)
   (tl:addintconq cooking :dur cook-demo)
   (tl:addintconq dining :dur cook-demo)
   (tl:addintconq dishes :dur cook-demo)

   (tl:addintconq cooking :m dining)
   (tl:addintconq dining :m dishes)
)


;;; Cooking-dining-dishes + context-dependent inference

(defun tl-demo-2 ()
   (tl:timelogic-init)
   (tl:timelogic-reset-props :display :on
			     :sort :on
			     :trace :verbose
			     :wait :on)
   (tl:defintq cooking cook-demo)
   (tl:defintq dining cook-demo)
   (tl:defintq coffee cook-demo)
   (tl:defintq dishes cook-demo)
   (tl:addintconq cooking :dur cook-demo)
   (tl:addintconq dining :dur cook-demo)
   (tl:addintconq dishes :dur cook-demo)

   (tl:addintconq cooking :m dining)
   (tl:addintconq dining (:b :m) coffee)
   (tl:addintconq dining (:b :m) dishes)
   (tl:addintconq coffee :dis dishes)

   (tl:switch-context (tl:create-context 'coffee-first t))
   (tl:addintconq dining :m coffee)

   (tl:switch-context (tl:create-context 'dishes-first t))
   (tl:addintconq dining :m dishes)

  (format t "~2&In root context         :   coffee ~S dishes~%"
	(tl:get-interval-constraint 'coffee 'dishes :context t))
  (format t "In coffee-first context :   coffee ~S dishes~%"
	(tl:get-interval-constraint 'coffee 'dishes :context 'coffee-first))
  (format t "In dishes-first context :   coffee ~S dishes~%"
	(tl:get-interval-constraint 'coffee 'dishes :context 'dishes-first))
)


;;; Switching the top two blocks on a stack (cf. Allen & Koomen, IJCAI-83)

(defvar tl-demo-stack-ints '(stack-demo i Oba Oac Obc
					Oab g Oby Mbcy
					Mbca PUba Hb PDbc
					Oaz Mabz Mabc PUac
					Ha PDab))

(defun tl-demo-3 (&key autorefs stats graph trace display sort wait)
  (unless autorefs
    (setf autorefs (cond ((y-or-n-p "~6T~A~20T" "Auto Refs?") :on)
			 (t :off))))
  (unless stats
    (setf stats (cond ((y-or-n-p "~6T~A~20T" "Stats?") :reset)
		      (t :off))))
  (unless graph
    (setf graph (cond ((y-or-n-p "~6T~A~20T" "Graph?") :on)
		      (t :off))))
  (unless trace
    (setf trace (cond ((not (y-or-n-p "~6T~A~20T" "Trace?")) :off)
		      ((not (y-or-n-p "~9T~A~20T" "Verbose?")) :all)
		      (t :verbose))))
  (unless display
    (setf display (cond ((y-or-n-p "~6T~A~20T" "Display?") :on)
			(t :off))))
  (unless sort
    (setf sort (cond ((eq display :off) :off)
		     ((y-or-n-p "~6T~A~20T" "Sort?") :on)
		     (t :off))))
  (unless wait
    (setf wait (cond ((eq trace :off) :off)
		     ((not (y-or-n-p "~6T~A~20T" "Wait?")) :off)
		     ((y-or-n-p "~9T~A~20T" "1 second?") 1)
		     (t :on))))
   (tl:timelogic-init)
   (tl:timelogic-reset-props :auto-reference autorefs
			     :stats stats
			     :trace trace
			     :display display
			     :sort sort
			     :wait wait)
   (unless (eq trace :off)
     (dolist (int tl-demo-stack-ints)
       (tl:trace-interval int)))
   (tl-demo-stack-switch)
   (unless (eq display :off)
     (tl:display-intervals :ints tl-demo-stack-ints))
   (unless (eq stats :off)
     (tl:timelogic-stats))
   (unless (eq graph :off)
     (tl:graph-intervals))
)

(defun tl-demo-stack-switch NIL
  ;; Switch top two blocks on stack (cf. Allen & Koomen, IJCAI-83)
  (tl:timelogic-init)
  (tl:define-interval 'I 'stack-demo)
  (tl:define-interval 'G 'stack-demo)
  (tl:add-interval-constraint 'I :S 'stack-demo)
  (tl:add-interval-constraint 'G :F 'stack-demo)
  (tl:add-interval-constraint 'I :B 'G)
  (tl:define-interval 'Oba 'stack-demo)
  (tl:define-interval 'Oac 'stack-demo)
  (tl:define-interval 'Cb1 'stack-demo)
  (tl:add-interval-constraint 'Oba :Con 'I)
  (tl:add-interval-constraint 'Oac :Con 'I)
  (tl:add-interval-constraint 'Cb1 :Con 'I)
  (tl:define-interval 'Oab 'stack-demo)
  (tl:add-interval-constraint 'Oab :Dis 'Oac)
  (tl:add-interval-constraint 'Cb1 :Dis 'Oab)
  (tl:add-interval-constraint 'Cb1 :Dis 'Oab)
  (tl:define-interval 'Obc 'stack-demo)
  (tl:add-interval-constraint 'Obc :Dis 'Oac)
  (tl:add-interval-constraint 'Obc :Dis 'Oba)
  (tl:define-interval 'Ca1 'stack-demo)
  (tl:add-interval-constraint 'Ca1 :Dis 'Oba)
  (tl:add-interval-constraint 'Oab :Con 'G)
  (tl:add-interval-constraint 'Obc :Con 'G)
  (tl:add-interval-constraint 'Ca1 :Con 'G)
  (tl:define-interval 'Mabz 'stack-demo)
  (tl:add-interval-constraint 'Mabz '(:A :Mi) 'I)
  (tl:add-interval-constraint 'Mabz '(:B :M) 'G)
  (tl:define-interval 'Ca2 'stack-demo)
  (tl:add-interval-constraint 'Ca2 '(:A :B :E) 'Ca1)
  (tl:add-interval-constraint 'Ca2 :Dis 'Oba)
  (tl:add-interval-constraint 'Mabz :Mi 'Ca2)
  (tl:define-interval 'Oaz 'stack-demo)
  (tl:add-interval-constraint 'Mabz :Oi 'Oaz)
  (tl:define-interval 'Cb2 'stack-demo)
  (tl:add-interval-constraint 'Cb2 '(:A :B :E) 'Cb1)
  (tl:add-interval-constraint 'Cb2 :Dis 'Oab)
  (tl:add-interval-constraint 'Mabz '(:C :O :Si) 'Cb2)
  (tl:define-interval 'Cz2 'stack-demo)
  (tl:add-interval-constraint 'Mabz '(:C :Fi :O) 'Cz2)
  (tl:define-interval 'Ca3 'stack-demo)
  (tl:add-interval-constraint 'Ca3 '(:A :B :E) 'Ca1)
  (tl:add-interval-constraint 'Ca3 '(:A :B :E) 'Ca2)
  (tl:add-interval-constraint 'Ca3 :Dis 'Oba)
  (tl:add-interval-constraint 'Mabz :M 'Ca3)
  (tl:add-interval-constraint 'Mabz :O 'Oab)
  (tl:define-interval 'Mbcy 'stack-demo)
  (tl:add-interval-constraint 'Mbcy '(:A :Mi) 'I)
  (tl:add-interval-constraint 'Mbcy '(:B :M) 'G)
  (tl:define-interval 'Cb4 'stack-demo)
  (tl:add-interval-constraint 'Cb4 '(:A :B :E) 'Cb1)
  (tl:add-interval-constraint 'Cb4 '(:A :B :E) 'Cb2)
  (tl:add-interval-constraint 'Cb4 :Dis 'Oab)
  (tl:add-interval-constraint 'Mbcy :Mi 'Cb4)
  (tl:define-interval 'Oby 'stack-demo)
  (tl:add-interval-constraint 'Mbcy :Oi 'Oby)
  (tl:define-interval 'Cc4 'stack-demo)
  (tl:add-interval-constraint 'Cc4 :Dis 'Oac)
  (tl:add-interval-constraint 'Cc4 :Dis 'Obc)
  (tl:add-interval-constraint 'Mbcy '(:C :O :Si) 'Cc4)
  (tl:define-interval 'Cy4 'stack-demo)
  (tl:add-interval-constraint 'Mbcy '(:C :Fi :O) 'Cy4)
  (tl:define-interval 'Cb5 'stack-demo)
  (tl:add-interval-constraint 'Cb5 '(:A :B :E) 'Cb1)
  (tl:add-interval-constraint 'Cb5 '(:A :B :E) 'Cb2)
  (tl:add-interval-constraint 'Cb5 '(:A :B :E) 'Cb4)
  (tl:add-interval-constraint 'Cb5 :Dis 'Oab)
  (tl:add-interval-constraint 'Mbcy :M 'Cb5)
  (tl:add-interval-constraint 'Mbcy :O 'Obc)
  (tl:show-interval-constraints 'Mbca :with-ints '(Mabc))
  (tl:show-interval-constraints 'Mbcy :with-ints '(Mabz))
  (tl:show-interval-constraints 'PUac :with-ints '(PUba))
  (tl:show-interval-constraints 'Ha :with-ints '(Hb))
  (tl:show-interval-constraints 'PDab :with-ints '(PDbc))
  (tl:show-interval-constraints 'PUac :with-ints '(PDbc))
  (tl:show-interval-constraints 'PUba :with-ints '(PDab))
  (tl:add-interval-constraint 'Oby :E 'Oba)
  (tl:define-interval 'Mbca 'stack-demo)
  (tl:add-interval-constraint 'Mbca :E 'Mbcy)
  (tl:add-interval-constraint 'Oaz :E 'Oac)
  (tl:define-interval 'Mabc 'stack-demo)
  (tl:add-interval-constraint 'Mabc :E 'Mabz)
  (tl:define-interval 'PUac 'stack-demo)
  (tl:add-interval-constraint 'PUac :S 'Mabc)
  (tl:define-interval 'Ha 'stack-demo)
  (tl:add-interval-constraint 'Ca1 :Dis 'Ha)
  (tl:add-interval-constraint 'Ca2 :Dis 'Ha)
  (tl:add-interval-constraint 'Ca3 :Dis 'Ha)
  (tl:add-interval-constraint 'Ha :Dis 'Oba)
  (tl:add-interval-constraint 'PUac :M 'Ha)
  (tl:define-interval 'PDab 'stack-demo)
  (tl:add-interval-constraint 'Ha :M 'PDab)
  (tl:add-interval-constraint 'PDab :F 'Mabc)
  (tl:add-interval-constraint 'PUac :Mi 'Ca2)
  (tl:add-interval-constraint 'PUac :F 'Oac)
  (tl:add-interval-constraint 'PUac :M 'Ha)
  (tl:define-interval 'Cc6 'stack-demo)
  (tl:add-interval-constraint 'Cc6 '(:A :B :E) 'Cc4)
  (tl:add-interval-constraint 'Cc6 :Dis 'Oac)
  (tl:add-interval-constraint 'Cc6 :Dis 'Obc)
  (tl:add-interval-constraint 'PUac :M 'Cc6)
  (tl:define-interval 'Cb6 'stack-demo)
  (tl:add-interval-constraint 'Cb6 '(:A :B :E) 'Cb1)
  (tl:add-interval-constraint 'Cb6 '(:A :B :E) 'Cb2)
  (tl:add-interval-constraint 'Cb6 '(:A :B :E) 'Cb4)
  (tl:add-interval-constraint 'Cb6 '(:A :B :E) 'Cb5)
  (tl:add-interval-constraint 'Cb6 :Dis 'Oab)
  (tl:add-interval-constraint 'PDab :Mi 'Cb6)
  (tl:add-interval-constraint 'PDab :Mi 'Ha)
  (tl:add-interval-constraint 'PDab :S 'Oab)
  (tl:add-interval-constraint 'PDab :M 'Ca3)
  (tl:define-interval 'PUba 'stack-demo)
  (tl:add-interval-constraint 'PUba :S 'Mbca)
  (tl:define-interval 'Hb 'stack-demo)
  (tl:add-interval-constraint 'Cb1 :Dis 'Hb)
  (tl:add-interval-constraint 'Cb2 :Dis 'Hb)
  (tl:add-interval-constraint 'Cb4 :Dis 'Hb)
  (tl:add-interval-constraint 'Cb5 :Dis 'Hb)
  (tl:add-interval-constraint 'Cb6 :Dis 'Hb)
  (tl:add-interval-constraint 'Hb :Dis 'Oab)
  (tl:add-interval-constraint 'PUba :M 'Hb)
  (tl:define-interval 'PDbc 'stack-demo)
  (tl:add-interval-constraint 'Hb :M 'PDbc)
  (tl:add-interval-constraint 'PDbc :F 'Mbca)
  (tl:add-interval-constraint 'PUba :Mi 'Cb4)
  (tl:add-interval-constraint 'PUba :F 'Oba)
  (tl:add-interval-constraint 'PUba :M 'Hb)
  (tl:define-interval 'Ca7 'stack-demo)
  (tl:add-interval-constraint 'Ca7 '(:A :B :E) 'Ca1)
  (tl:add-interval-constraint 'Ca7 '(:A :B :E) 'Ca2)
  (tl:add-interval-constraint 'Ca7 '(:A :B :E) 'Ca3)
  (tl:add-interval-constraint 'Ca7 :Dis 'Oba)
  (tl:add-interval-constraint 'Ca7 :Dis 'Ha)
  (tl:add-interval-constraint 'PUba :M 'Ca7)
  (tl:define-interval 'Cc7 'stack-demo)
  (tl:add-interval-constraint 'Cc7 '(:A :B :E) 'Cc4)
  (tl:add-interval-constraint 'Cc7 '(:A :B :E) 'Cc6)
  (tl:add-interval-constraint 'Cc7 :Dis 'Oac)
  (tl:add-interval-constraint 'Cc7 :Dis 'Obc)
  (tl:add-interval-constraint 'PDbc :Mi 'Cc7)
  (tl:add-interval-constraint 'PDbc :Mi 'Hb)
  (tl:add-interval-constraint 'PDbc :S 'Obc)
  (tl:add-interval-constraint 'PDbc :M 'Cb5)
  (tl:show-interval-constraints 'Mbca :with-ints '(Mabc))
  (tl:show-interval-constraints 'Mbcy :with-ints '(Mabz))
  (tl:show-interval-constraints 'PUac :with-ints '(PUba))
  (tl:show-interval-constraints 'Ha :with-ints '(Hb))
  (tl:show-interval-constraints 'PDab :with-ints '(PDbc))
  (tl:show-interval-constraints 'PUac :with-ints '(PDbc))
  (tl:show-interval-constraints 'PUba :with-ints '(PDab))
  tl::*tlr-assert-count*
)

;;; Xerox Lisp window goodies

#+xerox
(defvar *tl-demo-buttons* nil)

#+xerox
(defvar *tl-demo-intervals-sketchwindow* nil)

#+xerox
(defvar *tl-demo-stack-switch-sketchwindow* nil)

#+xerox
(defvar *tl-demo-intervals-sketch*
  '((il:sketch nil il:sketchcontext
	       ((round 2 0)
		(il:modern 12 (il:medium il:regular il:regular))
		(il:left il:baseline)
		(il:line 30.0 12)
		nil nil (il:center il:center)
		(nil nil)
		t nil nil 1.0))
    (((0.024 68.0 nil)
      (il:group
	(156.0 228.0 136.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((160.0 . 248.0) (160.0 . 232.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 64.0 nil)
	  (il:wire ((160.0 . 240.0) (288.0 . 240.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((288.0 . 248.0) (288.0 . 232.0))
		   (round 2 0)
		   nil nil 1.0)))
	(224.0 . 240.0)))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 232.0)
	("X")
	1.0
	(il:left il:baseline)
	(il:modern 12 (il:medium il:regular il:regular))
	((16.0 229.0 8.0 14.0))
	nil))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 208.0)
	("X  before  Y")
	1.0
	(il:left il:baseline)
	(il:modern 12 (il:medium il:regular il:regular))
	((16.0 205.0 66.0 14.0))
	nil))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 176.0)
	("X  meets  Y")
	1.0
	(il:left il:baseline)
	(il:modern 12 (il:medium il:regular il:regular))
	((16.0 173.0 63.0 14.0))
	nil))
     ((0.024 44.0 nil)
      (il:group
	(324.0 204.0 88.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((408.0 . 224.0) (408.0 . 208.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 40.0 nil)
	  (il:wire ((328.0 . 216.0) (408.0 . 216.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((328.0 . 224.0) (328.0 . 208.0))
		   (round 2 0)
		   nil nil 1.0)))
	(368.0 . 216.0)))
     ((0.024 44.0 nil)
      (il:group
	(284.0 172.0 88.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((368.0 . 192.0) (368.0 . 176.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 40.0 nil)
	  (il:wire ((288.0 . 184.0) (368.0 . 184.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((288.0 . 192.0) (288.0 . 176.0))
		   (round 2 0)
		   nil nil 1.0)))
	(328.0 . 184.0)))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 144.0)
	("X  overlaps  Y")
	1.0
	(il:left il:baseline)
	(il:modern 12 (il:medium il:regular il:regular))
	((16.0 141.0 75.0 14.0))
	nil))
     ((0.024 44.0 nil)
      (il:group
	(252.0 140.0 88.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((336.0 . 160.0) (336.0 . 144.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 40.0 nil)
	  (il:wire ((256.0 . 152.0) (336.0 . 152.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((256.0 . 160.0) (256.0 . 144.0))
		   (round 2 0)
		   nil nil 1.0)))
	(296.0 . 152.0)))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 112.0)
	("X  finished by  Y")
	1.0
	(il:left il:baseline)
	(il:modern 12
		   (il:medium il:regular il:regular))
	((16.0 109.0 88.0 14.0))
	nil))
     ((0.024 44.0 nil)
      (il:group
	(204.0 108.0 88.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((288.0 . 128.0) (288.0 . 112.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 40.0 nil)
	  (il:wire ((208.0 . 120.0) (288.0 . 120.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((208.0 . 128.0) (208.0 . 112.0))
		   (round 2 0)
		   nil nil 1.0)))
	(248.0 . 120.0)))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 80.0)
	("X  contains  Y")
	1.0
	(il:left il:baseline)
	(il:modern 12 (il:medium il:regular il:regular))
	((16.0 77.0 74.0 14.0))
	nil))
     ((0.024 44.0 nil)
      (il:group
	(180.0 76.0 88.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((264.0 . 96.0) (264.0 . 80.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 40.0 nil)
	  (il:wire ((184.0 . 88.0) (264.0 . 88.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((184.0 . 96.0) (184.0 . 80.0))
		   (round 2 0)
		   nil nil 1.0)))
	(224.0 . 88.0)))
     ((0.05 14.0 nil)
      (il:text
	(16.0 . 48.0)
	("X  started by  Y")
	1.0
	(il:left il:baseline)
	(il:modern 12 (il:medium il:regular il:regular))
	((16.0 45.0 85.0 14.0))
	nil))
     ((0.024 44.0 nil)
      (il:group
	(156.0 36.0 88.0 24.0)
	(((0.0 8.0 nil)
	  (il:wire ((240.0 . 56.0) (240.0 . 40.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 40.0 nil)
	  (il:wire ((160.0 . 48.0) (240.0 . 48.0))
		   (round 2 0)
		   nil nil 1.0))
	 ((0.0 8.0 nil)
	  (il:wire ((160.0 . 56.0) (160.0 . 40.0))
		   (round 2 0)
		   nil nil 1.0)))
	(200.0 . 48.0))))
    ((0.024 44.0 nil)
     (il:group
       (156.0 36.0 88.0 24.0)
       (((0.0 8.0 nil)
	 (il:wire ((240.0 . 56.0) (240.0 . 40.0))
		  (round 2 0)
		  nil nil 1.0))
	((0.0 40.0 nil)
	 (il:wire ((160.0 . 48.0) (240.0 . 48.0))
		  (round 2 0)
		  nil nil 1.0))
	((0.0 8.0 nil)
	 (il:wire ((160.0 . 56.0) (160.0 . 40.0))
		  (round 2 0)
		  nil nil 1.0)))
       (200.0 . 48.0))))
)

#+xerox
(defvar *tl-demo-stack-switch-sketch*
  '((il:sketch nil il:sketchcontext
	       ((round 2 0)
		(il:modern 12 (il:bold il:regular il:regular))
		(il:center il:baseline)
		(il:line 30.0 12)
		nil nil (il:center il:center)
		(nil nil)
		t nil nil 1.0))
    (((0.0 56.0 nil)
      (il:wire ((120.0 . 48.0) (232.0 . 48.0))
	       (round 3 0)
	       nil nil 1.0))
     ((0.0 56.0 nil)
      (il:wire ((400.0 . 48.0) (512.0 . 48.0))
	       (round 3 0)
	       nil nil 1.0))
     ((0.0 80.0 nil)
      (il:wire ((144.0 . 48.0) (144.0 . 208.0)
		(208.0 . 208.0) (208.0 . 48.0))
	       (round 2 0)
	       nil nil 1.0))
     ((0.0 80.0 nil)
      (il:wire ((424.0 . 48.0) (424.0 . 208.0)
		(488.0 . 208.0) (488.0 . 48.0))
	       (round 2 0)
	       nil nil 1.0))
     ((0.0 32.0 nil)
      (il:wire ((144.0 . 160.0) (208.0 . 160.0))
	       (round 2 0)
	       nil nil 1.0))
     ((0.0 32.0 nil)
      (il:wire ((144.0 . 104.0) (208.0 . 104.0))
	       (round 2 0)
	       nil nil 1.0))
     ((0.0 32.0 nil)
      (il:wire ((424.0 . 160.0) (488.0 . 160.0))
	       (round 2 0)
	       nil nil 1.0))
     ((0.0 32.0 nil)
      (il:wire ((424.0 . 104.0) (488.0 . 104.0))
	       (round 2 0)
	       nil nil 1.0))
     ((0.05 13.0 nil)
      (il:text
	(176.0 . 184.0)
	("B")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((172.5 180.5 7.0 13.0))
	nil))
     ((0.05 13.0 nil)
      (il:text
	(176.0 . 128.0)
	("A")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((171.5 124.5 9.0 13.0))
	nil))
     ((0.05 13.0 nil)
      (il:text
	(176.0 . 72.0)
	("C")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((172.5 68.5 7.0 13.0))
	nil))
     ((0.05 13.0 nil)
      (il:text
	(456.0 . 184.0)
	("A")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((451.5 180.5 9.0 13.0))
	nil))
     ((0.05 13.0 nil)
      (il:text
	(456.0 . 128.0)
	("B")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((452.5 124.5 7.0 13.0))
	nil))
     ((0.05 13.0 nil)
      (il:text
	(456.0 . 72.0)
	("C")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((452.5 68.5 7.0 13.0))
	nil))
     ((0.0 52.0 nil)
      (il:wire
	((264.0 . 128.0) (368.0 . 128.0))
	(round 2 0)
	(nil (il:line 30.0 12.0))
	nil 1.0))
     ((0.05 13.0 nil)
      (il:text
	(320.0 . 8.0)
	("Planning a block switch")
	1.0
	(il:center il:baseline)
	(il:modern 12 (il:bold il:regular il:regular))
	((255.5 4.5 129.0 13.0))
	nil)))
    ((0.05 13.0 nil)
     (il:text
       (320.0 . 8.0)
       ("Planning a block switch")
       1.0
       (il:center il:baseline)
       (il:modern 12 (il:bold il:regular il:regular))
       ((255.5 4.5 129.0 13.0))
       nil)))
)

#+xerox
(defun tl-demo-setup (&optional nobuttons? create?)
   (cond (create?
	   (if (il:windowp *tl-demo-intervals-sketchwindow*)
	       (il:closew *tl-demo-intervals-sketchwindow*))
	   (setq *tl-demo-intervals-sketchwindow* nil)))
   (if *tl-demo-intervals-sketchwindow*
       (il:openw *tl-demo-intervals-sketchwindow*)
     (setq *tl-demo-intervals-sketchwindow*
	   (il:sketchw.create *tl-demo-intervals-sketch*
		      '(9.0 25.0 425.0 235.0)
		      (let ((w 450)
			    (h 250))
			(il:createregion (- il:screenwidth w) 500 w h))
		      "6 basic interval relations")))
   (cond (create?
	  (if (il:windowp *tl-demo-stack-switch-sketchwindow*)
	       (il:closew *tl-demo-stack-switch-sketchwindow*))
	   (setq *tl-demo-stack-switch-sketchwindow* nil)))
   (if *tl-demo-stack-switch-sketchwindow*
       (il:openw *tl-demo-stack-switch-sketchwindow*)
     (setq *tl-demo-stack-switch-sketchwindow*
	   (il:sketchw.create *tl-demo-stack-switch-sketch*
		      '(46.0 -18.0 545.0 279.0)
		      (let ((w 550)
			    (h 300))
			(il:createregion (- il:screenwidth w) 500 w h))
		      "Overlapping actions:  switching two blocks")))
   (cond ((or create? nobuttons?)
	  (mapc #'il:delete-button *tl-demo-buttons*)
	  (setq *tl-demo-buttons* nil)))
   (unless (fboundp #'il:create-button) (setq nobuttons? t))
   (setq nobuttons? t)
   (cond (nobuttons?
	   (il:windowprop (il:shrinkw *tl-demo-intervals-sketchwindow*
				      nil '(500 . 500))
			  'il:title "Intervals")
	   (il:windowprop (il:shrinkw *tl-demo-stack-switch-sketchwindow*
				      nil '(575 . 500))
			  'il:title "Block Switch"))
	 (t (il:closew *tl-demo-intervals-sketchwindow*)
	    (il:closew *tl-demo-stack-switch-sketchwindow*)
	    (if *tl-demo-buttons*
		(mapc #'il:redisplay-button *tl-demo-buttons*)
	      (setq *tl-demo-buttons*
		    (list (il:create-button
			    '(if (il:openwp *tl-demo-intervals-sketchwindow*)
				 (il:closew *tl-demo-intervals-sketchwindow*)
			       (il:openw *tl-demo-Intervals-SketchWindow*))
			    "Intervals"
			    '(249 . 113))
			  (il:create-button
			    '(if (il:openwp *tl-demo-stack-switch-sketchwindow*)
				 (il:closew *tl-demo-stack-switch-sketchwindow*)
			       (il:openw *tl-demo-stack-switch-sketchwindow*))
			    "Block Switch"
			    '(249 . 64))
			  (il:create-button
			    '(tl::tl-graph-refs
			       "TimeLogic Reference Hierarchy")
			    "Graph Refs"
			    '(249 . 15))
			  (il:create-button
			    "(tl-demo)"
			    "TimeLogic Demo"
			    '(366 . 15)))))))
   (tl:timelogic-prop :auto-reference :on)
)

#+xerox
(eval-when (load)
  (tl-demo-setup)
)

;;; End of file TL-DEMOS
