;;; 
;;; Author: Stephen P Spackman <spackman@dfki.uni-sb.de>
;;; load this file, then tempos-test2.lisp in this directory.
;;; This file provides some rudimentary ability to work with absolute times in
;;; tempos/timelogic. The external interface is as follows:
;;;
;;; Times are still represented as type *t-time. This package, however, builds
;;; reference times of its own, so don't assume that everything that exists is
;;; something you declared!
;;;
;;; It is ok to use all of the tempos stuff, like time-reln, time-durn,
;;; time-disjoint, time-intersects etc..
;;;
;;; [the-second*t-time]
;;;     Constant - reference object, by convention one second long and situated
;;;     at the Common Lisp epoch.
;;; [starts-at ?time*t-time ?universal-time*t-integer]
;;;     Predicate, queryable, assertable. ?time starts at ?universal-time.
;;; [ends-at ?time*t-time ?universal-time*t-integer]
;;;     Predicate, queryable, assertable. ?time ends at ?time*t-time.
;;; [length ?t*t-time ?duration*t-integer]
;;;     Predicate, queryable, assertable. ?time is of length ?duration seconds.
;;; [starts-before ?time*t-time ?universal-time*t-integer]
;;;     Predicate, assertable only.
;;; [starts-after ?time*t-time ?universal-time*t-integer]
;;;     Predicate, assertable only.
;;; [ends-before ?time*t-time ?universal-time*t-integer]
;;;     Predicate, assertable only.
;;; [ends-after ?time*t-time ?universal-time*t-integer]
;;;     Predicate, assertable only.
;;; [longer-than ?time*t-time ?universal-time*t-integer]
;;;     Predicate, assertable only.
;;; [shorter-than ?time*t-time ?universal-time*t-integer]
;;;     Predicate, assertable only.
;;;
;;; [overlaps ?time1*t-time ?time2*t-time]
;;;     Predicate. Do ?time1 and ?time2 overlap? => prefer time-reln or
;;;     time-intersects, though, maybe.

(in-package :rhet-user)

;;; The Canonical Second (THE-SECOND) reaches from time zero (midnight on
;;; january the first, 1900, as I recall - but whatever Common Lisp uses) to
;;; time one (one second later). We can engineer absolute times and durations
;;; relative to this point - without sacrificing the ability to do relative
;;; ones to our hearts' content.
;;;
;;; This saves us from trouble in doing absolute arithmetic (all VALUES are
;;; actually stored as relative durations!) but it does have the disadvantage
;;; that each fully specified event will end up using three instances of
;;; *t-time - which is pretty grim given the performance rolloff we can expect
;;; from TimeLogic.

(define-time [the-second])

(rassert
 
 [[the-time ?x*t-time ?t*t-integer] 
  <temporal
  [the-time-internal ?x ?t]
  [cut]]
 
 [[the-time-internal the-second 1] <temporal]
 
 ;; For each actual absolute instant, we define and cache a reference time.
 ;; Timelogic can't handle point times, though, so outlaw zero.
 [[the-time-internal ?x*t-time ?t*t-integer] 
  <temporal
  [bound ?t]
  [> ?t 0]
  [var ?x]
  [cut]
  [time-skolem ?x]
  [forall ?y*t-time [and [the-time-internal ?y ?u*t-integer] [bound ?u]]
	  [or [and [< ?t ?u] [assert-fact [time-reln ?x :s ?y]]]
	      [and [== ?t ?u] [assert-fact [time-reln ?x :e ?y]]]
	      [and [< ?u ?t] [assert-fact [time-reln ?y :s ?x]]]
	      [call (error "Time is out of order!")]]]
  [assert-fact [time-durn ?x ?t the-second]
	       [the-time-internal ?x ?t]]]
 
 [[ends-at ?x*t-time ?t*t-integer]
  <temporal
  [bound ?x]
  [the-time ?x ?t]]
 
 [[starts-at ?x*time 0]
  <temporal
  [bound ?x]
  [the-time ?x ?n*t-integer]]

 [[time-reln ?x*t-time (:d :e :f :oi :s :si) ?r*t-time]
  <temporal
  [prove [the-time ?r ?t*t-integer]]
  :forward
  [starts-before ?x ?t]] 
 
 [[starts-before ?x*t-time ?t*t-integer]
  <temporal
  [the-time ?r*t-time ?t]
  [time-reln ?x (:d :e :f :oi :s :si) ?r]]

 [[time-reln ?x*t-time (:mi) ?r*t-time]
  <temporal
  [prove [the-time ?r ?t*t-integer]]
  :forward
  [starts-at ?x ?t]]
 
 [[starts-at ?x*t-time ?t*t-integer]
  <temporal
  [the-time ?r*t-time ?t]
  [time-reln ?x (:mi) ?r]]
 
 [[time-reln ?x*t-time (:a) ?r*t-time]
  <temporal
  [prove [the-time ?r ?t*t-integer]]
  :forward
  [starts-after ?x ?t]]
 
 [[time-reln ?x*t-time (:d :s) ?r*t-time]
  <temporal
  [prove [the-time ?r ?t*t-integer]]
  :forward
  [ends-before ?x ?t]] 
 
 [[ends-before ?x*t-time ?t*t-integer]
  <temporal
  [the-time ?r*t-time ?t]
  [time-reln ?x (:d :s) ?r]] 
 
 [[time-reln ?x*t-time (:e :f) ?r*t-time]
  <temporal
  [prove [the-time ?r ?t*t-integer]]
  :forward
  [ends-at ?x ?t]]
 
 [[ends-at ?x*t-time ?t*t-integer]
  <temporal
  [the-time ?r*t-time ?t]
  [time-reln ?x (:e :f) ?r]]
 
 [[time-reln ?x*t-time (:a :mi :oi :si) ?r*t-time]
  <temporal
  [prove [the-time ?r ?t*t-integer]]
  :forward
  [ends-after ?x ?t]]
 
 [[ends-after ?x*t-time ?t*t-integer]
  <temporal
  [the-time ?r*t-time ?t]
  [time-reln ?x (:a :mi :oi :si) ?r]]
 
 [[time-durn ?x*t-time (0 (?n*t-integer)) the-second]
  <temporal
  :forward
  [shorter-than ?x ?n]]
 
 [[shorter-than ?x*t-time ?n*t-integer]
  <temporal
  [time-durn ?x (0 (?n)) the-second]]
 
 [[time-durn ?x*t-time ?n*t-integer the-second]
  <temporal
  :forward
  [length ?x ?n]]

 [[ends-at ?x*t-time ?e*t-integer]
  <temporal
  [bound ?s*t-integer]
  [length ?x ?l*t-integer] ; prove?
  [bound ?l]
  [:= ?e (+ ?s ?l)]
  :forward
  [starts-at ?x ?s]]
 
 [[ends-at ?x*t-time ?e*t-integer]
  <temporal
  [bound ?l*t-integer]
  [starts-at ?x ?s*t-integer] ; prove?
  [bound ?s]
  [:= ?e (+ ?s ?l)]
  :forward
  [length ?x ?l]]
 
 [[starts-at ?x*t-time ?s*t-integer]
  <temporal
  [bound ?e]
  [length ?x ?l*t-integer] ; prove?
  [bound ?l*t-integer]
  [:= ?s (- ?e ?l)]
  :forward
  [ends-at ?x ?e]]
 
 [[starts-at ?x*t-time ?s*t-integer]
  <temporal
  [bound ?l*t-integer]
  [ends-at ?x ?e*t-integer] ; prove?
  [bound ?e]
  [:= ?s (- ?e ?l)]
  :forward
  [length ?x ?l]]
 
 [[length ?x*t-time ?n*t-integer]
  <temporal
  [time-durn ?x ?n the-second]]
 
 [[time-durn ?x*t-time ((?n*t-integer) :inf) the-second]
  <temporal
  :forward
  [longer-than ?x ?n]]
 
 [[longer-than ?x*t-time ?n*t-integer]
  <temporal
  [time-durn ?x ((?n) :inf) the-second]]
 
 [[length ?x*t-time ?n*t-integer]
  <temporal
  [time-reln ?x (:mi) ?b*t-time]
  [time-reln ?b (:e :si) the-second]
  [time-reln ?x (:e :f) ?e*t-time]
  [time-reln ?e (:e :si) the-second]
  [time-durn ?b ?s*t-integer the-second]
  [time-durn ?e ?f*t-integer the-second]
  [bound ?s]
  [bound ?f]
  [cut]
  [:= ?n (- ?f ?s)]
  [assert-fact [length ?x ?n]]]
 
 [[starts-at ?x*t-time ?n*t-integer]
  <temporal
  [var ?n]
  [time-reln ?x (:mi) ?b*t-time]
  [time-reln ?b (:e :si) the-second]
  [time-durn ?b ?n the-second]
  [bound ?n]
  [cut]
  [assert-fact [starts-at ?x ?n]]]

 [[starts-at ?x*t-time ?n*t-integer]
  <temporal
  [var ?n]
  [time-reln ?x (:f) ?e*t-time] ; :e fails because 0 length is illegal!
  [time-reln ?e (:e :si) the-second]
  [time-durn ?e ?f*t-integer the-second]
  [time-durn ?x ?l*t-integer the-second]
  [bound ?f]
  [bound ?l]
  [cut]
  [:= ?n (- ?f ?l)]
  [assert-fact [starts-at ?x ?n]]]
 
 [[ends-at ?x*t-time ?n*t-integer]
  <temporal
  [var ?n]
  [time-reln ?x (:e :f) ?e*t-time]
  [time-reln ?e (:e :si) the-second]
  [time-durn ?e ?n the-second]
  [bound ?n]
  [cut]
  [assert-fact [ends-at ?x ?n]]]

 [[ends-at ?x*t-time ?n*t-integer]
  <temporal
  [var ?n]
  [time-reln ?x (:mi) ?b*t-time]
  [time-reln ?b (:e :si) the-second]
  [time-durn ?b ?s*t-integer the-second]
  [time-durn ?x ?l*t-integer the-second]
  [bound ?s]
  [bound ?l]
  [cut]
  [:= ?n (+ ?s ?l)]
  [assert-fact [ends-at ?x ?n]]]

 [[overlaps ?x*t-time ?y*t-time]
  <temporal
  [time-reln ?x (:c :d :e :f :fi :o :oi :s :si) ?y]]
  )
